/*
 * Decompiled with CFR 0.152.
 */
package com.graphshift.ast;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.github.javaparser.JavaParser;
import com.github.javaparser.ParserConfiguration;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.resolution.TypeSolver;
import com.github.javaparser.resolution.UnsolvedSymbolException;
import com.github.javaparser.resolution.declarations.ResolvedConstructorDeclaration;
import com.github.javaparser.resolution.declarations.ResolvedMethodDeclaration;
import com.github.javaparser.symbolsolver.JavaSymbolSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.CombinedTypeSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.ReflectionTypeSolver;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

public class GSAnalyzer {
    private static final ObjectMapper objectMapper = new ObjectMapper().enable(SerializationFeature.INDENT_OUTPUT);
    private static Map<String, DeprecationInfo> knowledgeBase = new HashMap<String, DeprecationInfo>();

    public static void main(String[] args) {
        Options options = GSAnalyzer.createOptions();
        DefaultParser parser = new DefaultParser();
        try {
            CommandLine cmd = parser.parse(options, args);
            if (cmd.hasOption("h")) {
                GSAnalyzer.printHelp(options);
                return;
            }
            String directory = cmd.getOptionValue("d");
            String targetJdk = cmd.getOptionValue("t", "17");
            String outputPath = cmd.getOptionValue("o");
            String targetDirectory = cmd.getOptionValue("td");
            String extensionKbPath = cmd.getOptionValue("e");
            String scope = cmd.getOptionValue("s", "all-deprecations");
            if (extensionKbPath != null) {
                GSAnalyzer.loadExtensionKnowledgeBase(extensionKbPath);
            }
            if (outputPath == null) {
                outputPath = GSAnalyzer.generateDefaultOutputPath(directory);
            } else if (!outputPath.contains("/") && !outputPath.contains("\\")) {
                outputPath = "oss/outputs/" + outputPath;
            }
            Path outputDir = Paths.get(outputPath, new String[0]).getParent();
            if (outputDir != null) {
                Files.createDirectories(outputDir, new FileAttribute[0]);
            }
            System.err.println("GraphShift Analyzer - OSS Version");
            System.err.println("==================================");
            System.err.println("Directory: " + directory);
            System.err.println("Target JDK: " + targetJdk);
            System.err.println("Output: " + outputPath);
            if (targetDirectory != null) {
                System.err.println("Target Directory: " + targetDirectory);
            }
            System.err.println();
            List<Map<String, Object>> results = GSAnalyzer.analyzeDirectory(directory, targetJdk, targetDirectory, scope);
            objectMapper.writeValue(new File(outputPath), results);
            System.err.println("Analysis complete. Results written to: " + outputPath);
            System.err.println("Total findings: " + results.size());
        }
        catch (ParseException e) {
            System.err.println("Error parsing command line: " + e.getMessage());
            GSAnalyzer.printHelp(options);
            System.exit(1);
        }
        catch (Exception e) {
            System.err.println("Error during analysis: " + e.getMessage());
            e.printStackTrace();
            System.exit(1);
        }
    }

    private static Options createOptions() {
        Options options = new Options();
        options.addOption(Option.builder("d").longOpt("directory").desc("Directory to analyze").hasArg().required().build());
        options.addOption(Option.builder("t").longOpt("target-jdk").desc("Target JDK version (8, 11, 17, 21). Default: 17").hasArg().build());
        options.addOption(Option.builder("o").longOpt("output").desc("Output file path. Default: oss/outputs/{repo}_gs_insights_{timestamp}.json").hasArg().build());
        options.addOption(Option.builder("td").longOpt("target-directory").desc("Specific subdirectory within the repository to analyze").hasArg().build());
        options.addOption(Option.builder("e").longOpt("extend-kb").desc("Path to extension knowledge base JSON file").hasArg().build());
        options.addOption(Option.builder("s").longOpt("scope").desc("Analysis scope: upgrade-blockers (only breaking changes) or all-deprecations (complete audit). Default: all-deprecations").hasArg().build());
        options.addOption(Option.builder("h").longOpt("help").desc("Show help").build());
        return options;
    }

    private static void printHelp(Options options) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp("java -jar gs-analyzer-1.0.0.jar", "\nGraphShift Analyzer - OSS Version\nUniversal deprecation detection powered by comprehensive Knowledge Base\n\nExamples:\n  java -jar gs-analyzer-1.0.0.jar -d ./myproject -t 17\n  java -jar gs-analyzer-1.0.0.jar -d ./myproject -t 21 -o my-report.json\n  java -jar gs-analyzer-1.0.0.jar -d ./myproject -t 17 -td src/main/java\n\n", options, "\nFor more information, visit: https://docs.graphshift.dev");
    }

    private static String generateDefaultOutputPath(String directoryPath) {
        Path path = Paths.get(directoryPath, new String[0]).toAbsolutePath().normalize();
        String repoName = path.getFileName().toString();
        String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH-mm-ss"));
        String filename = repoName + "_gs_insights_" + timestamp + ".json";
        return "oss/outputs/" + filename;
    }

    private static void loadKnowledgeBase() {
        try {
            JsonNode thirdParty;
            InputStream kbStream = GSAnalyzer.class.getResourceAsStream("/graphshift-kb.json");
            if (kbStream == null) {
                System.err.println("Warning: Built-in knowledge base not found");
                return;
            }
            JsonNode kbJson = objectMapper.readTree(kbStream);
            JsonNode jdkCore = kbJson.get("JDK-CORE");
            if (jdkCore != null) {
                jdkCore.fields().forEachRemaining(entry -> knowledgeBase.put((String)entry.getKey(), new DeprecationInfo((JsonNode)entry.getValue())));
            }
            if ((thirdParty = kbJson.get("THIRD-PARTY")) != null) {
                thirdParty.fields().forEachRemaining(entry -> knowledgeBase.put((String)entry.getKey(), new DeprecationInfo((JsonNode)entry.getValue())));
            }
            System.err.println("Loaded " + knowledgeBase.size() + " deprecation entries from knowledge base");
        }
        catch (Exception e) {
            System.err.println("Warning: Could not load knowledge base: " + e.getMessage());
        }
    }

    private static void loadExtensionKnowledgeBase(String extensionPath) {
        try {
            JsonNode extensionJson = objectMapper.readTree(new File(extensionPath));
            extensionJson.fields().forEachRemaining(entry -> knowledgeBase.put((String)entry.getKey(), new DeprecationInfo((JsonNode)entry.getValue())));
            System.err.println("Loaded extension knowledge base from: " + extensionPath);
        }
        catch (Exception e) {
            System.err.println("Warning: Could not load extension knowledge base: " + e.getMessage());
        }
    }

    private static List<Map<String, Object>> analyzeDirectory(String directoryPath, String targetJdk, String targetDirectory, String scope) throws IOException {
        ArrayList<Map<String, Object>> allResults = new ArrayList<Map<String, Object>>();
        CombinedTypeSolver typeSolver = new CombinedTypeSolver(new TypeSolver[0]);
        typeSolver.add(new ReflectionTypeSolver());
        JavaParser parser = new JavaParser(new ParserConfiguration().setSymbolResolver(new JavaSymbolSolver(typeSolver)));
        Path basePath = Paths.get(directoryPath, new String[0]);
        Path searchPath = targetDirectory != null ? basePath.resolve(targetDirectory) : basePath;
        System.err.println("Scanning Java files in: " + searchPath);
        try (Stream<Path> paths = Files.walk(searchPath, new FileVisitOption[0]);){
            paths.filter(path -> path.toString().endsWith(".java")).forEach(javaFile -> {
                try {
                    List<Map<String, Object>> fileResults = GSAnalyzer.analyzeJavaFile(javaFile, parser, targetJdk, basePath, scope);
                    allResults.addAll(fileResults);
                }
                catch (Exception e) {
                    System.err.println("Error analyzing " + javaFile + ": " + e.getMessage());
                }
            });
        }
        return allResults;
    }

    private static boolean shouldIncludeInScope(String severity, String scope) {
        if ("upgrade-blockers".equals(scope)) {
            return "error".equals(severity);
        }
        return true;
    }

    private static List<Map<String, Object>> analyzeJavaFile(Path javaFile, JavaParser parser, String targetJdk, Path basePath, String scope) throws IOException {
        ArrayList<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        try {
            CompilationUnit cu = parser.parse(javaFile).getResult().orElse(null);
            if (cu == null) {
                return results;
            }
            String relativePath = basePath.relativize(javaFile).toString().replace('\\', '/');
            cu.findAll(MethodCallExpr.class).forEach(call -> {
                Map<String, Object> result = GSAnalyzer.analyzeMethodCall(call, relativePath, targetJdk, scope);
                if (result != null) {
                    results.add(result);
                }
            });
            cu.findAll(ObjectCreationExpr.class).forEach(creation -> {
                Map<String, Object> result = GSAnalyzer.analyzeConstructorCall(creation, relativePath, targetJdk, scope);
                if (result != null) {
                    results.add(result);
                }
            });
            return results;
        }
        catch (StackOverflowError e) {
            System.err.println("StackOverflowError analyzing " + javaFile + ": File too complex, skipping");
            return results;
        }
        catch (Exception e) {
            System.err.println("Error analyzing " + javaFile + ": " + e.getMessage());
            return results;
        }
    }

    private static Map<String, Object> analyzeMethodCall(MethodCallExpr call, String filePath, String targetJdk, String scope) {
        HashMap<String, Object> result = new HashMap<String, Object>();
        result.put("file", filePath);
        result.put("line_number", call.getBegin().map(pos -> pos.line).orElse(0));
        String methodName = call.getNameAsString();
        result.put("target_jdk", targetJdk);
        try {
            ResolvedMethodDeclaration resolved = call.resolve();
            String signature = resolved.getQualifiedSignature();
            result.put("signature", signature);
            if (signature != null && signature.contains("(")) {
                try {
                    String actualMethodName;
                    String beforeParams = signature.split("\\(")[0];
                    if (beforeParams.contains(".") && !(actualMethodName = beforeParams.substring(beforeParams.lastIndexOf(46) + 1).trim()).isEmpty() && !actualMethodName.equals(methodName)) {
                        methodName = actualMethodName;
                    }
                }
                catch (Exception beforeParams) {
                    // empty catch block
                }
            }
            result.put("method_name", methodName);
            DeprecationInfo deprecationInfo = knowledgeBase.get(signature);
            if (deprecationInfo != null) {
                result.put("reason", deprecationInfo.reason);
                result.put("deprecated_since", deprecationInfo.deprecatedSince);
                result.put("for_removal", deprecationInfo.forRemoval);
                if (deprecationInfo.isRemovedInTarget(targetJdk)) {
                    result.put("removed_in", deprecationInfo.removedIn);
                } else {
                    result.put("removed_in", "N/A");
                }
                result.put("years_deprecated", deprecationInfo.yearsDeprecated);
                String severity = deprecationInfo.getSeverity(targetJdk);
                result.put("severity", severity);
                result.put("source", deprecationInfo.source);
                if (!GSAnalyzer.shouldIncludeInScope(severity, scope)) {
                    return null;
                }
                return result;
            }
            result.put("status", "RESOLVED");
            return null;
        }
        catch (UnsolvedSymbolException e) {
            String unresolvedSignature = "unresolved:" + call.getNameAsString();
            result.put("method_name", methodName);
            for (Map.Entry<String, DeprecationInfo> entry : knowledgeBase.entrySet()) {
                if (!entry.getKey().contains(call.getNameAsString())) continue;
                DeprecationInfo deprecationInfo = entry.getValue();
                result.put("signature", entry.getKey());
                String kbSignature = entry.getKey();
                if (kbSignature != null && kbSignature.contains("(")) {
                    try {
                        String actualMethodName;
                        String beforeParams = kbSignature.split("\\(")[0];
                        if (beforeParams.contains(".") && !(actualMethodName = beforeParams.substring(beforeParams.lastIndexOf(46) + 1).trim()).isEmpty()) {
                            result.put("method_name", actualMethodName);
                        }
                    }
                    catch (Exception beforeParams) {
                        // empty catch block
                    }
                }
                result.put("reason", deprecationInfo.reason);
                result.put("deprecated_since", deprecationInfo.deprecatedSince);
                result.put("for_removal", deprecationInfo.forRemoval);
                if (deprecationInfo.isRemovedInTarget(targetJdk)) {
                    result.put("removed_in", deprecationInfo.removedIn);
                } else {
                    result.put("removed_in", "N/A");
                }
                result.put("years_deprecated", deprecationInfo.yearsDeprecated);
                String severity = deprecationInfo.getSeverity(targetJdk);
                result.put("severity", severity);
                result.put("source", deprecationInfo.source);
                if (!GSAnalyzer.shouldIncludeInScope(severity, scope)) {
                    return null;
                }
                return result;
            }
            result.put("signature", unresolvedSignature);
            result.put("status", "UNRESOLVED");
            return null;
        }
        catch (Exception e) {
            result.put("method_name", methodName);
            result.put("signature", "error:" + call.getNameAsString());
            result.put("status", "ERROR");
            result.put("error", e.getMessage());
            return null;
        }
    }

    private static Map<String, Object> analyzeConstructorCall(ObjectCreationExpr creation, String filePath, String targetJdk, String scope) {
        HashMap<String, Object> result = new HashMap<String, Object>();
        result.put("file", filePath);
        result.put("line_number", creation.getBegin().map(pos -> pos.line).orElse(0));
        result.put("constructor", creation.getTypeAsString());
        result.put("target_jdk", targetJdk);
        try {
            ResolvedConstructorDeclaration resolved = creation.resolve();
            String signature = resolved.getQualifiedSignature();
            result.put("signature", signature);
            DeprecationInfo deprecationInfo = knowledgeBase.get(signature);
            if (deprecationInfo != null) {
                result.put("reason", deprecationInfo.reason);
                result.put("deprecated_since", deprecationInfo.deprecatedSince);
                result.put("for_removal", deprecationInfo.forRemoval);
                if (deprecationInfo.isRemovedInTarget(targetJdk)) {
                    result.put("removed_in", deprecationInfo.removedIn);
                } else {
                    result.put("removed_in", "N/A");
                }
                result.put("years_deprecated", deprecationInfo.yearsDeprecated);
                String severity = deprecationInfo.getSeverity(targetJdk);
                result.put("severity", severity);
                result.put("source", deprecationInfo.source);
                if (!GSAnalyzer.shouldIncludeInScope(severity, scope)) {
                    return null;
                }
                return result;
            }
            result.put("status", "RESOLVED");
            return null;
        }
        catch (UnsolvedSymbolException e) {
            String constructorName = creation.getTypeAsString();
            for (Map.Entry<String, DeprecationInfo> entry : knowledgeBase.entrySet()) {
                if (!entry.getKey().contains(constructorName)) continue;
                DeprecationInfo deprecationInfo = entry.getValue();
                result.put("signature", entry.getKey());
                result.put("reason", deprecationInfo.reason);
                result.put("deprecated_since", deprecationInfo.deprecatedSince);
                result.put("for_removal", deprecationInfo.forRemoval);
                if (deprecationInfo.isRemovedInTarget(targetJdk)) {
                    result.put("removed_in", deprecationInfo.removedIn);
                } else {
                    result.put("removed_in", "N/A");
                }
                result.put("years_deprecated", deprecationInfo.yearsDeprecated);
                String severity = deprecationInfo.getSeverity(targetJdk);
                result.put("severity", severity);
                result.put("source", deprecationInfo.source);
                if (!GSAnalyzer.shouldIncludeInScope(severity, scope)) {
                    return null;
                }
                return result;
            }
            result.put("signature", "unresolved:" + constructorName);
            result.put("status", "UNRESOLVED");
            return null;
        }
        catch (Exception e) {
            result.put("signature", "error:" + creation.getTypeAsString());
            result.put("status", "ERROR");
            result.put("error", e.getMessage());
            return null;
        }
    }

    static {
        GSAnalyzer.loadKnowledgeBase();
    }

    private static class DeprecationInfo {
        String reason;
        String deprecatedSince;
        boolean forRemoval;
        String removalVersion;
        String removedIn;
        int yearsDeprecated;
        String source;
        Map<String, String> severityByJdk;

        DeprecationInfo(JsonNode node) {
            this.reason = node.has("reason") ? node.get("reason").asText() : "Deprecated";
            this.deprecatedSince = node.has("deprecated_since") ? node.get("deprecated_since").asText() : null;
            this.forRemoval = node.has("for_removal") && node.get("for_removal").asBoolean();
            this.removalVersion = node.has("removal_version") ? node.get("removal_version").asText() : null;
            this.removedIn = node.has("removed_in") ? node.get("removed_in").asText() : "N/A";
            this.yearsDeprecated = node.has("years_deprecated") ? node.get("years_deprecated").asInt() : 0;
            this.source = node.has("source") ? node.get("source").asText() : "manual";
            this.severityByJdk = new HashMap<String, String>();
            if (node.has("severity_by_jdk")) {
                JsonNode severityNode = node.get("severity_by_jdk");
                severityNode.fields().forEachRemaining(entry -> this.severityByJdk.put((String)entry.getKey(), ((JsonNode)entry.getValue()).asText()));
            }
        }

        String getSeverity(String targetJdk) {
            if (this.severityByJdk != null && this.severityByJdk.containsKey(targetJdk)) {
                return this.severityByJdk.get(targetJdk);
            }
            return "info";
        }

        boolean isRemovedInTarget(String targetJdk) {
            if (this.removedIn == null || "N/A".equals(this.removedIn)) {
                return false;
            }
            try {
                int targetVersion = Integer.parseInt(targetJdk);
                int removedVersion = Integer.parseInt(this.removedIn);
                return targetVersion >= removedVersion;
            }
            catch (NumberFormatException e) {
                return false;
            }
        }
    }
}

