#!/usr/bin/env python3
"""
Extract python_module fields from object_info.1 and create a simplified mapping.
Groups nodes by their source module.

Can enhance in the future to automatically get input data from:
/api/object_info
"""

import json
from collections import defaultdict
from pathlib import Path


def extract_node_modules(input_file: str, output_file: str):
    """
    Extract python_module mapping from object_info file.

    Creates two outputs:
    1. node_name -> python_module mapping
    2. python_module -> list of nodes grouping
    """
    # Read the large JSON file
    with open(input_file, 'r') as f:
        data = json.load(f)

    # Extract node -> module mapping
    node_to_module = {}
    module_to_nodes = defaultdict(list)

    for node_name, node_info in data.items():
        if isinstance(node_info, dict) and 'python_module' in node_info:
            python_module = node_info['python_module']
            node_to_module[node_name] = python_module
            module_to_nodes[python_module].append(node_name)

    # Create output structure
    output = {
        "node_to_module": node_to_module,
        "module_to_nodes": dict(module_to_nodes),
        "summary": {
            "total_nodes": len(node_to_module),
            "total_modules": len(module_to_nodes),
            "nodes_per_module": {
                module: len(nodes)
                for module, nodes in sorted(
                    module_to_nodes.items(),
                    key=lambda x: len(x[1]),
                    reverse=True
                )
            }
        }
    }

    # Write output
    with open(output_file, 'w') as f:
        json.dump(output, f, indent=2)

    # Print summary
    print(f"Extracted {len(node_to_module)} nodes from {len(module_to_nodes)} modules")
    print(f"\nTop 10 modules by node count:")
    for module, count in list(output["summary"]["nodes_per_module"].items())[:10]:
        print(f"  {module}: {count} nodes")

    print(f"\nOutput written to: {output_file}")

    # Also create a compact version with just node->module mapping
    compact_file = output_file.replace('.json', '_compact.json')
    with open(compact_file, 'w') as f:
        json.dump(node_to_module, f, indent=2)
    print(f"Compact mapping written to: {compact_file}")


if __name__ == "__main__":
    input_file = "object_info.1"
    output_file = "node_modules_mapping.json"

    extract_node_modules(input_file, output_file)
