"""
"@Author  :   Siddharth Mittal",
"@Version :   0.1.0",
"@Contact :   siddharth.mittal@meduniwien.ac.at",
"@License :   (C)Copyright 2025, Siddharth Mittal",
"@Desc    :   None",     
"""

import json
import re
import pandas as pd

import os
import joblib

# -------------------------
# Replace BIDS entities dynamically
# -------------------------
def replace_entity(filename, key, value):
    # Replace existing key-value pair, or insert if not present
    pattern = rf'{key}-[a-zA-Z0-9]+'
    if re.search(pattern, filename):
        filename = re.sub(pattern, f'{key}-{value}', filename)
    else:
        # If entity missing, add it before '_estimates.json'
        insert_pos = filename.find('_estimates.json')
        if insert_pos != -1:
            filename = filename[:insert_pos] + f'_{key}-{value}' + filename[insert_pos:]
    return filename


def _load_single_combination_prf_estimation_results(pRF_estimations_basepath, sub, ses, run, task, hemi, sample_filename):
    # Replace only relevant entities
    filename = sample_filename
    for key, value in [
        ("sub", sub),
        ("ses", ses),
        ("task", task),
        ("run", run),
        ("hemi", hemi)
    ]:
        if value is not None:
            filename = replace_entity(filename, key, value)

    # -------------------------
    # Build full filepath
    # -------------------------
    # Expected subdirectory pattern: base/sub-*/ses-*/
    subdir_parts = [f"sub-{sub}"]
    if ses:
        subdir_parts.append(f"ses-{ses}")
    subdir = "/".join(subdir_parts)
    pRF_estimations_filepath = os.path.join(pRF_estimations_basepath, subdir, filename)
    
    pRF_estimations_json_data_list = []
    if os.path.exists(pRF_estimations_filepath):
        with open(pRF_estimations_filepath, 'r') as file:
            data = json.load(file)
            for voxel_num, voxel_data in enumerate(data):
                voxel_data.update({
                    "Sub": sub,
                    "Ses": ses,
                    "Run": run,
                    "Task": task,
                    "Hemi": hemi,
                    "VoxelNum": voxel_num
                })
                pRF_estimations_json_data_list.append(voxel_data)

    return pRF_estimations_json_data_list

class PRFEstimationLoader:
    @classmethod
    def load_prf_estimations_data(cls, pRF_estimations_basepath, subs, sessions, runs, tasks, hemis, sample_filename, nJobs = -1):
        # Use joblib's Parallel to process combinations in parallel
        results = joblib.Parallel(n_jobs=nJobs)(
        joblib.delayed(_load_single_combination_prf_estimation_results)(pRF_estimations_basepath, sub, ses, run, task, hemi, sample_filename)
        for sub in subs
        for ses in sessions
        for run in runs
        for task in tasks
        for hemi in hemis
        )

        # Flatten the list of lists
        pRF_estimations_json_data_list = [item for sublist in results for item in sublist]
        
        return pd.DataFrame(pRF_estimations_json_data_list)
    

if __name__ == "__main__":
    # Example usage
    pRF_estimations_basepath = "/ceph/mri.meduniwien.ac.at/projects/physics/fmri/data/stimsim24/BIDS/derivatives/prfanalyze-gem/analysis-GEMpRF_test/"
    subs = ["001"]
    sessions = ["001nn"]
    runs = ["01", "02"]
    tasks = ["bar"]
    hemis = ["L", "R"]

    df = PRFEstimationLoader.load_prf_estimations_data(pRF_estimations_basepath, subs, sessions, runs, tasks, hemis, sample_filename="sub-001_ses-001nn_task-bar_run-01_hemi-L_estimates.json", nJobs=1)
    print(df.head())