"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImmuKVStack = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const s3 = require("aws-cdk-lib/aws-s3");
const iam = require("aws-cdk-lib/aws-iam");
/**
 * AWS CDK Stack for ImmuKV infrastructure
 *
 * Creates an S3 bucket with versioning enabled and IAM policies for
 * read/write and read-only access.
 */
class ImmuKVStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const logVersionRetentionDays = props?.logVersionRetentionDays ?? 365;
        const logVersionsToRetain = props?.logVersionsToRetain ?? 1000;
        const keyVersionRetentionDays = props?.keyVersionRetentionDays ?? 365;
        const keyVersionsToRetain = props?.keyVersionsToRetain ?? 100;
        const s3Prefix = props?.s3Prefix ?? '';
        // S3 Bucket with versioning
        this.bucket = new s3.Bucket(this, 'ImmuKVBucket', {
            bucketName: props?.bucketName,
            versioned: true,
            encryption: props?.useKmsEncryption
                ? s3.BucketEncryption.KMS_MANAGED
                : s3.BucketEncryption.S3_MANAGED,
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            lifecycleRules: [
                {
                    id: 'delete-old-log-versions',
                    enabled: true,
                    noncurrentVersionExpiration: cdk.Duration.days(logVersionRetentionDays),
                    noncurrentVersionsToRetain: logVersionsToRetain,
                    prefix: `${s3Prefix}_log.json`,
                },
                {
                    id: 'delete-old-key-versions',
                    enabled: true,
                    noncurrentVersionExpiration: cdk.Duration.days(keyVersionRetentionDays),
                    noncurrentVersionsToRetain: keyVersionsToRetain,
                    prefix: `${s3Prefix}keys/`,
                },
            ],
            removalPolicy: cdk.RemovalPolicy.RETAIN,
        });
        // IAM Policy for read/write access (Lambda, EC2, ECS, etc.)
        this.readWritePolicy = new iam.ManagedPolicy(this, 'ImmuKVReadWritePolicy', {
            managedPolicyName: `immukv-readwrite-${this.stackName}`,
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        's3:GetObject',
                        's3:GetObjectVersion',
                        's3:PutObject',
                        's3:ListBucket',
                        's3:ListBucketVersions',
                        's3:HeadObject',
                    ],
                    resources: [this.bucket.bucketArn, `${this.bucket.bucketArn}/*`],
                }),
            ],
        });
        // IAM Policy for read-only devices (sensors, IoT devices, etc.)
        this.readOnlyPolicy = new iam.ManagedPolicy(this, 'ImmuKVReadOnlyPolicy', {
            managedPolicyName: `immukv-readonly-${this.stackName}`,
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        's3:GetObject',
                        's3:GetObjectVersion',
                        's3:ListBucket',
                        's3:ListBucketVersions',
                        's3:HeadObject',
                    ],
                    resources: [this.bucket.bucketArn, `${this.bucket.bucketArn}/*`],
                }),
            ],
        });
        // Outputs
        new cdk.CfnOutput(this, 'BucketName', {
            value: this.bucket.bucketName,
            description: 'ImmuKV S3 Bucket Name',
        });
        new cdk.CfnOutput(this, 'BucketArn', {
            value: this.bucket.bucketArn,
            description: 'ImmuKV S3 Bucket ARN',
        });
        new cdk.CfnOutput(this, 'ReadWritePolicyArn', {
            value: this.readWritePolicy.managedPolicyArn,
            description: 'IAM Policy ARN for read/write access',
        });
        new cdk.CfnOutput(this, 'ReadOnlyPolicyArn', {
            value: this.readOnlyPolicy.managedPolicyArn,
            description: 'IAM Policy ARN for read-only access',
        });
    }
}
exports.ImmuKVStack = ImmuKVStack;
_a = JSII_RTTI_SYMBOL_1;
ImmuKVStack[_a] = { fqn: "cdk-immukv.ImmuKVStack", version: "0.1.11" };
//# sourceMappingURL=data:application/json;base64,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