"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenIdConnectProvider = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const RESOURCE_TYPE = 'Custom::AWSCDKOpenIdConnectProvider';
/**
 * IAM OIDC identity providers are entities in IAM that describe an external
 * identity provider (IdP) service that supports the OpenID Connect (OIDC)
 * standard, such as Google or Salesforce. You use an IAM OIDC identity provider
 * when you want to establish trust between an OIDC-compatible IdP and your AWS
 * account. This is useful when creating a mobile app or web application that
 * requires access to AWS resources, but you don't want to create custom sign-in
 * code or manage your own user identities.
 *
 * @see http://openid.net/connect
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc.html
 *
 * @experimental
 * @resource AWS::CloudFormation::CustomResource
 */
class OpenIdConnectProvider extends core_1.Resource {
    /**
     * Defines an OpenID Connect provider.
     * @param scope The definition scope
     * @param id Construct ID
     * @param props Initialization properties
     */
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: RESOURCE_TYPE,
            serviceToken: this.getOrCreateProvider(),
            properties: {
                ClientIDList: props.clientIds,
                ThumbprintList: props.thumbprints,
                Url: props.url,
            },
        });
        this.openIdConnectProviderArn = core_1.Token.asString(resource.ref);
    }
    /**
     * Imports an Open ID connect provider from an ARN.
     * @param scope The definition scope
     * @param id ID of the construct
     * @param openIdConnectProviderArn the ARN to import
     */
    static fromOpenIdConnectProviderArn(scope, id, openIdConnectProviderArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.openIdConnectProviderArn = openIdConnectProviderArn;
            }
        }
        return new Import(scope, id);
    }
    getOrCreateProvider() {
        return core_1.CustomResourceProvider.getOrCreate(this, RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'oidc-provider'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                {
                    Effect: 'Allow',
                    Resource: '*',
                    Action: [
                        'iam:CreateOpenIDConnectProvider',
                        'iam:DeleteOpenIDConnectProvider',
                        'iam:UpdateOpenIDConnectProviderThumbprint',
                        'iam:AddClientIDToOpenIDConnectProvider',
                        'iam:RemoveClientIDFromOpenIDConnectProvider',
                    ],
                },
            ],
        });
    }
}
exports.OpenIdConnectProvider = OpenIdConnectProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib2lkYy1wcm92aWRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm9pZGMtcHJvdmlkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBQzdCLHdDQUE2STtBQUU3SSxNQUFNLGFBQWEsR0FBRyxxQ0FBcUMsQ0FBQztBQTBFNUQ7Ozs7Ozs7Ozs7Ozs7O0dBY0c7QUFDSCxNQUFhLHFCQUFzQixTQUFRLGVBQVE7SUFtQmpEOzs7OztPQUtHO0lBQ0gsWUFBbUIsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBaUM7UUFDaEYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLFFBQVEsR0FBRyxJQUFJLHFCQUFjLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNwRCxZQUFZLEVBQUUsYUFBYTtZQUMzQixZQUFZLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixFQUFFO1lBQ3hDLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUUsS0FBSyxDQUFDLFNBQVM7Z0JBQzdCLGNBQWMsRUFBRSxLQUFLLENBQUMsV0FBVztnQkFDakMsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO2FBQ2Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsd0JBQXdCLEdBQUcsWUFBSyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDL0QsQ0FBQztJQXRDRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyw0QkFBNEIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSx3QkFBZ0M7UUFDdkcsTUFBTSxNQUFPLFNBQVEsZUFBUTtZQUE3Qjs7Z0JBQ2tCLDZCQUF3QixHQUFHLHdCQUF3QixDQUFDO1lBQ3RFLENBQUM7U0FBQTtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUE2Qk8sbUJBQW1CO1FBQ3pCLE9BQU8sNkJBQXNCLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDN0QsYUFBYSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGVBQWUsQ0FBQztZQUNwRCxPQUFPLEVBQUUsb0NBQTZCLENBQUMsU0FBUztZQUNoRCxnQkFBZ0IsRUFBRTtnQkFDaEI7b0JBQ0UsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFLEdBQUc7b0JBQ2IsTUFBTSxFQUFFO3dCQUNOLGlDQUFpQzt3QkFDakMsaUNBQWlDO3dCQUNqQywyQ0FBMkM7d0JBQzNDLHdDQUF3Qzt3QkFDeEMsNkNBQTZDO3FCQUM5QztpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBNURELHNEQTREQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIEN1c3RvbVJlc291cmNlLCBDdXN0b21SZXNvdXJjZVByb3ZpZGVyLCBDdXN0b21SZXNvdXJjZVByb3ZpZGVyUnVudGltZSwgSVJlc291cmNlLCBSZXNvdXJjZSwgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuY29uc3QgUkVTT1VSQ0VfVFlQRSA9ICdDdXN0b206OkFXU0NES09wZW5JZENvbm5lY3RQcm92aWRlcic7XG5cbi8qKlxuICogUmVwcmVzZW50cyBhbiBJQU0gT3BlbklEIENvbm5lY3QgcHJvdmlkZXIuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIElPcGVuSWRDb25uZWN0UHJvdmlkZXIgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAvKipcbiAgICogVGhlIEFtYXpvbiBSZXNvdXJjZSBOYW1lIChBUk4pIG9mIHRoZSBJQU0gT3BlbklEIENvbm5lY3QgcHJvdmlkZXIuXG4gICAqL1xuICByZWFkb25seSBvcGVuSWRDb25uZWN0UHJvdmlkZXJBcm46IHN0cmluZztcbn1cblxuLyoqXG4gKiBJbml0aWFsaXphdGlvbiBwcm9wZXJ0aWVzIGZvciBgT3BlbklkQ29ubmVjdFByb3ZpZGVyYC5cbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBPcGVuSWRDb25uZWN0UHJvdmlkZXJQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgVVJMIG9mIHRoZSBpZGVudGl0eSBwcm92aWRlci4gVGhlIFVSTCBtdXN0IGJlZ2luIHdpdGggaHR0cHM6Ly8gYW5kXG4gICAqIHNob3VsZCBjb3JyZXNwb25kIHRvIHRoZSBpc3MgY2xhaW0gaW4gdGhlIHByb3ZpZGVyJ3MgT3BlbklEIENvbm5lY3QgSURcbiAgICogdG9rZW5zLiBQZXIgdGhlIE9JREMgc3RhbmRhcmQsIHBhdGggY29tcG9uZW50cyBhcmUgYWxsb3dlZCBidXQgcXVlcnlcbiAgICogcGFyYW1ldGVycyBhcmUgbm90LiBUeXBpY2FsbHkgdGhlIFVSTCBjb25zaXN0cyBvZiBvbmx5IGEgaG9zdG5hbWUsIGxpa2VcbiAgICogaHR0cHM6Ly9zZXJ2ZXIuZXhhbXBsZS5vcmcgb3IgaHR0cHM6Ly9leGFtcGxlLmNvbS5cbiAgICpcbiAgICogWW91IGNhbm5vdCByZWdpc3RlciB0aGUgc2FtZSBwcm92aWRlciBtdWx0aXBsZSB0aW1lcyBpbiBhIHNpbmdsZSBBV1NcbiAgICogYWNjb3VudC4gSWYgeW91IHRyeSB0byBzdWJtaXQgYSBVUkwgdGhhdCBoYXMgYWxyZWFkeSBiZWVuIHVzZWQgZm9yIGFuXG4gICAqIE9wZW5JRCBDb25uZWN0IHByb3ZpZGVyIGluIHRoZSBBV1MgYWNjb3VudCwgeW91IHdpbGwgZ2V0IGFuIGVycm9yLlxuICAgKi9cbiAgcmVhZG9ubHkgdXJsOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBjbGllbnQgSURzIChhbHNvIGtub3duIGFzIGF1ZGllbmNlcykuIFdoZW4gYSBtb2JpbGUgb3Igd2ViIGFwcFxuICAgKiByZWdpc3RlcnMgd2l0aCBhbiBPcGVuSUQgQ29ubmVjdCBwcm92aWRlciwgdGhleSBlc3RhYmxpc2ggYSB2YWx1ZSB0aGF0XG4gICAqIGlkZW50aWZpZXMgdGhlIGFwcGxpY2F0aW9uLiAoVGhpcyBpcyB0aGUgdmFsdWUgdGhhdCdzIHNlbnQgYXMgdGhlIGNsaWVudF9pZFxuICAgKiBwYXJhbWV0ZXIgb24gT0F1dGggcmVxdWVzdHMuKVxuICAgKlxuICAgKiBZb3UgY2FuIHJlZ2lzdGVyIG11bHRpcGxlIGNsaWVudCBJRHMgd2l0aCB0aGUgc2FtZSBwcm92aWRlci4gRm9yIGV4YW1wbGUsXG4gICAqIHlvdSBtaWdodCBoYXZlIG11bHRpcGxlIGFwcGxpY2F0aW9ucyB0aGF0IHVzZSB0aGUgc2FtZSBPSURDIHByb3ZpZGVyLiBZb3VcbiAgICogY2Fubm90IHJlZ2lzdGVyIG1vcmUgdGhhbiAxMDAgY2xpZW50IElEcyB3aXRoIGEgc2luZ2xlIElBTSBPSURDIHByb3ZpZGVyLlxuICAgKlxuICAgKiBDbGllbnQgSURzIGFyZSB1cCB0byAyNTUgY2hhcmFjdGVycyBsb25nLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGNsaWVudHMgYXJlIGFsbG93ZWRcbiAgICovXG4gIHJlYWRvbmx5IGNsaWVudElkcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2Ygc2VydmVyIGNlcnRpZmljYXRlIHRodW1icHJpbnRzIGZvciB0aGUgT3BlbklEIENvbm5lY3QgKE9JREMpXG4gICAqIGlkZW50aXR5IHByb3ZpZGVyJ3Mgc2VydmVyIGNlcnRpZmljYXRlcy5cbiAgICpcbiAgICogVHlwaWNhbGx5IHRoaXMgbGlzdCBpbmNsdWRlcyBvbmx5IG9uZSBlbnRyeS4gSG93ZXZlciwgSUFNIGxldHMgeW91IGhhdmUgdXBcbiAgICogdG8gZml2ZSB0aHVtYnByaW50cyBmb3IgYW4gT0lEQyBwcm92aWRlci4gVGhpcyBsZXRzIHlvdSBtYWludGFpbiBtdWx0aXBsZVxuICAgKiB0aHVtYnByaW50cyBpZiB0aGUgaWRlbnRpdHkgcHJvdmlkZXIgaXMgcm90YXRpbmcgY2VydGlmaWNhdGVzLlxuICAgKlxuICAgKiBUaGUgc2VydmVyIGNlcnRpZmljYXRlIHRodW1icHJpbnQgaXMgdGhlIGhleC1lbmNvZGVkIFNIQS0xIGhhc2ggdmFsdWUgb2ZcbiAgICogdGhlIFguNTA5IGNlcnRpZmljYXRlIHVzZWQgYnkgdGhlIGRvbWFpbiB3aGVyZSB0aGUgT3BlbklEIENvbm5lY3QgcHJvdmlkZXJcbiAgICogbWFrZXMgaXRzIGtleXMgYXZhaWxhYmxlLiBJdCBpcyBhbHdheXMgYSA0MC1jaGFyYWN0ZXIgc3RyaW5nLlxuICAgKlxuICAgKiBZb3UgbXVzdCBwcm92aWRlIGF0IGxlYXN0IG9uZSB0aHVtYnByaW50IHdoZW4gY3JlYXRpbmcgYW4gSUFNIE9JRENcbiAgICogcHJvdmlkZXIuIEZvciBleGFtcGxlLCBhc3N1bWUgdGhhdCB0aGUgT0lEQyBwcm92aWRlciBpcyBzZXJ2ZXIuZXhhbXBsZS5jb21cbiAgICogYW5kIHRoZSBwcm92aWRlciBzdG9yZXMgaXRzIGtleXMgYXRcbiAgICogaHR0cHM6Ly9rZXlzLnNlcnZlci5leGFtcGxlLmNvbS9vcGVuaWQtY29ubmVjdC4gSW4gdGhhdCBjYXNlLCB0aGVcbiAgICogdGh1bWJwcmludCBzdHJpbmcgd291bGQgYmUgdGhlIGhleC1lbmNvZGVkIFNIQS0xIGhhc2ggdmFsdWUgb2YgdGhlXG4gICAqIGNlcnRpZmljYXRlIHVzZWQgYnkgaHR0cHM6Ly9rZXlzLnNlcnZlci5leGFtcGxlLmNvbS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBJZiBubyB0aHVtYnByaW50cyBhcmUgc3BlY2lmaWVkIChhbiBlbXB0eSBhcnJheSBvciBgdW5kZWZpbmVkYCksXG4gICAqIHRoZSB0aHVtYnByaW50IG9mIHRoZSByb290IGNlcnRpZmljYXRlIGF1dGhvcml0eSB3aWxsIGJlIG9idGFpbmVkIGZyb20gdGhlXG4gICAqIHByb3ZpZGVyJ3Mgc2VydmVyIGFzIGRlc2NyaWJlZCBpbiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvaWRfcm9sZXNfcHJvdmlkZXJzX2NyZWF0ZV9vaWRjX3ZlcmlmeS10aHVtYnByaW50Lmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHRodW1icHJpbnRzPzogc3RyaW5nW107XG59XG5cbi8qKlxuICogSUFNIE9JREMgaWRlbnRpdHkgcHJvdmlkZXJzIGFyZSBlbnRpdGllcyBpbiBJQU0gdGhhdCBkZXNjcmliZSBhbiBleHRlcm5hbFxuICogaWRlbnRpdHkgcHJvdmlkZXIgKElkUCkgc2VydmljZSB0aGF0IHN1cHBvcnRzIHRoZSBPcGVuSUQgQ29ubmVjdCAoT0lEQylcbiAqIHN0YW5kYXJkLCBzdWNoIGFzIEdvb2dsZSBvciBTYWxlc2ZvcmNlLiBZb3UgdXNlIGFuIElBTSBPSURDIGlkZW50aXR5IHByb3ZpZGVyXG4gKiB3aGVuIHlvdSB3YW50IHRvIGVzdGFibGlzaCB0cnVzdCBiZXR3ZWVuIGFuIE9JREMtY29tcGF0aWJsZSBJZFAgYW5kIHlvdXIgQVdTXG4gKiBhY2NvdW50LiBUaGlzIGlzIHVzZWZ1bCB3aGVuIGNyZWF0aW5nIGEgbW9iaWxlIGFwcCBvciB3ZWIgYXBwbGljYXRpb24gdGhhdFxuICogcmVxdWlyZXMgYWNjZXNzIHRvIEFXUyByZXNvdXJjZXMsIGJ1dCB5b3UgZG9uJ3Qgd2FudCB0byBjcmVhdGUgY3VzdG9tIHNpZ24taW5cbiAqIGNvZGUgb3IgbWFuYWdlIHlvdXIgb3duIHVzZXIgaWRlbnRpdGllcy5cbiAqXG4gKiBAc2VlIGh0dHA6Ly9vcGVuaWQubmV0L2Nvbm5lY3RcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2lkX3JvbGVzX3Byb3ZpZGVyc19vaWRjLmh0bWxcbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKiBAcmVzb3VyY2UgQVdTOjpDbG91ZEZvcm1hdGlvbjo6Q3VzdG9tUmVzb3VyY2VcbiAqL1xuZXhwb3J0IGNsYXNzIE9wZW5JZENvbm5lY3RQcm92aWRlciBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSU9wZW5JZENvbm5lY3RQcm92aWRlciB7XG4gIC8qKlxuICAgKiBJbXBvcnRzIGFuIE9wZW4gSUQgY29ubmVjdCBwcm92aWRlciBmcm9tIGFuIEFSTi5cbiAgICogQHBhcmFtIHNjb3BlIFRoZSBkZWZpbml0aW9uIHNjb3BlXG4gICAqIEBwYXJhbSBpZCBJRCBvZiB0aGUgY29uc3RydWN0XG4gICAqIEBwYXJhbSBvcGVuSWRDb25uZWN0UHJvdmlkZXJBcm4gdGhlIEFSTiB0byBpbXBvcnRcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbU9wZW5JZENvbm5lY3RQcm92aWRlckFybihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBvcGVuSWRDb25uZWN0UHJvdmlkZXJBcm46IHN0cmluZyk6IElPcGVuSWRDb25uZWN0UHJvdmlkZXIge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSU9wZW5JZENvbm5lY3RQcm92aWRlciB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgb3BlbklkQ29ubmVjdFByb3ZpZGVyQXJuID0gb3BlbklkQ29ubmVjdFByb3ZpZGVyQXJuO1xuICAgIH1cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBvZiB0aGUgSUFNIE9wZW5JRCBDb25uZWN0IHByb3ZpZGVyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG9wZW5JZENvbm5lY3RQcm92aWRlckFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGFuIE9wZW5JRCBDb25uZWN0IHByb3ZpZGVyLlxuICAgKiBAcGFyYW0gc2NvcGUgVGhlIGRlZmluaXRpb24gc2NvcGVcbiAgICogQHBhcmFtIGlkIENvbnN0cnVjdCBJRFxuICAgKiBAcGFyYW0gcHJvcHMgSW5pdGlhbGl6YXRpb24gcHJvcGVydGllc1xuICAgKi9cbiAgcHVibGljIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBPcGVuSWRDb25uZWN0UHJvdmlkZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDdXN0b21SZXNvdXJjZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICByZXNvdXJjZVR5cGU6IFJFU09VUkNFX1RZUEUsXG4gICAgICBzZXJ2aWNlVG9rZW46IHRoaXMuZ2V0T3JDcmVhdGVQcm92aWRlcigpLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBDbGllbnRJRExpc3Q6IHByb3BzLmNsaWVudElkcyxcbiAgICAgICAgVGh1bWJwcmludExpc3Q6IHByb3BzLnRodW1icHJpbnRzLFxuICAgICAgICBVcmw6IHByb3BzLnVybCxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0aGlzLm9wZW5JZENvbm5lY3RQcm92aWRlckFybiA9IFRva2VuLmFzU3RyaW5nKHJlc291cmNlLnJlZik7XG4gIH1cblxuICBwcml2YXRlIGdldE9yQ3JlYXRlUHJvdmlkZXIoKSB7XG4gICAgcmV0dXJuIEN1c3RvbVJlc291cmNlUHJvdmlkZXIuZ2V0T3JDcmVhdGUodGhpcywgUkVTT1VSQ0VfVFlQRSwge1xuICAgICAgY29kZURpcmVjdG9yeTogcGF0aC5qb2luKF9fZGlybmFtZSwgJ29pZGMtcHJvdmlkZXInKSxcbiAgICAgIHJ1bnRpbWU6IEN1c3RvbVJlc291cmNlUHJvdmlkZXJSdW50aW1lLk5PREVKU18xMixcbiAgICAgIHBvbGljeVN0YXRlbWVudHM6IFtcbiAgICAgICAge1xuICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICBSZXNvdXJjZTogJyonLFxuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgJ2lhbTpDcmVhdGVPcGVuSURDb25uZWN0UHJvdmlkZXInLFxuICAgICAgICAgICAgJ2lhbTpEZWxldGVPcGVuSURDb25uZWN0UHJvdmlkZXInLFxuICAgICAgICAgICAgJ2lhbTpVcGRhdGVPcGVuSURDb25uZWN0UHJvdmlkZXJUaHVtYnByaW50JyxcbiAgICAgICAgICAgICdpYW06QWRkQ2xpZW50SURUb09wZW5JRENvbm5lY3RQcm92aWRlcicsXG4gICAgICAgICAgICAnaWFtOlJlbW92ZUNsaWVudElERnJvbU9wZW5JRENvbm5lY3RQcm92aWRlcicsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG4gIH1cbn1cbiJdfQ==