"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const iam = require("../lib");
let stack;
let resource;
beforeEach(() => {
    stack = new core_1.Stack();
    resource = new core_1.CfnResource(stack, 'SomeResource', {
        type: 'CDK::Test::SomeResource',
    });
});
describe('IAM grant', () => {
    test('Grant.drop() returns a no-op grant', () => {
        const user = new iam.User(stack, 'poo');
        const grant = iam.Grant.drop(user, 'dropping me');
        expect(grant.success).toBeFalsy();
        expect(grant.principalStatement).toBeUndefined();
        expect(grant.resourceStatement).toBeUndefined();
    });
});
describe('Grant dependencies', () => {
    test('can depend on grant added to role', () => {
        // GIVEN
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('bla.amazonaws.com'),
        });
        // WHEN
        applyGrantWithDependencyTo(role);
        // THEN
        expectDependencyOn('RoleDefaultPolicy5FFB7DAB');
    });
    test('can depend on grant added to role wrapped with conditions', () => {
        // GIVEN
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('bla.amazonaws.com'),
        });
        // WHEN
        applyGrantWithDependencyTo(new iam.PrincipalWithConditions(role, {
            StringEquals: { 'aws:something': 'happy' },
        }));
        // THEN
        expectDependencyOn('RoleDefaultPolicy5FFB7DAB');
    });
    test('can depend on grant added to lazy role', () => {
        // GIVEN
        const role = new iam.LazyRole(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('bla.amazonaws.com'),
        });
        // WHEN
        applyGrantWithDependencyTo(role);
        Array.isArray(role.roleArn); // Force instantiation
        // THEN
        expectDependencyOn('RoleDefaultPolicy5FFB7DAB');
    });
    test('can depend on grant added to resource', () => {
        // WHEN
        iam.Grant.addToPrincipalOrResource({
            actions: ['service:DoAThing'],
            grantee: new iam.ServicePrincipal('bla.amazonaws.com'),
            resourceArns: ['*'],
            resource: new FakeResourceWithPolicy(stack, 'Buckaroo'),
        }).applyBefore(resource);
        // THEN
        expectDependencyOn('BuckarooPolicy74174DA4');
    });
    test('can depend on grant added to principal AND resource', () => {
        // GIVEN
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('bla.amazonaws.com'),
        });
        // WHEN
        iam.Grant.addToPrincipalAndResource({
            actions: ['service:DoAThing'],
            grantee: role,
            resourceArns: ['*'],
            resource: new FakeResourceWithPolicy(stack, 'Buckaroo'),
        }).applyBefore(resource);
        // THEN
        expectDependencyOn('RoleDefaultPolicy5FFB7DAB');
        expectDependencyOn('BuckarooPolicy74174DA4');
    });
});
function applyGrantWithDependencyTo(principal) {
    iam.Grant.addToPrincipal({
        actions: ['service:DoAThing'],
        grantee: principal,
        resourceArns: ['*'],
    }).applyBefore(resource);
}
function expectDependencyOn(id) {
    expect(stack).toHaveResource('CDK::Test::SomeResource', (props) => {
        var _a;
        return ((_a = props === null || props === void 0 ? void 0 : props.DependsOn) !== null && _a !== void 0 ? _a : []).includes(id);
    }, assert_1.ResourcePart.CompleteDefinition);
}
class FakeResourceWithPolicy extends core_1.Resource {
    constructor(scope, id) {
        super(scope, id);
        this.policy = new core_1.CfnResource(this, 'Policy', {
            type: 'CDK::Test::BuckarooPolicy',
        });
    }
    addToResourcePolicy(_statement) {
        return { statementAdded: true, policyDependable: this.policy };
    }
}
//# sourceMappingURL=data:application/json;base64,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