"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const iam = require("../lib");
/* eslint-disable quote-props */
describe('ImmutableRole', () => {
    let stack;
    let mutableRole;
    let immutableRole;
    beforeEach(() => {
        stack = new core_1.Stack();
        mutableRole = new iam.Role(stack, 'MutableRole', {
            assumedBy: new iam.AnyPrincipal(),
        });
        immutableRole = mutableRole.withoutPolicyUpdates();
    });
    test('ignores calls to attachInlinePolicy', () => {
        const user = new iam.User(stack, 'User');
        const policy = new iam.Policy(stack, 'Policy', {
            statements: [new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:*'],
                })],
            users: [user],
        });
        immutableRole.attachInlinePolicy(policy);
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 's3:*',
                        'Resource': '*',
                        'Effect': 'Allow',
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'Policy23B91518',
            'Users': [
                {
                    'Ref': 'User00B015A1',
                },
            ],
        });
    });
    test('ignores calls to addManagedPolicy', () => {
        mutableRole.addManagedPolicy({ managedPolicyArn: 'Arn1' });
        immutableRole.addManagedPolicy({ managedPolicyArn: 'Arn2' });
        expect(stack).toHaveResourceLike('AWS::IAM::Role', {
            'ManagedPolicyArns': [
                'Arn1',
            ],
        });
    });
    test('ignores calls to addToPolicy', () => {
        immutableRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['iam:*'],
        }));
        mutableRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['s3:*'],
        }));
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Version': '2012-10-17',
                'Statement': [
                    {
                        'Resource': '*',
                        'Action': 's3:*',
                        'Effect': 'Allow',
                    },
                ],
            },
        });
    });
    test('ignores grants', () => {
        iam.Grant.addToPrincipal({
            grantee: immutableRole,
            actions: ['s3:*'],
            resourceArns: ['*'],
        });
        expect(stack).not.toHaveResourceLike('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Resource': '*',
                        'Action': 's3:*',
                        'Effect': 'Allow',
                    },
                ],
            },
        });
    });
    // this pattern is used here:
    // aws-codepipeline-actions/lib/cloudformation/pipeline-actions.ts#L517
    test('immutable role is a construct', () => {
        new core_1.Construct(immutableRole, 'Child');
        new core_1.Construct(mutableRole.withoutPolicyUpdates(), 'Child2');
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW1tdXRhYmxlLXJvbGUudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImltbXV0YWJsZS1yb2xlLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxnQ0FBOEI7QUFDOUIsd0NBQWlEO0FBQ2pELDhCQUE4QjtBQUU5QixnQ0FBZ0M7QUFFaEMsUUFBUSxDQUFDLGVBQWUsRUFBRSxHQUFHLEVBQUU7SUFDN0IsSUFBSSxLQUFZLENBQUM7SUFDakIsSUFBSSxXQUFxQixDQUFDO0lBQzFCLElBQUksYUFBd0IsQ0FBQztJQUU3QixVQUFVLENBQUMsR0FBRyxFQUFFO1FBQ2QsS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDcEIsV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO1lBQy9DLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxZQUFZLEVBQUU7U0FDbEMsQ0FBQyxDQUFDO1FBQ0gsYUFBYSxHQUFHLFdBQVcsQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO0lBQ3JELENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHFDQUFxQyxFQUFFLEdBQUcsRUFBRTtRQUMvQyxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3pDLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQzdDLFVBQVUsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDbkMsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO29CQUNoQixPQUFPLEVBQUUsQ0FBQyxNQUFNLENBQUM7aUJBQ2xCLENBQUMsQ0FBQztZQUNILEtBQUssRUFBRSxDQUFDLElBQUksQ0FBQztTQUNkLENBQUMsQ0FBQztRQUVILGFBQWEsQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUV6QyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtCQUFrQixFQUFFO1lBQy9DLGdCQUFnQixFQUFFO2dCQUNoQixXQUFXLEVBQUU7b0JBQ1g7d0JBQ0UsUUFBUSxFQUFFLE1BQU07d0JBQ2hCLFVBQVUsRUFBRSxHQUFHO3dCQUNmLFFBQVEsRUFBRSxPQUFPO3FCQUNsQjtpQkFDRjtnQkFDRCxTQUFTLEVBQUUsWUFBWTthQUN4QjtZQUNELFlBQVksRUFBRSxnQkFBZ0I7WUFDOUIsT0FBTyxFQUFFO2dCQUNQO29CQUNFLEtBQUssRUFBRSxjQUFjO2lCQUN0QjthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsbUNBQW1DLEVBQUUsR0FBRyxFQUFFO1FBQzdDLFdBQVcsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLGdCQUFnQixFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUM7UUFFM0QsYUFBYSxDQUFDLGdCQUFnQixDQUFDLEVBQUUsZ0JBQWdCLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQztRQUU3RCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCLEVBQUU7WUFDakQsbUJBQW1CLEVBQUU7Z0JBQ25CLE1BQU07YUFDUDtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLDhCQUE4QixFQUFFLEdBQUcsRUFBRTtRQUN4QyxhQUFhLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNoRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7WUFDaEIsT0FBTyxFQUFFLENBQUMsT0FBTyxDQUFDO1NBQ25CLENBQUMsQ0FBQyxDQUFDO1FBRUosV0FBVyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDOUMsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1lBQ2hCLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQztTQUNsQixDQUFDLENBQUMsQ0FBQztRQUVKLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsa0JBQWtCLEVBQUU7WUFDL0MsZ0JBQWdCLEVBQUU7Z0JBQ2hCLFNBQVMsRUFBRSxZQUFZO2dCQUN2QixXQUFXLEVBQUU7b0JBQ1g7d0JBQ0UsVUFBVSxFQUFFLEdBQUc7d0JBQ2YsUUFBUSxFQUFFLE1BQU07d0JBQ2hCLFFBQVEsRUFBRSxPQUFPO3FCQUNsQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxFQUFFO1FBRTFCLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQ3ZCLE9BQU8sRUFBRSxhQUFhO1lBQ3RCLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQztZQUNqQixZQUFZLEVBQUUsQ0FBQyxHQUFHLENBQUM7U0FDcEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRTtZQUN2RCxnQkFBZ0IsRUFBRTtnQkFDaEIsV0FBVyxFQUFFO29CQUNYO3dCQUNFLFVBQVUsRUFBRSxHQUFHO3dCQUNmLFFBQVEsRUFBRSxNQUFNO3dCQUNoQixRQUFRLEVBQUUsT0FBTztxQkFDbEI7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsNkJBQTZCO0lBQzdCLHVFQUF1RTtJQUN2RSxJQUFJLENBQUMsK0JBQStCLEVBQUUsR0FBRyxFQUFFO1FBQ3pDLElBQUksZ0JBQVMsQ0FBQyxhQUFxQyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQzlELElBQUksZ0JBQVMsQ0FBQyxXQUFXLENBQUMsb0JBQW9CLEVBQTBCLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDdEYsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uL2xpYic7XG5cbi8qIGVzbGludC1kaXNhYmxlIHF1b3RlLXByb3BzICovXG5cbmRlc2NyaWJlKCdJbW11dGFibGVSb2xlJywgKCkgPT4ge1xuICBsZXQgc3RhY2s6IFN0YWNrO1xuICBsZXQgbXV0YWJsZVJvbGU6IGlhbS5Sb2xlO1xuICBsZXQgaW1tdXRhYmxlUm9sZTogaWFtLklSb2xlO1xuXG4gIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgIHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgbXV0YWJsZVJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdNdXRhYmxlUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5BbnlQcmluY2lwYWwoKSxcbiAgICB9KTtcbiAgICBpbW11dGFibGVSb2xlID0gbXV0YWJsZVJvbGUud2l0aG91dFBvbGljeVVwZGF0ZXMoKTtcbiAgfSk7XG5cbiAgdGVzdCgnaWdub3JlcyBjYWxscyB0byBhdHRhY2hJbmxpbmVQb2xpY3knLCAoKSA9PiB7XG4gICAgY29uc3QgdXNlciA9IG5ldyBpYW0uVXNlcihzdGFjaywgJ1VzZXInKTtcbiAgICBjb25zdCBwb2xpY3kgPSBuZXcgaWFtLlBvbGljeShzdGFjaywgJ1BvbGljeScsIHtcbiAgICAgIHN0YXRlbWVudHM6IFtuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgIGFjdGlvbnM6IFsnczM6KiddLFxuICAgICAgfSldLFxuICAgICAgdXNlcnM6IFt1c2VyXSxcbiAgICB9KTtcblxuICAgIGltbXV0YWJsZVJvbGUuYXR0YWNoSW5saW5lUG9saWN5KHBvbGljeSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgJ1BvbGljeURvY3VtZW50Jzoge1xuICAgICAgICAnU3RhdGVtZW50JzogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgICdBY3Rpb24nOiAnczM6KicsXG4gICAgICAgICAgICAnUmVzb3VyY2UnOiAnKicsXG4gICAgICAgICAgICAnRWZmZWN0JzogJ0FsbG93JyxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICAnVmVyc2lvbic6ICcyMDEyLTEwLTE3JyxcbiAgICAgIH0sXG4gICAgICAnUG9saWN5TmFtZSc6ICdQb2xpY3kyM0I5MTUxOCcsXG4gICAgICAnVXNlcnMnOiBbXG4gICAgICAgIHtcbiAgICAgICAgICAnUmVmJzogJ1VzZXIwMEIwMTVBMScsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdpZ25vcmVzIGNhbGxzIHRvIGFkZE1hbmFnZWRQb2xpY3knLCAoKSA9PiB7XG4gICAgbXV0YWJsZVJvbGUuYWRkTWFuYWdlZFBvbGljeSh7IG1hbmFnZWRQb2xpY3lBcm46ICdBcm4xJyB9KTtcblxuICAgIGltbXV0YWJsZVJvbGUuYWRkTWFuYWdlZFBvbGljeSh7IG1hbmFnZWRQb2xpY3lBcm46ICdBcm4yJyB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6Um9sZScsIHtcbiAgICAgICdNYW5hZ2VkUG9saWN5QXJucyc6IFtcbiAgICAgICAgJ0FybjEnLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnaWdub3JlcyBjYWxscyB0byBhZGRUb1BvbGljeScsICgpID0+IHtcbiAgICBpbW11dGFibGVSb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICBhY3Rpb25zOiBbJ2lhbToqJ10sXG4gICAgfSkpO1xuXG4gICAgbXV0YWJsZVJvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgIGFjdGlvbnM6IFsnczM6KiddLFxuICAgIH0pKTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICAnUG9saWN5RG9jdW1lbnQnOiB7XG4gICAgICAgICdWZXJzaW9uJzogJzIwMTItMTAtMTcnLFxuICAgICAgICAnU3RhdGVtZW50JzogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgICdSZXNvdXJjZSc6ICcqJyxcbiAgICAgICAgICAgICdBY3Rpb24nOiAnczM6KicsXG4gICAgICAgICAgICAnRWZmZWN0JzogJ0FsbG93JyxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnaWdub3JlcyBncmFudHMnLCAoKSA9PiB7XG5cbiAgICBpYW0uR3JhbnQuYWRkVG9QcmluY2lwYWwoe1xuICAgICAgZ3JhbnRlZTogaW1tdXRhYmxlUm9sZSxcbiAgICAgIGFjdGlvbnM6IFsnczM6KiddLFxuICAgICAgcmVzb3VyY2VBcm5zOiBbJyonXSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykubm90LnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICdQb2xpY3lEb2N1bWVudCc6IHtcbiAgICAgICAgJ1N0YXRlbWVudCc6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnUmVzb3VyY2UnOiAnKicsXG4gICAgICAgICAgICAnQWN0aW9uJzogJ3MzOionLFxuICAgICAgICAgICAgJ0VmZmVjdCc6ICdBbGxvdycsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIC8vIHRoaXMgcGF0dGVybiBpcyB1c2VkIGhlcmU6XG4gIC8vIGF3cy1jb2RlcGlwZWxpbmUtYWN0aW9ucy9saWIvY2xvdWRmb3JtYXRpb24vcGlwZWxpbmUtYWN0aW9ucy50cyNMNTE3XG4gIHRlc3QoJ2ltbXV0YWJsZSByb2xlIGlzIGEgY29uc3RydWN0JywgKCkgPT4ge1xuICAgIG5ldyBDb25zdHJ1Y3QoaW1tdXRhYmxlUm9sZSBhcyB1bmtub3duIGFzIENvbnN0cnVjdCwgJ0NoaWxkJyk7XG4gICAgbmV3IENvbnN0cnVjdChtdXRhYmxlUm9sZS53aXRob3V0UG9saWN5VXBkYXRlcygpIGFzIHVua25vd24gYXMgQ29uc3RydWN0LCAnQ2hpbGQyJyk7XG4gIH0pO1xufSk7Il19