"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const sinon = require("sinon");
const iam = require("../lib");
const diff_1 = require("../lib/oidc-provider/diff");
const external_1 = require("../lib/oidc-provider/external");
const handler = require("../lib/oidc-provider/index");
describe('OpenIdConnectProvider resource', () => {
    test('minimal configuration (no clients and no thumbprint)', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new iam.OpenIdConnectProvider(stack, 'MyProvider', {
            url: 'https://openid-endpoint',
        });
        // THEN
        expect(stack).toHaveResource('Custom::AWSCDKOpenIdConnectProvider', {
            Url: 'https://openid-endpoint',
        });
    });
    test('"openIdConnectProviderArn" resolves to the ref', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const provider = new iam.OpenIdConnectProvider(stack, 'MyProvider', {
            url: 'https://openid-endpoint',
        });
        // THEN
        expect(stack.resolve(provider.openIdConnectProviderArn)).toStrictEqual({ Ref: 'MyProvider730BA1C8' });
    });
    test('static fromOpenIdConnectProviderArn can be used to import a provider', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const provider = iam.OpenIdConnectProvider.fromOpenIdConnectProviderArn(stack, 'MyProvider', 'arn:of:provider');
        // THEN
        expect(stack.resolve(provider.openIdConnectProviderArn)).toStrictEqual('arn:of:provider');
    });
    test('thumbprint list and client ids can be specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new iam.OpenIdConnectProvider(stack, 'MyProvider', {
            url: 'https://my-url',
            clientIds: ['client1', 'client2'],
            thumbprints: ['thumb1'],
        });
        // THEN
        expect(stack).toHaveResource('Custom::AWSCDKOpenIdConnectProvider', {
            Url: 'https://my-url',
            ClientIDList: ['client1', 'client2'],
            ThumbprintList: ['thumb1'],
        });
    });
});
describe('custom resource provider infrastructure', () => {
    test('two resources share the same cr provider', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'stack');
        // WHEN
        new iam.OpenIdConnectProvider(stack, 'Provider1', { url: 'provider1' });
        new iam.OpenIdConnectProvider(stack, 'Provider2', { url: 'provider2' });
        // THEN
        const template = app.synth().getStackArtifact(stack.artifactId).template;
        const resourceTypes = Object.values(template.Resources).map((r) => r.Type).sort();
        expect(resourceTypes).toStrictEqual([
            // custom resource perovider resources
            'AWS::IAM::Role',
            'AWS::Lambda::Function',
            // open id connect resources
            'Custom::AWSCDKOpenIdConnectProvider',
            'Custom::AWSCDKOpenIdConnectProvider',
        ]);
    });
    test('iam policy', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new iam.OpenIdConnectProvider(stack, 'Provider1', { url: 'provider1' });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Role', {
            Policies: [
                {
                    PolicyName: 'Inline',
                    PolicyDocument: {
                        Version: '2012-10-17',
                        Statement: [
                            {
                                Effect: 'Allow',
                                Resource: '*',
                                Action: [
                                    'iam:CreateOpenIDConnectProvider',
                                    'iam:DeleteOpenIDConnectProvider',
                                    'iam:UpdateOpenIDConnectProviderThumbprint',
                                    'iam:AddClientIDToOpenIDConnectProvider',
                                    'iam:RemoveClientIDFromOpenIDConnectProvider',
                                ],
                            },
                        ],
                    },
                },
            ],
        });
    });
});
describe('custom resource provider handler', () => {
    external_1.external.log = () => { return; }; // disable verbosity for tests
    const downloadThumbprint = external_1.external.downloadThumbprint = sinon.fake.returns('FAKE-THUMBPRINT');
    const createOpenIDConnectProvider = external_1.external.createOpenIDConnectProvider = sinon.fake.resolves({ OpenIDConnectProviderArn: 'FAKE-ARN' });
    const deleteOpenIDConnectProvider = external_1.external.deleteOpenIDConnectProvider = sinon.fake.resolves({});
    const updateOpenIDConnectProviderThumbprint = external_1.external.updateOpenIDConnectProviderThumbprint = sinon.fake.resolves({});
    const addClientIDToOpenIDConnectProvider = external_1.external.addClientIDToOpenIDConnectProvider = sinon.fake.resolves({});
    const removeClientIDFromOpenIDConnectProvider = external_1.external.removeClientIDFromOpenIDConnectProvider = sinon.fake.resolves({});
    beforeEach(() => sinon.reset());
    test('create with url will download thumbprint from host', async () => {
        // WHEN
        const response = await invokeHandler({
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://my-urlx',
                ThumbprintList: ['MyThumbprint'],
            },
        });
        // THEN
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.calledWithExactly(createOpenIDConnectProvider, {
            ClientIDList: [],
            Url: 'https://my-urlx',
            ThumbprintList: ['MyThumbprint'],
        });
        expect(response).toStrictEqual({
            PhysicalResourceId: 'FAKE-ARN',
        });
    });
    test('create without thumbprint will download from host', async () => {
        // WHEN
        const response = await invokeHandler({
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://my-urlx',
            },
        });
        // THEN
        sinon.assert.calledWithExactly(downloadThumbprint, 'https://my-urlx');
        sinon.assert.calledWithExactly(createOpenIDConnectProvider, {
            ClientIDList: [],
            Url: 'https://my-urlx',
            ThumbprintList: ['FAKE-THUMBPRINT'],
        });
        expect(response).toStrictEqual({
            PhysicalResourceId: 'FAKE-ARN',
        });
    });
    test('delete', async () => {
        // WHEN
        await invokeHandler({
            RequestType: 'Delete',
            PhysicalResourceId: 'FAKE-ARN',
        });
        // THEN
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.notCalled(createOpenIDConnectProvider);
        sinon.assert.calledWithExactly(deleteOpenIDConnectProvider, {
            OpenIDConnectProviderArn: 'FAKE-ARN',
        });
    });
    test('update url with explicit thumbprints (replace)', async () => {
        // WHEN
        const response = await invokeHandler({
            RequestType: 'Update',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://new',
                ThumbprintList: ['THUMB1', 'THUMB2'],
            },
            OldResourceProperties: {
                Url: 'https://old',
            },
        });
        // THEN
        expect(response).toStrictEqual({
            PhysicalResourceId: 'FAKE-ARN',
        });
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.calledWithExactly(createOpenIDConnectProvider, {
            ClientIDList: [],
            Url: 'https://new',
            ThumbprintList: ['THUMB1', 'THUMB2'],
        });
    });
    test('update url with no thumbprint (replace)', async () => {
        // WHEN
        const response = await invokeHandler({
            RequestType: 'Update',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://new',
            },
            OldResourceProperties: {
                Url: 'https://old',
            },
        });
        // THEN
        expect(response).toStrictEqual({
            PhysicalResourceId: 'FAKE-ARN',
        });
        sinon.assert.calledOnceWithExactly(downloadThumbprint, 'https://new');
        sinon.assert.calledOnceWithExactly(createOpenIDConnectProvider, {
            ClientIDList: [],
            Url: 'https://new',
            ThumbprintList: ['FAKE-THUMBPRINT'],
        });
        sinon.assert.notCalled(deleteOpenIDConnectProvider);
    });
    test('update thumbprint list', async () => {
        // WHEN
        await invokeHandler({
            RequestType: 'Update',
            PhysicalResourceId: 'FAKE-PhysicalResourceId',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://url',
                ThumbprintList: ['Foo', 'Bar'],
            },
            OldResourceProperties: {
                Url: 'https://url',
                ThumbprintList: ['Foo'],
            },
        });
        // THEN
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.notCalled(createOpenIDConnectProvider);
        sinon.assert.notCalled(deleteOpenIDConnectProvider);
        sinon.assert.calledOnceWithExactly(updateOpenIDConnectProviderThumbprint, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId',
            ThumbprintList: ['Bar', 'Foo'],
        });
    });
    test('add/remove client ids', async () => {
        // WHEN
        await invokeHandler({
            RequestType: 'Update',
            PhysicalResourceId: 'FAKE-PhysicalResourceId',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://url',
                ClientIDList: ['A', 'B', 'C'],
            },
            OldResourceProperties: {
                Url: 'https://url',
                ClientIDList: ['A', 'D'],
            },
        });
        // THEN
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.notCalled(createOpenIDConnectProvider);
        sinon.assert.notCalled(deleteOpenIDConnectProvider);
        sinon.assert.notCalled(updateOpenIDConnectProviderThumbprint);
        sinon.assert.calledTwice(addClientIDToOpenIDConnectProvider);
        sinon.assert.calledWithExactly(addClientIDToOpenIDConnectProvider, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId', ClientID: 'B',
        });
        sinon.assert.calledWithExactly(addClientIDToOpenIDConnectProvider, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId', ClientID: 'C',
        });
        sinon.assert.calledOnceWithExactly(removeClientIDFromOpenIDConnectProvider, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId', ClientID: 'D',
        });
    });
    test('multiple in-place updates (no replace)', async () => {
        // WHEN
        await invokeHandler({
            RequestType: 'Update',
            PhysicalResourceId: 'FAKE-PhysicalResourceId',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://url',
                ThumbprintList: ['NEW-LIST'],
                ClientIDList: ['A'],
            },
            OldResourceProperties: {
                Url: 'https://url',
                ThumbprintList: ['OLD-LIST'],
                ClientIDList: [],
            },
        });
        // THEN
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.notCalled(createOpenIDConnectProvider);
        sinon.assert.notCalled(deleteOpenIDConnectProvider);
        sinon.assert.notCalled(removeClientIDFromOpenIDConnectProvider);
        sinon.assert.calledOnceWithExactly(updateOpenIDConnectProviderThumbprint, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId',
            ThumbprintList: ['NEW-LIST'],
        });
        sinon.assert.calledOnceWithExactly(addClientIDToOpenIDConnectProvider, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId',
            ClientID: 'A',
        });
    });
    test('multiple updates that include a url update, which means replacement', async () => {
        // WHEN
        await invokeHandler({
            RequestType: 'Update',
            PhysicalResourceId: 'FAKE-PhysicalResourceId',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://new-url',
                ClientIDList: ['A'],
            },
            OldResourceProperties: {
                Url: 'https://old-url',
                ThumbprintList: ['OLD-LIST'],
                ClientIDList: [],
            },
        });
        // THEN
        sinon.assert.notCalled(deleteOpenIDConnectProvider);
        sinon.assert.notCalled(removeClientIDFromOpenIDConnectProvider);
        sinon.assert.notCalled(updateOpenIDConnectProviderThumbprint);
        sinon.assert.notCalled(addClientIDToOpenIDConnectProvider);
        sinon.assert.calledOnceWithExactly(downloadThumbprint, 'https://new-url'); // since thumbprint list is empty
        sinon.assert.calledOnceWithExactly(createOpenIDConnectProvider, {
            ClientIDList: ['A'],
            ThumbprintList: ['FAKE-THUMBPRINT'],
            Url: 'https://new-url',
        });
    });
});
describe('arrayDiff', () => {
    test('calculates the difference between two arrays', () => {
        expect(diff_1.arrayDiff(['a', 'b', 'c'], ['a', 'd'])).toStrictEqual({ adds: ['d'], deletes: ['b', 'c'] });
        expect(diff_1.arrayDiff(['a', 'b', 'c'], [])).toStrictEqual({ adds: [], deletes: ['a', 'b', 'c'] });
        expect(diff_1.arrayDiff(['a', 'b', 'c'], ['a', 'c', 'b'])).toStrictEqual({ adds: [], deletes: [] });
        expect(diff_1.arrayDiff([], ['a', 'c', 'b'])).toStrictEqual({ adds: ['a', 'c', 'b'], deletes: [] });
        expect(diff_1.arrayDiff(['x', 'y'], ['a', 'c', 'b'])).toStrictEqual({ adds: ['a', 'c', 'b'], deletes: ['x', 'y'] });
        expect(diff_1.arrayDiff([], [])).toStrictEqual({ adds: [], deletes: [] });
        expect(diff_1.arrayDiff(['a', 'a'], ['a', 'b', 'a', 'b', 'b'])).toStrictEqual({ adds: ['b'], deletes: [] });
    });
});
async function invokeHandler(event) {
    return handler.handler(event);
}
//# sourceMappingURL=data:application/json;base64,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