"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('IAM policy document', () => {
    test('the Permission class is a programming model for iam', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addActions('sqs:SendMessage');
        p.addActions('dynamodb:CreateTable', 'dynamodb:DeleteTable');
        p.addResources('myQueue');
        p.addResources('yourQueue');
        p.addAllResources();
        p.addAwsAccountPrincipal(`my${core_1.Token.asString({ account: 'account' })}name`);
        p.addAccountCondition('12221121221');
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Action: ['sqs:SendMessage',
                'dynamodb:CreateTable',
                'dynamodb:DeleteTable'],
            Resource: ['myQueue', 'yourQueue', '*'],
            Effect: 'Allow',
            Principal: {
                AWS: {
                    'Fn::Join': ['',
                        ['arn:',
                            { Ref: 'AWS::Partition' },
                            ':iam::my',
                            { account: 'account' },
                            'name:root']],
                },
            },
            Condition: { StringEquals: { 'sts:ExternalId': '12221121221' } },
        });
    });
    test('the PolicyDocument class is a dom for iam policy documents', () => {
        const stack = new core_1.Stack();
        const doc = new lib_1.PolicyDocument();
        const p1 = new lib_1.PolicyStatement();
        p1.addActions('sqs:SendMessage');
        p1.addNotResources('arn:aws:sqs:us-east-1:123456789012:forbidden_queue');
        const p2 = new lib_1.PolicyStatement();
        p2.effect = lib_1.Effect.DENY;
        p2.addActions('cloudformation:CreateStack');
        const p3 = new lib_1.PolicyStatement();
        p3.effect = lib_1.Effect.ALLOW;
        p3.addNotActions('cloudformation:UpdateTerminationProtection');
        const p4 = new lib_1.PolicyStatement();
        p4.effect = lib_1.Effect.DENY;
        p4.addNotPrincipals(new lib_1.CanonicalUserPrincipal('OnlyAuthorizedUser'));
        doc.addStatements(p1);
        doc.addStatements(p2);
        doc.addStatements(p3);
        doc.addStatements(p4);
        expect(stack.resolve(doc)).toEqual({
            Version: '2012-10-17',
            Statement: [{ Effect: 'Allow', Action: 'sqs:SendMessage', NotResource: 'arn:aws:sqs:us-east-1:123456789012:forbidden_queue' },
                { Effect: 'Deny', Action: 'cloudformation:CreateStack' },
                { Effect: 'Allow', NotAction: 'cloudformation:UpdateTerminationProtection' },
                { Effect: 'Deny', NotPrincipal: { CanonicalUser: 'OnlyAuthorizedUser' } }],
        });
    });
    test('Cannot combine Actions and NotActions', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                actions: ['abc:def'],
                notActions: ['abc:def'],
            });
        }).toThrow(/Cannot add 'NotActions' to policy statement if 'Actions' have been added/);
    });
    test('Throws with invalid actions', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                actions: ['service:action', '*', 'service:acti*', 'in:val:id'],
            });
        }).toThrow(/Action 'in:val:id' is invalid/);
    });
    test('Throws with invalid not actions', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                notActions: ['service:action', '*', 'service:acti*', 'in:val:id'],
            });
        }).toThrow(/Action 'in:val:id' is invalid/);
    });
    test('Cannot combine Resources and NotResources', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                resources: ['abc'],
                notResources: ['def'],
            });
        }).toThrow(/Cannot add 'NotResources' to policy statement if 'Resources' have been added/);
    });
    test('Cannot add NotPrincipals when Principals exist', () => {
        const stmt = new lib_1.PolicyStatement({
            principals: [new lib_1.CanonicalUserPrincipal('abc')],
        });
        expect(() => {
            stmt.addNotPrincipals(new lib_1.CanonicalUserPrincipal('def'));
        }).toThrow(/Cannot add 'NotPrincipals' to policy statement if 'Principals' have been added/);
    });
    test('Cannot add Principals when NotPrincipals exist', () => {
        const stmt = new lib_1.PolicyStatement({
            notPrincipals: [new lib_1.CanonicalUserPrincipal('abc')],
        });
        expect(() => {
            stmt.addPrincipals(new lib_1.CanonicalUserPrincipal('def'));
        }).toThrow(/Cannot add 'Principals' to policy statement if 'NotPrincipals' have been added/);
    });
    test('Permission allows specifying multiple actions upon construction', () => {
        const stack = new core_1.Stack();
        const perm = new lib_1.PolicyStatement();
        perm.addResources('MyResource');
        perm.addActions('Action1', 'Action2', 'Action3');
        expect(stack.resolve(perm.toStatementJson())).toEqual({
            Effect: 'Allow',
            Action: ['Action1', 'Action2', 'Action3'],
            Resource: 'MyResource',
        });
    });
    test('PolicyDoc resolves to undefined if there are no permissions', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyDocument();
        expect(stack.resolve(p)).toBeUndefined();
    });
    test('canonicalUserPrincipal adds a principal to a policy with the passed canonical user id', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        const canoncialUser = 'averysuperduperlongstringfor';
        p.addPrincipals(new lib_1.CanonicalUserPrincipal(canoncialUser));
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                CanonicalUser: canoncialUser,
            },
        });
    });
    test('addAccountRootPrincipal adds a principal with the current account root', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAccountRootPrincipal();
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                AWS: {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            { Ref: 'AWS::Partition' },
                            ':iam::',
                            { Ref: 'AWS::AccountId' },
                            ':root',
                        ],
                    ],
                },
            },
        });
    });
    test('addFederatedPrincipal adds a Federated principal with the passed value', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addFederatedPrincipal('com.amazon.cognito', { StringEquals: { key: 'value' } });
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                Federated: 'com.amazon.cognito',
            },
            Condition: {
                StringEquals: { key: 'value' },
            },
        });
    });
    test('addAwsAccountPrincipal can be used multiple times', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAwsAccountPrincipal('1234');
        p.addAwsAccountPrincipal('5678');
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::1234:root']] },
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::5678:root']] },
                ],
            },
        });
    });
    describe('hasResource', () => {
        test('false if there are no resources', () => {
            expect(new lib_1.PolicyStatement().hasResource).toEqual(false);
        });
        test('true if there is one resource', () => {
            expect(new lib_1.PolicyStatement({ resources: ['one-resource'] }).hasResource).toEqual(true);
        });
        test('true for multiple resources', () => {
            const p = new lib_1.PolicyStatement();
            p.addResources('r1');
            p.addResources('r2');
            expect(p.hasResource).toEqual(true);
        });
    });
    describe('hasPrincipal', () => {
        test('false if there is no principal', () => {
            expect(new lib_1.PolicyStatement().hasPrincipal).toEqual(false);
        });
        test('true if there is a principal', () => {
            const p = new lib_1.PolicyStatement();
            p.addArnPrincipal('bla');
            expect(p.hasPrincipal).toEqual(true);
        });
        test('true if there is a notPrincipal', () => {
            const p = new lib_1.PolicyStatement();
            p.addNotPrincipals(new lib_1.CanonicalUserPrincipal('test'));
            expect(p.hasPrincipal).toEqual(true);
        });
    });
    test('statementCount returns the number of statement in the policy document', () => {
        const p = new lib_1.PolicyDocument();
        expect(p.statementCount).toEqual(0);
        p.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'] }));
        expect(p.statementCount).toEqual(1);
        p.addStatements(new lib_1.PolicyStatement({ actions: ['service:action2'] }));
        expect(p.statementCount).toEqual(2);
    });
    describe('{ AWS: "*" } principal', () => {
        test('is represented as `Anyone`', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatements(new lib_1.PolicyStatement({ principals: [new lib_1.Anyone()] }));
            expect(stack.resolve(p)).toEqual({
                Statement: [
                    { Effect: 'Allow', Principal: '*' },
                ],
                Version: '2012-10-17',
            });
        });
        test('is represented as `AnyPrincipal`', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatements(new lib_1.PolicyStatement({ principals: [new lib_1.AnyPrincipal()] }));
            expect(stack.resolve(p)).toEqual({
                Statement: [
                    { Effect: 'Allow', Principal: '*' },
                ],
                Version: '2012-10-17',
            });
        });
        test('is represented as `addAnyPrincipal`', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const s = new lib_1.PolicyStatement();
            s.addAnyPrincipal();
            p.addStatements(s);
            expect(stack.resolve(p)).toEqual({
                Statement: [
                    { Effect: 'Allow', Principal: '*' },
                ],
                Version: '2012-10-17',
            });
        });
    });
    test('addResources() will not break a list-encoded Token', () => {
        const stack = new core_1.Stack();
        const statement = new lib_1.PolicyStatement();
        statement.addActions(...core_1.Lazy.listValue({ produce: () => ['a', 'b', 'c'] }));
        statement.addResources(...core_1.Lazy.listValue({ produce: () => ['x', 'y', 'z'] }));
        expect(stack.resolve(statement.toStatementJson())).toEqual({
            Effect: 'Allow',
            Action: ['a', 'b', 'c'],
            Resource: ['x', 'y', 'z'],
        });
    });
    test('addCanonicalUserPrincipal can be used to add cannonical user principals', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyDocument();
        const s1 = new lib_1.PolicyStatement();
        s1.addCanonicalUserPrincipal('cannonical-user-1');
        const s2 = new lib_1.PolicyStatement();
        s2.addPrincipals(new lib_1.CanonicalUserPrincipal('cannonical-user-2'));
        p.addStatements(s1);
        p.addStatements(s2);
        expect(stack.resolve(p)).toEqual({
            Statement: [
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-1' } },
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-2' } },
            ],
            Version: '2012-10-17',
        });
    });
    test('addPrincipal correctly merges array in', () => {
        const stack = new core_1.Stack();
        const arrayPrincipal = {
            get grantPrincipal() { return this; },
            assumeRoleAction: 'sts:AssumeRole',
            policyFragment: new lib_1.PrincipalPolicyFragment({ AWS: ['foo', 'bar'] }),
            addToPolicy() { return false; },
            addToPrincipalPolicy() { return { statementAdded: false }; },
        };
        const s = new lib_1.PolicyStatement();
        s.addAccountRootPrincipal();
        s.addPrincipals(arrayPrincipal);
        expect(stack.resolve(s.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::', { Ref: 'AWS::AccountId' }, ':root']] },
                    'foo', 'bar',
                ],
            },
        });
    });
    // https://github.com/aws/aws-cdk/issues/1201
    test('policy statements with multiple principal types can be created using multiple addPrincipal calls', () => {
        const stack = new core_1.Stack();
        const s = new lib_1.PolicyStatement();
        s.addArnPrincipal('349494949494');
        s.addServicePrincipal('test.service');
        s.addResources('resource');
        s.addActions('action');
        expect(stack.resolve(s.toStatementJson())).toEqual({
            Action: 'action',
            Effect: 'Allow',
            Principal: { AWS: '349494949494', Service: 'test.service' },
            Resource: 'resource',
        });
    });
    describe('Service principals', () => {
        test('regional service principals resolve appropriately', () => {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('codedeploy.amazonaws.com');
            expect(stack.resolve(s.toStatementJson())).toEqual({
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' },
            });
        });
        test('regional service principals resolve appropriately (with user-set region)', () => {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-northeast-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('codedeploy.amazonaws.com', { region: 'cn-north-1' });
            expect(stack.resolve(s.toStatementJson())).toEqual({
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' },
            });
        });
        test('obscure service principals resolve to the user-provided value', () => {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('test.service-principal.dev');
            expect(stack.resolve(s.toStatementJson())).toEqual({
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'test.service-principal.dev' },
            });
        });
    });
    describe('CompositePrincipal can be used to represent a principal that has multiple types', () => {
        test('with a single principal', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.CompositePrincipal(new lib_1.ArnPrincipal('i:am:an:arn'));
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            expect(stack.resolve(statement.toStatementJson())).toEqual({ Effect: 'Allow', Principal: { AWS: 'i:am:an:arn' } });
        });
        test('conditions are not allowed on individual principals of a composite', () => {
            const p = new lib_1.CompositePrincipal(new lib_1.ArnPrincipal('i:am'));
            expect(() => p.addPrincipals(new lib_1.FederatedPrincipal('federated', { StringEquals: { 'aws:some-key': 'some-value' } })))
                .toThrow(/Components of a CompositePrincipal must not have conditions/);
        });
        test('principals and conditions are a big nice merge', () => {
            const stack = new core_1.Stack();
            // add via ctor
            const p = new lib_1.CompositePrincipal(new lib_1.ArnPrincipal('i:am:an:arn'), new lib_1.ServicePrincipal('amazon.com'));
            // add via `addPrincipals` (with condition)
            p.addPrincipals(new lib_1.Anyone(), new lib_1.ServicePrincipal('another.service'));
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            // add via policy statement
            statement.addArnPrincipal('aws-principal-3');
            statement.addCondition('cond2', { boom: '123' });
            expect(stack.resolve(statement.toStatementJson())).toEqual({
                Condition: {
                    cond2: { boom: '123' },
                },
                Effect: 'Allow',
                Principal: {
                    AWS: ['i:am:an:arn', '*', 'aws-principal-3'],
                    Service: ['amazon.com', 'another.service'],
                },
            });
        });
        test('cannot mix types of assumeRoleAction in a single composite', () => {
            // GIVEN
            const p = new lib_1.CompositePrincipal(new lib_1.ArnPrincipal('arn')); // assumeRoleAction is "sts:AssumeRule"
            // THEN
            expect(() => p.addPrincipals(new lib_1.FederatedPrincipal('fed', {}, 'sts:Boom')))
                .toThrow(/Cannot add multiple principals with different "assumeRoleAction". Expecting "sts:AssumeRole", got "sts:Boom"/);
        });
    });
    describe('PrincipalWithConditions can be used to add a principal with conditions', () => {
        test('includes conditions from both the wrapped principal and the wrapper', () => {
            const stack = new core_1.Stack();
            const principalOpts = {
                conditions: {
                    BinaryEquals: {
                        'principal-key': 'SGV5LCBmcmllbmQh',
                    },
                },
            };
            const p = new lib_1.ServicePrincipal('s3.amazonaws.com', principalOpts)
                .withConditions({ StringEquals: { 'wrapper-key': ['val-1', 'val-2'] } });
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            expect(stack.resolve(statement.toStatementJson())).toEqual({
                Condition: {
                    BinaryEquals: { 'principal-key': 'SGV5LCBmcmllbmQh' },
                    StringEquals: { 'wrapper-key': ['val-1', 'val-2'] },
                },
                Effect: 'Allow',
                Principal: {
                    Service: 's3.amazonaws.com',
                },
            });
        });
        test('conditions from addCondition are merged with those from the principal', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.AccountPrincipal('012345678900').withConditions({ StringEquals: { key: 'val' } });
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            statement.addCondition('Null', { 'banned-key': 'true' });
            expect(stack.resolve(statement.toStatementJson())).toEqual({
                Effect: 'Allow',
                Principal: { AWS: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::012345678900:root']] } },
                Condition: { StringEquals: { key: 'val' }, Null: { 'banned-key': 'true' } },
            });
        });
        test('adding conditions via `withConditions` does not affect the original principal', () => {
            const originalPrincipal = new lib_1.ArnPrincipal('iam:an:arn');
            const principalWithConditions = originalPrincipal.withConditions({ StringEquals: { key: 'val' } });
            expect(originalPrincipal.policyFragment.conditions).toEqual({});
            expect(principalWithConditions.policyFragment.conditions).toEqual({ StringEquals: { key: 'val' } });
        });
        test('conditions are merged when operators conflict', () => {
            const p = new lib_1.FederatedPrincipal('fed', {
                OperatorOne: { 'fed-key': 'fed-val' },
                OperatorTwo: { 'fed-key': 'fed-val' },
                OperatorThree: { 'fed-key': 'fed-val' },
            }).withConditions({
                OperatorTwo: { 'with-key': 'with-val' },
                OperatorThree: { 'with-key': 'with-val' },
            });
            const statement = new lib_1.PolicyStatement();
            statement.addCondition('OperatorThree', { 'add-key': 'add-val' });
            statement.addPrincipals(p);
            expect(statement.toStatementJson()).toEqual({
                Effect: 'Allow',
                Principal: { Federated: 'fed' },
                Condition: {
                    OperatorOne: { 'fed-key': 'fed-val' },
                    OperatorTwo: { 'fed-key': 'fed-val', 'with-key': 'with-val' },
                    OperatorThree: { 'fed-key': 'fed-val', 'with-key': 'with-val', 'add-key': 'add-val' },
                },
            });
        });
        test('tokens can be used in conditions', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const statement = new lib_1.PolicyStatement();
            // WHEN
            const p = new lib_1.ArnPrincipal('arn:of:principal').withConditions({
                StringEquals: core_1.Lazy.anyValue({ produce: () => ({ goo: 'zar' }) }),
            });
            statement.addPrincipals(p);
            // THEN
            const resolved = stack.resolve(statement.toStatementJson());
            expect(resolved).toEqual({
                Condition: {
                    StringEquals: {
                        goo: 'zar',
                    },
                },
                Effect: 'Allow',
                Principal: {
                    AWS: 'arn:of:principal',
                },
            });
        });
        test('conditions cannot be merged if they include tokens', () => {
            const p = new lib_1.FederatedPrincipal('fed', {
                StringEquals: { foo: 'bar' },
            }).withConditions({
                StringEquals: core_1.Lazy.anyValue({ produce: () => ({ goo: 'zar' }) }),
            });
            const statement = new lib_1.PolicyStatement();
            expect(() => statement.addPrincipals(p)).toThrow(/multiple "StringEquals" conditions cannot be merged if one of them contains an unresolved token/);
        });
        test('values passed to `withConditions` overwrite values from the wrapped principal ' +
            'when keys conflict within an operator', () => {
            const p = new lib_1.FederatedPrincipal('fed', {
                Operator: { key: 'p-val' },
            }).withConditions({
                Operator: { key: 'with-val' },
            });
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            expect(statement.toStatementJson()).toEqual({
                Effect: 'Allow',
                Principal: { Federated: 'fed' },
                Condition: {
                    Operator: { key: 'with-val' },
                },
            });
        });
    });
    describe('duplicate statements', () => {
        test('without tokens', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement = new lib_1.PolicyStatement();
            statement.addResources('resource1', 'resource2');
            statement.addActions('action1', 'action2');
            statement.addServicePrincipal('service');
            statement.addConditions({
                a: {
                    b: 'c',
                },
                d: {
                    e: 'f',
                },
            });
            // WHEN
            p.addStatements(statement);
            p.addStatements(statement);
            p.addStatements(statement);
            // THEN
            expect(stack.resolve(p).Statement).toHaveLength(1);
        });
        test('with tokens', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement1 = new lib_1.PolicyStatement();
            statement1.addResources(core_1.Lazy.stringValue({ produce: () => 'resource' }));
            statement1.addActions(core_1.Lazy.stringValue({ produce: () => 'action' }));
            const statement2 = new lib_1.PolicyStatement();
            statement2.addResources(core_1.Lazy.stringValue({ produce: () => 'resource' }));
            statement2.addActions(core_1.Lazy.stringValue({ produce: () => 'action' }));
            // WHEN
            p.addStatements(statement1);
            p.addStatements(statement2);
            // THEN
            expect(stack.resolve(p).Statement).toHaveLength(1);
        });
    });
    test('autoAssignSids enables auto-assignment of a unique SID for each statement', () => {
        // GIVEN
        const doc = new lib_1.PolicyDocument({
            assignSids: true,
        });
        // WHEN
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action2'], resources: ['resource2'] }));
        // THEN
        const stack = new core_1.Stack();
        expect(stack.resolve(doc)).toEqual({
            Version: '2012-10-17',
            Statement: [
                { Action: 'service:action1', Effect: 'Allow', Resource: 'resource1', Sid: '0' },
                { Action: 'service:action2', Effect: 'Allow', Resource: 'resource2', Sid: '1' },
            ],
        });
    });
    test('constructor args are equivalent to mutating in-place', () => {
        const stack = new core_1.Stack();
        const s = new lib_1.PolicyStatement();
        s.addActions('service:action1', 'service:action2');
        s.addAllResources();
        s.addArnPrincipal('arn');
        s.addCondition('key', { equals: 'value' });
        const doc1 = new lib_1.PolicyDocument();
        doc1.addStatements(s);
        const doc2 = new lib_1.PolicyDocument();
        doc2.addStatements(new lib_1.PolicyStatement({
            actions: ['service:action1', 'service:action2'],
            resources: ['*'],
            principals: [new lib_1.ArnPrincipal('arn')],
            conditions: {
                key: { equals: 'value' },
            },
        }));
        expect(stack.resolve(doc1)).toEqual(stack.resolve(doc2));
    });
    describe('fromJson', () => {
        test("throws error when Statement isn't an array", () => {
            expect(() => {
                lib_1.PolicyDocument.fromJson({
                    Statement: 'asdf',
                });
            }).toThrow(/Statement must be an array/);
        });
    });
    test('adding another condition with the same operator does not delete the original', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addCondition('StringEquals', { 'kms:ViaService': 'service' });
        p.addAccountCondition('12221121221');
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Condition: { StringEquals: { 'kms:ViaService': 'service', 'sts:ExternalId': '12221121221' } },
        });
    });
    test('validation error if policy statement has no actions', () => {
        const policyStatement = new lib_1.PolicyStatement({
            principals: [new lib_1.AnyPrincipal()],
        });
        // THEN
        const validationErrorsForResourcePolicy = policyStatement.validateForResourcePolicy();
        // const validationErrorsForIdentityPolicy: string[] = policyStatement.validateForIdentityPolicy();
        expect(validationErrorsForResourcePolicy).toEqual(['A PolicyStatement must specify at least one \'action\' or \'notAction\'.']);
    });
    test('validation error if policy statement for resource-based policy has no principals specified', () => {
        const policyStatement = new lib_1.PolicyStatement({
            actions: ['*'],
        });
        // THEN
        const validationErrors = policyStatement.validateForResourcePolicy();
        expect(validationErrors).toEqual(['A PolicyStatement used in a resource-based policy must specify at least one IAM principal.']);
    });
});
//# sourceMappingURL=data:application/json;base64,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