"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/* eslint-disable quote-props */
describe('IAM policy', () => {
    let app;
    let stack;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'MyStack');
    });
    test('fails when "forced" policy is empty', () => {
        new lib_1.Policy(stack, 'MyPolicy', { force: true });
        expect(() => app.synth()).toThrow(/is empty/);
    });
    test('policy with statements', () => {
        const policy = new lib_1.Policy(stack, 'MyPolicy', { policyName: 'MyPolicyName' });
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.attachInlinePolicy(policy);
        expect(stack).toMatchTemplate({
            Resources: {
                MyPolicy39D66CF6: {
                    Type: 'AWS::IAM::Policy',
                    Properties: {
                        Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: {
                            Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17',
                        },
                        PolicyName: 'MyPolicyName',
                    },
                },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' },
            },
        });
    });
    test('policy name can be omitted, in which case the logical id will be used', () => {
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const user = new lib_1.User(stack, 'MyUser');
        user.attachInlinePolicy(policy);
        expect(stack).toMatchTemplate({
            Resources: {
                MyPolicy39D66CF6: {
                    Type: 'AWS::IAM::Policy',
                    Properties: {
                        PolicyDocument: {
                            Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17',
                        },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }],
                    },
                },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
            },
        });
    });
    test('policy can be attached users, groups and roles and added permissions via props', () => {
        const user1 = new lib_1.User(stack, 'User1');
        const group1 = new lib_1.Group(stack, 'Group1');
        const role1 = new lib_1.Role(stack, 'Role1', {
            assumedBy: new lib_1.ServicePrincipal('test.service'),
        });
        new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
            users: [user1],
            groups: [group1],
            roles: [role1],
            statements: [new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:PutItem'] })],
        });
        expect(stack).toMatchTemplate({
            Resources: {
                User1E278A736: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' },
                                }],
                            Version: '2012-10-17',
                        },
                    },
                },
                MyTestPolicy316BDB50: {
                    Type: 'AWS::IAM::Policy',
                    Properties: {
                        Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: {
                            Statement: [{ Action: 'dynamodb:PutItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17',
                        },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }],
                    },
                },
            },
        });
    });
    test('idempotent if a principal (user/group/role) is attached twice', () => {
        const p = new lib_1.Policy(stack, 'MyPolicy');
        p.addStatements(new lib_1.PolicyStatement({ actions: ['*'], resources: ['*'] }));
        const user = new lib_1.User(stack, 'MyUser');
        p.attachToUser(user);
        p.attachToUser(user);
        expect(stack).toMatchTemplate({
            Resources: {
                MyPolicy39D66CF6: {
                    Type: 'AWS::IAM::Policy',
                    Properties: {
                        PolicyDocument: {
                            Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17',
                        },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }],
                    },
                },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
            },
        });
    });
    test('users, groups, roles and permissions can be added using methods', () => {
        const p = new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
        });
        p.attachToUser(new lib_1.User(stack, 'User1'));
        p.attachToUser(new lib_1.User(stack, 'User2'));
        p.attachToGroup(new lib_1.Group(stack, 'Group1'));
        p.attachToRole(new lib_1.Role(stack, 'Role1', { assumedBy: new lib_1.ServicePrincipal('test.service') }));
        p.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:GetItem'] }));
        expect(stack).toMatchTemplate({
            Resources: {
                MyTestPolicy316BDB50: {
                    Type: 'AWS::IAM::Policy',
                    Properties: {
                        Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: {
                            Statement: [{ Action: 'dynamodb:GetItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17',
                        },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }, { Ref: 'User21F1486D1' }],
                    },
                },
                User1E278A736: { Type: 'AWS::IAM::User' },
                User21F1486D1: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' },
                                }],
                            Version: '2012-10-17',
                        },
                    },
                },
            },
        });
    });
    test('policy can be attached to users, groups or role via methods on the principal', () => {
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.attachInlinePolicy(policy);
        group.attachInlinePolicy(policy);
        role.attachInlinePolicy(policy);
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        expect(stack).toMatchTemplate({
            Resources: {
                MyPolicy39D66CF6: {
                    Type: 'AWS::IAM::Policy',
                    Properties: {
                        Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: {
                            Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17',
                        },
                        PolicyName: 'MyPolicy39D66CF6',
                        Roles: [{ Ref: 'MyRoleF48FFE04' }],
                        Users: [{ Ref: 'MyUserDC45028B' }],
                    },
                },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' },
                MyRoleF48FFE04: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' },
                                }],
                            Version: '2012-10-17',
                        },
                    },
                },
            },
        });
    });
    test('fails if policy name is not unique within a user/group/role', () => {
        // create two policies named Foo and attach them both to the same user/group/role
        const p1 = new lib_1.Policy(stack, 'P1', { policyName: 'Foo' });
        const p2 = new lib_1.Policy(stack, 'P2', { policyName: 'Foo' });
        const p3 = new lib_1.Policy(stack, 'P3'); // uses logicalID as name
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        p1.attachToUser(user);
        p1.attachToGroup(group);
        p1.attachToRole(role);
        // try to attach p2 to all of these and expect to fail
        expect(() => p2.attachToUser(user)).toThrow(/A policy named "Foo" is already attached/);
        expect(() => p2.attachToGroup(group)).toThrow(/A policy named "Foo" is already attached/);
        expect(() => p2.attachToRole(role)).toThrow(/A policy named "Foo" is already attached/);
        p3.attachToUser(user);
        p3.attachToGroup(group);
        p3.attachToRole(role);
    });
    test('fails if "forced" policy is not attached to a principal', () => {
        new lib_1.Policy(stack, 'MyPolicy', { force: true });
        expect(() => app.synth()).toThrow(/attached to at least one principal: user, group or role/);
    });
    test("generated policy name is the same as the logical id if it's shorter than 128 characters", () => {
        createPolicyWithLogicalId(stack, 'Foo');
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            'PolicyName': 'Foo',
        });
    });
    test('generated policy name only uses the last 128 characters of the logical id', () => {
        const logicalId128 = 'a' + dup(128 - 2) + 'a';
        const logicalIdOver128 = 'PREFIX' + logicalId128;
        createPolicyWithLogicalId(stack, logicalIdOver128);
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            'PolicyName': logicalId128,
        });
        function dup(count) {
            let r = '';
            for (let i = 0; i < count; ++i) {
                r += 'x';
            }
            return r;
        }
    });
    test('force=false, dependency on empty Policy never materializes', () => {
        // GIVEN
        const pol = new lib_1.Policy(stack, 'Pol', { force: false });
        const res = new core_1.CfnResource(stack, 'Resource', {
            type: 'Some::Resource',
        });
        // WHEN
        res.node.addDependency(pol);
        // THEN
        expect(stack).toMatchTemplate({
            Resources: {
                Resource: {
                    Type: 'Some::Resource',
                },
            },
        });
    });
    test('force=false, dependency on attached and non-empty Policy can be taken', () => {
        // GIVEN
        const pol = new lib_1.Policy(stack, 'Pol', { force: false });
        pol.addStatements(new lib_1.PolicyStatement({
            actions: ['s3:*'],
            resources: ['*'],
        }));
        pol.attachToUser(new lib_1.User(stack, 'User'));
        const res = new core_1.CfnResource(stack, 'Resource', {
            type: 'Some::Resource',
        });
        // WHEN
        res.node.addDependency(pol);
        // THEN
        expect(stack).toHaveResource('Some::Resource', {
            Type: 'Some::Resource',
            DependsOn: ['Pol0FE9AD5D'],
        }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('empty policy is OK if force=false', () => {
        new lib_1.Policy(stack, 'Pol', { force: false });
        app.synth();
        // If we got here, all OK
    });
    test('reading policyName forces a Policy to materialize', () => {
        const pol = new lib_1.Policy(stack, 'Pol', { force: false });
        Array.isArray(pol.policyName);
        expect(() => app.synth()).toThrow(/must contain at least one statement/);
    });
    test('fails if policy document is invalid', () => {
        new lib_1.Policy(stack, 'MyRole', {
            statements: [new lib_1.PolicyStatement({
                    actions: ['*'],
                    principals: [new lib_1.ServicePrincipal('test.service')],
                })],
        });
        expect(() => app.synth()).toThrow(/A PolicyStatement used in an identity-based policy cannot specify any IAM principals/);
    });
});
function createPolicyWithLogicalId(stack, logicalId) {
    const policy = new lib_1.Policy(stack, logicalId);
    const cfnPolicy = policy.node.defaultChild;
    cfnPolicy.overrideLogicalId(logicalId); // force a particular logical ID
    // add statements & principal to satisfy validation
    policy.addStatements(new lib_1.PolicyStatement({
        actions: ['*'],
        resources: ['*'],
    }));
    policy.attachToRole(new lib_1.Role(stack, 'Role', { assumedBy: new lib_1.AnyPrincipal() }));
}
//# sourceMappingURL=data:application/json;base64,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