"""
kegg_search_pathway

Search KEGG pathways by keyword. Returns pathway IDs and descriptions matching the search term.
"""

from typing import Any, Optional, Callable
from ._shared_client import get_shared_client


def kegg_search_pathway(
    keyword: str,
    *,
    stream_callback: Optional[Callable[[str], None]] = None,
    use_cache: bool = False,
    validate: bool = True,
) -> dict[str, Any]:
    """
    Search KEGG pathways by keyword. Returns pathway IDs and descriptions matching the search term.

    Parameters
    ----------
    keyword : str
        Search keyword for pathway names or descriptions (e.g., 'diabetes', 'metaboli...
    stream_callback : Callable, optional
        Callback for streaming output
    use_cache : bool, default False
        Enable caching
    validate : bool, default True
        Validate parameters

    Returns
    -------
    dict[str, Any]
    """
    # Handle mutable defaults to avoid B006 linting error

    return get_shared_client().run_one_function(
        {"name": "kegg_search_pathway", "arguments": {"keyword": keyword}},
        stream_callback=stream_callback,
        use_cache=use_cache,
        validate=validate,
    )


__all__ = ["kegg_search_pathway"]
