"""
Pydantic base model for MAML version 1.0
"""

# generated by datamodel-codegen:
# filename:  v1p0.json
# timestamp: 2025-09-16T01:55:03+00:00
# datamodel-codegen --input schemas/v1p0.json --input-file-type jsonschema --output model_v1p0.py

from __future__ import annotations

from datetime import date as date_aliased
from typing import List, Optional, Union

from pydantic import BaseModel, Field, field_validator, model_validator
from astropy.io.votable.ucd import check_ucd

from .date_funcs import today


class DOIEntry(BaseModel):
    """
    DOI field that holds the DOI and the type of DOI.
    """

    DOI: str = Field(..., description="Valid DOI")
    type: str = Field(..., description="Type of DOI")


class DependEntry(BaseModel):
    """
    Depends field that tells you what tables the meta data depends on.
    """

    survey: Optional[str] = Field(None, description="The name of the dependent survey.")
    dataset: Optional[str] = Field(
        None, description="The name of the dependent dataset."
    )
    table: str = Field(..., description="The name of the dependent table.")
    version: Optional[Union[str, float]] = Field(
        None, description="The version of the dependent table."
    )


class QCEntry(BaseModel):
    """
    Quality Control field that allows for min/max checks during data validation.
    """

    min: Optional[Union[float, str]] = None
    max: Optional[Union[float, str]] = None
    miss: Optional[Union[float, str]] = None


class FieldEntry(BaseModel):
    """
    Fields field describing the columns of the table.
    """

    name: str = Field(..., description="Required field name")
    unit: Optional[str] = Field(None, description="Recommended unit of measurement")
    info: Optional[str] = Field(None, description="Recommended short description")
    ucd: Optional[Union[str, List[str]]] = Field(
        None, description="Recommended UCD string"
    )
    data_type: str = Field(
        ..., description="Required data type (e.g., int32, string, bool, double)"
    )
    array_size: Optional[Union[int, str]] = Field(
        None, description="Optional max length of string"
    )
    qc: Optional[QCEntry] = Field(
        None, description="Optional quality control parameters"
    )

    @field_validator("ucd")
    @classmethod
    def validate_ucd(cls, value, info):
        """Checks that the ucds are valid ucds."""
        if isinstance(value, list):
            ucd_string = ";".join(value)
        else:
            ucd_string = value
        if value is not None and not check_ucd(
            ucd_string, check_controlled_vocabulary=True
        ):
            raise ValueError(
                f"{ucd_string} is not valid UCD in field {info.data['name']}"
            )
        return value


class V1P0(BaseModel):
    """Pydantic Base model for version 1.0"""

    class Config:
        """Setting configuration. Not allowing extra fields"""

        extra = "forbid"

    survey: Optional[str] = Field(default=None, description="Optional survey name")
    dataset: Optional[str] = Field(None, description="Recommended dataset name")
    table: str = Field(..., description="Required table name")
    version: Union[str, float] = Field(
        ..., description="Required version (string, integer, or float)"
    )
    date: date_aliased = Field(
        ..., description="Required date in YYYY-MM-DD format (ISO-8601)"
    )
    author: str = Field(..., description="Required lead author name and <email>")
    coauthors: Optional[Union[str, List[str]]] = Field(
        None, description="Optional coauthor name and optionally <email>"
    )
    DOIs: Optional[List[DOIEntry]] = None
    depends: Optional[List[DependEntry]] = None
    description: Optional[str] = Field(
        None, description="Recommended short description of the table"
    )
    comments: Optional[Union[str, List[str]]] = Field(
        None, description="Optional comment or interesting fact"
    )
    license: Optional[str] = Field(None, description="Recommended license for the data")
    keywords: Optional[Union[str, List[str]]] = Field(
        None, description="Optional keyword tag"
    )
    MAML_version: float = Field(
        1.0, literal=True, description="Optional version of the MAML schema"
    )
    fields: List[FieldEntry]

    @classmethod
    def with_defaults(cls) -> "V1P0":
        """Factory that builds valid blank defaults."""
        return cls(
            table="__REQUIRED__: Table Name",
            version="__REQUIRED__: 0.1.0",
            date=today(),
            author="__REQUIRED__: Main Author",
            fields=[
                FieldEntry(
                    name="__REQUIRED__: field name", data_type="__REQUIRED__: data_type"
                )
            ],
        )

    @model_validator(mode="before")
    @classmethod
    def check_explicit_nones(cls, values):
        """Reject explicitly set None values for optional fields."""
        optional_fields = [
            "survey",
            "dataset",
            "description",
            "comments",
            "license",
            "keywords",
            "coauthors",
            "DOIs",
            "depends",
            "keyarray",
            "extra",
        ]

        for field in optional_fields:
            # Check if field exists in input AND is explicitly None
            if field in values and values[field] is None:
                raise ValueError(
                    f"{field} cannot be explicitly set to None. Omit the field instead."
                )

        return values
