# Agent Onboarding Cookbook

This guide provides step-by-step workflows for AI agents to coordinate effectively using MCP Agent Mail.

## About This Guide

**Example Format**: Code examples in this guide use simplified pseudo-JSON for clarity. When using MCP tools, your MCP client library will handle the actual JSON-RPC wire protocol. Focus on understanding the tool names, parameters, and workflows shown here.

**Terminology**:
- `{project_key}` can be either the **human_key** (e.g., `/data/projects/my-app`) or the **slug** (e.g., `my-app-abc123`). Both work interchangeably.
- Agent names are unique identifiers within a project (e.g., "BackendDev", "FrontendDev").

## Quick Start: Basic Coordination Workflow

### Step 1: Ensure Your Project Exists

Before doing anything, make sure your project is registered:

```json
{
  "tool": "ensure_project",
  "arguments": {
    "human_key": "/data/projects/my-project"
  }
}
```

**What this does:**
- Creates the project if it doesn't exist
- Initializes the Git archive for message storage
- Returns project metadata (slug, created_at, etc.)

**Tips:**
- Use an absolute path as the `human_key`
- The same `human_key` should be used consistently across all agents working on the project
- For monorepos with frontend/backend, use the same project for both (see "Project Boundaries" below)

### Step 2: Register Your Agent Identity

Create your unique identity in the project:

```json
{
  "tool": "register_agent",
  "arguments": {
    "project_key": "/data/projects/my-project",
    "program": "claude-code",
    "model": "sonnet-4.5",
    "name": "BackendDev",
    "task_description": "Backend API development"
  }
}
```

**What this does:**
- Registers you as an agent in the project
- Creates your mailbox (inbox/outbox)
- Updates your `last_active_ts` timestamp
- Writes your profile to Git at `agents/<Name>/profile.json`

**Tips:**
- `name` is optional - if omitted, a memorable name like "BlueLakeA" is auto-generated
- If you use the same `name` again, it updates your profile (doesn't create a duplicate)
- Choose descriptive names like "FrontendDev", "DatabaseAdmin", not generic ones like "Claude" or "ubuntu"

### Step 3: Discover Other Agents

Find out who else is working on the project:

**Method 1: Using the dedicated agent directory (RECOMMENDED)**

```json
{
  "resource": "resource://agents/my-project-slug",
  "method": "read"
}
```

**Method 2: Using the project resource**

```json
{
  "resource": "resource://project/my-project-slug",
  "method": "read"
}
```

**Response format:**
```json
{
  "project": {
    "slug": "my-project-abc123",
    "human_key": "/data/projects/my-project"
  },
  "agents": [
    {
      "name": "BackendDev",
      "program": "claude-code",
      "model": "sonnet-4.5",
      "task_description": "Backend API development",
      "inception_ts": "2025-10-25T10:00:00+00:00",
      "last_active_ts": "2025-10-25T10:30:00+00:00",
      "unread_count": 3
    },
    {
      "name": "FrontendDev",
      "program": "codex-cli",
      "model": "gpt5-codex",
      "task_description": "React UI components",
      "inception_ts": "2025-10-25T09:00:00+00:00",
      "last_active_ts": "2025-10-25T10:15:00+00:00",
      "unread_count": 0
    }
  ]
}
```

**Important notes:**
- Agent names shown here (e.g., "BackendDev", "FrontendDev") are the names to use in tools
- These are NOT the same as your program name or user name
- `unread_count` shows how many unread messages each agent has (useful for knowing if they're checking messages)
- `last_active_ts` shows when the agent was last active

### Step 4: Check Your Inbox

See if anyone has sent you messages:

```json
{
  "tool": "fetch_inbox",
  "arguments": {
    "project_key": "/data/projects/my-project",
    "agent_name": "BackendDev",
    "include_bodies": true,
    "limit": 20
  }
}
```

**Filtering options:**
- `urgent_only: true` - Only show high-priority messages
- `since_ts: "2025-10-25T10:00:00+00:00"` - Only messages after this timestamp
- `limit: 20` - Max number of messages to return

**Response:**
```json
[
  {
    "id": 123,
    "subject": "API endpoint design",
    "from": "FrontendDev",
    "to": ["BackendDev"],
    "created_ts": "2025-10-25T10:15:00+00:00",
    "importance": "normal",
    "ack_required": false,
    "body_md": "Can we discuss the /api/users endpoint?",
    "thread_id": "msg-123"
  }
]
```

## Common Workflows

### Workflow 1: Sending a Message

```json
{
  "tool": "send_message",
  "arguments": {
    "project_key": "/data/projects/my-project",
    "sender_name": "BackendDev",
    "to": ["FrontendDev"],
    "subject": "API endpoint ready",
    "body_md": "The /api/users endpoint is now available. See docs at `/docs/api.md`.",
    "importance": "normal",
    "ack_required": false
  }
}
```

**Best practices:**
- Keep subjects concise and specific (≤ 80 characters)
- Use Markdown for formatting in `body_md`
- Only request acknowledgement (`ack_required: true`) when you need confirmation
- Use `importance: "high"` or `"urgent"` sparingly

### Workflow 2: Replying to a Message

```json
{
  "tool": "reply_message",
  "arguments": {
    "project_key": "/data/projects/my-project",
    "message_id": 123,
    "sender_name": "BackendDev",
    "body_md": "Great question! The endpoint returns JSON with `{users: [...], total: int}`."
  }
}
```

**What this does:**
- Automatically sets `to` to the original sender
- Preserves the thread (uses original `thread_id` or creates one)
- Adds "Re:" prefix to subject if not already present
- Inherits `importance` and `ack_required` from original message

### Workflow 3: Requesting Contact Approval (OPTIONAL - NOT REQUIRED)

**NOTE: Contact approval is no longer required. Agents can send messages directly using `send_message` without requesting contact first.**

This workflow is maintained for backward compatibility. If you want to use it for explicit contact tracking:

```json
{
  "tool": "request_contact",
  "arguments": {
    "project_key": "/data/projects/my-project",
    "from_agent": "BackendDev",
    "to_agent": "DatabaseAdmin",
    "reason": "Need to coordinate database migration"
  }
}
```

**Note:** Contact policies are no longer enforced by default. All agents can message each other directly.

### Workflow 3.5: Discovering Related Projects

#### What You'll See

The web dashboard intelligently identifies projects that might be related—like `/data/projects/my-app-backend` and `/data/projects/my-app-frontend`. These appear as **suggestion cards** on the Projects page with:

- 🎯 **Confidence scores** (how likely they're related)
- 💬 **AI explanations** (why they might belong together)
- ✅ **Confirm Link** button (accept the relationship)
- ✖️ **Dismiss** button (hide the suggestion)

#### How It Works

The system analyzes multiple signals:

1. **Pattern matching**: Compares project names and directory structures
2. **AI analysis** (when enabled): Reads `README.md`, `AGENTS.md`, and other docs to understand each project's purpose
3. **Smart ranking**: Orders suggestions by confidence with clear rationales

#### Important: Discovery ≠ Authorization

> 💡 **Key Concept**: Confirming a sibling link updates your **UI navigation**, not your **messaging permissions**.

**What happens when you confirm:**
- ✅ Both projects show interactive badges for quick navigation
- ✅ You can easily jump between related codebases
- ❌ Agents **cannot** automatically message across projects

**Why the separation?**

Agent Mail uses **agent-centric routing** — every message delivery requires explicit permission:

```
Agent A sends message → System finds Agent B → Checks AgentLink → ✓ Delivers or ✗ Blocks
```

This ensures:
- **Security**: No surprise cross-project deliveries
- **Transparency**: Clear audit trail of who can message whom
- **Control**: You approve every communication path

**Why not auto-authorize with AI?**
If we let the LLM automatically grant messaging permissions based on project similarity, we'd:
- Risk misrouting messages to unintended recipients
- Bypass your contact policies without oversight
- Create hidden routing paths that are hard to audit
- Potentially connect wrong projects with similar names

Instead, we split the problem:
- **Discovery** (AI-powered): "These projects look related" → Safe, read-only
- **Authorization** (human/agent-controlled): "Agent A can message Agent B" → Explicit approval required

#### The Complete Flow

```
┌─────────────────────────────────────────────────────────────┐
│ 1. System suggests relationship (AI discovers patterns)     │
│    "my-app-frontend" ↔ "my-app-backend"                     │
└────────────────┬────────────────────────────────────────────┘
                 ↓
┌─────────────────────────────────────────────────────────────┐
│ 2. You confirm in UI (one-click acceptance)                 │
│    → Badges appear on both project cards                    │
└────────────────┬────────────────────────────────────────────┘
                 ↓
┌─────────────────────────────────────────────────────────────┐
│ 3. Messages flow (agents can communicate directly)          │
│    send_message with cross-project addressing               │
└─────────────────────────────────────────────────────────────┘
```

**NOTE:** Contact approval is no longer required. Steps 3-4 in the old workflow (request_contact/respond_contact) are now optional. Agents can message each other directly after project discovery.

**Think of it like your phone's contact list**:
- Discovery = "People you may know" suggestions
- UI confirmation = Visual organization (not authorization)

#### Your Next Steps

When you see a sibling suggestion you agree with:

1. **Confirm the link** in the UI (updates navigation badges for better organization)
2. **Agents can message directly** using cross-project addressing in `send_message`:
   ```json
   {
     "tool": "send_message",
     "arguments": {
       "project_key": "/data/projects/my-app-frontend",
       "sender_name": "FrontendDev",
       "to": ["project:my-app-backend#BackendDev"],
       "subject": "Need to coordinate API changes",
       "body_md": "Let's discuss the upcoming API updates..."
     }
   }
   ```

**NOTE:** No contact approval required. The `to` field can include cross-project addresses like `project:<slug>#<AgentName>`.

### Workflow 4: Reserving Files Before Editing

Signal your intent to edit files to avoid conflicts:

```json
{
  "tool": "file_reservation_paths",
  "arguments": {
    "project_key": "/data/projects/my-project",
    "agent_name": "BackendDev",
    "paths": ["app/api/*.py", "tests/test_api.py"],
    "ttl_seconds": 7200,  // 2 hours (default is 3600 if omitted)
    "exclusive": true,
    "reason": "Refactoring API endpoints"
  }
}
```

**Best practices:**
- Reserve specific paths, not broad globs like `**/*`
- Set realistic TTL (time to live) - default is 1 hour (3600s)
- Use `exclusive: true` when you need write access
- Use `exclusive: false` for read-only observation
- Release reservations when done with `release_file_reservations()`

### Workflow 5: Searching Messages

Find messages by full-text search:

```json
{
  "tool": "search_messages",
  "arguments": {
    "project_key": "/data/projects/my-project",
    "query": "\"API design\" AND migration",
    "limit": 50
  }
}
```

**Search syntax (SQLite FTS5):**
- Phrase: `"API design"`
- Prefix: `migrat*`
- Boolean: `plan AND users`
- Require urgent: `urgent AND deployment`

### Workflow 6: Thread Summaries

Get an overview of a long discussion:

```json
{
  "tool": "summarize_thread",
  "arguments": {
    "project_key": "/data/projects/my-project",
    "thread_id": "msg-123",
    "include_examples": true
  }
}
```

**Response:**
```json
{
  "thread_id": "msg-123",
  "summary": {
    "participants": ["BackendDev", "FrontendDev", "DatabaseAdmin"],
    "key_points": [
      "Discussed new API endpoint design",
      "Agreed on JSON response format",
      "Identified need for database migration"
    ],
    "action_items": [
      "BackendDev: Implement /api/users endpoint",
      "DatabaseAdmin: Create migration script",
      "FrontendDev: Update UI to use new endpoint"
    ],
    "total_messages": 15
  },
  "examples": [
    {
      "id": 123,
      "subject": "API endpoint design",
      "from": "FrontendDev",
      "created_ts": "2025-10-25T10:15:00+00:00"
    }
  ]
}
```

## Common Pitfalls and Solutions

### Pitfall 1: "Agent not registered" Error

**Error message:**
```
Agent 'Claude' not registered for project '/data/projects/my-project'
```

**Problem:** You tried to use a generic name like "Claude", "ubuntu", or assumed an agent name exists.

**Solution:**
1. Use `resource://agents/{project}` to discover registered agent names
2. Use the actual names returned (e.g., "BackendDev", "BlueLakeA")

### Pitfall 2: "Project not found" Error

**Error message:**
```
Project 'my-project' not found
```

**Problem:** The project doesn't exist yet.

**Solution:**
1. Call `ensure_project(human_key="/data/projects/my-project")` first
2. Use the same `human_key` consistently

### Pitfall 3: Agents Can't See Each Other

**Problem:** Frontend agent registered in `/data/projects/frontend`, backend in `/data/projects/backend` - they can't see each other.

**Why:** Projects are isolated namespaces. Agents in different projects cannot communicate.

**Solution:**
- Use ONE shared project for both frontend and backend: `/data/projects/my-project`
- Register all agents in the same project
- See "Project Boundaries" section below

### Pitfall 4: No Messages in Inbox

**Problem:** Sent a message but the recipient's inbox is empty.

**Possible causes:**
1. Recipient name is wrong - verify with `resource://agents/{project}`
2. Wrong project key - ensure both agents use the same `project_key`
3. Message was sent but not fetched - check `since_ts` parameter in `fetch_inbox()`

### Pitfall 5: File Reservation Conflicts

**Error message:**
```
Conflict: 'FrontendDev' holds exclusive reservation on 'app/api/*.py' until 2025-10-25T12:00:00
```

**Problem:** Another agent has reserved the files you want to edit.

**Solutions:**
1. Wait for the reservation to expire
2. Coordinate with the holder via messages
3. Use `resource://file_reservations/{project}?active_only=true` to see all active reservations
4. Reserve more specific paths to avoid overlaps

## Project Boundaries

### Single Project (Recommended for Monorepos)

Use this approach when frontend and backend are part of the same codebase:

```
Project: /data/projects/smartedgar
Agents:
  - FrontendDev (works on React app)
  - BackendDev (works on FastAPI)
  - DatabaseAdmin (works on migrations)
  - DevOpsEngineer (works on CI/CD)
```

**Pros:**
- Simple setup
- Agents can easily coordinate
- Shared message threads
- File reservations work across all code

**Cons:**
- Shared namespace (all agents see all messages)

### Multiple Projects (Currently Limited)

**IMPORTANT:** Cross-project coordination is NOT currently supported. Agents in different projects cannot see or message each other.

**Future Feature:** Cross-project messaging and agent links are planned for a future release.

**Current Workaround:** Use a single shared project for all related agents.

## Advanced Features

### Macro: Start Session

Bootstrap a project session in one call:

```json
{
  "tool": "macro_start_session",
  "arguments": {
    "human_key": "/data/projects/my-project",
    "program": "claude-code",
    "model": "sonnet-4.5",
    "agent_name": "BackendDev",
    "task_description": "Backend API development",
    "reserve_paths": ["app/api/*.py"],
    "inbox_limit": 10
  }
}
```

**What this does:**
1. Ensures project exists
2. Registers your agent
3. Reserves specified file paths
4. Fetches your inbox
5. Returns all results in one call

### Macro: Prepare Thread

Align with an existing discussion thread:

```json
{
  "tool": "macro_prepare_thread",
  "arguments": {
    "project_key": "/data/projects/my-project",
    "thread_id": "msg-123",
    "program": "claude-code",
    "model": "sonnet-4.5",
    "agent_name": "BackendDev"
  }
}
```

**What this does:**
1. Registers your agent if needed
2. Summarizes the thread
3. Fetches recent inbox
4. Returns context to jump into the discussion


## Getting Help

If you encounter issues:

1. **Check error messages** - They now include helpful details about what went wrong
2. **Verify agent names** - Use `resource://agents/{project}` to see registered names
3. **Check project boundaries** - Ensure all agents use the same `project_key`
4. **Review file reservations** - Use `resource://file_reservations/{project}?active_only=true` to see conflicts
5. **Search messages** - Use `search_messages()` to find related discussions

## Complete Example: Two Agents Coordinating

**Agent 1: BackendDev (starting fresh)**

```json
// Step 1: Ensure project
{"tool": "ensure_project", "arguments": {"human_key": "/data/projects/smartedgar"}}
// Returns: {"slug": "smartedgar-abc123", "human_key": "/data/projects/smartedgar", ...}

// Step 2: Register
{"tool": "register_agent", "arguments": {
  "project_key": "/data/projects/smartedgar",  // Can use human_key or slug
  "program": "claude-code",
  "model": "sonnet-4.5",
  "name": "BackendDev",
  "task_description": "API development"
}}

// Step 3: Discover others
{"resource": "resource://agents/smartedgar-abc123"}  // Use slug from Step 1 response
// Response: {"agents": [{"name": "FrontendDev", ...}, {"name": "BackendDev", ...}]}

// Step 4: Send message to FrontendDev
{"tool": "send_message", "arguments": {
  "project_key": "/data/projects/smartedgar",
  "sender_name": "BackendDev",
  "to": ["FrontendDev"],
  "subject": "API endpoint ready",
  "body_md": "The /api/users endpoint is ready for integration."
}}

// Step 5: Reserve files before editing
{"tool": "file_reservation_paths", "arguments": {
  "project_key": "/data/projects/smartedgar",
  "agent_name": "BackendDev",
  "paths": ["app/api/users.py"],
  "ttl_seconds": 3600,
  "exclusive": true,
  "reason": "Adding pagination"
}}
```

**Agent 2: FrontendDev (checking inbox)**

```json
// Step 1-2: Ensure and register (same as above)

// Step 3: Check inbox
{"tool": "fetch_inbox", "arguments": {
  "project_key": "/data/projects/smartedgar",
  "agent_name": "FrontendDev",
  "include_bodies": true
}}
// Response: [{"id": 456, "subject": "API endpoint ready", "from": "BackendDev", ...}]

// Step 4: Reply
{"tool": "reply_message", "arguments": {
  "project_key": "/data/projects/smartedgar",
  "message_id": 456,
  "sender_name": "FrontendDev",
  "body_md": "Great! I'll update the UI to use the new endpoint."
}}

// Step 5: Reserve frontend files
{"tool": "file_reservation_paths", "arguments": {
  "project_key": "/data/projects/smartedgar",
  "agent_name": "FrontendDev",
  "paths": ["src/components/UserList.tsx"],
  "ttl_seconds": 3600,
  "exclusive": true,
  "reason": "Integrating new API endpoint"
}}
```

## Resources Reference

- `resource://agents/{project_key}` - List agents in a project (RECOMMENDED for discovery)
- `resource://project/{project_key}` - Project details including agents
- `resource://projects` - List all projects
- `resource://file_reservations/{project_key}?active_only=true` - Active file reservations
- `resource://inbox/{project_key}/{agent_name}` - Agent's inbox
- `resource://outbox/{project_key}/{agent_name}` - Agent's sent messages
- `resource://message/{message_id}` - Single message details

## Next Steps

1. **Try the Quick Start workflow** above to get familiar with basic operations
2. **Experiment with messaging** - Send messages between agents
3. **Practice reserving files** - Use file reservations to signal editing intent
4. **Explore thread summaries** - Use `summarize_thread()` for long discussions
5. **Set up contact policies (optional)** - Configure how agents can reach you (not enforced by default)

For more details, see:
- [ROOT_CAUSE_ANALYSIS.md](./ROOT_CAUSE_ANALYSIS.md) - Understanding the system architecture
- [README.md](./README.md) - Full system documentation
- Tool descriptions in `src/mcp_agent_mail/app.py` - Complete API reference
