# -*- coding: utf-8 -*-
"""
地震情報取得スクリプト

EarthquakeDataProcessorを使用して
地震情報を取得・処理し、Redisに格納します。

使用方法:
    python get_earthquake.py
"""

import json
import sys
import os
from pathlib import Path

# パスを追加して直接実行にも対応
sys.path.insert(
    0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
)

from WIPServerPy.data.controllers.earthquake_data_processor import (
    EarthquakeDataProcessor,
)
from WIPServerPy.data.redis_manager import create_redis_manager

JSON_DIR = Path(__file__).resolve().parent.parent / "json"


def main():
    """
    地震情報処理のメイン関数

    EarthquakeDataProcessorを使用して地震情報を取得し、
    エリアコード変換、時間統合を行います。
    """
    print("=== 地震情報取得開始 ===")

    # EarthquakeDataProcessorのインスタンスを作成
    processor = EarthquakeDataProcessor()

    # 地震情報処理を実行
    try:
        # Step 1: XMLファイルリストの取得
        url_list = processor.get_earthquake_xml_list()
        if not url_list:
            print("No URLs found. Exiting.")
            return

        print(f"Found {len(url_list)} earthquake XML files to process.")

        # Step 2: 地震情報の取得・統合
        json_result = processor.get_earthquake_info(url_list)
        print("\n=== Earthquake Info Processing Complete ===")
        print(f"Debug: json_result type: {type(json_result)}")

        # Step 3: JSONデータの解析
        try:
            result_dict = json.loads(json_result)
        except json.JSONDecodeError as e:
            print(f"Error decoding JSON: {e}")
            print(f"Problematic JSON: {json_result}")
            return  # 処理を中断

        # Step 4: エリアコードデータの読み込み
        with open(JSON_DIR / "area_codes.json", "r", encoding="utf-8") as f:
            area_codes_data = json.load(f)
        print(f"Debug: area_codes_data type: {type(area_codes_data)}")
        print(f"Debug: area_codes_data is None: {area_codes_data is None}")

        # Step 5: エリアコード変換・統合処理
        converted_data, converted_report_times = (
            processor.convert_earthquake_keys_to_area_codes(
                result_dict, area_codes_data
            )
        )
        print(f"Debug: converted_data type: {type(converted_data)}")
        print(f"Debug: converted_data is None: {converted_data is None}")
        print(f"Debug: converted_report_times type: {type(converted_report_times)}")
        print(
            f"Debug: converted_report_times is None: {converted_report_times is None}"
        )

        # Step 6: 最終結果を新しいフォーマットで保存（ReportDateTime付き）
        final_formatted_data = processor.format_to_alert_style(
            converted_data, converted_report_times, area_codes_data
        )

        # 処理完了 - JSONファイル保存は削除

        print("=== 地震情報取得完了 ===")
        print("Processing completed successfully.")

        # Redis管理クラスを使用してデータを更新
        print("\n=== Redisデータ更新開始 ===")

        try:
            # Redis管理クラスのインスタンスを作成
            redis_manager = create_redis_manager(debug=True)

            # 地震情報を更新
            result = redis_manager.update_earthquakes(final_formatted_data)

            # 結果を表示
            print(f"\n=== Redis更新結果 ===")
            print(f"更新されたエリア: {result['updated']}件")
            print(f"新規作成されたエリア: {result['created']}件")
            print(f"エラー: {result['errors']}件")
            print(f"合計処理エリア: {result['updated'] + result['created']}件")

            # 接続を閉じる
            redis_manager.close()

            print("=== Redisデータ更新完了 ===")

        except Exception as e:
            print(f"Redis更新エラー: {e}")

    except Exception as e:
        print(f"Error in earthquake processing: {e}")


