"""
Created on 2023-09-10

@author: wf
"""

import webbrowser
from argparse import ArgumentParser
from datetime import datetime

from basemkit.base_cmd import BaseCmd

from ngwidgets.webserver import WebserverConfig


class WebserverCmd(BaseCmd):
    """
    Baseclass for command line handling of Webservers
    """

    def __init__(self, config: WebserverConfig, webserver_cls, debug: bool = False):
        """
        constructor
        """
        super().__init__(config.version, config.version.description)
        self.config = config
        self.version = config.version
        self.debug = debug
        self.webserver_cls = webserver_cls

    def add_arguments(self, parser: ArgumentParser):
        """
        Add webserver-specific CLI arguments.

        Args:
            parser (ArgumentParser): the parser to add arguments to.
        """
        super().add_arguments(parser)
        parser.add_argument(
            "--apache", help="create an apache configuration file for the given domain"
        )
        parser.add_argument("-c", "--client", action="store_true", help="start client")
        parser.add_argument(
            "-l",
            "--local",
            action="store_true",
            help="run with local file system access",
        )
        parser.add_argument("-i", "--input", help="input file")
        parser.add_argument(
            "-rol", "--render_on_load", action="store_true", help="render on load"
        )
        parser.add_argument(
            "--host",
            default="localhost",
            help="the host to serve / listen from (default: localhost)",
        )
        parser.add_argument(
            "--port",
            type=int,
            default=self.config.default_port,
            help=f"the port to serve from (default: {self.config.default_port})",
        )
        parser.add_argument(
            "-s", "--serve", action="store_true", help="start webserver"
        )

    def handle_args(self, args) -> bool:
        """
        Handle the given command line arguments.

        Args:
            args: The parsed command line arguments.

        Returns:
            bool: True if any argument was handled, False otherwise.
        """
        handled = super().handle_args(args)
        if args.apache:
            print(self.to_apache_config(self.config, args.apache))
            handled = True
        if args.client:
            url = f"http://{args.host}:{args.port}"
            webbrowser.open(url)
            handled = True
        if args.serve:
            ws = self.webserver_cls()
            ws.run(args)
            handled = True
        return handled

    def to_apache_config(self, config: WebserverConfig, domain: str) -> str:
        """
        Generate Apache configuration based on the given WebserverConfig.

        Args:
            config (WebserverConfig): The webserver configuration object.
            domain(str): the base domain to use
        Returns:
            str: The Apache configuration as a string.
        """
        iso_timestamp = datetime.now().isoformat()
        server_name = f"{config.short_name}.{domain}"
        admin_email = f"webmaster@{domain}"
        version_info = ""
        if config.version:
            version_info = f"""{config.version.name} Version {config.version.version} of {config.version.updated} ({config.version.description})"""

        header_comment = f"""# Apache Configuration for {server_name}
# {version_info}
# Generated by WebserverCmd at {iso_timestamp}
# http Port: {config.default_port}
# SSL Port: 443
# {config.copy_right}
# timeout: {config.timeout}
"""

        template = """<VirtualHost *:{port}>
    ServerName {server_name}
    ServerAdmin {admin_email}

    {ssl_config_part}
    ErrorLog ${{APACHE_LOG_DIR}}/{short_name}_error{log_suffix}.log
    CustomLog ${{APACHE_LOG_DIR}}/{short_name}{log_suffix}.log combined

    RewriteEngine On
    RewriteCond %{{HTTP:Upgrade}} =websocket [NC]
    RewriteRule /(.*) ws://localhost:{default_port}/$1 [P,L]
    RewriteCond %{{HTTP:Upgrade}} !=websocket [NC]
    RewriteRule /(.*) http://localhost:{default_port}/$1 [P,L]

    ProxyPassReverse / http://localhost:{default_port}/
</VirtualHost>
"""

        # For SSL Configuration
        ssl_config = template.format(
            port=443,
            server_name=server_name,
            admin_email=admin_email,
            short_name=config.short_name,
            log_suffix="_ssl",
            default_port=config.default_port,
            ssl_config_part="Include ssl.conf",
        )

        # For Non-SSL Configuration
        http_config = template.format(
            port=80,
            server_name=server_name,
            admin_email=admin_email,
            short_name=config.short_name,
            log_suffix="",
            default_port=config.default_port,
            ssl_config_part="",
        )

        apache_config = header_comment + ssl_config + http_config
        return apache_config

    def cmd_main(self, argv: list = None) -> int:
        """
        Main entry point (legacy contract).
        """
        exit_code = self.run(argv)
        return exit_code

    @classmethod
    def main(cls, config: WebserverConfig, webserver_cls, argv=None) -> int:
        """
        Main entry point to launch the webserver cmd with BaseCmd logic.
        """
        instance = cls(config=config, webserver_cls=webserver_cls)
        return instance.run(argv)
