# SPDX-FileCopyrightText: PyPSA Contributors
#
# SPDX-License-Identifier: MIT

import pandas as pd
import pytest


@pytest.fixture
def network(ac_dc_network):
    return ac_dc_network  # Change scope of existing fixture to function


@pytest.fixture
def network_add_snapshots(network):
    # Single dimension
    n = network.copy()
    snapshots = pd.date_range("2015-01-01", "2015-01-02", freq="h")
    n.set_snapshots(snapshots)
    assert n.snapshots.equals(snapshots)
    assert n.snapshots.names == ["snapshot"]  # TODO: Should be changed
    return n


@pytest.fixture
def network_add_snapshots_multiindex(network):
    # Multi dimension
    n = network.copy()
    snapshots = pd.MultiIndex.from_product(
        [[2015], pd.date_range("2015-01-01", "2015-01-02", freq="h")]
    )
    n.set_snapshots(snapshots)
    assert n.snapshots.equals(snapshots)
    assert n.snapshots.names == ["period", "timestep"]
    return n


@pytest.mark.parametrize(
    "network_fixture",
    ["network", "network_add_snapshots", "network_add_snapshots_multiindex"],
)
def test_snapshot_index_consistency(request, network_fixture):
    n = request.getfixturevalue(network_fixture)
    for component in n.all_components:
        dynamic = n.c[component].dynamic
        for k in dynamic.keys():
            assert dynamic[k].index.equals(n.snapshots)


@pytest.mark.parametrize(
    "network_fixture",
    ["network_add_snapshots", "network_add_snapshots_multiindex"],
)
def test_existing_value_casting(request, network_fixture):
    n = request.getfixturevalue(network_fixture)
    base_network = request.getfixturevalue("network")
    assert not isinstance(base_network.snapshots, pd.MultiIndex)
    snapshots = base_network.snapshots
    if isinstance(n.snapshots, pd.MultiIndex):
        vals = n.c.generators.dynamic.p_max_pu.xs(2015).loc[snapshots, :]
    else:
        vals = n.c.generators.dynamic.p_max_pu.loc[snapshots, :]
    assert vals.equals(base_network.c.generators.dynamic.p_max_pu)


# @pytest.mark.parametrize("meta", [{"test": "test"}, {"test": {"test": "test"}}])
def test_set_snapshots_checks(network):
    # Don't allow time zone aware snapshots
    snapshots_tz = pd.date_range("2020-01-01", "2020-01-02", freq="h", tz="UTC")
    with pytest.raises(ValueError):
        network.set_snapshots(snapshots_tz)

    # Don't allow more than two dimensions
    snapshots_more_dims = pd.MultiIndex.from_product([[2020], snapshots_tz, ["test"]])
    with pytest.raises(ValueError):
        network.set_snapshots(snapshots_more_dims)

    # Don't allow empty snapshots
    with pytest.raises(ValueError):
        network.set_snapshots(pd.Index([]))
    with pytest.raises(ValueError):
        network.set_snapshots(pd.MultiIndex.from_arrays([[], []]))
