"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CalculatedHealthCheck = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const route53 = require("aws-cdk-lib/aws-route53");
const health_check_1 = require("./health-check");
const health_check_type_1 = require("./health-check-type");
/**
 * Create a Route53 HealthCheck that monitors other Route53 HealthChecks.
 *
 * <b>Example</b>
 * ```typescript
 * const healthCheck = new EndpointHealthCheck(stack, "HealthCheck", {
 *   domainName: "pepperize.com",
 * });
 * new CalculatedHealthCheck(stack, "CalculatedHealthCheck", {
 *   childHealthChecks: [healthCheck],
 * });
 * ```
 * @link https://docs.aws.amazon.com/de_de/AWSCloudFormation/latest/UserGuide/aws-resource-route53-healthcheck.html#aws-resource-route53-healthcheck-properties
 *
 * @resource AWS::Route53::HealthCheck
 */
class CalculatedHealthCheck extends health_check_1.HealthCheckBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.childHealthChecks = props.childHealthChecks ?? [];
        if (this.childHealthChecks.length > 256) {
            aws_cdk_lib_1.Annotations.of(this).addError("ChildHealthChecks has to be smaller than 256");
        }
        if (undefined != props.healthThreshold && props.healthThreshold > 256) {
            aws_cdk_lib_1.Annotations.of(this).addError("HealthThreshold has to be smaller than 256");
        }
        const resource = new route53.CfnHealthCheck(this, "Resource", {
            healthCheckConfig: {
                childHealthChecks: aws_cdk_lib_1.Lazy.list({
                    produce: () => this.childHealthChecks.map((healthCheck) => healthCheck.healthCheckId),
                }),
                healthThreshold: props.healthThreshold,
                type: health_check_type_1.HealthCheckType.CALCULATED,
            },
            healthCheckTags: this.tags.renderedTags,
        });
        this.healthCheckId = resource.attrHealthCheckId;
        if (props.healthCheckName) {
            aws_cdk_lib_1.Tags.of(this).add("Name", props.healthCheckName);
        }
    }
    /**
     * The number of ChildHealthChecks that are healthy that Route53 is monitoring.
     *
     * Valid statistics: Average (recommended), Minimum, Maximum
     */
    metricChildHealthCheckHealthyCount(props) {
        return this.metric("ChildHealthCheckHealthyCount", { statistic: cloudwatch.Statistic.AVERAGE, ...props });
    }
    addChildHealthCheck(healthCheck) {
        this.childHealthChecks.push(healthCheck);
    }
}
exports.CalculatedHealthCheck = CalculatedHealthCheck;
_a = JSII_RTTI_SYMBOL_1;
CalculatedHealthCheck[_a] = { fqn: "@pepperize/cdk-route53-health-check.CalculatedHealthCheck", version: "0.0.273" };
//# sourceMappingURL=data:application/json;base64,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