"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheckerRegions = exports.Protocol = exports.EndpointHealthCheck = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const route53 = require("aws-cdk-lib/aws-route53");
const health_check_1 = require("./health-check");
const health_check_type_1 = require("./health-check-type");
/**
 * Create a Route53 HealthCheck that monitors an endpoint either by domain name or by IP address.
 *
 * <b>Example</b>
 * ```typescript
 * new EndpointHealthCheck(stack, "HealthCheck", {
 *   domainName: "pepperize.com",
 * });
 * ```
 * Generates
 * ```yaml
 * Resources:
 *   Type: AWS::Route53::HealthCheck
 *   Properties:
 *     HealthCheckConfig:
 *       FullyQualifiedDomainName: "pepperize.com"
 *       Port: 443
 *       Type: "HTTPS"
 *       EnableSNI: true
 * ```
 * @link https://docs.aws.amazon.com/de_de/AWSCloudFormation/latest/UserGuide/aws-resource-route53-healthcheck.html#aws-resource-route53-healthcheck-properties
 *
 * @resource AWS::Route53::HealthCheck
 */
class EndpointHealthCheck extends health_check_1.HealthCheckBase {
    constructor(scope, id, props) {
        super(scope, id);
        const type = this.healthCheckType(props.protocol, props.searchString);
        const port = this.defaultPort(props.port, type);
        const enableSni = this.enableSniForHttps(type, props.enableSni);
        if (!props.domainName && !props.ipAddress) {
            aws_cdk_lib_1.Annotations.of(this).addError("Either DomainName or IpAddress has to be specified");
        }
        if (props.ipAddress && !new RegExp(IP_ADDRESS_REGEX_PATTERN).test(props.ipAddress)) {
            aws_cdk_lib_1.Annotations.of(this).addError("IpAddress must be valid");
        }
        if (undefined != props.port && (props.port < 1 || props.port > 65535)) {
            aws_cdk_lib_1.Annotations.of(this).addError("Port has to be between 1 and 65535");
        }
        if (props.resourcePath && props.resourcePath.length > 255) {
            aws_cdk_lib_1.Annotations.of(this).addError("ResourcePath must be at least 255 character long");
        }
        if (props.searchString && props.searchString.length > 255) {
            aws_cdk_lib_1.Annotations.of(this).addError("SearchString must be at least 255 character long");
        }
        if (undefined != props.failureThreshold && (props.failureThreshold < 1 || props.failureThreshold > 10)) {
            aws_cdk_lib_1.Annotations.of(this).addError("FailureThreshold has to be between 10 and 30");
        }
        if (undefined != props.requestInterval && (props.requestInterval < 10 || props.requestInterval > 30)) {
            aws_cdk_lib_1.Annotations.of(this).addError("RequestInterval has to be between 10 and 30");
        }
        if (props.regions && props.regions.length < 3) {
            aws_cdk_lib_1.Annotations.of(this).addError("At least three HealthCheckerRegions have to be given");
        }
        const resource = new route53.CfnHealthCheck(this, "Resource", {
            healthCheckConfig: {
                enableSni: enableSni,
                fullyQualifiedDomainName: props.domainName,
                ipAddress: props.ipAddress,
                inverted: props.inverted,
                port: port,
                resourcePath: props.resourcePath,
                searchString: props.searchString,
                type: type,
                requestInterval: props.requestInterval,
                failureThreshold: props.failureThreshold,
                measureLatency: props.latencyGraphs,
                regions: props.regions,
            },
            healthCheckTags: this.tags.renderedTags,
        });
        this.healthCheckId = resource.attrHealthCheckId;
        const healthCheckName = props.healthCheckName || props.domainName;
        if (healthCheckName) {
            aws_cdk_lib_1.Tags.of(this).add("Name", healthCheckName);
        }
    }
    /**
     * Returns the CFN HealthCheckType for the given protocol. If undefined returns default HTTPS.
     */
    healthCheckType(protocol, searchString) {
        switch (true) {
            case Protocol.TCP == protocol:
                return health_check_type_1.HealthCheckType.TCP;
            case Protocol.HTTPS == protocol && !!searchString:
                return health_check_type_1.HealthCheckType.HTTPS_STR_MATCH;
            case Protocol.HTTPS == protocol:
                return health_check_type_1.HealthCheckType.HTTPS;
            case Protocol.HTTP == protocol && !!searchString:
                return health_check_type_1.HealthCheckType.HTTP_STR_MATCH;
            case Protocol.HTTP == protocol:
                return health_check_type_1.HealthCheckType.HTTP;
            default:
                return health_check_type_1.HealthCheckType.HTTPS;
        }
    }
    /**
     * Sets the default if undefined for HTTP and HTTPS
     */
    defaultPort(port, type) {
        switch (true) {
            case port && port > 1:
                return port;
            case health_check_type_1.HealthCheckType.HTTPS == type:
                return 443;
            case health_check_type_1.HealthCheckType.HTTPS_STR_MATCH == type:
                return 443;
            case health_check_type_1.HealthCheckType.HTTP == type:
                return 80;
            case health_check_type_1.HealthCheckType.HTTP_STR_MATCH == type:
                return 80;
            default:
                return undefined;
        }
    }
    /**
     * Enables SNI by default for HTTPS if omitted, otherwise undefined
     */
    enableSniForHttps(type, enableSni) {
        if (health_check_type_1.HealthCheckType.HTTPS != type && health_check_type_1.HealthCheckType.HTTPS_STR_MATCH != type) {
            return undefined;
        }
        if (undefined == enableSni) {
            return true;
        }
        return enableSni;
    }
    /**
     * The percentage of Route53 health checkers that report that the status of the health check is healthy
     *
     * <b>LatencyGraphs has to be enabled</b>
     *
     * Valid statistics: Average (recommended), Minimum, Maximum
     */
    metricHealthCheckPercentageHealthy(props) {
        return this.metric("HealthCheckPercentageHealthy", { statistic: cloudwatch.Statistic.AVERAGE, ...props });
    }
    /**
     * The time in milliseconds that it took Route53 health checkers to establish a TCP connection with the endpoint
     *
     * Valid statistics: Average (recommended), Minimum, Maximum
     */
    metricConnectionTime(props) {
        return this.metric("ConnectionTime", { statistic: cloudwatch.Statistic.AVERAGE, ...props });
    }
    /**
     * The time in milliseconds that it took Route53 health checkers to complete the SSL/TLS handshake
     *
     * Valid statistics: Average, Minimum, Maximum
     */
    metricSSLHandshakeTime(props) {
        return this.metric("SSLHandshakeTime", { statistic: cloudwatch.Statistic.AVERAGE, ...props });
    }
    /**
     * The time in milliseconds that it took Route53 health checkers to receive the first byte of the response to an HTTP or HTTPS request
     *
     * Valid statistics: Average (recommended), Minimum, Maximum
     */
    metricTimeToFirstByte(props) {
        return this.metric("TimeToFirstByte", { statistic: cloudwatch.Statistic.AVERAGE, ...props });
    }
}
exports.EndpointHealthCheck = EndpointHealthCheck;
_a = JSII_RTTI_SYMBOL_1;
EndpointHealthCheck[_a] = { fqn: "@pepperize/cdk-route53-health-check.EndpointHealthCheck", version: "0.0.273" };
/**
 * The protocol that Route53 uses to communicate with the endpoint.
 */
var Protocol;
(function (Protocol) {
    Protocol["HTTP"] = "HTTP";
    Protocol["HTTPS"] = "HTTPS";
    Protocol["TCP"] = "TCP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * The regions of health checker from which Route53 performs checks on the endpoint.
 */
var HealthCheckerRegions;
(function (HealthCheckerRegions) {
    HealthCheckerRegions["US_EAST_1"] = "us-east-1";
    HealthCheckerRegions["US_WEST_1"] = "us-west-1";
    HealthCheckerRegions["US_WEST_2"] = "us-west-2";
    HealthCheckerRegions["EU_WEST_1"] = "eu-west-1";
    HealthCheckerRegions["AP_SOUTHEAST_1"] = "ap-southeast-1";
    HealthCheckerRegions["AP_SOUTHEAST_2"] = "ap-southeast-2";
    HealthCheckerRegions["AP_NORTHEAST_1"] = "ap-northeast-1";
    HealthCheckerRegions["SA_EAST_1"] = "sa-east-1";
})(HealthCheckerRegions = exports.HealthCheckerRegions || (exports.HealthCheckerRegions = {}));
const IP_ADDRESS_REGEX_PATTERN = "(^((([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\\.){3}([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5]))$|^(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))$)";
//# sourceMappingURL=data:application/json;base64,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