import { ITaggable, Resource, TagManager } from "aws-cdk-lib";
import * as cloudwatch from "aws-cdk-lib/aws-cloudwatch";
import * as route53 from "aws-cdk-lib/aws-route53";
import { RecordSet } from "aws-cdk-lib/aws-route53";
import { Construct } from "constructs";
/**
 * @internal
 */
export interface HealthCheckOptions {
    /**
     * The display name of this Route53 HealthCheck
     */
    readonly healthCheckName?: string;
    /**
     * Whether to invert the status of the Route53 health check status.
     */
    readonly inverted?: boolean;
}
export interface IHealthCheck {
    readonly healthCheckId: string;
    /**
     * Sets `this.healthCheckId` as the value for `HealthCheckId` on the given RecordSet.
     *
     * <b>Applies only to alias, failover alias, geolocation alias, latency alias, and weighted alias resource record sets</b>
     *
     * @param recordSet The Route53 RecordSet to configure failover
     * @param evaluateTargetHealth Inherit the health of the referenced Alias RecordSet Target
     * @param failover Sets `PRIMARY` or `SECONDARY` as the value for `Failover` on the given RecordSet.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-route53-aliastarget.html#cfn-route53-aliastarget-evaluatetargethealth
     */
    failover(recordSet: RecordSet, evaluateTargetHealth?: boolean, failover?: Failover): void;
    /**
     * Sets `PRIMARY` as the value for `Failover` on the given RecordSet. Additionally, sets `this.healthCheckId` as the value for `HealthCheckId`.
     *
     * <b>Applies only to alias, failover alias, geolocation alias, latency alias, and weighted alias resource record sets</b>
     *
     * @param recordSet The Route53 RecordSet to configure failover
     * @param evaluateTargetHealth Inherit the health of the referenced Alias RecordSet Target
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-route53-aliastarget.html#cfn-route53-aliastarget-evaluatetargethealth
     */
    failoverPrimary(recordSet: route53.RecordSet, evaluateTargetHealth?: boolean): void;
    /**
     * Sets `PRIMARY` as the value for `Failover` on the given RecordSet. Additionally, sets `this.healthCheckId` as the value for `HealthCheckId`.
     *
     * <b>Applies only to alias, failover alias, geolocation alias, latency alias, and weighted alias resource record sets</b>
     *
     * @param recordSet The Route53 RecordSet to configure failover
     * @param evaluateTargetHealth Inherit the health of the referenced Alias RecordSet Target
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-route53-aliastarget.html#cfn-route53-aliastarget-evaluatetargethealth
     */
    failoverSecondary(recordSet: route53.RecordSet, evaluateTargetHealth?: boolean): void;
    /**
     * Return the given named metric for this HealthCheck.
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Route53 health checkers report that the HealthCheck is healthy or unhealthy.
     */
    metricHealthCheckStatus(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * @internal
 */
export declare abstract class HealthCheckBase extends Resource implements IHealthCheck, ITaggable {
    /**
     * Import an existing Route53 HealthCheck.
     */
    static fromHealthCheckId(scope: Construct, id: string, healthCheckId: string): IHealthCheck;
    abstract readonly healthCheckId: string;
    readonly tags: TagManager;
    failover(recordSet: RecordSet, evaluateTargetHealth?: boolean, failover?: Failover): void;
    failoverPrimary(recordSet: route53.RecordSet, evaluateTargetHealth?: boolean): void;
    failoverSecondary(recordSet: route53.RecordSet, evaluateTargetHealth?: boolean): void;
    /**
     * Return the given named metric for this HealthCheck.
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Route53 health checkers report that the HealthCheck is healthy or unhealthy.
     */
    metricHealthCheckStatus(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
export declare enum Failover {
    /**
     * The primary record set
     */
    PRIMARY = "PRIMARY",
    /**
     * The secondary record set
     */
    SECONDARY = "SECONDARY"
}
