from __future__ import annotations

import os
import shlex
import subprocess
import uuid
from typing import Optional, Sequence

import click
from distributed import Client
from packaging.version import Version
from rich import print

import coiled
from coiled.compatibility import DISTRIBUTED_VERSION

from .utils import CONTEXT_SETTINGS

MINIMUM_DISTRIBUTED_VERSION = Version("2023.6.0")


@click.command(context_settings=CONTEXT_SETTINGS)
@click.option(
    "--name",
    default=None,
    help="Run name. If not given, defaults to a unique name.",
)
@click.option(
    "--account",
    default=None,
    help="Coiled account (uses default account if not specified)",
)
@click.option(
    "--software",
    default=None,
    help=(
        "Software environment name to use. If neither software nor container is specified, "
        "all the currently-installed Python packages are replicated on the VM using package sync."
    ),
)
@click.option(
    "--container",
    default=None,
    help=(
        "Container image to use. If neither software nor container is specified, "
        "all the currently-installed Python packages are replicated on the VM using package sync."
    ),
)
@click.option(
    "--vm-type",
    default=[],
    multiple=True,
    help="VM type to use. Specify multiple times to provide multiple options.",
)
@click.option(
    "--gpu",
    default=False,
    is_flag=True,
    help="Have a GPU available.",
)
@click.option(
    "--file",
    "-f",
    default=[],
    multiple=True,
    help="Local files required to run command.",
)
@click.argument("command", nargs=-1)
def run(
    name: str,
    account: Optional[str],
    software: Optional[str],
    container: Optional[str],
    vm_type: Sequence[str],
    gpu: bool,
    file,
    command,
):
    """
    Run a command on the cloud.

    .. warning::

        ``coiled run`` is an experimental feature and is subject to breaking changes.

    """

    print(
        "[red]Warning:[/red] [bold]coiled run[/bold] is an experimental "
        "feature and is subject to breaking changes.\n"
    )

    if DISTRIBUTED_VERSION < MINIMUM_DISTRIBUTED_VERSION:
        # Relies on scheduler having scratch space, which was added in https://github.com/dask/distributed/pull/7802
        raise RuntimeError(
            f"`coiled run` requires distributed>={MINIMUM_DISTRIBUTED_VERSION} "
            f"(distributed={DISTRIBUTED_VERSION} is installed)"
        )

    env = None
    if container and "rapidsai" in container:
        env = {"DISABLE_JUPYTER": "true"}  # needed for "stable" RAPIDS image

    if not command:
        raise ValueError("command must be specified")

    print(f"Running [bold]{shlex.join(command)}[/bold]...")
    with coiled.Cluster(
        name=name or f"run-{uuid.uuid4().hex[:8]}",
        account=account,
        n_workers=0,
        software=software,
        container=container,
        scheduler_options={"idle_timeout": "24 hours"},
        scheduler_vm_types=list(vm_type) if vm_type else None,
        allow_ssh=True,
        environ=env,
        scheduler_gpu=gpu,
        tags={"coiled-cluster-type": "run"},
    ) as cluster:
        with Client(cluster) as client:
            # Extract and upload files from `command`
            command = shlex.split(" ".join(command))
            for idx, i in enumerate(command):
                if os.path.exists(i) and os.path.isfile(i):
                    client.upload_file(i, load=False)
                    command[idx] = os.path.basename(i)
            # Upload user-specified files too
            for f in file:
                client.upload_file(f, load=False)

            def run_command(dask_scheduler, command):
                subprocess.run(command, cwd=dask_scheduler.local_directory)

            client.run_on_scheduler(run_command, command=command)
