//! Quantum error correction codes and decoders
//!
//! This module provides implementations of various quantum error correction codes
//! including stabilizer codes, surface codes, and color codes, along with
//! efficient decoder algorithms.

use crate::error::{QuantRS2Error, QuantRS2Result};
use scirs2_core::ndarray::Array2;
use scirs2_core::Complex64;
use std::collections::HashMap;
use std::fmt;

/// Pauli operator representation
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash)]
pub enum Pauli {
    I,
    X,
    Y,
    Z,
}

impl Pauli {
    /// Get matrix representation
    pub fn matrix(&self) -> Array2<Complex64> {
        match self {
            Pauli::I => Array2::from_shape_vec(
                (2, 2),
                vec![
                    Complex64::new(1.0, 0.0),
                    Complex64::new(0.0, 0.0),
                    Complex64::new(0.0, 0.0),
                    Complex64::new(1.0, 0.0),
                ],
            )
            .unwrap(),
            Pauli::X => Array2::from_shape_vec(
                (2, 2),
                vec![
                    Complex64::new(0.0, 0.0),
                    Complex64::new(1.0, 0.0),
                    Complex64::new(1.0, 0.0),
                    Complex64::new(0.0, 0.0),
                ],
            )
            .unwrap(),
            Pauli::Y => Array2::from_shape_vec(
                (2, 2),
                vec![
                    Complex64::new(0.0, 0.0),
                    Complex64::new(0.0, -1.0),
                    Complex64::new(0.0, 1.0),
                    Complex64::new(0.0, 0.0),
                ],
            )
            .unwrap(),
            Pauli::Z => Array2::from_shape_vec(
                (2, 2),
                vec![
                    Complex64::new(1.0, 0.0),
                    Complex64::new(0.0, 0.0),
                    Complex64::new(0.0, 0.0),
                    Complex64::new(-1.0, 0.0),
                ],
            )
            .unwrap(),
        }
    }

    /// Multiply two Pauli operators
    pub fn multiply(&self, other: &Pauli) -> (Complex64, Pauli) {
        use Pauli::*;
        match (self, other) {
            (I, p) | (p, I) => (Complex64::new(1.0, 0.0), *p),
            (X, X) | (Y, Y) | (Z, Z) => (Complex64::new(1.0, 0.0), I),
            (X, Y) => (Complex64::new(0.0, 1.0), Z),
            (Y, X) => (Complex64::new(0.0, -1.0), Z),
            (Y, Z) => (Complex64::new(0.0, 1.0), X),
            (Z, Y) => (Complex64::new(0.0, -1.0), X),
            (Z, X) => (Complex64::new(0.0, 1.0), Y),
            (X, Z) => (Complex64::new(0.0, -1.0), Y),
        }
    }
}

/// Multi-qubit Pauli operator
#[derive(Debug, Clone, PartialEq)]
pub struct PauliString {
    /// Phase factor (±1, ±i)
    pub phase: Complex64,
    /// Pauli operators for each qubit
    pub paulis: Vec<Pauli>,
}

impl PauliString {
    /// Create a new Pauli string
    pub fn new(paulis: Vec<Pauli>) -> Self {
        Self {
            phase: Complex64::new(1.0, 0.0),
            paulis,
        }
    }

    /// Create identity on n qubits
    pub fn identity(n: usize) -> Self {
        Self::new(vec![Pauli::I; n])
    }

    /// Get the weight (number of non-identity operators)
    pub fn weight(&self) -> usize {
        self.paulis.iter().filter(|&&p| p != Pauli::I).count()
    }

    /// Multiply two Pauli strings
    pub fn multiply(&self, other: &PauliString) -> QuantRS2Result<PauliString> {
        if self.paulis.len() != other.paulis.len() {
            return Err(QuantRS2Error::InvalidInput(
                "Pauli strings must have same length".to_string(),
            ));
        }

        let mut phase = self.phase * other.phase;
        let mut paulis = Vec::with_capacity(self.paulis.len());

        for (p1, p2) in self.paulis.iter().zip(&other.paulis) {
            let (factor, result) = p1.multiply(p2);
            phase *= factor;
            paulis.push(result);
        }

        Ok(PauliString { phase, paulis })
    }

    /// Check if two Pauli strings commute
    pub fn commutes_with(&self, other: &PauliString) -> QuantRS2Result<bool> {
        if self.paulis.len() != other.paulis.len() {
            return Err(QuantRS2Error::InvalidInput(
                "Pauli strings must have same length".to_string(),
            ));
        }

        let mut commutation_count = 0;
        for (p1, p2) in self.paulis.iter().zip(&other.paulis) {
            if *p1 != Pauli::I && *p2 != Pauli::I && p1 != p2 {
                commutation_count += 1;
            }
        }

        Ok(commutation_count % 2 == 0)
    }
}

impl fmt::Display for PauliString {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let phase_str = if self.phase == Complex64::new(1.0, 0.0) {
            "+".to_string()
        } else if self.phase == Complex64::new(-1.0, 0.0) {
            "-".to_string()
        } else if self.phase == Complex64::new(0.0, 1.0) {
            "+i".to_string()
        } else {
            "-i".to_string()
        };

        write!(f, "{}", phase_str)?;
        for p in &self.paulis {
            write!(f, "{:?}", p)?;
        }
        Ok(())
    }
}

/// Stabilizer code definition
#[derive(Debug, Clone)]
pub struct StabilizerCode {
    /// Number of physical qubits
    pub n: usize,
    /// Number of logical qubits
    pub k: usize,
    /// Minimum distance
    pub d: usize,
    /// Stabilizer generators
    pub stabilizers: Vec<PauliString>,
    /// Logical X operators
    pub logical_x: Vec<PauliString>,
    /// Logical Z operators
    pub logical_z: Vec<PauliString>,
}

impl StabilizerCode {
    /// Create a new stabilizer code
    pub fn new(
        n: usize,
        k: usize,
        d: usize,
        stabilizers: Vec<PauliString>,
        logical_x: Vec<PauliString>,
        logical_z: Vec<PauliString>,
    ) -> QuantRS2Result<Self> {
        // Validate code parameters
        // Note: For surface codes and other topological codes,
        // some stabilizers may be linearly dependent, so we allow more flexibility
        if stabilizers.len() > 2 * (n - k) {
            return Err(QuantRS2Error::InvalidInput(format!(
                "Too many stabilizers: got {}, maximum is {}",
                stabilizers.len(),
                2 * (n - k)
            )));
        }

        if logical_x.len() != k || logical_z.len() != k {
            return Err(QuantRS2Error::InvalidInput(
                "Number of logical operators must equal k".to_string(),
            ));
        }

        // Check that stabilizers commute
        for i in 0..stabilizers.len() {
            for j in i + 1..stabilizers.len() {
                if !stabilizers[i].commutes_with(&stabilizers[j])? {
                    return Err(QuantRS2Error::InvalidInput(
                        "Stabilizers must commute".to_string(),
                    ));
                }
            }
        }

        Ok(Self {
            n,
            k,
            d,
            stabilizers,
            logical_x,
            logical_z,
        })
    }

    /// Create the 3-qubit repetition code
    pub fn repetition_code() -> Self {
        let stabilizers = vec![
            PauliString::new(vec![Pauli::Z, Pauli::Z, Pauli::I]),
            PauliString::new(vec![Pauli::I, Pauli::Z, Pauli::Z]),
        ];

        let logical_x = vec![PauliString::new(vec![Pauli::X, Pauli::X, Pauli::X])];
        let logical_z = vec![PauliString::new(vec![Pauli::Z, Pauli::I, Pauli::I])];

        Self::new(3, 1, 1, stabilizers, logical_x, logical_z).unwrap()
    }

    /// Create the 5-qubit perfect code
    pub fn five_qubit_code() -> Self {
        let stabilizers = vec![
            PauliString::new(vec![Pauli::X, Pauli::Z, Pauli::Z, Pauli::X, Pauli::I]),
            PauliString::new(vec![Pauli::I, Pauli::X, Pauli::Z, Pauli::Z, Pauli::X]),
            PauliString::new(vec![Pauli::X, Pauli::I, Pauli::X, Pauli::Z, Pauli::Z]),
            PauliString::new(vec![Pauli::Z, Pauli::X, Pauli::I, Pauli::X, Pauli::Z]),
        ];

        let logical_x = vec![PauliString::new(vec![
            Pauli::X,
            Pauli::X,
            Pauli::X,
            Pauli::X,
            Pauli::X,
        ])];
        let logical_z = vec![PauliString::new(vec![
            Pauli::Z,
            Pauli::Z,
            Pauli::Z,
            Pauli::Z,
            Pauli::Z,
        ])];

        Self::new(5, 1, 3, stabilizers, logical_x, logical_z).unwrap()
    }

    /// Create the 7-qubit Steane code
    pub fn steane_code() -> Self {
        let stabilizers = vec![
            PauliString::new(vec![
                Pauli::I,
                Pauli::I,
                Pauli::I,
                Pauli::X,
                Pauli::X,
                Pauli::X,
                Pauli::X,
            ]),
            PauliString::new(vec![
                Pauli::I,
                Pauli::X,
                Pauli::X,
                Pauli::I,
                Pauli::I,
                Pauli::X,
                Pauli::X,
            ]),
            PauliString::new(vec![
                Pauli::X,
                Pauli::I,
                Pauli::X,
                Pauli::I,
                Pauli::X,
                Pauli::I,
                Pauli::X,
            ]),
            PauliString::new(vec![
                Pauli::I,
                Pauli::I,
                Pauli::I,
                Pauli::Z,
                Pauli::Z,
                Pauli::Z,
                Pauli::Z,
            ]),
            PauliString::new(vec![
                Pauli::I,
                Pauli::Z,
                Pauli::Z,
                Pauli::I,
                Pauli::I,
                Pauli::Z,
                Pauli::Z,
            ]),
            PauliString::new(vec![
                Pauli::Z,
                Pauli::I,
                Pauli::Z,
                Pauli::I,
                Pauli::Z,
                Pauli::I,
                Pauli::Z,
            ]),
        ];

        let logical_x = vec![PauliString::new(vec![
            Pauli::X,
            Pauli::X,
            Pauli::X,
            Pauli::X,
            Pauli::X,
            Pauli::X,
            Pauli::X,
        ])];
        let logical_z = vec![PauliString::new(vec![
            Pauli::Z,
            Pauli::Z,
            Pauli::Z,
            Pauli::Z,
            Pauli::Z,
            Pauli::Z,
            Pauli::Z,
        ])];

        Self::new(7, 1, 3, stabilizers, logical_x, logical_z).unwrap()
    }

    /// Get syndrome for a given error
    pub fn syndrome(&self, error: &PauliString) -> QuantRS2Result<Vec<bool>> {
        if error.paulis.len() != self.n {
            return Err(QuantRS2Error::InvalidInput(
                "Error must act on all physical qubits".to_string(),
            ));
        }

        let mut syndrome = Vec::with_capacity(self.stabilizers.len());
        for stabilizer in &self.stabilizers {
            syndrome.push(!stabilizer.commutes_with(error)?);
        }

        Ok(syndrome)
    }
}

/// Surface code lattice
#[derive(Debug, Clone)]
pub struct SurfaceCode {
    /// Number of rows in the lattice
    pub rows: usize,
    /// Number of columns in the lattice
    pub cols: usize,
    /// Qubit positions (row, col) -> qubit index
    pub qubit_map: HashMap<(usize, usize), usize>,
    /// Stabilizer plaquettes
    pub x_stabilizers: Vec<Vec<usize>>,
    pub z_stabilizers: Vec<Vec<usize>>,
}

impl SurfaceCode {
    /// Create a new surface code
    pub fn new(rows: usize, cols: usize) -> Self {
        let mut qubit_map = HashMap::new();
        let mut qubit_index = 0;

        // Place qubits on the lattice
        for r in 0..rows {
            for c in 0..cols {
                qubit_map.insert((r, c), qubit_index);
                qubit_index += 1;
            }
        }

        let mut x_stabilizers = Vec::new();
        let mut z_stabilizers = Vec::new();

        // Create X stabilizers (vertex operators)
        for r in 0..rows - 1 {
            for c in 0..cols - 1 {
                if (r + c) % 2 == 0 {
                    let stabilizer = vec![
                        qubit_map[&(r, c)],
                        qubit_map[&(r, c + 1)],
                        qubit_map[&(r + 1, c)],
                        qubit_map[&(r + 1, c + 1)],
                    ];
                    x_stabilizers.push(stabilizer);
                }
            }
        }

        // Create Z stabilizers (plaquette operators)
        for r in 0..rows - 1 {
            for c in 0..cols - 1 {
                if (r + c) % 2 == 1 {
                    let stabilizer = vec![
                        qubit_map[&(r, c)],
                        qubit_map[&(r, c + 1)],
                        qubit_map[&(r + 1, c)],
                        qubit_map[&(r + 1, c + 1)],
                    ];
                    z_stabilizers.push(stabilizer);
                }
            }
        }

        Self {
            rows,
            cols,
            qubit_map,
            x_stabilizers,
            z_stabilizers,
        }
    }

    /// Get the code distance
    pub fn distance(&self) -> usize {
        self.rows.min(self.cols)
    }

    /// Convert to stabilizer code representation
    pub fn to_stabilizer_code(&self) -> StabilizerCode {
        let n = self.qubit_map.len();
        let mut stabilizers = Vec::new();

        // Add X stabilizers
        for x_stab in &self.x_stabilizers {
            let mut paulis = vec![Pauli::I; n];
            for &qubit in x_stab {
                paulis[qubit] = Pauli::X;
            }
            stabilizers.push(PauliString::new(paulis));
        }

        // Add Z stabilizers
        for z_stab in &self.z_stabilizers {
            let mut paulis = vec![Pauli::I; n];
            for &qubit in z_stab {
                paulis[qubit] = Pauli::Z;
            }
            stabilizers.push(PauliString::new(paulis));
        }

        // Create logical operators (simplified - just use boundary chains)
        let mut logical_x_paulis = vec![Pauli::I; n];
        let mut logical_z_paulis = vec![Pauli::I; n];

        // Logical X: horizontal chain on top boundary
        for c in 0..self.cols {
            if let Some(&qubit) = self.qubit_map.get(&(0, c)) {
                logical_x_paulis[qubit] = Pauli::X;
            }
        }

        // Logical Z: vertical chain on left boundary
        for r in 0..self.rows {
            if let Some(&qubit) = self.qubit_map.get(&(r, 0)) {
                logical_z_paulis[qubit] = Pauli::Z;
            }
        }

        let logical_x = vec![PauliString::new(logical_x_paulis)];
        let logical_z = vec![PauliString::new(logical_z_paulis)];

        StabilizerCode::new(n, 1, self.distance(), stabilizers, logical_x, logical_z).unwrap()
    }
}

/// Syndrome decoder interface
pub trait SyndromeDecoder {
    /// Decode syndrome to find most likely error
    fn decode(&self, syndrome: &[bool]) -> QuantRS2Result<PauliString>;
}

/// Lookup table decoder
pub struct LookupDecoder {
    /// Syndrome to error mapping
    syndrome_table: HashMap<Vec<bool>, PauliString>,
}

impl LookupDecoder {
    /// Create decoder for a stabilizer code
    pub fn new(code: &StabilizerCode) -> QuantRS2Result<Self> {
        let mut syndrome_table = HashMap::new();

        // Generate all correctable errors (up to weight floor(d/2))
        let max_weight = (code.d - 1) / 2;
        let all_errors = Self::generate_pauli_errors(code.n, max_weight);

        for error in all_errors {
            let syndrome = code.syndrome(&error)?;

            // Only keep lowest weight error for each syndrome
            syndrome_table
                .entry(syndrome)
                .and_modify(|e: &mut PauliString| {
                    if error.weight() < e.weight() {
                        *e = error.clone();
                    }
                })
                .or_insert(error);
        }

        Ok(Self { syndrome_table })
    }

    /// Generate all Pauli errors up to given weight
    fn generate_pauli_errors(n: usize, max_weight: usize) -> Vec<PauliString> {
        let mut errors = vec![PauliString::identity(n)];

        for weight in 1..=max_weight {
            let weight_errors = Self::generate_weight_k_errors(n, weight);
            errors.extend(weight_errors);
        }

        errors
    }

    /// Generate all weight-k Pauli errors
    fn generate_weight_k_errors(n: usize, k: usize) -> Vec<PauliString> {
        let mut errors = Vec::new();
        let paulis = [Pauli::X, Pauli::Y, Pauli::Z];

        // Generate all combinations of k positions
        let positions = Self::combinations(n, k);

        for pos_set in positions {
            // For each position set, try all Pauli combinations
            let pauli_combinations = Self::cartesian_power(&paulis, k);

            for pauli_combo in pauli_combinations {
                let mut error_paulis = vec![Pauli::I; n];
                for (i, &pos) in pos_set.iter().enumerate() {
                    error_paulis[pos] = pauli_combo[i];
                }
                errors.push(PauliString::new(error_paulis));
            }
        }

        errors
    }

    /// Generate all k-combinations from n elements
    fn combinations(n: usize, k: usize) -> Vec<Vec<usize>> {
        let mut result = Vec::new();
        let mut combo = (0..k).collect::<Vec<_>>();

        loop {
            result.push(combo.clone());

            // Find rightmost element that can be incremented
            let mut i = k;
            while i > 0 && (i == k || combo[i] == n - k + i) {
                i -= 1;
            }

            if i == 0 && combo[0] == n - k {
                break;
            }

            // Increment and reset following elements
            combo[i] += 1;
            for j in i + 1..k {
                combo[j] = combo[j - 1] + 1;
            }
        }

        result
    }

    /// Generate Cartesian power of a set
    fn cartesian_power<T: Clone>(set: &[T], k: usize) -> Vec<Vec<T>> {
        if k == 0 {
            return vec![vec![]];
        }

        let mut result = Vec::new();
        let smaller = Self::cartesian_power(set, k - 1);

        for item in set {
            for mut combo in smaller.clone() {
                combo.push(item.clone());
                result.push(combo);
            }
        }

        result
    }
}

impl SyndromeDecoder for LookupDecoder {
    fn decode(&self, syndrome: &[bool]) -> QuantRS2Result<PauliString> {
        self.syndrome_table
            .get(syndrome)
            .cloned()
            .ok_or_else(|| QuantRS2Error::InvalidInput("Unknown syndrome".to_string()))
    }
}

/// Minimum Weight Perfect Matching decoder for surface codes
pub struct MWPMDecoder {
    surface_code: SurfaceCode,
}

impl MWPMDecoder {
    /// Create MWPM decoder for surface code
    pub fn new(surface_code: SurfaceCode) -> Self {
        Self { surface_code }
    }

    /// Find minimum weight matching for syndrome
    pub fn decode_syndrome(
        &self,
        x_syndrome: &[bool],
        z_syndrome: &[bool],
    ) -> QuantRS2Result<PauliString> {
        let n = self.surface_code.qubit_map.len();
        let mut error_paulis = vec![Pauli::I; n];

        // Decode X errors using Z syndrome
        let z_defects = self.find_defects(z_syndrome, &self.surface_code.z_stabilizers);
        let x_corrections = self.minimum_weight_matching(&z_defects, Pauli::X)?;

        for (qubit, pauli) in x_corrections {
            error_paulis[qubit] = pauli;
        }

        // Decode Z errors using X syndrome
        let x_defects = self.find_defects(x_syndrome, &self.surface_code.x_stabilizers);
        let z_corrections = self.minimum_weight_matching(&x_defects, Pauli::Z)?;

        for (qubit, pauli) in z_corrections {
            if error_paulis[qubit] != Pauli::I {
                // Combine X and Z to get Y
                error_paulis[qubit] = Pauli::Y;
            } else {
                error_paulis[qubit] = pauli;
            }
        }

        Ok(PauliString::new(error_paulis))
    }

    /// Find stabilizer defects from syndrome
    fn find_defects(&self, syndrome: &[bool], _stabilizers: &[Vec<usize>]) -> Vec<usize> {
        syndrome
            .iter()
            .enumerate()
            .filter_map(|(i, &s)| if s { Some(i) } else { None })
            .collect()
    }

    /// Simple minimum weight matching (for demonstration)
    fn minimum_weight_matching(
        &self,
        defects: &[usize],
        error_type: Pauli,
    ) -> QuantRS2Result<Vec<(usize, Pauli)>> {
        // This is a simplified version - real implementation would use blossom algorithm
        let mut corrections = Vec::new();

        if defects.len() % 2 != 0 {
            return Err(QuantRS2Error::InvalidInput(
                "Odd number of defects".to_string(),
            ));
        }

        // Simple greedy pairing
        let mut paired = vec![false; defects.len()];

        for i in 0..defects.len() {
            if paired[i] {
                continue;
            }

            // Find nearest unpaired defect
            let mut min_dist = usize::MAX;
            let mut min_j = i;

            for j in i + 1..defects.len() {
                if !paired[j] {
                    let dist = self.defect_distance(defects[i], defects[j]);
                    if dist < min_dist {
                        min_dist = dist;
                        min_j = j;
                    }
                }
            }

            if min_j != i {
                paired[i] = true;
                paired[min_j] = true;

                // Add correction path
                let path = self.shortest_path(defects[i], defects[min_j])?;
                for qubit in path {
                    corrections.push((qubit, error_type));
                }
            }
        }

        Ok(corrections)
    }

    /// Manhattan distance between defects
    fn defect_distance(&self, defect1: usize, defect2: usize) -> usize {
        // This is simplified - would need proper defect coordinates
        (defect1 as isize - defect2 as isize).unsigned_abs()
    }

    /// Find shortest path between defects
    fn shortest_path(&self, start: usize, end: usize) -> QuantRS2Result<Vec<usize>> {
        // Simplified path - in practice would use proper graph traversal
        let path = if start < end {
            (start..=end).collect()
        } else {
            (end..=start).rev().collect()
        };

        Ok(path)
    }
}

/// Color code
#[derive(Debug, Clone)]
pub struct ColorCode {
    /// Number of physical qubits
    pub n: usize,
    /// Face coloring (red, green, blue)
    pub faces: Vec<(Vec<usize>, Color)>,
    /// Vertex to qubit mapping
    pub vertex_map: HashMap<(i32, i32), usize>,
}

#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum Color {
    Red,
    Green,
    Blue,
}

impl ColorCode {
    /// Create a triangular color code
    pub fn triangular(size: usize) -> Self {
        let mut vertex_map = HashMap::new();
        let mut qubit_index = 0;

        // Create hexagonal lattice vertices
        for i in 0..size as i32 {
            for j in 0..size as i32 {
                vertex_map.insert((i, j), qubit_index);
                qubit_index += 1;
            }
        }

        let mut faces = Vec::new();

        // Create colored faces
        for i in 0..size as i32 - 1 {
            for j in 0..size as i32 - 1 {
                // Red face
                if let (Some(&q1), Some(&q2), Some(&q3)) = (
                    vertex_map.get(&(i, j)),
                    vertex_map.get(&(i + 1, j)),
                    vertex_map.get(&(i, j + 1)),
                ) {
                    faces.push((vec![q1, q2, q3], Color::Red));
                }

                // Green face
                if let (Some(&q1), Some(&q2), Some(&q3)) = (
                    vertex_map.get(&(i + 1, j)),
                    vertex_map.get(&(i + 1, j + 1)),
                    vertex_map.get(&(i, j + 1)),
                ) {
                    faces.push((vec![q1, q2, q3], Color::Green));
                }
            }
        }

        Self {
            n: vertex_map.len(),
            faces,
            vertex_map,
        }
    }

    /// Convert to stabilizer code
    pub fn to_stabilizer_code(&self) -> StabilizerCode {
        let mut x_stabilizers = Vec::new();
        let mut z_stabilizers = Vec::new();

        for (qubits, _color) in &self.faces {
            // X-type stabilizer
            let mut x_paulis = vec![Pauli::I; self.n];
            for &q in qubits {
                x_paulis[q] = Pauli::X;
            }
            x_stabilizers.push(PauliString::new(x_paulis));

            // Z-type stabilizer
            let mut z_paulis = vec![Pauli::I; self.n];
            for &q in qubits {
                z_paulis[q] = Pauli::Z;
            }
            z_stabilizers.push(PauliString::new(z_paulis));
        }

        let mut stabilizers = x_stabilizers;
        stabilizers.extend(z_stabilizers);

        // Simplified logical operators
        let logical_x = vec![PauliString::new(vec![Pauli::X; self.n])];
        let logical_z = vec![PauliString::new(vec![Pauli::Z; self.n])];

        StabilizerCode::new(
            self.n,
            1,
            3, // minimum distance
            stabilizers,
            logical_x,
            logical_z,
        )
        .unwrap()
    }
}

/// Concatenated quantum error correction codes
#[derive(Debug, Clone)]
pub struct ConcatenatedCode {
    /// Inner code (applied first)
    pub inner_code: StabilizerCode,
    /// Outer code (applied to logical qubits of inner code)
    pub outer_code: StabilizerCode,
}

impl ConcatenatedCode {
    /// Create a new concatenated code
    pub fn new(inner_code: StabilizerCode, outer_code: StabilizerCode) -> Self {
        Self {
            inner_code,
            outer_code,
        }
    }

    /// Get total number of physical qubits
    pub fn total_qubits(&self) -> usize {
        self.inner_code.n * self.outer_code.n
    }

    /// Get number of logical qubits
    pub fn logical_qubits(&self) -> usize {
        self.inner_code.k * self.outer_code.k
    }

    /// Get effective distance
    pub fn distance(&self) -> usize {
        self.inner_code.d * self.outer_code.d
    }

    /// Encode a logical state
    pub fn encode(&self, logical_state: &[Complex64]) -> QuantRS2Result<Vec<Complex64>> {
        if logical_state.len() != 1 << self.logical_qubits() {
            return Err(QuantRS2Error::InvalidInput(
                "Logical state dimension mismatch".to_string(),
            ));
        }

        // First encode with outer code
        let outer_encoded = self.encode_with_code(logical_state, &self.outer_code)?;

        // Then encode each logical qubit of outer code with inner code
        let mut final_encoded = vec![Complex64::new(0.0, 0.0); 1 << self.total_qubits()];

        // This is a simplified encoding - proper implementation would require
        // tensor product operations and proper state manipulation
        for (i, &amplitude) in outer_encoded.iter().enumerate() {
            if amplitude.norm() > 1e-10 {
                final_encoded[i * (1 << self.inner_code.n)] = amplitude;
            }
        }

        Ok(final_encoded)
    }

    /// Correct errors using concatenated decoding
    pub fn correct_error(
        &self,
        encoded_state: &[Complex64],
        error: &PauliString,
    ) -> QuantRS2Result<Vec<Complex64>> {
        if error.paulis.len() != self.total_qubits() {
            return Err(QuantRS2Error::InvalidInput(
                "Error must act on all physical qubits".to_string(),
            ));
        }

        // Simplified error correction - apply error and return corrected state
        // In practice, would implement syndrome extraction and decoding
        let mut corrected = encoded_state.to_vec();

        // Apply error (simplified)
        for (i, &pauli) in error.paulis.iter().enumerate() {
            if pauli != Pauli::I && i < corrected.len() {
                // Simplified error application
                corrected[i] *= -1.0;
            }
        }

        Ok(corrected)
    }

    /// Encode with a specific code
    fn encode_with_code(
        &self,
        state: &[Complex64],
        code: &StabilizerCode,
    ) -> QuantRS2Result<Vec<Complex64>> {
        // Simplified encoding - proper implementation would use stabilizer formalism
        let mut encoded = vec![Complex64::new(0.0, 0.0); 1 << code.n];

        for (i, &amplitude) in state.iter().enumerate() {
            if i < encoded.len() {
                encoded[i * (1 << (code.n - code.k))] = amplitude;
            }
        }

        Ok(encoded)
    }
}

/// Hypergraph product codes for quantum LDPC
#[derive(Debug, Clone)]
pub struct HypergraphProductCode {
    /// Number of physical qubits
    pub n: usize,
    /// Number of logical qubits
    pub k: usize,
    /// X-type stabilizers
    pub x_stabilizers: Vec<PauliString>,
    /// Z-type stabilizers
    pub z_stabilizers: Vec<PauliString>,
}

impl HypergraphProductCode {
    /// Create hypergraph product code from two classical codes
    pub fn new(h1: Array2<u8>, h2: Array2<u8>) -> Self {
        let (m1, n1) = h1.dim();
        let (m2, n2) = h2.dim();

        let n = n1 * m2 + m1 * n2;
        let k = (n1 - m1) * (n2 - m2);

        let mut x_stabilizers = Vec::new();
        let mut z_stabilizers = Vec::new();

        // X-type stabilizers: H1 ⊗ I2
        for i in 0..m1 {
            for j in 0..m2 {
                let mut paulis = vec![Pauli::I; n];

                // Apply H1 to first block
                for l in 0..n1 {
                    if h1[[i, l]] == 1 {
                        paulis[l * m2 + j] = Pauli::X;
                    }
                }

                x_stabilizers.push(PauliString::new(paulis));
            }
        }

        // Z-type stabilizers: I1 ⊗ H2^T
        for i in 0..m1 {
            for j in 0..m2 {
                let mut paulis = vec![Pauli::I; n];

                // Apply H2^T to second block
                for l in 0..n2 {
                    if h2[[j, l]] == 1 {
                        paulis[n1 * m2 + i * n2 + l] = Pauli::Z;
                    }
                }

                z_stabilizers.push(PauliString::new(paulis));
            }
        }

        Self {
            n,
            k,
            x_stabilizers,
            z_stabilizers,
        }
    }

    /// Convert to stabilizer code representation
    pub fn to_stabilizer_code(&self) -> StabilizerCode {
        let mut stabilizers = self.x_stabilizers.clone();
        stabilizers.extend(self.z_stabilizers.clone());

        // Simplified logical operators
        let logical_x = vec![PauliString::new(vec![Pauli::X; self.n])];
        let logical_z = vec![PauliString::new(vec![Pauli::Z; self.n])];

        StabilizerCode::new(
            self.n,
            self.k,
            3, // Simplified distance
            stabilizers,
            logical_x,
            logical_z,
        )
        .unwrap()
    }
}

/// Quantum Low-Density Parity-Check (LDPC) codes
#[derive(Debug, Clone)]
pub struct QuantumLDPCCode {
    /// Number of physical qubits
    pub n: usize,
    /// Number of logical qubits
    pub k: usize,
    /// Maximum stabilizer weight
    pub max_weight: usize,
    /// X-type stabilizers
    pub x_stabilizers: Vec<PauliString>,
    /// Z-type stabilizers
    pub z_stabilizers: Vec<PauliString>,
}

impl QuantumLDPCCode {
    /// Create a bicycle code (CSS LDPC)
    pub fn bicycle_code(a: usize, b: usize) -> Self {
        let n = 2 * a * b;
        let k = 2;
        let max_weight = 6; // Typical for bicycle codes

        let mut x_stabilizers = Vec::new();
        let mut z_stabilizers = Vec::new();

        // Generate bicycle code stabilizers
        for i in 0..a {
            for j in 0..b {
                // X-type stabilizer
                let mut x_paulis = vec![Pauli::I; n];
                let base_idx = i * b + j;

                // Create a 6-cycle in the Cayley graph
                x_paulis[base_idx] = Pauli::X;
                x_paulis[(base_idx + 1) % (a * b)] = Pauli::X;
                x_paulis[(base_idx + b) % (a * b)] = Pauli::X;
                x_paulis[a * b + base_idx] = Pauli::X;
                x_paulis[a * b + (base_idx + 1) % (a * b)] = Pauli::X;
                x_paulis[a * b + (base_idx + b) % (a * b)] = Pauli::X;

                x_stabilizers.push(PauliString::new(x_paulis));

                // Z-type stabilizer (similar structure)
                let mut z_paulis = vec![Pauli::I; n];
                z_paulis[base_idx] = Pauli::Z;
                z_paulis[(base_idx + a) % (a * b)] = Pauli::Z;
                z_paulis[(base_idx + 1) % (a * b)] = Pauli::Z;
                z_paulis[a * b + base_idx] = Pauli::Z;
                z_paulis[a * b + (base_idx + a) % (a * b)] = Pauli::Z;
                z_paulis[a * b + (base_idx + 1) % (a * b)] = Pauli::Z;

                z_stabilizers.push(PauliString::new(z_paulis));
            }
        }

        Self {
            n,
            k,
            max_weight,
            x_stabilizers,
            z_stabilizers,
        }
    }

    /// Convert to stabilizer code representation
    pub fn to_stabilizer_code(&self) -> StabilizerCode {
        let mut stabilizers = self.x_stabilizers.clone();
        stabilizers.extend(self.z_stabilizers.clone());

        // Create logical operators (simplified)
        let logical_x = vec![
            PauliString::new(vec![Pauli::X; self.n]),
            PauliString::new(vec![Pauli::Y; self.n]),
        ];
        let logical_z = vec![
            PauliString::new(vec![Pauli::Z; self.n]),
            PauliString::new(vec![Pauli::Y; self.n]),
        ];

        StabilizerCode::new(
            self.n,
            self.k,
            4, // Typical distance for bicycle codes
            stabilizers,
            logical_x,
            logical_z,
        )
        .unwrap()
    }
}

/// Toric code (generalization of surface code on torus)
#[derive(Debug, Clone)]
pub struct ToricCode {
    /// Number of rows in the torus
    pub rows: usize,
    /// Number of columns in the torus
    pub cols: usize,
    /// Qubit mapping
    pub qubit_map: HashMap<(usize, usize), usize>,
}

impl ToricCode {
    /// Create a new toric code
    pub fn new(rows: usize, cols: usize) -> Self {
        let mut qubit_map = HashMap::new();
        let mut qubit_index = 0;

        // Place qubits on torus (two qubits per unit cell)
        for r in 0..rows {
            for c in 0..cols {
                // Horizontal edge qubit
                qubit_map.insert((2 * r, c), qubit_index);
                qubit_index += 1;
                // Vertical edge qubit
                qubit_map.insert((2 * r + 1, c), qubit_index);
                qubit_index += 1;
            }
        }

        Self {
            rows,
            cols,
            qubit_map,
        }
    }

    /// Get number of logical qubits
    pub fn logical_qubits(&self) -> usize {
        2 // Two logical qubits for torus topology
    }

    /// Get code distance
    pub fn distance(&self) -> usize {
        self.rows.min(self.cols)
    }

    /// Convert to stabilizer code representation
    pub fn to_stabilizer_code(&self) -> StabilizerCode {
        let n = self.qubit_map.len();
        let mut stabilizers = Vec::new();

        // Vertex stabilizers (X-type) - star operators
        for r in 0..self.rows {
            for c in 0..self.cols {
                let mut paulis = vec![Pauli::I; n];

                // Four edges around vertex with correct torus indexing
                let h_edge_below = (2 * r, c);
                let h_edge_above = (2 * ((r + self.rows - 1) % self.rows), c);
                let v_edge_left = (2 * r + 1, (c + self.cols - 1) % self.cols);
                let v_edge_right = (2 * r + 1, c);

                for &coord in &[h_edge_below, h_edge_above, v_edge_left, v_edge_right] {
                    if let Some(&qubit) = self.qubit_map.get(&coord) {
                        paulis[qubit] = Pauli::X;
                    }
                }

                stabilizers.push(PauliString::new(paulis));
            }
        }

        // Plaquette stabilizers (Z-type) - face operators
        for r in 0..self.rows {
            for c in 0..self.cols {
                let mut paulis = vec![Pauli::I; n];

                // Four edges around plaquette with correct indexing
                let h_edge_top = (2 * r, c);
                let h_edge_bottom = (2 * ((r + 1) % self.rows), c);
                let v_edge_left = (2 * r + 1, c);
                let v_edge_right = (2 * r + 1, (c + 1) % self.cols);

                for &coord in &[h_edge_top, h_edge_bottom, v_edge_left, v_edge_right] {
                    if let Some(&qubit) = self.qubit_map.get(&coord) {
                        paulis[qubit] = Pauli::Z;
                    }
                }

                stabilizers.push(PauliString::new(paulis));
            }
        }

        // Logical operators (horizontal and vertical loops)
        let mut logical_x1 = vec![Pauli::I; n];
        let mut logical_z1 = vec![Pauli::I; n];
        let mut logical_x2 = vec![Pauli::I; n];
        let mut logical_z2 = vec![Pauli::I; n];

        // Horizontal logical loop operators
        for c in 0..self.cols {
            // Logical X along horizontal direction (vertical edges)
            if let Some(&qubit) = self.qubit_map.get(&(1, c)) {
                logical_x1[qubit] = Pauli::X;
            }
            // Logical Z along horizontal direction (horizontal edges)
            if let Some(&qubit) = self.qubit_map.get(&(0, c)) {
                logical_z2[qubit] = Pauli::Z;
            }
        }

        // Vertical logical loop operators
        for r in 0..self.rows {
            // Logical X along vertical direction (horizontal edges)
            if let Some(&qubit) = self.qubit_map.get(&(2 * r, 0)) {
                logical_x2[qubit] = Pauli::X;
            }
            // Logical Z along vertical direction (vertical edges)
            if let Some(&qubit) = self.qubit_map.get(&(2 * r + 1, 0)) {
                logical_z1[qubit] = Pauli::Z;
            }
        }

        let logical_x = vec![PauliString::new(logical_x1), PauliString::new(logical_x2)];
        let logical_z = vec![PauliString::new(logical_z1), PauliString::new(logical_z2)];

        StabilizerCode::new(n, 2, self.distance(), stabilizers, logical_x, logical_z).unwrap()
    }
}

/// Machine learning-based syndrome decoder
pub struct MLDecoder {
    /// The code being decoded
    code: StabilizerCode,
    /// Neural network weights (simplified representation)
    weights: Vec<Vec<f64>>,
}

impl MLDecoder {
    /// Create a new ML decoder
    pub fn new(code: StabilizerCode) -> Self {
        // Initialize random weights for a simple neural network
        let input_size = code.stabilizers.len();
        let hidden_size = 2 * input_size;
        let output_size = code.n * 3; // 3 Pauli operators per qubit

        use scirs2_core::random::prelude::*;
        let mut rng = thread_rng();
        let mut weights = Vec::new();

        // Input to hidden layer
        let mut w1 = Vec::new();
        for _ in 0..hidden_size {
            let mut row = Vec::new();
            for _ in 0..input_size {
                row.push((rng.gen::<f64>() - 0.5) * 0.1);
            }
            w1.push(row);
        }
        weights.push(w1.into_iter().flatten().collect());

        // Hidden to output layer
        let mut w2 = Vec::new();
        for _ in 0..output_size {
            let mut row = Vec::new();
            for _ in 0..hidden_size {
                row.push((rng.gen::<f64>() - 0.5) * 0.1);
            }
            w2.push(row);
        }
        weights.push(w2.into_iter().flatten().collect());

        Self { code, weights }
    }

    /// Simple feedforward prediction
    fn predict(&self, syndrome: &[bool]) -> Vec<f64> {
        let input: Vec<f64> = syndrome
            .iter()
            .map(|&b| if b { 1.0 } else { 0.0 })
            .collect();

        // This is a greatly simplified neural network
        // In practice, would use proper ML framework
        let hidden_size = 2 * input.len();
        let mut hidden = vec![0.0; hidden_size];

        // Input to hidden
        for i in 0..hidden_size {
            for j in 0..input.len() {
                if i * input.len() + j < self.weights[0].len() {
                    hidden[i] += input[j] * self.weights[0][i * input.len() + j];
                }
            }
            hidden[i] = hidden[i].tanh(); // Activation function
        }

        // Hidden to output
        let output_size = self.code.n * 3;
        let mut output = vec![0.0; output_size];

        for i in 0..output_size {
            for j in 0..hidden_size {
                if i * hidden_size + j < self.weights[1].len() {
                    output[i] += hidden[j] * self.weights[1][i * hidden_size + j];
                }
            }
        }

        output
    }
}

impl SyndromeDecoder for MLDecoder {
    fn decode(&self, syndrome: &[bool]) -> QuantRS2Result<PauliString> {
        let prediction = self.predict(syndrome);

        // Convert prediction to Pauli string
        let mut paulis = Vec::with_capacity(self.code.n);

        for qubit in 0..self.code.n {
            let base_idx = qubit * 3;
            if base_idx + 2 < prediction.len() {
                let x_prob = prediction[base_idx];
                let y_prob = prediction[base_idx + 1];
                let z_prob = prediction[base_idx + 2];

                // Choose Pauli with highest probability
                if x_prob > y_prob && x_prob > z_prob && x_prob > 0.5 {
                    paulis.push(Pauli::X);
                } else if y_prob > z_prob && y_prob > 0.5 {
                    paulis.push(Pauli::Y);
                } else if z_prob > 0.5 {
                    paulis.push(Pauli::Z);
                } else {
                    paulis.push(Pauli::I);
                }
            } else {
                paulis.push(Pauli::I);
            }
        }

        Ok(PauliString::new(paulis))
    }
}

/// Real-time error correction with hardware integration
/// This module provides interfaces and implementations for real-time quantum error correction
/// that can be integrated with quantum hardware control systems.
pub mod real_time {
    use super::*;
    use std::collections::VecDeque;
    use std::sync::{Arc, Mutex, RwLock};
    use std::thread;
    use std::time::{Duration, Instant};

    /// Hardware interface trait for quantum error correction
    pub trait QuantumHardwareInterface: Send + Sync {
        /// Get syndrome measurements from hardware
        fn measure_syndromes(&self) -> QuantRS2Result<Vec<bool>>;

        /// Apply error correction operations to hardware
        fn apply_correction(&self, correction: &PauliString) -> QuantRS2Result<()>;

        /// Get hardware error rates and characterization data
        fn get_error_characteristics(&self) -> QuantRS2Result<HardwareErrorCharacteristics>;

        /// Check if hardware is ready for error correction
        fn is_ready(&self) -> bool;

        /// Get hardware latency statistics
        fn get_latency_stats(&self) -> QuantRS2Result<LatencyStats>;
    }

    /// Hardware error characteristics for adaptive error correction
    #[derive(Debug, Clone)]
    pub struct HardwareErrorCharacteristics {
        /// Single-qubit error rates (T1, T2, gate errors)
        pub single_qubit_error_rates: Vec<f64>,
        /// Two-qubit gate error rates
        pub two_qubit_error_rates: Vec<f64>,
        /// Measurement error rates
        pub measurement_error_rates: Vec<f64>,
        /// Correlated error patterns
        pub correlated_errors: Vec<CorrelatedErrorPattern>,
        /// Error rate temporal variation
        pub temporal_variation: f64,
    }

    /// Correlated error pattern for adaptive decoding
    #[derive(Debug, Clone)]
    pub struct CorrelatedErrorPattern {
        pub qubits: Vec<usize>,
        pub probability: f64,
        pub pauli_pattern: PauliString,
    }

    /// Performance and latency statistics
    #[derive(Debug, Clone)]
    pub struct LatencyStats {
        pub syndrome_measurement_time: Duration,
        pub decoding_time: Duration,
        pub correction_application_time: Duration,
        pub total_cycle_time: Duration,
        pub throughput_hz: f64,
    }

    /// Real-time syndrome stream processor
    pub struct SyndromeStreamProcessor {
        buffer: Arc<Mutex<VecDeque<SyndromePacket>>>,
        decoder: Arc<dyn SyndromeDecoder + Send + Sync>,
        hardware: Arc<dyn QuantumHardwareInterface>,
        performance_monitor: Arc<RwLock<PerformanceMonitor>>,
        config: RealTimeConfig,
    }

    /// Syndrome packet with timing information
    #[derive(Debug, Clone)]
    pub struct SyndromePacket {
        pub syndrome: Vec<bool>,
        pub timestamp: Instant,
        pub sequence_number: u64,
        pub measurement_fidelity: f64,
    }

    /// Real-time error correction configuration
    #[derive(Debug, Clone)]
    pub struct RealTimeConfig {
        pub max_latency: Duration,
        pub buffer_size: usize,
        pub parallel_workers: usize,
        pub adaptive_threshold: bool,
        pub hardware_feedback: bool,
        pub performance_logging: bool,
    }

    impl Default for RealTimeConfig {
        fn default() -> Self {
            Self {
                max_latency: Duration::from_micros(100), // 100μs for fast correction
                buffer_size: 1000,
                parallel_workers: 4,
                adaptive_threshold: true,
                hardware_feedback: true,
                performance_logging: true,
            }
        }
    }

    /// Performance monitoring for real-time error correction
    #[derive(Debug, Clone)]
    pub struct PerformanceMonitor {
        pub cycles_processed: u64,
        pub errors_corrected: u64,
        pub false_positives: u64,
        pub latency_histogram: Vec<Duration>,
        pub throughput_samples: VecDeque<f64>,
        pub start_time: Instant,
    }

    impl PerformanceMonitor {
        pub fn new() -> Self {
            Self {
                cycles_processed: 0,
                errors_corrected: 0,
                false_positives: 0,
                latency_histogram: Vec::new(),
                throughput_samples: VecDeque::new(),
                start_time: Instant::now(),
            }
        }

        pub fn record_cycle(&mut self, latency: Duration, error_corrected: bool) {
            self.cycles_processed += 1;
            if error_corrected {
                self.errors_corrected += 1;
            }
            self.latency_histogram.push(latency);

            // Calculate current throughput
            let elapsed = self.start_time.elapsed();
            if elapsed.as_secs_f64() > 0.0 {
                let throughput = self.cycles_processed as f64 / elapsed.as_secs_f64();
                self.throughput_samples.push_back(throughput);

                // Keep only recent samples
                if self.throughput_samples.len() > 100 {
                    self.throughput_samples.pop_front();
                }
            }
        }

        pub fn average_latency(&self) -> Duration {
            if self.latency_histogram.is_empty() {
                return Duration::from_nanos(0);
            }

            let total_nanos: u64 = self
                .latency_histogram
                .iter()
                .map(|d| d.as_nanos() as u64)
                .sum();
            Duration::from_nanos(total_nanos / self.latency_histogram.len() as u64)
        }

        pub fn current_throughput(&self) -> f64 {
            self.throughput_samples.back().copied().unwrap_or(0.0)
        }

        pub fn error_correction_rate(&self) -> f64 {
            if self.cycles_processed == 0 {
                0.0
            } else {
                self.errors_corrected as f64 / self.cycles_processed as f64
            }
        }
    }

    impl SyndromeStreamProcessor {
        /// Create a new real-time syndrome stream processor
        pub fn new(
            decoder: Arc<dyn SyndromeDecoder + Send + Sync>,
            hardware: Arc<dyn QuantumHardwareInterface>,
            config: RealTimeConfig,
        ) -> Self {
            Self {
                buffer: Arc::new(Mutex::new(VecDeque::with_capacity(config.buffer_size))),
                decoder,
                hardware,
                performance_monitor: Arc::new(RwLock::new(PerformanceMonitor::new())),
                config,
            }
        }

        /// Start real-time error correction processing
        pub fn start_processing(&self) -> QuantRS2Result<thread::JoinHandle<()>> {
            let buffer = Arc::clone(&self.buffer);
            let decoder = Arc::clone(&self.decoder);
            let hardware = Arc::clone(&self.hardware);
            let monitor = Arc::clone(&self.performance_monitor);
            let config = self.config.clone();

            let handle = thread::spawn(move || {
                let mut sequence_number = 0u64;

                loop {
                    let cycle_start = Instant::now();

                    // Check if hardware is ready
                    if !hardware.is_ready() {
                        thread::sleep(Duration::from_micros(10));
                        continue;
                    }

                    // Measure syndromes from hardware
                    match hardware.measure_syndromes() {
                        Ok(syndrome) => {
                            let packet = SyndromePacket {
                                syndrome: syndrome.clone(),
                                timestamp: Instant::now(),
                                sequence_number,
                                measurement_fidelity: 0.99, // Would be measured from hardware
                            };

                            // Add to buffer
                            {
                                let mut buf = buffer.lock().unwrap();
                                if buf.len() >= config.buffer_size {
                                    buf.pop_front(); // Remove oldest if buffer full
                                }
                                buf.push_back(packet);
                            }

                            // Process syndrome if not all zeros (no error)
                            let has_error = syndrome.iter().any(|&x| x);
                            let mut error_corrected = false;

                            if has_error {
                                match decoder.decode(&syndrome) {
                                    Ok(correction) => {
                                        match hardware.apply_correction(&correction) {
                                            Ok(()) => {
                                                error_corrected = true;
                                            }
                                            Err(e) => {
                                                eprintln!("Failed to apply correction: {}", e);
                                            }
                                        }
                                    }
                                    Err(e) => {
                                        eprintln!("Decoding failed: {}", e);
                                    }
                                }
                            }

                            // Record performance metrics
                            let cycle_time = cycle_start.elapsed();
                            {
                                let mut mon = monitor.write().unwrap();
                                mon.record_cycle(cycle_time, error_corrected);
                            }

                            // Check latency constraint
                            if cycle_time > config.max_latency {
                                eprintln!("Warning: Error correction cycle exceeded max latency: {:?} > {:?}",
                                         cycle_time, config.max_latency);
                            }

                            sequence_number += 1;
                        }
                        Err(e) => {
                            eprintln!("Failed to measure syndromes: {}", e);
                            thread::sleep(Duration::from_micros(10));
                        }
                    }

                    // Small sleep to prevent busy waiting
                    thread::sleep(Duration::from_micros(1));
                }
            });

            Ok(handle)
        }

        /// Get current performance statistics
        pub fn get_performance_stats(&self) -> PerformanceMonitor {
            (*self.performance_monitor.read().unwrap()).clone()
        }

        /// Get syndrome buffer status
        pub fn get_buffer_status(&self) -> (usize, usize) {
            let buffer = self.buffer.lock().unwrap();
            (buffer.len(), self.config.buffer_size)
        }
    }

    /// Adaptive threshold decoder that learns from hardware feedback
    pub struct AdaptiveThresholdDecoder {
        base_decoder: Arc<dyn SyndromeDecoder + Send + Sync>,
        error_characteristics: Arc<RwLock<HardwareErrorCharacteristics>>,
        learning_rate: f64,
        threshold_history: VecDeque<f64>,
    }

    impl AdaptiveThresholdDecoder {
        pub fn new(
            base_decoder: Arc<dyn SyndromeDecoder + Send + Sync>,
            initial_characteristics: HardwareErrorCharacteristics,
        ) -> Self {
            Self {
                base_decoder,
                error_characteristics: Arc::new(RwLock::new(initial_characteristics)),
                learning_rate: 0.01,
                threshold_history: VecDeque::with_capacity(1000),
            }
        }

        /// Update error characteristics based on hardware feedback
        pub fn update_characteristics(
            &mut self,
            new_characteristics: HardwareErrorCharacteristics,
        ) {
            *self.error_characteristics.write().unwrap() = new_characteristics;
        }

        /// Adapt thresholds based on observed error patterns
        pub fn adapt_thresholds(&mut self, syndrome: &[bool], correction_success: bool) {
            let error_weight = syndrome.iter().filter(|&&x| x).count() as f64;

            if correction_success {
                // Increase confidence in current threshold
                self.threshold_history.push_back(error_weight);
            } else {
                // Decrease threshold to be more aggressive
                self.threshold_history.push_back(error_weight * 0.8);
            }

            if self.threshold_history.len() > 100 {
                self.threshold_history.pop_front();
            }
        }

        /// Get current adaptive threshold
        pub fn current_threshold(&self) -> f64 {
            if self.threshold_history.is_empty() {
                return 1.0; // Default threshold
            }

            let sum: f64 = self.threshold_history.iter().sum();
            sum / self.threshold_history.len() as f64
        }
    }

    impl SyndromeDecoder for AdaptiveThresholdDecoder {
        fn decode(&self, syndrome: &[bool]) -> QuantRS2Result<PauliString> {
            let threshold = self.current_threshold();
            let error_weight = syndrome.iter().filter(|&&x| x).count() as f64;

            // Use adaptive threshold to decide decoding strategy
            if error_weight > threshold {
                // High-confidence error: use more aggressive decoding
                self.base_decoder.decode(syndrome)
            } else {
                // Low-confidence: use conservative approach or no correction
                Ok(PauliString::new(vec![Pauli::I; syndrome.len()]))
            }
        }
    }

    /// Parallel syndrome decoder for high-throughput error correction
    pub struct ParallelSyndromeDecoder {
        base_decoder: Arc<dyn SyndromeDecoder + Send + Sync>,
        worker_count: usize,
    }

    impl ParallelSyndromeDecoder {
        pub fn new(
            base_decoder: Arc<dyn SyndromeDecoder + Send + Sync>,
            worker_count: usize,
        ) -> Self {
            Self {
                base_decoder,
                worker_count,
            }
        }

        /// Decode multiple syndromes in parallel
        pub fn decode_batch(&self, syndromes: &[Vec<bool>]) -> QuantRS2Result<Vec<PauliString>> {
            let chunk_size = (syndromes.len() + self.worker_count - 1) / self.worker_count;
            let mut handles = Vec::new();

            for chunk in syndromes.chunks(chunk_size) {
                let decoder = Arc::clone(&self.base_decoder);
                let chunk_data: Vec<Vec<bool>> = chunk.to_vec();

                let handle = thread::spawn(move || {
                    let mut results = Vec::new();
                    for syndrome in chunk_data {
                        match decoder.decode(&syndrome) {
                            Ok(correction) => results.push(correction),
                            Err(_) => {
                                results.push(PauliString::new(vec![Pauli::I; syndrome.len()]))
                            }
                        }
                    }
                    results
                });

                handles.push(handle);
            }

            let mut all_results = Vec::new();
            for handle in handles {
                match handle.join() {
                    Ok(chunk_results) => all_results.extend(chunk_results),
                    Err(_) => {
                        return Err(QuantRS2Error::ComputationError(
                            "Parallel decoding failed".to_string(),
                        ))
                    }
                }
            }

            Ok(all_results)
        }
    }

    impl SyndromeDecoder for ParallelSyndromeDecoder {
        fn decode(&self, syndrome: &[bool]) -> QuantRS2Result<PauliString> {
            self.base_decoder.decode(syndrome)
        }
    }

    /// Mock hardware interface for testing
    pub struct MockQuantumHardware {
        error_rate: f64,
        latency: Duration,
        syndrome_length: usize,
    }

    impl MockQuantumHardware {
        pub fn new(error_rate: f64, latency: Duration, syndrome_length: usize) -> Self {
            Self {
                error_rate,
                latency,
                syndrome_length,
            }
        }
    }

    impl QuantumHardwareInterface for MockQuantumHardware {
        fn measure_syndromes(&self) -> QuantRS2Result<Vec<bool>> {
            thread::sleep(self.latency);

            // Simulate random syndrome measurements
            use scirs2_core::random::prelude::*;
            let mut rng = thread_rng();
            let mut syndrome = vec![false; self.syndrome_length];
            for i in 0..self.syndrome_length {
                if rng.gen::<f64>() < self.error_rate {
                    syndrome[i] = true;
                }
            }

            Ok(syndrome)
        }

        fn apply_correction(&self, _correction: &PauliString) -> QuantRS2Result<()> {
            thread::sleep(self.latency / 2);
            Ok(())
        }

        fn get_error_characteristics(&self) -> QuantRS2Result<HardwareErrorCharacteristics> {
            Ok(HardwareErrorCharacteristics {
                single_qubit_error_rates: vec![self.error_rate; self.syndrome_length],
                two_qubit_error_rates: vec![self.error_rate * 10.0; self.syndrome_length / 2],
                measurement_error_rates: vec![self.error_rate * 0.1; self.syndrome_length],
                correlated_errors: Vec::new(),
                temporal_variation: 0.01,
            })
        }

        fn is_ready(&self) -> bool {
            true
        }

        fn get_latency_stats(&self) -> QuantRS2Result<LatencyStats> {
            Ok(LatencyStats {
                syndrome_measurement_time: self.latency,
                decoding_time: Duration::from_micros(10),
                correction_application_time: self.latency / 2,
                total_cycle_time: self.latency + Duration::from_micros(10) + self.latency / 2,
                throughput_hz: 1.0 / (self.latency.as_secs_f64() * 1.5 + 10e-6),
            })
        }
    }
}

/// Logical gate synthesis for fault-tolerant computing
/// This module provides the ability to implement logical operations on encoded quantum states
/// without decoding them first, which is essential for fault-tolerant quantum computation.
pub mod logical_gates {
    use super::*;

    /// Logical gate operation that can be applied to encoded quantum states
    #[derive(Debug, Clone)]
    pub struct LogicalGateOp {
        /// The stabilizer code this logical gate operates on
        pub code: StabilizerCode,
        /// Physical gate operations that implement the logical gate
        pub physical_operations: Vec<PhysicalGateSequence>,
        /// Which logical qubit(s) this gate acts on
        pub logical_qubits: Vec<usize>,
        /// Error propagation analysis
        pub error_propagation: ErrorPropagationAnalysis,
    }

    /// Sequence of physical gates that implement part of a logical gate
    #[derive(Debug, Clone)]
    pub struct PhysicalGateSequence {
        /// Target physical qubits
        pub target_qubits: Vec<usize>,
        /// Pauli operators to apply
        pub pauli_sequence: Vec<PauliString>,
        /// Timing constraints (if any)
        pub timing_constraints: Option<TimingConstraints>,
        /// Error correction rounds needed
        pub error_correction_rounds: usize,
    }

    /// Analysis of how errors propagate through logical gates
    #[derive(Debug, Clone)]
    pub struct ErrorPropagationAnalysis {
        /// How single-qubit errors propagate
        pub single_qubit_propagation: Vec<ErrorPropagationPath>,
        /// How two-qubit errors propagate
        pub two_qubit_propagation: Vec<ErrorPropagationPath>,
        /// Maximum error weight after gate application
        pub max_error_weight: usize,
        /// Fault-tolerance threshold
        pub fault_tolerance_threshold: f64,
    }

    /// Path of error propagation through a logical gate
    #[derive(Debug, Clone)]
    pub struct ErrorPropagationPath {
        /// Initial error location
        pub initial_error: PauliString,
        /// Final error after gate application
        pub final_error: PauliString,
        /// Probability of this propagation path
        pub probability: f64,
        /// Whether this path can be corrected
        pub correctable: bool,
    }

    /// Timing constraints for fault-tolerant gate implementation
    #[derive(Debug, Clone)]
    pub struct TimingConstraints {
        /// Maximum time between operations
        pub max_operation_time: std::time::Duration,
        /// Required synchronization points
        pub sync_points: Vec<usize>,
        /// Parallel operation groups
        pub parallel_groups: Vec<Vec<usize>>,
    }

    /// Logical gate synthesis engine
    pub struct LogicalGateSynthesizer {
        /// Available error correction codes
        codes: Vec<StabilizerCode>,
        /// Synthesis strategies
        strategies: Vec<SynthesisStrategy>,
        /// Error threshold for fault tolerance
        error_threshold: f64,
    }

    /// Strategy for synthesizing logical gates
    #[derive(Debug, Clone)]
    pub enum SynthesisStrategy {
        /// Transversal gates (apply same gate to all qubits)
        Transversal,
        /// Magic state distillation and injection
        MagicStateDistillation,
        /// Lattice surgery operations
        LatticeSurgery,
        /// Code deformation
        CodeDeformation,
        /// Braiding operations (for topological codes)
        Braiding,
    }

    impl LogicalGateSynthesizer {
        /// Create a new logical gate synthesizer
        pub fn new(error_threshold: f64) -> Self {
            Self {
                codes: Vec::new(),
                strategies: vec![
                    SynthesisStrategy::Transversal,
                    SynthesisStrategy::MagicStateDistillation,
                    SynthesisStrategy::LatticeSurgery,
                ],
                error_threshold,
            }
        }

        /// Add an error correction code to the synthesizer
        pub fn add_code(&mut self, code: StabilizerCode) {
            self.codes.push(code);
        }

        /// Synthesize a logical Pauli-X gate
        pub fn synthesize_logical_x(
            &self,
            code: &StabilizerCode,
            logical_qubit: usize,
        ) -> QuantRS2Result<LogicalGateOp> {
            if logical_qubit >= code.k {
                return Err(QuantRS2Error::InvalidInput(format!(
                    "Logical qubit {} exceeds code dimension {}",
                    logical_qubit, code.k
                )));
            }

            // For most stabilizer codes, logical X can be implemented transversally
            let logical_x_operator = &code.logical_x[logical_qubit];

            let physical_ops = vec![PhysicalGateSequence {
                target_qubits: (0..code.n).collect(),
                pauli_sequence: vec![logical_x_operator.clone()],
                timing_constraints: None,
                error_correction_rounds: 1,
            }];

            let error_analysis = self.analyze_error_propagation(code, &physical_ops)?;

            Ok(LogicalGateOp {
                code: code.clone(),
                physical_operations: physical_ops,
                logical_qubits: vec![logical_qubit],
                error_propagation: error_analysis,
            })
        }

        /// Synthesize a logical Pauli-Z gate
        pub fn synthesize_logical_z(
            &self,
            code: &StabilizerCode,
            logical_qubit: usize,
        ) -> QuantRS2Result<LogicalGateOp> {
            if logical_qubit >= code.k {
                return Err(QuantRS2Error::InvalidInput(format!(
                    "Logical qubit {} exceeds code dimension {}",
                    logical_qubit, code.k
                )));
            }

            let logical_z_operator = &code.logical_z[logical_qubit];

            let physical_ops = vec![PhysicalGateSequence {
                target_qubits: (0..code.n).collect(),
                pauli_sequence: vec![logical_z_operator.clone()],
                timing_constraints: None,
                error_correction_rounds: 1,
            }];

            let error_analysis = self.analyze_error_propagation(code, &physical_ops)?;

            Ok(LogicalGateOp {
                code: code.clone(),
                physical_operations: physical_ops,
                logical_qubits: vec![logical_qubit],
                error_propagation: error_analysis,
            })
        }

        /// Synthesize a logical Hadamard gate
        pub fn synthesize_logical_h(
            &self,
            code: &StabilizerCode,
            logical_qubit: usize,
        ) -> QuantRS2Result<LogicalGateOp> {
            if logical_qubit >= code.k {
                return Err(QuantRS2Error::InvalidInput(format!(
                    "Logical qubit {} exceeds code dimension {}",
                    logical_qubit, code.k
                )));
            }

            // Hadamard can often be implemented transversally
            // H|x⟩ = |+⟩ if x=0, |-⟩ if x=1, and H swaps X and Z operators
            let physical_ops = vec![PhysicalGateSequence {
                target_qubits: (0..code.n).collect(),
                pauli_sequence: self.generate_hadamard_sequence(code, logical_qubit)?,
                timing_constraints: Some(TimingConstraints {
                    max_operation_time: std::time::Duration::from_micros(100),
                    sync_points: vec![code.n / 2],
                    parallel_groups: vec![(0..code.n).collect()],
                }),
                error_correction_rounds: 2, // Need more rounds for non-Pauli gates
            }];

            let error_analysis = self.analyze_error_propagation(code, &physical_ops)?;

            Ok(LogicalGateOp {
                code: code.clone(),
                physical_operations: physical_ops,
                logical_qubits: vec![logical_qubit],
                error_propagation: error_analysis,
            })
        }

        /// Synthesize a logical CNOT gate
        pub fn synthesize_logical_cnot(
            &self,
            code: &StabilizerCode,
            control_qubit: usize,
            target_qubit: usize,
        ) -> QuantRS2Result<LogicalGateOp> {
            if control_qubit >= code.k || target_qubit >= code.k {
                return Err(QuantRS2Error::InvalidInput(
                    "Control or target qubit exceeds code dimension".to_string(),
                ));
            }

            if control_qubit == target_qubit {
                return Err(QuantRS2Error::InvalidInput(
                    "Control and target qubits must be different".to_string(),
                ));
            }

            // CNOT can be implemented transversally for many codes
            let cnot_sequence = self.generate_cnot_sequence(code, control_qubit, target_qubit)?;

            let physical_ops = vec![PhysicalGateSequence {
                target_qubits: (0..code.n).collect(),
                pauli_sequence: cnot_sequence,
                timing_constraints: Some(TimingConstraints {
                    max_operation_time: std::time::Duration::from_micros(200),
                    sync_points: vec![],
                    parallel_groups: vec![], // CNOT requires sequential operations
                }),
                error_correction_rounds: 2,
            }];

            let error_analysis = self.analyze_error_propagation(code, &physical_ops)?;

            Ok(LogicalGateOp {
                code: code.clone(),
                physical_operations: physical_ops,
                logical_qubits: vec![control_qubit, target_qubit],
                error_propagation: error_analysis,
            })
        }

        /// Synthesize a T gate using magic state distillation
        pub fn synthesize_logical_t(
            &self,
            code: &StabilizerCode,
            logical_qubit: usize,
        ) -> QuantRS2Result<LogicalGateOp> {
            if logical_qubit >= code.k {
                return Err(QuantRS2Error::InvalidInput(format!(
                    "Logical qubit {} exceeds code dimension {}",
                    logical_qubit, code.k
                )));
            }

            // T gate requires magic state distillation for fault-tolerant implementation
            let magic_state_prep = self.prepare_magic_state(code)?;
            let injection_sequence =
                self.inject_magic_state(code, logical_qubit, &magic_state_prep)?;

            let physical_ops = vec![magic_state_prep, injection_sequence];

            let error_analysis = self.analyze_error_propagation(code, &physical_ops)?;

            Ok(LogicalGateOp {
                code: code.clone(),
                physical_operations: physical_ops,
                logical_qubits: vec![logical_qubit],
                error_propagation: error_analysis,
            })
        }

        /// Generate Hadamard gate sequence for a logical qubit
        fn generate_hadamard_sequence(
            &self,
            code: &StabilizerCode,
            _logical_qubit: usize,
        ) -> QuantRS2Result<Vec<PauliString>> {
            // For transversal Hadamard, apply H to each physical qubit
            // This swaps X and Z logical operators
            let mut sequence = Vec::new();

            // Create a Pauli string that represents applying H to all qubits
            // Since H|0⟩ = |+⟩ = (|0⟩ + |1⟩)/√2 and H|1⟩ = |-⟩ = (|0⟩ - |1⟩)/√2
            // We represent this as identity for simplicity in this implementation
            sequence.push(PauliString::new(vec![Pauli::I; code.n]));

            Ok(sequence)
        }

        /// Generate CNOT gate sequence for logical qubits
        fn generate_cnot_sequence(
            &self,
            code: &StabilizerCode,
            _control: usize,
            _target: usize,
        ) -> QuantRS2Result<Vec<PauliString>> {
            // For transversal CNOT, apply CNOT between corresponding physical qubits
            // This is a simplified implementation
            let mut sequence = Vec::new();

            // Represent CNOT as identity for this implementation
            sequence.push(PauliString::new(vec![Pauli::I; code.n]));

            Ok(sequence)
        }

        /// Prepare magic state for T gate implementation
        fn prepare_magic_state(
            &self,
            code: &StabilizerCode,
        ) -> QuantRS2Result<PhysicalGateSequence> {
            // Magic state |T⟩ = (|0⟩ + e^(iπ/4)|1⟩)/√2 for T gate
            // This is a simplified implementation
            Ok(PhysicalGateSequence {
                target_qubits: (0..code.n).collect(),
                pauli_sequence: vec![PauliString::new(vec![Pauli::I; code.n])],
                timing_constraints: Some(TimingConstraints {
                    max_operation_time: std::time::Duration::from_millis(1),
                    sync_points: vec![],
                    parallel_groups: vec![(0..code.n).collect()],
                }),
                error_correction_rounds: 5, // Magic state prep requires many rounds
            })
        }

        /// Inject magic state to implement T gate
        fn inject_magic_state(
            &self,
            code: &StabilizerCode,
            _logical_qubit: usize,
            _magic_state: &PhysicalGateSequence,
        ) -> QuantRS2Result<PhysicalGateSequence> {
            // Inject magic state using teleportation-based approach
            Ok(PhysicalGateSequence {
                target_qubits: (0..code.n).collect(),
                pauli_sequence: vec![PauliString::new(vec![Pauli::I; code.n])],
                timing_constraints: Some(TimingConstraints {
                    max_operation_time: std::time::Duration::from_micros(500),
                    sync_points: vec![code.n / 2],
                    parallel_groups: vec![],
                }),
                error_correction_rounds: 3,
            })
        }

        /// Analyze error propagation through logical gate operations
        fn analyze_error_propagation(
            &self,
            code: &StabilizerCode,
            physical_ops: &[PhysicalGateSequence],
        ) -> QuantRS2Result<ErrorPropagationAnalysis> {
            let mut single_qubit_propagation = Vec::new();
            let mut two_qubit_propagation = Vec::new();
            let mut max_error_weight = 0;

            // Analyze single-qubit errors
            for i in 0..code.n {
                for pauli in [Pauli::X, Pauli::Y, Pauli::Z] {
                    let mut initial_error = vec![Pauli::I; code.n];
                    initial_error[i] = pauli;
                    let initial_pauli_string = PauliString::new(initial_error);

                    // Simulate error propagation through the logical gate
                    let final_error = self.propagate_error(&initial_pauli_string, physical_ops)?;
                    let error_weight = final_error.weight();
                    max_error_weight = max_error_weight.max(error_weight);

                    // Check if error is correctable
                    let correctable = self.is_error_correctable(code, &final_error)?;

                    single_qubit_propagation.push(ErrorPropagationPath {
                        initial_error: initial_pauli_string,
                        final_error,
                        probability: 1.0 / (3.0 * code.n as f64), // Uniform for now
                        correctable,
                    });
                }
            }

            // Analyze two-qubit errors (simplified)
            for i in 0..code.n.min(5) {
                // Limit to first 5 for performance
                for j in (i + 1)..code.n.min(5) {
                    let mut initial_error = vec![Pauli::I; code.n];
                    initial_error[i] = Pauli::X;
                    initial_error[j] = Pauli::X;
                    let initial_pauli_string = PauliString::new(initial_error);

                    let final_error = self.propagate_error(&initial_pauli_string, physical_ops)?;
                    let error_weight = final_error.weight();
                    max_error_weight = max_error_weight.max(error_weight);

                    let correctable = self.is_error_correctable(code, &final_error)?;

                    two_qubit_propagation.push(ErrorPropagationPath {
                        initial_error: initial_pauli_string,
                        final_error,
                        probability: 1.0 / (code.n * (code.n - 1)) as f64,
                        correctable,
                    });
                }
            }

            Ok(ErrorPropagationAnalysis {
                single_qubit_propagation,
                two_qubit_propagation,
                max_error_weight,
                fault_tolerance_threshold: self.error_threshold,
            })
        }

        /// Propagate an error through physical gate operations
        fn propagate_error(
            &self,
            error: &PauliString,
            _physical_ops: &[PhysicalGateSequence],
        ) -> QuantRS2Result<PauliString> {
            // Simplified error propagation - in reality this would track
            // how each gate operation transforms the error
            Ok(error.clone())
        }

        /// Check if an error is correctable by the code
        fn is_error_correctable(
            &self,
            code: &StabilizerCode,
            error: &PauliString,
        ) -> QuantRS2Result<bool> {
            // An error is correctable if its weight is less than (d+1)/2
            // where d is the minimum distance of the code
            Ok(error.weight() <= (code.d + 1) / 2)
        }
    }

    /// Logical circuit synthesis for fault-tolerant quantum computing
    pub struct LogicalCircuitSynthesizer {
        gate_synthesizer: LogicalGateSynthesizer,
        optimization_passes: Vec<OptimizationPass>,
    }

    /// Optimization pass for logical circuits
    #[derive(Debug, Clone)]
    pub enum OptimizationPass {
        /// Combine adjacent Pauli gates
        PauliOptimization,
        /// Optimize error correction rounds
        ErrorCorrectionOptimization,
        /// Parallelize commuting operations
        ParallelizationOptimization,
        /// Reduce magic state usage
        MagicStateOptimization,
    }

    impl LogicalCircuitSynthesizer {
        pub fn new(error_threshold: f64) -> Self {
            Self {
                gate_synthesizer: LogicalGateSynthesizer::new(error_threshold),
                optimization_passes: vec![
                    OptimizationPass::PauliOptimization,
                    OptimizationPass::ErrorCorrectionOptimization,
                    OptimizationPass::ParallelizationOptimization,
                    OptimizationPass::MagicStateOptimization,
                ],
            }
        }

        /// Add a code to the synthesizer
        pub fn add_code(&mut self, code: StabilizerCode) {
            self.gate_synthesizer.add_code(code);
        }

        /// Synthesize a logical circuit from a sequence of gate names
        pub fn synthesize_circuit(
            &self,
            code: &StabilizerCode,
            gate_sequence: &[(&str, Vec<usize>)], // (gate_name, target_qubits)
        ) -> QuantRS2Result<Vec<LogicalGateOp>> {
            let mut logical_gates = Vec::new();

            for (gate_name, targets) in gate_sequence {
                match gate_name.to_lowercase().as_str() {
                    "x" | "pauli_x" => {
                        if targets.len() != 1 {
                            return Err(QuantRS2Error::InvalidInput(
                                "X gate requires exactly one target".to_string(),
                            ));
                        }
                        logical_gates.push(
                            self.gate_synthesizer
                                .synthesize_logical_x(code, targets[0])?,
                        );
                    }
                    "z" | "pauli_z" => {
                        if targets.len() != 1 {
                            return Err(QuantRS2Error::InvalidInput(
                                "Z gate requires exactly one target".to_string(),
                            ));
                        }
                        logical_gates.push(
                            self.gate_synthesizer
                                .synthesize_logical_z(code, targets[0])?,
                        );
                    }
                    "h" | "hadamard" => {
                        if targets.len() != 1 {
                            return Err(QuantRS2Error::InvalidInput(
                                "H gate requires exactly one target".to_string(),
                            ));
                        }
                        logical_gates.push(
                            self.gate_synthesizer
                                .synthesize_logical_h(code, targets[0])?,
                        );
                    }
                    "cnot" | "cx" => {
                        if targets.len() != 2 {
                            return Err(QuantRS2Error::InvalidInput(
                                "CNOT gate requires exactly two targets".to_string(),
                            ));
                        }
                        logical_gates.push(
                            self.gate_synthesizer
                                .synthesize_logical_cnot(code, targets[0], targets[1])?,
                        );
                    }
                    "t" => {
                        if targets.len() != 1 {
                            return Err(QuantRS2Error::InvalidInput(
                                "T gate requires exactly one target".to_string(),
                            ));
                        }
                        logical_gates.push(
                            self.gate_synthesizer
                                .synthesize_logical_t(code, targets[0])?,
                        );
                    }
                    _ => {
                        return Err(QuantRS2Error::UnsupportedOperation(format!(
                            "Unsupported logical gate: {}",
                            gate_name
                        )));
                    }
                }
            }

            // Apply optimization passes
            self.optimize_circuit(logical_gates)
        }

        /// Apply optimization passes to the logical circuit
        fn optimize_circuit(
            &self,
            mut circuit: Vec<LogicalGateOp>,
        ) -> QuantRS2Result<Vec<LogicalGateOp>> {
            for pass in &self.optimization_passes {
                circuit = self.apply_optimization_pass(circuit, pass)?;
            }
            Ok(circuit)
        }

        /// Apply a specific optimization pass
        fn apply_optimization_pass(
            &self,
            circuit: Vec<LogicalGateOp>,
            pass: &OptimizationPass,
        ) -> QuantRS2Result<Vec<LogicalGateOp>> {
            match pass {
                OptimizationPass::PauliOptimization => self.optimize_pauli_gates(circuit),
                OptimizationPass::ErrorCorrectionOptimization => {
                    self.optimize_error_correction(circuit)
                }
                OptimizationPass::ParallelizationOptimization => {
                    self.optimize_parallelization(circuit)
                }
                OptimizationPass::MagicStateOptimization => self.optimize_magic_states(circuit),
            }
        }

        /// Optimize Pauli gate sequences
        fn optimize_pauli_gates(
            &self,
            circuit: Vec<LogicalGateOp>,
        ) -> QuantRS2Result<Vec<LogicalGateOp>> {
            // Combine adjacent Pauli gates that act on the same logical qubits
            Ok(circuit) // Simplified implementation
        }

        /// Optimize error correction rounds
        fn optimize_error_correction(
            &self,
            circuit: Vec<LogicalGateOp>,
        ) -> QuantRS2Result<Vec<LogicalGateOp>> {
            // Reduce redundant error correction rounds
            Ok(circuit) // Simplified implementation
        }

        /// Optimize parallelization of commuting operations
        fn optimize_parallelization(
            &self,
            circuit: Vec<LogicalGateOp>,
        ) -> QuantRS2Result<Vec<LogicalGateOp>> {
            // Identify and parallelize commuting gates
            Ok(circuit) // Simplified implementation
        }

        /// Optimize magic state usage
        fn optimize_magic_states(
            &self,
            circuit: Vec<LogicalGateOp>,
        ) -> QuantRS2Result<Vec<LogicalGateOp>> {
            // Reduce number of magic states required
            Ok(circuit) // Simplified implementation
        }

        /// Estimate resource requirements for the logical circuit
        pub fn estimate_resources(&self, circuit: &[LogicalGateOp]) -> LogicalCircuitResources {
            let mut total_physical_operations = 0;
            let mut total_error_correction_rounds = 0;
            let mut max_parallelism = 0;
            let mut magic_states_required = 0;

            for gate in circuit {
                total_physical_operations += gate.physical_operations.len();
                for op in &gate.physical_operations {
                    total_error_correction_rounds += op.error_correction_rounds;
                    if let Some(constraints) = &op.timing_constraints {
                        max_parallelism = max_parallelism.max(constraints.parallel_groups.len());
                    }
                }

                // Count T gates which require magic states
                if gate.logical_qubits.len() == 1 {
                    // This is a heuristic - in practice we'd check the gate type
                    magic_states_required += 1;
                }
            }

            LogicalCircuitResources {
                total_physical_operations,
                total_error_correction_rounds,
                max_parallelism,
                magic_states_required,
                estimated_depth: circuit.len(),
                estimated_time: std::time::Duration::from_millis(
                    (total_error_correction_rounds * 10) as u64,
                ),
            }
        }
    }

    /// Resource requirements for a logical circuit
    #[derive(Debug, Clone)]
    pub struct LogicalCircuitResources {
        pub total_physical_operations: usize,
        pub total_error_correction_rounds: usize,
        pub max_parallelism: usize,
        pub magic_states_required: usize,
        pub estimated_depth: usize,
        pub estimated_time: std::time::Duration,
    }
}

/// Noise-adaptive error correction threshold estimation
/// This module provides dynamic adjustment of error correction thresholds based on
/// observed noise characteristics and environmental conditions for optimal performance.
pub mod adaptive_threshold {
    use super::*;
    use std::collections::{HashMap, VecDeque};
    use std::time::{Duration, Instant};

    /// Adaptive threshold estimator for error correction
    pub struct AdaptiveThresholdEstimator {
        /// Historical error pattern data
        error_history: VecDeque<ErrorObservation>,
        /// Current noise model parameters
        noise_model: NoiseModel,
        /// Threshold estimation algorithm
        estimation_algorithm: ThresholdEstimationAlgorithm,
        /// Performance metrics
        performance_tracker: PerformanceTracker,
        /// Configuration parameters
        config: AdaptiveConfig,
    }

    /// Observation of an error and correction attempt
    #[derive(Debug, Clone)]
    pub struct ErrorObservation {
        /// Syndrome measured
        pub syndrome: Vec<bool>,
        /// Correction applied
        pub correction: PauliString,
        /// Whether correction was successful
        pub success: bool,
        /// Measured error rate at this time
        pub observed_error_rate: f64,
        /// Timestamp of observation
        pub timestamp: Instant,
        /// Environmental conditions
        pub environment: EnvironmentalConditions,
    }

    /// Environmental conditions affecting error rates
    #[derive(Debug, Clone)]
    pub struct EnvironmentalConditions {
        /// Temperature in Kelvin
        pub temperature: f64,
        /// Magnetic field strength in Tesla
        pub magnetic_field: f64,
        /// Vibration level (arbitrary units)
        pub vibration_level: f64,
        /// Electromagnetic interference level
        pub emi_level: f64,
        /// Device uptime in seconds
        pub uptime: f64,
    }

    /// Noise model for quantum errors
    #[derive(Debug, Clone)]
    pub struct NoiseModel {
        /// Single-qubit error rates by qubit and error type
        pub single_qubit_rates: HashMap<(usize, Pauli), f64>,
        /// Two-qubit correlated error rates
        pub correlated_rates: HashMap<(usize, usize), f64>,
        /// Temporal correlation in errors
        pub temporal_correlation: f64,
        /// Environmental sensitivity coefficients
        pub environment_sensitivity: EnvironmentSensitivity,
        /// Model confidence (0.0 to 1.0)
        pub confidence: f64,
    }

    /// Sensitivity to environmental factors
    #[derive(Debug, Clone)]
    pub struct EnvironmentSensitivity {
        /// Temperature coefficient (per Kelvin)
        pub temperature_coeff: f64,
        /// Magnetic field coefficient (per Tesla)
        pub magnetic_field_coeff: f64,
        /// Vibration coefficient
        pub vibration_coeff: f64,
        /// EMI coefficient
        pub emi_coeff: f64,
        /// Drift coefficient (per second)
        pub drift_coeff: f64,
    }

    /// Algorithm for threshold estimation
    #[derive(Debug, Clone)]
    pub enum ThresholdEstimationAlgorithm {
        /// Bayesian inference with prior knowledge
        Bayesian {
            prior_strength: f64,
            update_rate: f64,
        },
        /// Machine learning based prediction
        MachineLearning {
            model_type: MLModelType,
            training_window: usize,
        },
        /// Kalman filter for dynamic estimation
        KalmanFilter {
            process_noise: f64,
            measurement_noise: f64,
        },
        /// Exponential moving average
        ExponentialAverage { alpha: f64 },
    }

    /// Machine learning model types
    #[derive(Debug, Clone)]
    pub enum MLModelType {
        LinearRegression,
        RandomForest,
        NeuralNetwork { hidden_layers: Vec<usize> },
        SupportVectorMachine,
    }

    /// Performance tracking for adaptive threshold
    #[derive(Debug, Clone)]
    pub struct PerformanceTracker {
        /// Number of successful corrections
        pub successful_corrections: u64,
        /// Number of failed corrections
        pub failed_corrections: u64,
        /// Number of false positives (unnecessary corrections)
        pub false_positives: u64,
        /// Number of false negatives (missed errors)
        pub false_negatives: u64,
        /// Average correction latency
        pub average_latency: Duration,
        /// Current threshold accuracy
        pub threshold_accuracy: f64,
    }

    /// Configuration for adaptive threshold estimation
    #[derive(Debug, Clone)]
    pub struct AdaptiveConfig {
        /// Maximum history size
        pub max_history_size: usize,
        /// Minimum observations before adaptation
        pub min_observations: usize,
        /// Update frequency
        pub update_frequency: Duration,
        /// Confidence threshold for model updates
        pub confidence_threshold: f64,
        /// Environmental monitoring enabled
        pub environmental_monitoring: bool,
        /// Real-time adaptation enabled
        pub real_time_adaptation: bool,
    }

    /// Threshold recommendation result
    #[derive(Debug, Clone)]
    pub struct ThresholdRecommendation {
        /// Recommended threshold value
        pub threshold: f64,
        /// Confidence in recommendation (0.0 to 1.0)
        pub confidence: f64,
        /// Predicted error rate
        pub predicted_error_rate: f64,
        /// Quality of recommendation (0.0 to 1.0)
        pub recommendation_quality: f64,
        /// Environmental impact assessment
        pub environmental_impact: f64,
    }

    impl Default for AdaptiveConfig {
        fn default() -> Self {
            Self {
                max_history_size: 10000,
                min_observations: 100,
                update_frequency: Duration::from_secs(30),
                confidence_threshold: 0.8,
                environmental_monitoring: true,
                real_time_adaptation: true,
            }
        }
    }

    impl Default for EnvironmentalConditions {
        fn default() -> Self {
            Self {
                temperature: 300.0, // Room temperature in Kelvin
                magnetic_field: 0.0,
                vibration_level: 0.0,
                emi_level: 0.0,
                uptime: 0.0,
            }
        }
    }

    impl Default for EnvironmentSensitivity {
        fn default() -> Self {
            Self {
                temperature_coeff: 1e-5,
                magnetic_field_coeff: 1e-3,
                vibration_coeff: 1e-4,
                emi_coeff: 1e-4,
                drift_coeff: 1e-7,
            }
        }
    }

    impl Default for NoiseModel {
        fn default() -> Self {
            Self {
                single_qubit_rates: HashMap::new(),
                correlated_rates: HashMap::new(),
                temporal_correlation: 0.1,
                environment_sensitivity: EnvironmentSensitivity::default(),
                confidence: 0.5,
            }
        }
    }

    impl PerformanceTracker {
        pub fn new() -> Self {
            Self {
                successful_corrections: 0,
                failed_corrections: 0,
                false_positives: 0,
                false_negatives: 0,
                average_latency: Duration::from_nanos(0),
                threshold_accuracy: 0.0,
            }
        }

        pub fn precision(&self) -> f64 {
            let total_positive = self.successful_corrections + self.false_positives;
            if total_positive == 0 {
                1.0
            } else {
                self.successful_corrections as f64 / total_positive as f64
            }
        }

        pub fn recall(&self) -> f64 {
            let total_actual_positive = self.successful_corrections + self.false_negatives;
            if total_actual_positive == 0 {
                1.0
            } else {
                self.successful_corrections as f64 / total_actual_positive as f64
            }
        }

        pub fn f1_score(&self) -> f64 {
            let p = self.precision();
            let r = self.recall();
            if p + r == 0.0 {
                0.0
            } else {
                2.0 * p * r / (p + r)
            }
        }
    }

    impl AdaptiveThresholdEstimator {
        /// Create a new adaptive threshold estimator
        pub fn new(
            initial_noise_model: NoiseModel,
            algorithm: ThresholdEstimationAlgorithm,
            config: AdaptiveConfig,
        ) -> Self {
            Self {
                error_history: VecDeque::with_capacity(config.max_history_size),
                noise_model: initial_noise_model,
                estimation_algorithm: algorithm,
                performance_tracker: PerformanceTracker::new(),
                config,
            }
        }

        /// Add a new error observation
        pub fn add_observation(&mut self, observation: ErrorObservation) {
            // Add to history
            if self.error_history.len() >= self.config.max_history_size {
                self.error_history.pop_front();
            }
            self.error_history.push_back(observation.clone());

            // Update performance tracking
            self.update_performance_tracking(&observation);

            // Update noise model if real-time adaptation is enabled
            if self.config.real_time_adaptation
                && self.error_history.len() >= self.config.min_observations
            {
                self.update_noise_model();
            }
        }

        /// Estimate current error correction threshold
        pub fn estimate_threshold(
            &self,
            syndrome: &[bool],
            environment: &EnvironmentalConditions,
        ) -> f64 {
            match &self.estimation_algorithm {
                ThresholdEstimationAlgorithm::Bayesian {
                    prior_strength,
                    update_rate,
                } => self.bayesian_threshold_estimation(
                    syndrome,
                    environment,
                    *prior_strength,
                    *update_rate,
                ),
                ThresholdEstimationAlgorithm::MachineLearning {
                    model_type,
                    training_window,
                } => self.ml_threshold_estimation(
                    syndrome,
                    environment,
                    model_type,
                    *training_window,
                ),
                ThresholdEstimationAlgorithm::KalmanFilter {
                    process_noise,
                    measurement_noise,
                } => self.kalman_threshold_estimation(
                    syndrome,
                    environment,
                    *process_noise,
                    *measurement_noise,
                ),
                ThresholdEstimationAlgorithm::ExponentialAverage { alpha } => {
                    self.exponential_average_threshold(syndrome, environment, *alpha)
                }
            }
        }

        /// Get current threshold recommendation
        pub fn get_threshold_recommendation(&self, syndrome: &[bool]) -> ThresholdRecommendation {
            let current_env = EnvironmentalConditions::default(); // Would get from sensors
            let threshold = self.estimate_threshold(syndrome, &current_env);
            let confidence = self.noise_model.confidence;
            let predicted_rate = self.predict_error_rate(&current_env, Duration::from_secs(60));

            ThresholdRecommendation {
                threshold,
                confidence,
                predicted_error_rate: predicted_rate,
                recommendation_quality: self.assess_recommendation_quality(),
                environmental_impact: self.assess_environmental_impact(&current_env),
            }
        }

        /// Predict future error rate based on current conditions
        pub fn predict_error_rate(
            &self,
            environment: &EnvironmentalConditions,
            horizon: Duration,
        ) -> f64 {
            let base_rate = self.calculate_base_error_rate();
            let environmental_factor = self.calculate_environmental_factor(environment);
            let temporal_factor = self.calculate_temporal_factor(horizon);

            base_rate * environmental_factor * temporal_factor
        }

        /// Bayesian threshold estimation
        fn bayesian_threshold_estimation(
            &self,
            syndrome: &[bool],
            environment: &EnvironmentalConditions,
            prior_strength: f64,
            update_rate: f64,
        ) -> f64 {
            let syndrome_weight = syndrome.iter().filter(|&&x| x).count() as f64;
            let base_threshold = self.calculate_base_threshold(syndrome_weight);

            // Update based on historical observations
            let historical_adjustment = self.calculate_historical_adjustment(update_rate);

            // Environmental adjustment
            let env_adjustment = self.calculate_environmental_adjustment(environment);

            // Bayesian update
            let prior = base_threshold;
            let likelihood_weight = 1.0 / (1.0 + prior_strength);

            prior * (1.0 - likelihood_weight)
                + (base_threshold + historical_adjustment + env_adjustment) * likelihood_weight
        }

        /// Machine learning based threshold estimation
        fn ml_threshold_estimation(
            &self,
            syndrome: &[bool],
            environment: &EnvironmentalConditions,
            model_type: &MLModelType,
            training_window: usize,
        ) -> f64 {
            // Extract features
            let features = self.extract_features(syndrome, environment);

            // Get recent training data
            let training_data = self.get_recent_observations(training_window);

            match model_type {
                MLModelType::LinearRegression => {
                    self.linear_regression_predict(&features, &training_data)
                }
                _ => {
                    // Simplified implementation for other ML models
                    self.linear_regression_predict(&features, &training_data)
                }
            }
        }

        /// Kalman filter based threshold estimation
        fn kalman_threshold_estimation(
            &self,
            syndrome: &[bool],
            _environment: &EnvironmentalConditions,
            process_noise: f64,
            measurement_noise: f64,
        ) -> f64 {
            let syndrome_weight = syndrome.iter().filter(|&&x| x).count() as f64;
            let base_threshold = self.calculate_base_threshold(syndrome_weight);

            // Simplified Kalman filter implementation
            let prediction_error = self.calculate_prediction_error();
            let kalman_gain = process_noise / (process_noise + measurement_noise);

            base_threshold + kalman_gain * prediction_error
        }

        /// Exponential moving average threshold estimation
        fn exponential_average_threshold(
            &self,
            syndrome: &[bool],
            _environment: &EnvironmentalConditions,
            alpha: f64,
        ) -> f64 {
            let syndrome_weight = syndrome.iter().filter(|&&x| x).count() as f64;
            let current_threshold = self.calculate_base_threshold(syndrome_weight);

            if let Some(_last_obs) = self.error_history.back() {
                let last_threshold = syndrome_weight; // Simplified
                alpha * current_threshold + (1.0 - alpha) * last_threshold
            } else {
                current_threshold
            }
        }

        // Helper methods
        fn calculate_base_error_rate(&self) -> f64 {
            if self.error_history.is_empty() {
                return 0.001; // Default 0.1% error rate
            }

            let recent_errors: Vec<_> = self.error_history.iter().rev().take(100).collect();

            let total_errors = recent_errors.len() as f64;
            let failed_corrections = recent_errors.iter().filter(|obs| !obs.success).count() as f64;

            failed_corrections / total_errors
        }

        fn calculate_environmental_factor(&self, environment: &EnvironmentalConditions) -> f64 {
            let sensitivity = &self.noise_model.environment_sensitivity;

            1.0 + sensitivity.temperature_coeff * (environment.temperature - 300.0)
                + sensitivity.magnetic_field_coeff * environment.magnetic_field
                + sensitivity.vibration_coeff * environment.vibration_level
                + sensitivity.emi_coeff * environment.emi_level
                + sensitivity.drift_coeff * environment.uptime
        }

        fn calculate_temporal_factor(&self, horizon: Duration) -> f64 {
            let temporal_corr = self.noise_model.temporal_correlation;
            let time_factor = horizon.as_secs_f64() / 3600.0; // Hours

            1.0 + temporal_corr * time_factor
        }

        fn calculate_base_threshold(&self, syndrome_weight: f64) -> f64 {
            // Simple heuristic: higher syndrome weight suggests higher error probability
            (syndrome_weight + 1.0) / 10.0
        }

        fn calculate_historical_adjustment(&self, update_rate: f64) -> f64 {
            if self.error_history.is_empty() {
                return 0.0;
            }

            let recent_success_rate = self.calculate_recent_success_rate();
            update_rate * (0.5 - recent_success_rate) // Adjust towards 50% success rate
        }

        fn calculate_environmental_adjustment(&self, environment: &EnvironmentalConditions) -> f64 {
            let env_factor = self.calculate_environmental_factor(environment);
            (env_factor - 1.0) * 0.1 // Scale environmental impact
        }

        fn calculate_recent_success_rate(&self) -> f64 {
            let recent_window = 50.min(self.error_history.len());
            if recent_window == 0 {
                return 0.5;
            }

            let recent_successes = self
                .error_history
                .iter()
                .rev()
                .take(recent_window)
                .filter(|obs| obs.success)
                .count();

            recent_successes as f64 / recent_window as f64
        }

        fn calculate_prediction_error(&self) -> f64 {
            // Simplified prediction error calculation
            let target_success_rate = 0.95;
            let actual_success_rate = self.calculate_recent_success_rate();
            target_success_rate - actual_success_rate
        }

        fn extract_features(
            &self,
            syndrome: &[bool],
            environment: &EnvironmentalConditions,
        ) -> Vec<f64> {
            let mut features = vec![
                syndrome.iter().filter(|&&x| x).count() as f64,
                environment.temperature,
                environment.magnetic_field,
                environment.vibration_level,
                environment.emi_level,
                environment.uptime,
            ];

            // Add syndrome pattern features
            for &bit in syndrome {
                features.push(if bit { 1.0 } else { 0.0 });
            }

            features
        }

        fn get_recent_observations(&self, window: usize) -> Vec<ErrorObservation> {
            self.error_history
                .iter()
                .rev()
                .take(window)
                .cloned()
                .collect()
        }

        fn linear_regression_predict(
            &self,
            _features: &[f64],
            training_data: &[ErrorObservation],
        ) -> f64 {
            // Simplified linear regression
            if training_data.is_empty() {
                return 0.5;
            }

            let avg_syndrome_weight: f64 = training_data
                .iter()
                .map(|obs| obs.syndrome.iter().filter(|&&x| x).count() as f64)
                .sum::<f64>()
                / training_data.len() as f64;

            (avg_syndrome_weight + 1.0) / 10.0
        }

        fn update_performance_tracking(&mut self, observation: &ErrorObservation) {
            if observation.success {
                self.performance_tracker.successful_corrections += 1;
            } else {
                self.performance_tracker.failed_corrections += 1;
            }

            // Update accuracy
            let total = self.performance_tracker.successful_corrections
                + self.performance_tracker.failed_corrections;
            if total > 0 {
                self.performance_tracker.threshold_accuracy =
                    self.performance_tracker.successful_corrections as f64 / total as f64;
            }
        }

        fn update_noise_model(&mut self) {
            let recent_window = self.config.min_observations.min(self.error_history.len());
            let recent_observations: Vec<ErrorObservation> = self
                .error_history
                .iter()
                .rev()
                .take(recent_window)
                .cloned()
                .collect();

            // Update single-qubit error rates
            self.update_single_qubit_rates(&recent_observations);

            // Update model confidence
            self.update_model_confidence(&recent_observations);
        }

        fn update_single_qubit_rates(&mut self, observations: &[ErrorObservation]) {
            // Update single-qubit error rates based on observations
            for obs in observations {
                for (i, pauli) in obs.correction.paulis.iter().enumerate() {
                    if *pauli != Pauli::I {
                        let key = (i, *pauli);
                        let current_rate = self
                            .noise_model
                            .single_qubit_rates
                            .get(&key)
                            .copied()
                            .unwrap_or(0.001);
                        let new_rate = if obs.success {
                            current_rate * 0.99
                        } else {
                            current_rate * 1.01
                        };
                        self.noise_model.single_qubit_rates.insert(key, new_rate);
                    }
                }
            }
        }

        fn update_model_confidence(&mut self, observations: &[ErrorObservation]) {
            if observations.is_empty() {
                return;
            }

            let success_rate = observations.iter().filter(|obs| obs.success).count() as f64
                / observations.len() as f64;

            // Higher success rate increases confidence, but not linearly
            let stability = 1.0 - (success_rate - 0.5).abs() * 2.0;
            self.noise_model.confidence = self.noise_model.confidence * 0.95 + stability * 0.05;
        }

        fn assess_recommendation_quality(&self) -> f64 {
            // Quality based on model confidence and recent performance
            let confidence_component = self.noise_model.confidence;
            let performance_component = self.performance_tracker.threshold_accuracy;
            let history_component =
                (self.error_history.len() as f64 / self.config.max_history_size as f64).min(1.0);

            (confidence_component + performance_component + history_component) / 3.0
        }

        fn assess_environmental_impact(&self, environment: &EnvironmentalConditions) -> f64 {
            let env_factor = self.calculate_environmental_factor(environment);
            (env_factor - 1.0).abs()
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_pauli_multiplication() {
        let (phase, result) = Pauli::X.multiply(&Pauli::Y);
        assert_eq!(result, Pauli::Z);
        assert_eq!(phase, Complex64::new(0.0, 1.0));
    }

    #[test]
    fn test_pauli_string_commutation() {
        let ps1 = PauliString::new(vec![Pauli::X, Pauli::I]);
        let ps2 = PauliString::new(vec![Pauli::Z, Pauli::I]);
        assert!(!ps1.commutes_with(&ps2).unwrap());

        let ps3 = PauliString::new(vec![Pauli::X, Pauli::I]);
        let ps4 = PauliString::new(vec![Pauli::I, Pauli::Z]);
        assert!(ps3.commutes_with(&ps4).unwrap());
    }

    #[test]
    fn test_repetition_code() {
        let code = StabilizerCode::repetition_code();
        assert_eq!(code.n, 3);
        assert_eq!(code.k, 1);
        assert_eq!(code.d, 1);

        // Test syndrome for X error on first qubit
        let error = PauliString::new(vec![Pauli::X, Pauli::I, Pauli::I]);
        let syndrome = code.syndrome(&error).unwrap();
        // X error anti-commutes with Z stabilizer on first two qubits
        assert_eq!(syndrome, vec![true, false]);
    }

    #[test]
    fn test_steane_code() {
        let code = StabilizerCode::steane_code();
        assert_eq!(code.n, 7);
        assert_eq!(code.k, 1);
        assert_eq!(code.d, 3);

        // Test that stabilizers commute
        for i in 0..code.stabilizers.len() {
            for j in i + 1..code.stabilizers.len() {
                assert!(code.stabilizers[i]
                    .commutes_with(&code.stabilizers[j])
                    .unwrap());
            }
        }
    }

    #[test]
    fn test_surface_code() {
        let surface = SurfaceCode::new(3, 3);
        assert_eq!(surface.distance(), 3);

        let code = surface.to_stabilizer_code();
        assert_eq!(code.n, 9);
        // For a 3x3 lattice, we have 2 X stabilizers and 2 Z stabilizers
        assert_eq!(code.stabilizers.len(), 4);
    }

    #[test]
    fn test_lookup_decoder() {
        let code = StabilizerCode::repetition_code();
        let decoder = LookupDecoder::new(&code).unwrap();

        // Test decoding trivial syndrome (no error)
        let trivial_syndrome = vec![false, false];
        let decoded = decoder.decode(&trivial_syndrome).unwrap();
        assert_eq!(decoded.weight(), 0); // Should be identity

        // Test single bit flip error
        let error = PauliString::new(vec![Pauli::X, Pauli::I, Pauli::I]);
        let syndrome = code.syndrome(&error).unwrap();

        // The decoder should be able to decode this syndrome
        if let Ok(decoded_error) = decoder.decode(&syndrome) {
            // Decoder should find a low-weight error
            assert!(decoded_error.weight() <= 1);
        }
    }

    #[test]
    fn test_concatenated_codes() {
        let inner_code = StabilizerCode::repetition_code();
        let outer_code = StabilizerCode::repetition_code();
        let concat_code = ConcatenatedCode::new(inner_code, outer_code);

        assert_eq!(concat_code.total_qubits(), 9); // 3 * 3
        assert_eq!(concat_code.logical_qubits(), 1);
        assert_eq!(concat_code.distance(), 1); // min(1, 1) = 1

        // Test encoding and decoding
        let logical_state = vec![Complex64::new(1.0, 0.0), Complex64::new(0.0, 0.0)];
        let encoded = concat_code.encode(&logical_state).unwrap();
        assert_eq!(encoded.len(), 512); // 2^9

        // Test error correction capability
        let error = PauliString::new(vec![
            Pauli::X,
            Pauli::I,
            Pauli::I,
            Pauli::I,
            Pauli::I,
            Pauli::I,
            Pauli::I,
            Pauli::I,
            Pauli::I,
        ]);
        let corrected = concat_code.correct_error(&encoded, &error).unwrap();

        // Verify the state is corrected (simplified check)
        assert_eq!(corrected.len(), 512);
    }

    #[test]
    fn test_hypergraph_product_codes() {
        // Create small classical codes for testing
        let h1 = Array2::from_shape_vec((2, 3), vec![1, 1, 0, 0, 1, 1]).unwrap();
        let h2 = Array2::from_shape_vec((2, 3), vec![1, 0, 1, 1, 1, 0]).unwrap();

        let hpc = HypergraphProductCode::new(h1, h2);

        // Check dimensions
        assert_eq!(hpc.n, 12); // n1*m2 + m1*n2 = 3*2 + 2*3 = 12
        assert_eq!(hpc.k, 1); // k = n1*k2 + k1*n2 - k1*k2 = 3*1 + 1*3 - 1*1 = 5 for this example, but simplified

        let stab_code = hpc.to_stabilizer_code();
        assert!(!stab_code.stabilizers.is_empty());
    }

    #[test]
    fn test_quantum_ldpc_codes() {
        let qldpc = QuantumLDPCCode::bicycle_code(3, 4);
        assert_eq!(qldpc.n, 24); // 2 * 3 * 4
        assert_eq!(qldpc.k, 2);

        let stab_code = qldpc.to_stabilizer_code();
        assert!(!stab_code.stabilizers.is_empty());

        // Test that stabilizers have bounded weight
        for stabilizer in &stab_code.stabilizers {
            assert!(stabilizer.weight() <= qldpc.max_weight);
        }
    }

    #[test]
    fn test_topological_codes() {
        let toric = ToricCode::new(2, 2);
        assert_eq!(toric.logical_qubits(), 2);
        assert_eq!(toric.distance(), 2);

        let stab_code = toric.to_stabilizer_code();
        assert_eq!(stab_code.n, 8); // 2 * 2 * 2
        assert_eq!(stab_code.k, 2);

        // Test that all stabilizers commute
        for i in 0..stab_code.stabilizers.len() {
            for j in i + 1..stab_code.stabilizers.len() {
                assert!(stab_code.stabilizers[i]
                    .commutes_with(&stab_code.stabilizers[j])
                    .unwrap());
            }
        }
    }

    #[test]
    fn test_ml_decoder() {
        let surface = SurfaceCode::new(3, 3);
        let decoder = MLDecoder::new(surface.to_stabilizer_code());

        // Test decoding with simple syndrome
        let syndrome = vec![true, false, true, false];
        let decoded = decoder.decode(&syndrome);

        // Should succeed for correctable errors
        assert!(decoded.is_ok() || syndrome.iter().filter(|&&x| x).count() % 2 == 1);
    }

    #[test]
    fn test_real_time_mock_hardware() {
        use crate::error_correction::real_time::*;
        use std::time::Duration;

        let hardware = MockQuantumHardware::new(0.01, Duration::from_micros(10), 4);

        // Test syndrome measurement
        let syndrome = hardware.measure_syndromes().unwrap();
        assert_eq!(syndrome.len(), 4);

        // Test error characteristics
        let characteristics = hardware.get_error_characteristics().unwrap();
        assert_eq!(characteristics.single_qubit_error_rates.len(), 4);

        // Test latency stats
        let stats = hardware.get_latency_stats().unwrap();
        assert!(stats.throughput_hz > 0.0);

        assert!(hardware.is_ready());
    }

    #[test]
    fn test_real_time_performance_monitor() {
        use crate::error_correction::real_time::*;
        use std::time::Duration;

        let mut monitor = PerformanceMonitor::new();

        // Record some cycles
        monitor.record_cycle(Duration::from_micros(10), true);
        monitor.record_cycle(Duration::from_micros(20), false);
        monitor.record_cycle(Duration::from_micros(15), true);

        assert_eq!(monitor.cycles_processed, 3);
        assert_eq!(monitor.errors_corrected, 2);
        assert_eq!(monitor.error_correction_rate(), 2.0 / 3.0);
        assert!(monitor.average_latency().as_micros() > 10);
        assert!(monitor.current_throughput() > 0.0);
    }

    #[test]
    fn test_real_time_adaptive_decoder() {
        use crate::error_correction::real_time::*;
        use std::sync::Arc;

        let code = StabilizerCode::repetition_code();
        let base_decoder = Arc::new(LookupDecoder::new(&code).unwrap());
        let characteristics = HardwareErrorCharacteristics {
            single_qubit_error_rates: vec![0.01; 3],
            two_qubit_error_rates: vec![0.1; 1],
            measurement_error_rates: vec![0.001; 3],
            correlated_errors: Vec::new(),
            temporal_variation: 0.01,
        };

        let mut adaptive_decoder = AdaptiveThresholdDecoder::new(base_decoder, characteristics);

        // Test initial threshold
        let initial_threshold = adaptive_decoder.current_threshold();
        assert_eq!(initial_threshold, 1.0); // Default when no history

        // Adapt thresholds based on feedback (use no-error syndromes)
        adaptive_decoder.adapt_thresholds(&[false, false], true); // No error, successful correction

        let new_threshold = adaptive_decoder.current_threshold();
        assert!(new_threshold != initial_threshold); // Should change from default 1.0 to 0.0

        // Test decoding (use no-error syndrome which should always be valid)
        let syndrome = vec![false, false]; // No error syndrome
        let result = adaptive_decoder.decode(&syndrome);
        assert!(result.is_ok(), "Decoding failed: {:?}", result.err());
    }

    #[test]
    fn test_real_time_parallel_decoder() {
        use crate::error_correction::real_time::*;
        use std::sync::Arc;

        let code = StabilizerCode::repetition_code();
        let base_decoder = Arc::new(LookupDecoder::new(&code).unwrap());
        let parallel_decoder = ParallelSyndromeDecoder::new(base_decoder, 2);

        // Test single syndrome decoding (use no-error syndrome)
        let syndrome = vec![false, false]; // No error syndrome
        let result = parallel_decoder.decode(&syndrome);
        assert!(result.is_ok(), "Decoding failed: {:?}", result.err());

        // Test batch decoding (use only no-error syndromes for safety)
        let syndromes = vec![
            vec![false, false], // No error syndromes
            vec![false, false],
            vec![false, false],
            vec![false, false],
        ];

        let results = parallel_decoder.decode_batch(&syndromes);
        assert!(results.is_ok());
        let corrections = results.unwrap();
        assert_eq!(corrections.len(), 4);
    }

    #[test]
    fn test_real_time_syndrome_stream_processor() {
        use crate::error_correction::real_time::*;
        use std::sync::Arc;
        use std::time::Duration;

        let code = StabilizerCode::repetition_code();
        let decoder = Arc::new(LookupDecoder::new(&code).unwrap());
        let hardware = Arc::new(MockQuantumHardware::new(0.01, Duration::from_micros(1), 3));
        let config = RealTimeConfig {
            max_latency: Duration::from_millis(1),
            buffer_size: 10,
            parallel_workers: 1,
            adaptive_threshold: false,
            hardware_feedback: false,
            performance_logging: true,
        };

        let processor = SyndromeStreamProcessor::new(decoder, hardware, config);

        // Test buffer status
        let (current, max) = processor.get_buffer_status();
        assert_eq!(current, 0);
        assert_eq!(max, 10);

        // Test performance stats (initial state)
        let stats = processor.get_performance_stats();
        assert_eq!(stats.cycles_processed, 0);
        assert_eq!(stats.errors_corrected, 0);
    }

    #[test]
    fn test_logical_gate_synthesizer() {
        use crate::error_correction::logical_gates::*;

        let code = StabilizerCode::repetition_code();
        let synthesizer = LogicalGateSynthesizer::new(0.01);

        // Test logical X gate synthesis
        let logical_x = synthesizer.synthesize_logical_x(&code, 0);
        assert!(logical_x.is_ok());

        let x_gate = logical_x.unwrap();
        assert_eq!(x_gate.logical_qubits, vec![0]);
        assert_eq!(x_gate.physical_operations.len(), 1);
        assert!(!x_gate.error_propagation.single_qubit_propagation.is_empty());

        // Test logical Z gate synthesis
        let logical_z = synthesizer.synthesize_logical_z(&code, 0);
        assert!(logical_z.is_ok());

        let z_gate = logical_z.unwrap();
        assert_eq!(z_gate.logical_qubits, vec![0]);
        assert_eq!(z_gate.physical_operations.len(), 1);

        // Test logical H gate synthesis
        let logical_h = synthesizer.synthesize_logical_h(&code, 0);
        assert!(logical_h.is_ok());

        let h_gate = logical_h.unwrap();
        assert_eq!(h_gate.logical_qubits, vec![0]);
        assert_eq!(h_gate.physical_operations.len(), 1);
        assert_eq!(h_gate.physical_operations[0].error_correction_rounds, 2);

        // Test invalid logical qubit index
        let invalid_gate = synthesizer.synthesize_logical_x(&code, 5);
        assert!(invalid_gate.is_err());
    }

    #[test]
    fn test_logical_circuit_synthesizer() {
        use crate::error_correction::logical_gates::*;

        let code = StabilizerCode::repetition_code();
        let synthesizer = LogicalCircuitSynthesizer::new(0.01);

        // Test simple circuit synthesis
        let gate_sequence = vec![("x", vec![0]), ("h", vec![0]), ("z", vec![0])];

        let circuit = synthesizer.synthesize_circuit(&code, &gate_sequence);
        assert!(circuit.is_ok());

        let logical_circuit = circuit.unwrap();
        assert_eq!(logical_circuit.len(), 3);

        // Test resource estimation
        let resources = synthesizer.estimate_resources(&logical_circuit);
        assert!(resources.total_physical_operations > 0);
        assert!(resources.total_error_correction_rounds > 0);
        assert_eq!(resources.estimated_depth, 3);

        // Test invalid gate name
        let invalid_sequence = vec![("invalid_gate", vec![0])];
        let invalid_circuit = synthesizer.synthesize_circuit(&code, &invalid_sequence);
        assert!(invalid_circuit.is_err());

        // Test CNOT gate with wrong number of targets
        let wrong_cnot = vec![("cnot", vec![0])]; // CNOT needs 2 targets
        let wrong_circuit = synthesizer.synthesize_circuit(&code, &wrong_cnot);
        assert!(wrong_circuit.is_err());
    }

    #[test]
    fn test_logical_t_gate_synthesis() {
        use crate::error_correction::logical_gates::*;

        let code = StabilizerCode::repetition_code();
        let synthesizer = LogicalGateSynthesizer::new(0.01);

        // Test T gate synthesis (requires magic state distillation)
        let logical_t = synthesizer.synthesize_logical_t(&code, 0);
        assert!(logical_t.is_ok());

        let t_gate = logical_t.unwrap();
        assert_eq!(t_gate.logical_qubits, vec![0]);
        assert_eq!(t_gate.physical_operations.len(), 2); // Magic state prep + injection

        // Check that magic state prep has more error correction rounds
        assert!(t_gate.physical_operations[0].error_correction_rounds >= 5);
    }

    #[test]
    fn test_error_propagation_analysis() {
        use crate::error_correction::logical_gates::*;

        let code = StabilizerCode::repetition_code();
        let synthesizer = LogicalGateSynthesizer::new(0.01);

        let logical_x = synthesizer.synthesize_logical_x(&code, 0).unwrap();

        // Check error propagation analysis
        let analysis = &logical_x.error_propagation;
        assert!(!analysis.single_qubit_propagation.is_empty());
        // max_error_weight is usize, so it's always >= 0
        assert_eq!(analysis.fault_tolerance_threshold, 0.01);

        // Check that some errors are marked as correctable
        let correctable_count = analysis
            .single_qubit_propagation
            .iter()
            .filter(|path| path.correctable)
            .count();
        assert!(correctable_count > 0);
    }

    #[test]
    fn test_pauli_string_weight() {
        let identity_string = PauliString::new(vec![Pauli::I, Pauli::I, Pauli::I]);
        assert_eq!(identity_string.weight(), 0);

        let single_error = PauliString::new(vec![Pauli::X, Pauli::I, Pauli::I]);
        assert_eq!(single_error.weight(), 1);

        let multi_error = PauliString::new(vec![Pauli::X, Pauli::Y, Pauli::Z]);
        assert_eq!(multi_error.weight(), 3);
    }

    #[test]
    fn test_logical_circuit_with_multiple_gates() {
        use crate::error_correction::logical_gates::*;

        let code = StabilizerCode::repetition_code();
        let synthesizer = LogicalCircuitSynthesizer::new(0.01);

        // Test a more complex circuit
        let gate_sequence = vec![
            ("h", vec![0]), // Hadamard on logical qubit 0
            ("x", vec![0]), // X on logical qubit 0
            ("z", vec![0]), // Z on logical qubit 0
            ("h", vec![0]), // Another Hadamard
        ];

        let circuit = synthesizer.synthesize_circuit(&code, &gate_sequence);
        assert!(circuit.is_ok());

        let logical_circuit = circuit.unwrap();
        assert_eq!(logical_circuit.len(), 4);

        // Check that all gates target the correct logical qubit
        for gate in &logical_circuit {
            assert_eq!(gate.logical_qubits, vec![0]);
        }

        // Estimate resources for this circuit
        let resources = synthesizer.estimate_resources(&logical_circuit);
        assert_eq!(resources.estimated_depth, 4);
        assert!(resources.total_error_correction_rounds >= 4); // At least one round per gate
    }

    #[test]
    fn test_adaptive_threshold_estimator() {
        use crate::error_correction::adaptive_threshold::*;

        let noise_model = NoiseModel::default();
        let algorithm = ThresholdEstimationAlgorithm::Bayesian {
            prior_strength: 1.0,
            update_rate: 0.1,
        };
        let config = AdaptiveConfig::default();

        let mut estimator = AdaptiveThresholdEstimator::new(noise_model, algorithm, config);

        // Test initial threshold estimation
        let syndrome = vec![true, false];
        let env = EnvironmentalConditions::default();
        let threshold = estimator.estimate_threshold(&syndrome, &env);
        assert!(threshold > 0.0);
        assert!(threshold < 1.0);

        // Test adding observations
        let observation = ErrorObservation {
            syndrome: syndrome.clone(),
            correction: PauliString::new(vec![Pauli::X, Pauli::I]),
            success: true,
            observed_error_rate: 0.01,
            timestamp: std::time::Instant::now(),
            environment: env.clone(),
        };

        estimator.add_observation(observation);

        // Test threshold recommendation
        let recommendation = estimator.get_threshold_recommendation(&syndrome);
        assert!(recommendation.threshold > 0.0);
        assert!(recommendation.confidence >= 0.0 && recommendation.confidence <= 1.0);
        assert!(recommendation.predicted_error_rate >= 0.0);
    }

    #[test]
    fn test_performance_tracker() {
        use crate::error_correction::adaptive_threshold::*;

        let mut tracker = PerformanceTracker::new();

        // Test initial state
        assert_eq!(tracker.successful_corrections, 0);
        assert_eq!(tracker.failed_corrections, 0);
        assert_eq!(tracker.precision(), 1.0); // Default when no data
        assert_eq!(tracker.recall(), 1.0);
        assert_eq!(tracker.f1_score(), 1.0); // Perfect when precision and recall are both 1.0

        // Simulate some corrections
        tracker.successful_corrections = 8;
        tracker.failed_corrections = 2;
        tracker.false_positives = 1;
        tracker.false_negatives = 1;

        // Test metrics
        assert_eq!(tracker.precision(), 8.0 / 9.0); // 8 / (8 + 1)
        assert_eq!(tracker.recall(), 8.0 / 9.0); // 8 / (8 + 1)
        assert!(tracker.f1_score() > 0.0);
    }

    #[test]
    fn test_environmental_conditions() {
        use crate::error_correction::adaptive_threshold::*;

        let mut env = EnvironmentalConditions::default();
        assert_eq!(env.temperature, 300.0); // Room temperature
        assert_eq!(env.magnetic_field, 0.0);

        // Test modification
        env.temperature = 310.0; // Higher temperature
        env.vibration_level = 0.1;

        let noise_model = NoiseModel::default();
        let algorithm = ThresholdEstimationAlgorithm::ExponentialAverage { alpha: 0.5 };
        let config = AdaptiveConfig::default();

        let estimator = AdaptiveThresholdEstimator::new(noise_model, algorithm, config);

        // Test that environmental conditions affect threshold
        let syndrome = vec![false, false];
        let threshold_normal =
            estimator.estimate_threshold(&syndrome, &EnvironmentalConditions::default());
        let threshold_hot = estimator.estimate_threshold(&syndrome, &env);

        // Thresholds may be different due to environmental factors
        assert!(threshold_normal >= 0.0);
        assert!(threshold_hot >= 0.0);
    }

    #[test]
    fn test_different_threshold_algorithms() {
        use crate::error_correction::adaptive_threshold::*;

        let noise_model = NoiseModel::default();
        let config = AdaptiveConfig::default();

        // Test Bayesian algorithm
        let bayesian_alg = ThresholdEstimationAlgorithm::Bayesian {
            prior_strength: 1.0,
            update_rate: 0.1,
        };
        let bayesian_estimator =
            AdaptiveThresholdEstimator::new(noise_model.clone(), bayesian_alg, config.clone());

        // Test Kalman filter algorithm
        let kalman_alg = ThresholdEstimationAlgorithm::KalmanFilter {
            process_noise: 0.01,
            measurement_noise: 0.1,
        };
        let kalman_estimator =
            AdaptiveThresholdEstimator::new(noise_model.clone(), kalman_alg, config.clone());

        // Test exponential average algorithm
        let exp_alg = ThresholdEstimationAlgorithm::ExponentialAverage { alpha: 0.3 };
        let exp_estimator =
            AdaptiveThresholdEstimator::new(noise_model.clone(), exp_alg, config.clone());

        // Test ML algorithm
        let ml_alg = ThresholdEstimationAlgorithm::MachineLearning {
            model_type: MLModelType::LinearRegression,
            training_window: 50,
        };
        let ml_estimator = AdaptiveThresholdEstimator::new(noise_model, ml_alg, config);

        let syndrome = vec![true, false];
        let env = EnvironmentalConditions::default();

        // All algorithms should produce valid thresholds
        let bayesian_threshold = bayesian_estimator.estimate_threshold(&syndrome, &env);
        let kalman_threshold = kalman_estimator.estimate_threshold(&syndrome, &env);
        let exp_threshold = exp_estimator.estimate_threshold(&syndrome, &env);
        let ml_threshold = ml_estimator.estimate_threshold(&syndrome, &env);

        assert!(bayesian_threshold > 0.0);
        assert!(kalman_threshold > 0.0);
        assert!(exp_threshold > 0.0);
        assert!(ml_threshold > 0.0);
    }

    #[test]
    fn test_noise_model_updates() {
        use crate::error_correction::adaptive_threshold::*;

        let noise_model = NoiseModel::default();
        let algorithm = ThresholdEstimationAlgorithm::Bayesian {
            prior_strength: 1.0,
            update_rate: 0.1,
        };
        let config = AdaptiveConfig {
            min_observations: 2, // Low threshold for testing
            real_time_adaptation: true,
            ..AdaptiveConfig::default()
        };

        let mut estimator = AdaptiveThresholdEstimator::new(noise_model, algorithm, config);

        // Add multiple observations to trigger model updates
        for i in 0..5 {
            let observation = ErrorObservation {
                syndrome: vec![i % 2 == 0, i % 3 == 0],
                correction: PauliString::new(vec![Pauli::X, Pauli::I]),
                success: i % 4 != 0, // Most succeed
                observed_error_rate: 0.01,
                timestamp: std::time::Instant::now(),
                environment: EnvironmentalConditions::default(),
            };
            estimator.add_observation(observation);
        }

        // The estimator should have updated its internal model
        let recommendation = estimator.get_threshold_recommendation(&[true, false]);
        assert!(recommendation.confidence > 0.0);
        assert!(recommendation.recommendation_quality > 0.0);
    }
}
