//! Neutral Atom Quantum Computing Platform
//!
//! This module implements neutral atom quantum computing using ultracold atoms
//! trapped in optical tweezers. It includes:
//! - Rydberg atom interactions for two-qubit gates
//! - Optical tweezer array management
//! - Laser pulse sequences for gate operations
//! - Atom loading and arrangement protocols
//! - Error models for neutral atom platforms

use crate::{
    error::{QuantRS2Error, QuantRS2Result},
    qubit::QubitId,
};
use scirs2_core::ndarray::{Array1, Array2};
use scirs2_core::Complex64;
use rustc_hash::FxHashMap;
use std::collections::HashMap;
use std::f64::consts::PI;

/// Types of neutral atoms used for quantum computing
#[derive(Debug, Clone, PartialEq)]
pub enum AtomSpecies {
    /// Rubidium-87 (commonly used)
    Rb87,
    /// Cesium-133
    Cs133,
    /// Strontium-88
    Sr88,
    /// Ytterbium-171
    Yb171,
    /// Custom atom with specified properties
    Custom {
        mass: f64,             // atomic mass in u
        ground_state: String,  // ground state configuration
        rydberg_state: String, // Rydberg state used
    },
}

impl AtomSpecies {
    /// Get atomic mass in atomic mass units
    pub fn mass(&self) -> f64 {
        match self {
            AtomSpecies::Rb87 => 86.909183,
            AtomSpecies::Cs133 => 132.905447,
            AtomSpecies::Sr88 => 87.905614,
            AtomSpecies::Yb171 => 170.936426,
            AtomSpecies::Custom { mass, .. } => *mass,
        }
    }

    /// Get typical trap depth in μK
    pub fn typical_trap_depth(&self) -> f64 {
        match self {
            AtomSpecies::Rb87 => 1000.0,
            AtomSpecies::Cs133 => 800.0,
            AtomSpecies::Sr88 => 1200.0,
            AtomSpecies::Yb171 => 900.0,
            AtomSpecies::Custom { .. } => 1000.0,
        }
    }

    /// Get Rydberg state energy in GHz
    pub fn rydberg_energy(&self) -> f64 {
        match self {
            AtomSpecies::Rb87 => 100.0, // Example value
            AtomSpecies::Cs133 => 95.0,
            AtomSpecies::Sr88 => 110.0,
            AtomSpecies::Yb171 => 105.0,
            AtomSpecies::Custom { .. } => 100.0,
        }
    }
}

/// Position in 3D space (in micrometers)
#[derive(Debug, Clone, Copy, PartialEq)]
pub struct Position3D {
    pub x: f64,
    pub y: f64,
    pub z: f64,
}

impl Position3D {
    pub fn new(x: f64, y: f64, z: f64) -> Self {
        Self { x, y, z }
    }

    /// Calculate distance to another position
    pub fn distance_to(&self, other: &Position3D) -> f64 {
        ((self.x - other.x).powi(2) + (self.y - other.y).powi(2) + (self.z - other.z).powi(2))
            .sqrt()
    }

    /// Calculate Rydberg interaction strength at this distance
    pub fn rydberg_interaction(&self, other: &Position3D, c6: f64) -> f64 {
        let distance = self.distance_to(other);
        if distance == 0.0 {
            f64::INFINITY
        } else {
            c6 / distance.powi(6) // van der Waals interaction
        }
    }
}

/// State of a neutral atom
#[derive(Debug, Clone, Copy, PartialEq)]
pub enum AtomState {
    /// Ground state |g⟩
    Ground,
    /// Rydberg state |r⟩
    Rydberg,
    /// Intermediate state during laser transitions
    Intermediate,
    /// Atom is missing (loading failure)
    Missing,
}

/// Individual neutral atom in an optical tweezer
#[derive(Debug, Clone)]
pub struct NeutralAtom {
    /// Atom species
    pub species: AtomSpecies,
    /// Position in the tweezer array
    pub position: Position3D,
    /// Current quantum state
    pub state: AtomState,
    /// Loading probability (success rate)
    pub loading_probability: f64,
    /// Lifetime in trap (seconds)
    pub lifetime: f64,
    /// Coherence time (seconds)
    pub coherence_time: f64,
}

impl NeutralAtom {
    /// Create a new neutral atom
    pub fn new(species: AtomSpecies, position: Position3D) -> Self {
        Self {
            species,
            position,
            state: AtomState::Ground,
            loading_probability: 0.9, // 90% loading success
            lifetime: 100.0,          // 100 seconds
            coherence_time: 1.0,      // 1 second
        }
    }

    /// Check if atom is successfully loaded
    pub fn is_loaded(&self) -> bool {
        self.state != AtomState::Missing
    }

    /// Calculate interaction strength with another atom
    pub fn interaction_with(&self, other: &NeutralAtom) -> f64 {
        // C6 coefficient for Rydberg interactions (MHz·μm^6)
        let c6 = match (&self.species, &other.species) {
            (AtomSpecies::Rb87, AtomSpecies::Rb87) => 5000.0,
            (AtomSpecies::Cs133, AtomSpecies::Cs133) => 6000.0,
            (AtomSpecies::Sr88, AtomSpecies::Sr88) => 4500.0,
            (AtomSpecies::Yb171, AtomSpecies::Yb171) => 4800.0,
            _ => 5000.0, // Default for mixed species
        };

        self.position.rydberg_interaction(&other.position, c6)
    }
}

/// Optical tweezer for trapping individual atoms
#[derive(Debug, Clone)]
pub struct OpticalTweezer {
    /// Position of the tweezer focus
    pub position: Position3D,
    /// Laser power (mW)
    pub power: f64,
    /// Wavelength (nm)
    pub wavelength: f64,
    /// Numerical aperture of focusing objective
    pub numerical_aperture: f64,
    /// Whether tweezer is active
    pub active: bool,
    /// Trapped atom (if any)
    pub atom: Option<NeutralAtom>,
}

impl OpticalTweezer {
    /// Create a new optical tweezer
    pub fn new(position: Position3D, power: f64, wavelength: f64, numerical_aperture: f64) -> Self {
        Self {
            position,
            power,
            wavelength,
            numerical_aperture,
            active: true,
            atom: None,
        }
    }

    /// Calculate trap depth in μK
    pub fn trap_depth(&self) -> f64 {
        if !self.active {
            return 0.0;
        }

        // Simplified calculation: depth ∝ power / (wavelength * beam_waist²)
        let beam_waist = self.wavelength / (PI * self.numerical_aperture); // Rough approximation
        (self.power * 1000.0) / (self.wavelength * beam_waist.powi(2))
    }

    /// Load an atom into the tweezer
    pub fn load_atom(&mut self, mut atom: NeutralAtom) -> bool {
        if !self.active || self.atom.is_some() {
            return false;
        }

        // Simulate probabilistic loading
        use scirs2_core::random::prelude::*;
        let mut rng = thread_rng();
        let success = rng.gen::<f64>() < atom.loading_probability;

        if success {
            atom.position = self.position;
            self.atom = Some(atom);
            true
        } else {
            false
        }
    }

    /// Remove atom from tweezer
    pub fn remove_atom(&mut self) -> Option<NeutralAtom> {
        self.atom.take()
    }

    /// Check if tweezer has an atom
    pub fn has_atom(&self) -> bool {
        self.atom.is_some()
    }
}

/// Laser system for controlling neutral atoms
#[derive(Debug, Clone)]
pub struct LaserSystem {
    /// Wavelength in nm
    pub wavelength: f64,
    /// Power in mW
    pub power: f64,
    /// Beam waist in μm
    pub beam_waist: f64,
    /// Detuning from atomic transition in MHz
    pub detuning: f64,
    /// Laser linewidth in kHz
    pub linewidth: f64,
    /// Whether laser is currently on
    pub active: bool,
}

impl LaserSystem {
    /// Create a new laser system
    pub fn new(wavelength: f64, power: f64, beam_waist: f64, detuning: f64) -> Self {
        Self {
            wavelength,
            power,
            beam_waist,
            detuning,
            linewidth: 1.0, // 1 kHz linewidth
            active: false,
        }
    }

    /// Calculate Rabi frequency in MHz
    pub fn rabi_frequency(&self) -> f64 {
        if !self.active {
            return 0.0;
        }

        // Simplified: Ω ∝ √(power) / beam_waist
        (self.power.sqrt() * 10.0) / self.beam_waist
    }

    /// Set laser detuning
    pub fn set_detuning(&mut self, detuning: f64) {
        self.detuning = detuning;
    }

    /// Turn laser on/off
    pub fn set_active(&mut self, active: bool) {
        self.active = active;
    }
}

/// Neutral atom quantum computer platform
#[derive(Debug)]
pub struct NeutralAtomQC {
    /// Array of optical tweezers
    pub tweezers: Vec<OpticalTweezer>,
    /// Map from qubit ID to tweezer index
    pub qubit_map: FxHashMap<QubitId, usize>,
    /// Laser systems for different transitions
    pub lasers: HashMap<String, LaserSystem>,
    /// Current quantum state (simplified representation)
    pub state: Array1<Complex64>,
    /// Number of qubits
    pub num_qubits: usize,
    /// Global phase accumulation
    pub global_phase: f64,
    /// Error model parameters
    pub error_model: NeutralAtomErrorModel,
}

impl NeutralAtomQC {
    /// Create a new neutral atom quantum computer
    pub fn new(num_qubits: usize) -> Self {
        let mut tweezers = Vec::new();
        let mut qubit_map = FxHashMap::default();

        // Create a linear array of tweezers
        for i in 0..num_qubits {
            let position = Position3D::new(i as f64 * 5.0, 0.0, 0.0); // 5 μm spacing
            let tweezer = OpticalTweezer::new(
                position, 1.0,    // 1 mW power
                1064.0, // 1064 nm wavelength
                0.75,   // NA = 0.75
            );
            tweezers.push(tweezer);
            qubit_map.insert(QubitId(i as u32), i);
        }

        // Setup laser systems
        let mut lasers = HashMap::new();

        // Cooling laser (for Rb87)
        lasers.insert(
            "cooling".to_string(),
            LaserSystem::new(780.0, 10.0, 100.0, -10.0),
        );

        // Rydberg excitation laser
        lasers.insert(
            "rydberg".to_string(),
            LaserSystem::new(480.0, 1.0, 2.0, 0.0),
        );

        // Two-photon Raman laser
        lasers.insert("raman".to_string(), LaserSystem::new(795.0, 5.0, 10.0, 0.0));

        // Initialize quantum state to |00...0⟩
        let dim = 1 << num_qubits;
        let mut state = Array1::zeros(dim);
        state[0] = Complex64::new(1.0, 0.0);

        Self {
            tweezers,
            qubit_map,
            lasers,
            state,
            num_qubits,
            global_phase: 0.0,
            error_model: NeutralAtomErrorModel::default(),
        }
    }

    /// Load atoms into the tweezer array
    pub fn load_atoms(&mut self, species: AtomSpecies) -> QuantRS2Result<usize> {
        let mut loaded_count = 0;

        for tweezer in &mut self.tweezers {
            let atom = NeutralAtom::new(species.clone(), tweezer.position);
            if tweezer.load_atom(atom) {
                loaded_count += 1;
            }
        }

        Ok(loaded_count)
    }

    /// Perform atom rearrangement to fill gaps
    pub fn rearrange_atoms(&mut self) -> QuantRS2Result<()> {
        // Collect all loaded atoms
        let mut atoms = Vec::new();
        for tweezer in &mut self.tweezers {
            if let Some(atom) = tweezer.remove_atom() {
                atoms.push(atom);
            }
        }

        // Redistribute atoms to fill tweezers from the beginning
        for (i, atom) in atoms.into_iter().enumerate() {
            if i < self.tweezers.len() {
                self.tweezers[i].load_atom(atom);
            }
        }

        Ok(())
    }

    /// Apply single-qubit rotation gate
    pub fn apply_single_qubit_gate(
        &mut self,
        qubit: QubitId,
        gate_matrix: &Array2<Complex64>,
    ) -> QuantRS2Result<()> {
        let tweezer_idx = *self
            .qubit_map
            .get(&qubit)
            .ok_or_else(|| QuantRS2Error::InvalidInput(format!("Qubit {:?} not found", qubit)))?;

        // Check if atom is present
        if !self.tweezers[tweezer_idx].has_atom() {
            return Err(QuantRS2Error::InvalidOperation(
                "No atom in tweezer".to_string(),
            ));
        }

        // Apply gate using tensor product
        let qubit_index = qubit.0 as usize;
        let new_state = self.apply_single_qubit_tensor(qubit_index, gate_matrix)?;
        self.state = new_state;

        // Apply errors
        self.apply_single_qubit_errors(qubit_index)?;

        Ok(())
    }

    /// Apply two-qubit Rydberg gate
    pub fn apply_rydberg_gate(&mut self, control: QubitId, target: QubitId) -> QuantRS2Result<()> {
        let control_idx = *self.qubit_map.get(&control).ok_or_else(|| {
            QuantRS2Error::InvalidInput(format!("Control qubit {:?} not found", control))
        })?;
        let target_idx = *self.qubit_map.get(&target).ok_or_else(|| {
            QuantRS2Error::InvalidInput(format!("Target qubit {:?} not found", target))
        })?;

        // Check if both atoms are present
        if !self.tweezers[control_idx].has_atom() || !self.tweezers[target_idx].has_atom() {
            return Err(QuantRS2Error::InvalidOperation(
                "Missing atoms for two-qubit gate".to_string(),
            ));
        }

        // Check interaction strength
        let control_atom = self.tweezers[control_idx].atom.as_ref().unwrap();
        let target_atom = self.tweezers[target_idx].atom.as_ref().unwrap();
        let interaction = control_atom.interaction_with(target_atom);

        if interaction < 1.0 {
            // Minimum interaction strength in MHz
            return Err(QuantRS2Error::InvalidOperation(
                "Insufficient Rydberg interaction".to_string(),
            ));
        }

        // Apply Rydberg CZ gate (simplified)
        let cz_matrix = self.create_rydberg_cz_matrix()?;
        let new_state =
            self.apply_two_qubit_tensor(control.0 as usize, target.0 as usize, &cz_matrix)?;
        self.state = new_state;

        // Apply errors
        self.apply_two_qubit_errors(control.0 as usize, target.0 as usize)?;

        Ok(())
    }

    /// Create Rydberg CZ gate matrix
    fn create_rydberg_cz_matrix(&self) -> QuantRS2Result<Array2<Complex64>> {
        let mut cz = Array2::eye(4);
        cz[[3, 3]] = Complex64::new(-1.0, 0.0); // |11⟩ → -|11⟩
        Ok(cz)
    }

    /// Apply single-qubit gate using tensor product
    fn apply_single_qubit_tensor(
        &self,
        qubit_idx: usize,
        gate: &Array2<Complex64>,
    ) -> QuantRS2Result<Array1<Complex64>> {
        let dim = 1 << self.num_qubits;
        let mut new_state = Array1::zeros(dim);

        for state in 0..dim {
            let qubit_bit = (state >> qubit_idx) & 1;
            let other_bits = state & !(1 << qubit_idx);

            for new_qubit_bit in 0..2 {
                let new_state_idx = other_bits | (new_qubit_bit << qubit_idx);
                let gate_element = gate[[new_qubit_bit, qubit_bit]];
                new_state[new_state_idx] += gate_element * self.state[state];
            }
        }

        Ok(new_state)
    }

    /// Apply two-qubit gate using tensor product
    fn apply_two_qubit_tensor(
        &self,
        qubit1: usize,
        qubit2: usize,
        gate: &Array2<Complex64>,
    ) -> QuantRS2Result<Array1<Complex64>> {
        let dim = 1 << self.num_qubits;
        let mut new_state = Array1::zeros(dim);

        for state in 0..dim {
            let bit1 = (state >> qubit1) & 1;
            let bit2 = (state >> qubit2) & 1;
            let other_bits = state & !((1 << qubit1) | (1 << qubit2));
            let two_qubit_state = (bit1 << 1) | bit2;

            for new_two_qubit_state in 0..4 {
                let new_bit1 = (new_two_qubit_state >> 1) & 1;
                let new_bit2 = new_two_qubit_state & 1;
                let new_state_idx = other_bits | (new_bit1 << qubit1) | (new_bit2 << qubit2);

                let gate_element = gate[[new_two_qubit_state, two_qubit_state]];
                new_state[new_state_idx] += gate_element * self.state[state];
            }
        }

        Ok(new_state)
    }

    /// Apply single-qubit error models
    fn apply_single_qubit_errors(&mut self, _qubit: usize) -> QuantRS2Result<()> {
        // Simplified error model - in practice would include:
        // - Laser intensity fluctuations
        // - Decoherence from atom motion
        // - Spontaneous emission
        // - AC Stark shifts

        use scirs2_core::random::prelude::*;
        let mut rng = thread_rng();

        // Add small random phase error
        let phase_error = rng.gen_range(-0.01..0.01);
        self.global_phase += phase_error;

        Ok(())
    }

    /// Apply two-qubit error models
    fn apply_two_qubit_errors(&mut self, _qubit1: usize, _qubit2: usize) -> QuantRS2Result<()> {
        // Simplified error model for Rydberg gates
        use scirs2_core::random::prelude::*;
        let mut rng = thread_rng();

        // Rydberg gate errors are typically higher
        let phase_error = rng.gen_range(-0.05..0.05);
        self.global_phase += phase_error;

        Ok(())
    }

    /// Measure a qubit
    pub fn measure_qubit(&mut self, qubit: QubitId) -> QuantRS2Result<u8> {
        let qubit_idx = qubit.0 as usize;
        if qubit_idx >= self.num_qubits {
            return Err(QuantRS2Error::InvalidInput(
                "Qubit index out of range".to_string(),
            ));
        }

        // Calculate probabilities
        let mut prob_0 = 0.0;
        let mut prob_1 = 0.0;

        for state in 0..(1 << self.num_qubits) {
            let amplitude_sq = self.state[state].norm_sqr();
            if (state >> qubit_idx) & 1 == 0 {
                prob_0 += amplitude_sq;
            } else {
                prob_1 += amplitude_sq;
            }
        }

        // Sample measurement result
        use scirs2_core::random::prelude::*;
        let mut rng = thread_rng();
        let result: usize = if rng.gen::<f64>() < prob_0 / (prob_0 + prob_1) {
            0
        } else {
            1
        };

        // Collapse state
        let mut new_state = Array1::zeros(1 << self.num_qubits);
        let normalization = if result == 0 {
            prob_0.sqrt()
        } else {
            prob_1.sqrt()
        };

        for state in 0..(1 << self.num_qubits) {
            if ((state >> qubit_idx) & 1) == result {
                new_state[state] = self.state[state] / normalization;
            }
        }

        self.state = new_state;
        Ok(result as u8)
    }

    /// Get number of loaded atoms
    pub fn loaded_atom_count(&self) -> usize {
        self.tweezers.iter().filter(|t| t.has_atom()).count()
    }

    /// Get quantum state probabilities
    pub fn get_probabilities(&self) -> Vec<f64> {
        self.state.iter().map(|c| c.norm_sqr()).collect()
    }

    /// Reset to ground state
    pub fn reset(&mut self) {
        let dim = 1 << self.num_qubits;
        self.state = Array1::zeros(dim);
        self.state[0] = Complex64::new(1.0, 0.0);
        self.global_phase = 0.0;
    }

    /// Get atom positions for visualization
    pub fn get_atom_positions(&self) -> Vec<(QubitId, Position3D, bool)> {
        self.tweezers
            .iter()
            .enumerate()
            .filter_map(|(i, tweezer)| {
                self.qubit_map
                    .iter()
                    .find(|(_, &idx)| idx == i)
                    .map(|(&qubit_id, _)| (qubit_id, tweezer.position, tweezer.has_atom()))
            })
            .collect()
    }
}

/// Error model for neutral atom quantum computing
#[derive(Debug, Clone)]
pub struct NeutralAtomErrorModel {
    /// Loading fidelity (probability of successful atom loading)
    pub loading_fidelity: f64,
    /// Single-qubit gate fidelity
    pub single_qubit_fidelity: f64,
    /// Two-qubit gate fidelity
    pub two_qubit_fidelity: f64,
    /// Measurement fidelity
    pub measurement_fidelity: f64,
    /// Coherence time in seconds
    pub coherence_time: f64,
    /// Rydberg blockade radius in μm
    pub blockade_radius: f64,
}

impl Default for NeutralAtomErrorModel {
    fn default() -> Self {
        Self {
            loading_fidelity: 0.95,
            single_qubit_fidelity: 0.999,
            two_qubit_fidelity: 0.98,
            measurement_fidelity: 0.99,
            coherence_time: 1.0,
            blockade_radius: 10.0,
        }
    }
}

/// Neutral atom gate library
pub struct NeutralAtomGates;

impl NeutralAtomGates {
    /// X gate (π rotation around X-axis)
    pub fn x_gate() -> Array2<Complex64> {
        let mut x = Array2::zeros((2, 2));
        x[[0, 1]] = Complex64::new(1.0, 0.0);
        x[[1, 0]] = Complex64::new(1.0, 0.0);
        x
    }

    /// Y gate (π rotation around Y-axis)
    pub fn y_gate() -> Array2<Complex64> {
        let mut y = Array2::zeros((2, 2));
        y[[0, 1]] = Complex64::new(0.0, -1.0);
        y[[1, 0]] = Complex64::new(0.0, 1.0);
        y
    }

    /// Z gate (π rotation around Z-axis)
    pub fn z_gate() -> Array2<Complex64> {
        let mut z = Array2::zeros((2, 2));
        z[[0, 0]] = Complex64::new(1.0, 0.0);
        z[[1, 1]] = Complex64::new(-1.0, 0.0);
        z
    }

    /// Hadamard gate
    pub fn h_gate() -> Array2<Complex64> {
        let h_val = 1.0 / 2.0_f64.sqrt();
        let mut h = Array2::zeros((2, 2));
        h[[0, 0]] = Complex64::new(h_val, 0.0);
        h[[0, 1]] = Complex64::new(h_val, 0.0);
        h[[1, 0]] = Complex64::new(h_val, 0.0);
        h[[1, 1]] = Complex64::new(-h_val, 0.0);
        h
    }

    /// Rotation around X-axis
    pub fn rx_gate(theta: f64) -> Array2<Complex64> {
        let cos_half = (theta / 2.0).cos();
        let sin_half = (theta / 2.0).sin();

        let mut rx = Array2::zeros((2, 2));
        rx[[0, 0]] = Complex64::new(cos_half, 0.0);
        rx[[0, 1]] = Complex64::new(0.0, -sin_half);
        rx[[1, 0]] = Complex64::new(0.0, -sin_half);
        rx[[1, 1]] = Complex64::new(cos_half, 0.0);
        rx
    }

    /// Rotation around Y-axis
    pub fn ry_gate(theta: f64) -> Array2<Complex64> {
        let cos_half = (theta / 2.0).cos();
        let sin_half = (theta / 2.0).sin();

        let mut ry = Array2::zeros((2, 2));
        ry[[0, 0]] = Complex64::new(cos_half, 0.0);
        ry[[0, 1]] = Complex64::new(-sin_half, 0.0);
        ry[[1, 0]] = Complex64::new(sin_half, 0.0);
        ry[[1, 1]] = Complex64::new(cos_half, 0.0);
        ry
    }

    /// Rotation around Z-axis
    pub fn rz_gate(theta: f64) -> Array2<Complex64> {
        let exp_neg = Complex64::new(0.0, -theta / 2.0).exp();
        let exp_pos = Complex64::new(0.0, theta / 2.0).exp();

        let mut rz = Array2::zeros((2, 2));
        rz[[0, 0]] = exp_neg;
        rz[[1, 1]] = exp_pos;
        rz
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_atom_species_properties() {
        let rb87 = AtomSpecies::Rb87;
        assert!((rb87.mass() - 86.909183).abs() < 1e-6);
        assert!(rb87.typical_trap_depth() > 0.0);
        assert!(rb87.rydberg_energy() > 0.0);

        let custom = AtomSpecies::Custom {
            mass: 100.0,
            ground_state: "5s".to_string(),
            rydberg_state: "50s".to_string(),
        };
        assert_eq!(custom.mass(), 100.0);
    }

    #[test]
    fn test_position_calculations() {
        let pos1 = Position3D::new(0.0, 0.0, 0.0);
        let pos2 = Position3D::new(3.0, 4.0, 0.0);

        assert_eq!(pos1.distance_to(&pos2), 5.0);

        let interaction = pos1.rydberg_interaction(&pos2, 1000.0);
        assert!(interaction > 0.0);
        assert!(interaction.is_finite());
    }

    #[test]
    fn test_optical_tweezer() {
        let position = Position3D::new(0.0, 0.0, 0.0);
        let mut tweezer = OpticalTweezer::new(position, 1.0, 1064.0, 0.75);

        assert!(tweezer.active);
        assert!(!tweezer.has_atom());
        assert!(tweezer.trap_depth() > 0.0);

        let atom = NeutralAtom::new(AtomSpecies::Rb87, position);
        let loaded = tweezer.load_atom(atom);
        // Loading is probabilistic, so we can't guarantee success
        // Test that loading returns a valid boolean value (no panic)
        let _result = loaded; // Simply verify the operation completes
    }

    #[test]
    fn test_laser_system() {
        let mut laser = LaserSystem::new(780.0, 10.0, 100.0, -10.0);

        assert!(!laser.active);
        assert_eq!(laser.rabi_frequency(), 0.0);

        laser.set_active(true);
        assert!(laser.rabi_frequency() > 0.0);

        laser.set_detuning(5.0);
        assert_eq!(laser.detuning, 5.0);
    }

    #[test]
    fn test_neutral_atom_creation() {
        let position = Position3D::new(1.0, 2.0, 3.0);
        let atom = NeutralAtom::new(AtomSpecies::Rb87, position);

        assert_eq!(atom.species, AtomSpecies::Rb87);
        assert_eq!(atom.position, position);
        assert_eq!(atom.state, AtomState::Ground);
        assert!(atom.is_loaded());
    }

    #[test]
    fn test_neutral_atom_qc_initialization() {
        let qc = NeutralAtomQC::new(3);

        assert_eq!(qc.num_qubits, 3);
        assert_eq!(qc.tweezers.len(), 3);
        assert_eq!(qc.qubit_map.len(), 3);
        assert_eq!(qc.state.len(), 8); // 2^3 = 8

        // Initial state should be |000⟩
        assert!((qc.state[0].norm_sqr() - 1.0).abs() < 1e-10);
        for i in 1..8 {
            assert!(qc.state[i].norm_sqr() < 1e-10);
        }
    }

    #[test]
    fn test_atom_loading() {
        let mut qc = NeutralAtomQC::new(2);
        let loaded = qc.load_atoms(AtomSpecies::Rb87).unwrap();

        // Should attempt to load into both tweezers
        assert!(loaded <= 2);
        assert!(qc.loaded_atom_count() <= 2);
    }

    #[test]
    fn test_single_qubit_gates() {
        let mut qc = NeutralAtomQC::new(1);
        qc.load_atoms(AtomSpecies::Rb87).unwrap();

        if qc.loaded_atom_count() > 0 {
            let x_gate = NeutralAtomGates::x_gate();
            let result = qc.apply_single_qubit_gate(QubitId(0), &x_gate);

            // If atom is loaded, gate should succeed
            assert!(result.is_ok());
        }
    }

    #[test]
    fn test_two_qubit_rydberg_gate() {
        let mut qc = NeutralAtomQC::new(2);
        qc.load_atoms(AtomSpecies::Rb87).unwrap();

        if qc.loaded_atom_count() == 2 {
            let result = qc.apply_rydberg_gate(QubitId(0), QubitId(1));
            // Should succeed if both atoms are loaded and close enough
            assert!(result.is_ok() || result.is_err()); // Test interface
        }
    }

    #[test]
    fn test_measurement() {
        let mut qc = NeutralAtomQC::new(1);
        qc.load_atoms(AtomSpecies::Rb87).unwrap();

        if qc.loaded_atom_count() > 0 {
            let result = qc.measure_qubit(QubitId(0));
            if result.is_ok() {
                let measurement = result.unwrap();
                assert!(measurement == 0 || measurement == 1);
            }
        }
    }

    #[test]
    fn test_gate_matrices() {
        let x = NeutralAtomGates::x_gate();
        let y = NeutralAtomGates::y_gate();
        let z = NeutralAtomGates::z_gate();
        let h = NeutralAtomGates::h_gate();

        // Test basic properties
        assert_eq!(x.dim(), (2, 2));
        assert_eq!(y.dim(), (2, 2));
        assert_eq!(z.dim(), (2, 2));
        assert_eq!(h.dim(), (2, 2));

        // Test Pauli X matrix elements
        assert_eq!(x[[0, 1]], Complex64::new(1.0, 0.0));
        assert_eq!(x[[1, 0]], Complex64::new(1.0, 0.0));
        assert_eq!(x[[0, 0]], Complex64::new(0.0, 0.0));
        assert_eq!(x[[1, 1]], Complex64::new(0.0, 0.0));
    }

    #[test]
    fn test_rotation_gates() {
        let rx_pi = NeutralAtomGates::rx_gate(PI);
        let _ry_pi = NeutralAtomGates::ry_gate(PI);
        let _rz_pi = NeutralAtomGates::rz_gate(PI);

        // RX(π) should be approximately -iX
        let x = NeutralAtomGates::x_gate();
        let expected_rx = x.mapv(|x| Complex64::new(0.0, -1.0) * x);

        for i in 0..2 {
            for j in 0..2 {
                assert!((rx_pi[[i, j]] - expected_rx[[i, j]]).norm() < 1e-10);
            }
        }
    }

    #[test]
    fn test_atom_rearrangement() {
        let mut qc = NeutralAtomQC::new(3);

        // Manually add some atoms to non-consecutive tweezers with 100% loading probability
        let mut atom1 = NeutralAtom::new(AtomSpecies::Rb87, qc.tweezers[0].position);
        let mut atom2 = NeutralAtom::new(AtomSpecies::Rb87, qc.tweezers[2].position);
        atom1.loading_probability = 1.0; // Ensure 100% loading success for test
        atom2.loading_probability = 1.0; // Ensure 100% loading success for test
        qc.tweezers[0].atom = Some(atom1);
        qc.tweezers[2].atom = Some(atom2);

        assert_eq!(qc.loaded_atom_count(), 2);

        // Rearrange should succeed
        assert!(qc.rearrange_atoms().is_ok());

        // After rearrangement, atoms should be in consecutive tweezers
        assert!(qc.tweezers[0].has_atom());
        assert!(qc.tweezers[1].has_atom());
        assert!(!qc.tweezers[2].has_atom());
    }

    #[test]
    fn test_error_model_defaults() {
        let error_model = NeutralAtomErrorModel::default();

        assert!(error_model.loading_fidelity > 0.0 && error_model.loading_fidelity <= 1.0);
        assert!(
            error_model.single_qubit_fidelity > 0.0 && error_model.single_qubit_fidelity <= 1.0
        );
        assert!(error_model.two_qubit_fidelity > 0.0 && error_model.two_qubit_fidelity <= 1.0);
        assert!(error_model.measurement_fidelity > 0.0 && error_model.measurement_fidelity <= 1.0);
        assert!(error_model.coherence_time > 0.0);
        assert!(error_model.blockade_radius > 0.0);
    }

    #[test]
    fn test_quantum_state_probabilities() {
        let qc = NeutralAtomQC::new(2);
        let probs = qc.get_probabilities();

        assert_eq!(probs.len(), 4); // 2^2 = 4 states

        // Probabilities should sum to 1
        let total: f64 = probs.iter().sum();
        assert!((total - 1.0).abs() < 1e-10);

        // Initial state |00⟩ should have probability 1
        assert!((probs[0] - 1.0).abs() < 1e-10);
    }

    #[test]
    fn test_atom_position_retrieval() {
        let qc = NeutralAtomQC::new(2);
        let positions = qc.get_atom_positions();

        assert_eq!(positions.len(), 2);

        // Check that positions are returned correctly
        for (qubit_id, position, has_atom) in positions {
            assert!(qubit_id.0 < 2);
            assert!(position.x >= 0.0); // Should be at positive x coordinates
                                        // has_atom depends on loading success, so we just test the interface
                                        // Test that has_atom returns a boolean value
                                        // has_atom is a boolean, so this assertion is always true
                                        // We're just exercising the has_atom method
            let _ = has_atom;
        }
    }
}
