//! Post-Quantum Cryptography Primitives
//!
//! Quantum-resistant cryptographic operations with lattice-based and code-based quantum gates.

use crate::error::QuantRS2Error;
// use crate::matrix_ops::{DenseMatrix, QuantumMatrix};
// use crate::qubit::QubitId;
use scirs2_core::Complex64;
// use scirs2_linalg::{matrix_exp, qr_decompose};
use scirs2_core::ndarray::{Array1, Array2};
// use std::collections::HashMap;
use std::f64::consts::PI;
// use sha3::{Digest, Sha3_256, Sha3_512};
use scirs2_core::random::{Rng, SeedableRng};
use scirs2_core::random::ChaCha20Rng;

/// Quantum hash function implementations
#[derive(Debug, Clone)]
pub struct QuantumHashFunction {
    pub num_qubits: usize,
    pub hash_size: usize,
    pub compression_function: CompressionFunction,
    pub quantum_circuit: Array2<Complex64>,
}

#[derive(Debug, Clone)]
pub enum CompressionFunction {
    QuantumSponge { rate: usize, capacity: usize },
    QuantumMerkleTree { depth: usize, arity: usize },
    QuantumGrover { iterations: usize },
}

impl QuantumHashFunction {
    /// Create a new quantum hash function
    pub fn new(
        num_qubits: usize,
        hash_size: usize,
        compression_function: CompressionFunction,
    ) -> Result<Self, QuantRS2Error> {
        let quantum_circuit = Self::build_hash_circuit(num_qubits, &compression_function)?;

        Ok(Self {
            num_qubits,
            hash_size,
            compression_function,
            quantum_circuit,
        })
    }

    /// Build the quantum hash circuit
    fn build_hash_circuit(
        num_qubits: usize,
        compression_function: &CompressionFunction,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        let _dim = 2_usize.pow(num_qubits as u32);
        let circuit = match compression_function {
            CompressionFunction::QuantumSponge { rate, capacity } => {
                Self::build_sponge_circuit(num_qubits, *rate, *capacity)?
            }
            CompressionFunction::QuantumMerkleTree { depth, arity } => {
                Self::build_merkle_circuit(num_qubits, *depth, *arity)?
            }
            CompressionFunction::QuantumGrover { iterations } => {
                Self::build_grover_circuit(num_qubits, *iterations)?
            }
        };

        Ok(circuit)
    }

    /// Build quantum sponge construction circuit
    fn build_sponge_circuit(
        num_qubits: usize,
        rate: usize,
        capacity: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        if rate + capacity != num_qubits {
            return Err(QuantRS2Error::InvalidParameter(
                "Rate + capacity must equal number of qubits".to_string(),
            ));
        }

        let dim = 2_usize.pow(num_qubits as u32);
        let mut circuit = Array2::eye(dim);

        // Absorption phase
        for round in 0..3 {
            // Add input to rate portion
            circuit = circuit.dot(&Self::create_absorption_layer(num_qubits, rate)?);

            // Apply permutation
            circuit = circuit.dot(&Self::create_permutation_layer(num_qubits, round)?);
        }

        // Squeezing phase
        circuit = circuit.dot(&Self::create_squeezing_layer(num_qubits, capacity)?);

        Ok(circuit)
    }

    /// Build quantum Merkle tree circuit
    fn build_merkle_circuit(
        num_qubits: usize,
        depth: usize,
        arity: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        let dim = 2_usize.pow(num_qubits as u32);
        let mut circuit = Array2::eye(dim);

        // Build tree level by level
        for level in 0..depth {
            let nodes_at_level = arity.pow(level as u32);

            for _node in 0..nodes_at_level {
                let compression_circuit = Self::create_compression_node(num_qubits, arity)?;
                circuit = circuit.dot(&compression_circuit);
            }
        }

        Ok(circuit)
    }

    /// Build Grover-based hash circuit
    fn build_grover_circuit(
        num_qubits: usize,
        iterations: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        let dim = 2_usize.pow(num_qubits as u32);
        let mut circuit = Array2::eye(dim);

        // Initial superposition
        circuit = circuit.dot(&Self::create_hadamard_layer(num_qubits)?);

        // Grover iterations
        for _ in 0..iterations {
            // Oracle
            circuit = circuit.dot(&Self::create_oracle_layer(num_qubits)?);

            // Diffusion operator
            circuit = circuit.dot(&Self::create_diffusion_layer(num_qubits)?);
        }

        Ok(circuit)
    }

    /// Create absorption layer for sponge construction
    fn create_absorption_layer(
        num_qubits: usize,
        rate: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        let dim = 2_usize.pow(num_qubits as u32);
        let mut layer = Array2::eye(dim);

        // Apply controlled operations on rate qubits
        for i in 0..rate {
            let angle = PI / (i + 1) as f64;
            let rotation = Self::ry_gate(angle);
            layer = Self::apply_single_qubit_gate(&layer, &rotation, i, num_qubits)?;
        }

        Ok(layer)
    }

    /// Create permutation layer
    fn create_permutation_layer(
        num_qubits: usize,
        round: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        let dim = 2_usize.pow(num_qubits as u32);
        let mut layer = Array2::eye(dim);

        // Round-dependent permutation
        for i in 0..num_qubits - 1 {
            let target = (i + round + 1) % num_qubits;
            let cnot = Self::cnot_gate();
            layer = Self::apply_two_qubit_gate(&layer, &cnot, i, target, num_qubits)?;
        }

        Ok(layer)
    }

    /// Create squeezing layer
    fn create_squeezing_layer(
        num_qubits: usize,
        capacity: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        let dim = 2_usize.pow(num_qubits as u32);
        let mut layer = Array2::eye(dim);

        // Extract hash from capacity qubits
        let start_qubit = num_qubits - capacity;
        for i in start_qubit..num_qubits {
            let measurement_basis = Self::create_measurement_basis(i as f64);
            layer = Self::apply_single_qubit_gate(&layer, &measurement_basis, i, num_qubits)?;
        }

        Ok(layer)
    }

    /// Create compression node for Merkle tree
    fn create_compression_node(
        num_qubits: usize,
        arity: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        let dim = 2_usize.pow(num_qubits as u32);
        let mut node = Array2::eye(dim);

        // Compress arity inputs into single output
        let qubits_per_input = num_qubits / arity;

        for group in 0..arity {
            let start_qubit = group * qubits_per_input;
            for i in 0..qubits_per_input - 1 {
                let cnot = Self::cnot_gate();
                node = Self::apply_two_qubit_gate(
                    &node,
                    &cnot,
                    start_qubit + i,
                    start_qubit + i + 1,
                    num_qubits,
                )?;
            }
        }

        Ok(node)
    }

    /// Create Hadamard layer
    fn create_hadamard_layer(num_qubits: usize) -> Result<Array2<Complex64>, QuantRS2Error> {
        let dim = 2_usize.pow(num_qubits as u32);
        let mut layer = Array2::eye(dim);

        let hadamard = Self::hadamard_gate();
        for i in 0..num_qubits {
            layer = Self::apply_single_qubit_gate(&layer, &hadamard, i, num_qubits)?;
        }

        Ok(layer)
    }

    /// Create oracle layer for Grover
    fn create_oracle_layer(num_qubits: usize) -> Result<Array2<Complex64>, QuantRS2Error> {
        let dim = 2_usize.pow(num_qubits as u32);
        let mut oracle = Array2::eye(dim);

        // Mark target state |11...1⟩ by applying Z gate
        let target_index = dim - 1;
        oracle[[target_index, target_index]] = Complex64::new(-1.0, 0.0);

        Ok(oracle)
    }

    /// Create diffusion layer for Grover
    fn create_diffusion_layer(num_qubits: usize) -> Result<Array2<Complex64>, QuantRS2Error> {
        let dim = 2_usize.pow(num_qubits as u32);
        let mut diffusion = Array2::eye(dim);

        // 2|s⟩⟨s| - I where |s⟩ is uniform superposition
        let coeff = 2.0 / (dim as f64);
        for i in 0..dim {
            for j in 0..dim {
                if i == j {
                    diffusion[[i, j]] = Complex64::new(coeff - 1.0, 0.0);
                } else {
                    diffusion[[i, j]] = Complex64::new(coeff, 0.0);
                }
            }
        }

        Ok(diffusion)
    }

    /// Hash input data
    pub fn hash(&self, input: &[u8]) -> Result<Vec<u8>, QuantRS2Error> {
        // Convert input to quantum state
        let input_state = self.classical_to_quantum(input)?;

        // Apply quantum hash circuit
        let output_state = self.quantum_circuit.dot(&input_state);

        // Extract classical hash
        self.quantum_to_classical(&output_state)
    }

    /// Convert classical input to quantum state
    fn classical_to_quantum(&self, input: &[u8]) -> Result<Array1<Complex64>, QuantRS2Error> {
        let dim = 2_usize.pow(self.num_qubits as u32);
        let mut state = Array1::zeros(dim);

        // Encode input bits into quantum state amplitudes
        let input_bits = self.bytes_to_bits(input);
        let effective_bits = input_bits.len().min(self.num_qubits);

        // Create superposition based on input
        for i in 0..effective_bits {
            if input_bits[i] {
                let basis_state = 1 << i;
                if basis_state < dim {
                    state[basis_state] = Complex64::new(1.0, 0.0);
                }
            }
        }

        // Normalize
        let norm = state.dot(&state.mapv(|x| x.conj())).norm();
        if norm > 0.0 {
            state = state / norm;
        } else {
            state[0] = Complex64::new(1.0, 0.0); // Default to |0⟩
        }

        Ok(state)
    }

    /// Convert quantum state to classical hash
    fn quantum_to_classical(&self, state: &Array1<Complex64>) -> Result<Vec<u8>, QuantRS2Error> {
        let probabilities: Vec<f64> = state.iter().map(|amp| amp.norm_sqr()).collect();

        // Extract bits from measurement probabilities
        let mut hash_bits = Vec::new();
        for (i, &prob) in probabilities.iter().enumerate() {
            if prob > 0.5 {
                hash_bits.push(i % 2 == 1);
            }
        }

        // Pad or truncate to desired hash size
        hash_bits.resize(self.hash_size * 8, false);

        Ok(self.bits_to_bytes(&hash_bits))
    }

    /// Convert bytes to bit vector
    fn bytes_to_bits(&self, bytes: &[u8]) -> Vec<bool> {
        bytes
            .iter()
            .flat_map(|&byte| (0..8).map(move |i| (byte >> i) & 1 == 1))
            .collect()
    }

    /// Convert bit vector to bytes
    fn bits_to_bytes(&self, bits: &[bool]) -> Vec<u8> {
        bits.chunks(8)
            .map(|chunk| {
                chunk.iter().enumerate().fold(
                    0u8,
                    |acc, (i, &bit)| {
                        if bit {
                            acc | (1 << i)
                        } else {
                            acc
                        }
                    },
                )
            })
            .collect()
    }

    /// Helper gate implementations
    fn ry_gate(angle: f64) -> Array2<Complex64> {
        let cos_half = (angle / 2.0).cos();
        let sin_half = (angle / 2.0).sin();

        scirs2_core::ndarray::array![
            [
                Complex64::new(cos_half, 0.0),
                Complex64::new(-sin_half, 0.0)
            ],
            [Complex64::new(sin_half, 0.0), Complex64::new(cos_half, 0.0)]
        ]
    }

    fn cnot_gate() -> Array2<Complex64> {
        scirs2_core::ndarray::array![
            [
                Complex64::new(1.0, 0.0),
                Complex64::new(0.0, 0.0),
                Complex64::new(0.0, 0.0),
                Complex64::new(0.0, 0.0)
            ],
            [
                Complex64::new(0.0, 0.0),
                Complex64::new(1.0, 0.0),
                Complex64::new(0.0, 0.0),
                Complex64::new(0.0, 0.0)
            ],
            [
                Complex64::new(0.0, 0.0),
                Complex64::new(0.0, 0.0),
                Complex64::new(0.0, 0.0),
                Complex64::new(1.0, 0.0)
            ],
            [
                Complex64::new(0.0, 0.0),
                Complex64::new(0.0, 0.0),
                Complex64::new(1.0, 0.0),
                Complex64::new(0.0, 0.0)
            ]
        ]
    }

    fn hadamard_gate() -> Array2<Complex64> {
        let inv_sqrt2 = 1.0 / 2.0_f64.sqrt();
        scirs2_core::ndarray::array![
            [
                Complex64::new(inv_sqrt2, 0.0),
                Complex64::new(inv_sqrt2, 0.0)
            ],
            [
                Complex64::new(inv_sqrt2, 0.0),
                Complex64::new(-inv_sqrt2, 0.0)
            ]
        ]
    }

    fn create_measurement_basis(phase: f64) -> Array2<Complex64> {
        let exp_phase = Complex64::from_polar(1.0, phase);
        scirs2_core::ndarray::array![
            [Complex64::new(1.0, 0.0), Complex64::new(0.0, 0.0)],
            [Complex64::new(0.0, 0.0), exp_phase]
        ]
    }

    /// Apply single qubit gate (simplified implementation)
    fn apply_single_qubit_gate(
        circuit: &Array2<Complex64>,
        gate: &Array2<Complex64>,
        target_qubit: usize,
        num_qubits: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        // Create tensor product gate for multi-qubit system
        let dim = 2_usize.pow(num_qubits as u32);
        let mut full_gate = Array2::eye(dim);

        // Simple implementation: apply gate only to computational basis states
        // This is a simplified version - a full implementation would need proper tensor products
        for i in 0..dim {
            let target_bit = (i >> target_qubit) & 1;
            if target_bit == 0 {
                // Apply gate[0,0] and gate[1,0] components
                let j = i | (1 << target_qubit);
                if j < dim {
                    full_gate[[i, i]] = gate[[0, 0]];
                    full_gate[[j, i]] = gate[[1, 0]];
                }
            } else {
                // Apply gate[0,1] and gate[1,1] components
                let j = i & !(1 << target_qubit);
                if j < dim {
                    full_gate[[j, i]] = gate[[0, 1]];
                    full_gate[[i, i]] = gate[[1, 1]];
                }
            }
        }

        Ok(circuit.dot(&full_gate))
    }

    /// Apply two qubit gate (simplified implementation)
    fn apply_two_qubit_gate(
        circuit: &Array2<Complex64>,
        gate: &Array2<Complex64>,
        control: usize,
        target: usize,
        num_qubits: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        // Create tensor product gate for multi-qubit system
        let dim = 2_usize.pow(num_qubits as u32);
        let mut full_gate = Array2::eye(dim);

        // Simplified implementation: apply gate to control-target qubit pairs
        // This is a placeholder - full implementation would need proper tensor products
        for i in 0..dim {
            let control_bit = (i >> control) & 1;
            let target_bit = (i >> target) & 1;
            let two_qubit_state = (control_bit << 1) | target_bit;

            // Apply 4x4 gate to the two-qubit subspace
            if two_qubit_state < 4 {
                for j in 0..4 {
                    let new_control = (j >> 1) & 1;
                    let new_target = j & 1;
                    let new_i = (i & !((1 << control) | (1 << target)))
                        | (new_control << control)
                        | (new_target << target);
                    if new_i < dim {
                        full_gate[[new_i, i]] = gate[[j, two_qubit_state]];
                    }
                }
            }
        }

        Ok(circuit.dot(&full_gate))
    }
}

/// Quantum digital signature verification gates
#[derive(Debug, Clone)]
pub struct QuantumDigitalSignature {
    pub public_key: Array2<Complex64>,
    pub private_key: Array1<Complex64>,
    pub signature_length: usize,
    pub security_parameter: usize,
}

impl QuantumDigitalSignature {
    /// Create a new quantum digital signature scheme
    pub fn new(signature_length: usize, security_parameter: usize) -> Result<Self, QuantRS2Error> {
        let (public_key, private_key) =
            Self::generate_key_pair(signature_length, security_parameter)?;

        Ok(Self {
            public_key,
            private_key,
            signature_length,
            security_parameter,
        })
    }

    /// Generate quantum key pair
    fn generate_key_pair(
        signature_length: usize,
        security_parameter: usize,
    ) -> Result<(Array2<Complex64>, Array1<Complex64>), QuantRS2Error> {
        let mut rng = ChaCha20Rng::from_seed([0u8; 32]); // Use fixed seed for deterministic behavior

        // Generate random private key
        let private_key = Array1::from_shape_fn(signature_length, |_| {
            Complex64::new(rng.random_range(-1.0..1.0), rng.random_range(-1.0..1.0))
        });

        // Generate public key using quantum one-way function
        let public_key = Self::quantum_one_way_function(&private_key, security_parameter)?;

        Ok((public_key, private_key))
    }

    /// Quantum one-way function for key generation
    fn quantum_one_way_function(
        private_key: &Array1<Complex64>,
        security_parameter: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        let n = private_key.len();
        let mut public_key = Array2::eye(n);

        // Apply sequence of quantum gates based on private key
        for (i, &key_element) in private_key.iter().enumerate() {
            let angle = key_element.arg() * (security_parameter as f64);
            let quantum_gate = Self::parameterized_quantum_gate(angle, i, n)?;
            public_key = public_key.dot(&quantum_gate);
        }

        Ok(public_key)
    }

    /// Create parameterized quantum gate
    fn parameterized_quantum_gate(
        angle: f64,
        index: usize,
        matrix_size: usize,
    ) -> Result<Array2<Complex64>, QuantRS2Error> {
        let cos_val = angle.cos();
        let sin_val = angle.sin();
        let phase = Complex64::from_polar(1.0, (index as f64) * PI / 4.0);

        // Create identity matrix of the correct size
        let mut gate = Array2::eye(matrix_size);

        // Apply rotation to specific indices (simplified approach)
        let i = index % matrix_size;
        let j = (index + 1) % matrix_size;

        // Apply 2x2 rotation to positions (i,j)
        gate[[i, i]] = Complex64::new(cos_val, 0.0);
        gate[[i, j]] = Complex64::new(-sin_val, 0.0) * phase;
        gate[[j, i]] = Complex64::new(sin_val, 0.0) * phase.conj();
        gate[[j, j]] = Complex64::new(cos_val, 0.0);

        Ok(gate)
    }

    /// Sign a quantum message
    pub fn sign(&self, message: &Array1<Complex64>) -> Result<QuantumSignature, QuantRS2Error> {
        // Hash the message
        let num_qubits = (message.len().next_power_of_two().trailing_zeros() as usize).max(8);
        let hash_function = QuantumHashFunction::new(
            num_qubits,
            self.signature_length / 8,
            CompressionFunction::QuantumSponge {
                rate: 4,
                capacity: 4,
            },
        )?;

        let message_bytes = self.quantum_state_to_bytes(message)?;
        let message_hash = hash_function.hash(&message_bytes)?;

        // Create quantum signature using private key
        let signature_state = self.create_signature_state(&message_hash)?;

        Ok(QuantumSignature {
            signature_state,
            message_hash,
            timestamp: std::time::SystemTime::now(),
        })
    }

    /// Create quantum signature state
    fn create_signature_state(
        &self,
        message_hash: &[u8],
    ) -> Result<Array1<Complex64>, QuantRS2Error> {
        let hash_bits = self.bytes_to_bits(message_hash);
        let mut signature_state = Array1::zeros(self.signature_length);

        // Encode signature using private key and message hash
        for (i, &bit) in hash_bits.iter().enumerate() {
            if i < self.signature_length {
                if bit {
                    signature_state[i] = self.private_key[i % self.private_key.len()];
                } else {
                    signature_state[i] = self.private_key[i % self.private_key.len()].conj();
                }
            }
        }

        // Normalize
        let norm = signature_state
            .dot(&signature_state.mapv(|x| x.conj()))
            .norm();
        if norm > 0.0 {
            signature_state = signature_state / norm;
        }

        Ok(signature_state)
    }

    /// Verify quantum signature
    pub fn verify(
        &self,
        message: &Array1<Complex64>,
        signature: &QuantumSignature,
    ) -> Result<bool, QuantRS2Error> {
        // Recompute message hash
        let num_qubits = (message.len().next_power_of_two().trailing_zeros() as usize).max(8);
        let hash_function = QuantumHashFunction::new(
            num_qubits,
            self.signature_length / 8,
            CompressionFunction::QuantumSponge {
                rate: 4,
                capacity: 4,
            },
        )?;

        let message_bytes = self.quantum_state_to_bytes(message)?;
        let computed_hash = hash_function.hash(&message_bytes)?;

        // Check hash consistency
        if computed_hash != signature.message_hash {
            return Ok(false);
        }

        // Verify quantum signature using public key
        let verification_result =
            self.quantum_signature_verification(&signature.signature_state)?;

        Ok(verification_result)
    }

    /// Quantum signature verification
    fn quantum_signature_verification(
        &self,
        signature_state: &Array1<Complex64>,
    ) -> Result<bool, QuantRS2Error> {
        // Apply public key transformation
        let transformed_state = self.public_key.dot(signature_state);

        // Check if transformed state satisfies verification condition
        let verification_observable = self.create_verification_observable()?;
        let expectation = transformed_state
            .t()
            .dot(&verification_observable.dot(&transformed_state));

        // Signature is valid if expectation value is above threshold
        Ok(expectation.re > 0.5)
    }

    /// Create verification observable
    fn create_verification_observable(&self) -> Result<Array2<Complex64>, QuantRS2Error> {
        let n = self.signature_length;
        let mut observable = Array2::zeros((n, n));

        // Create observable that measures signature validity
        for i in 0..n {
            observable[[i, i]] = Complex64::new(1.0, 0.0);
            if i > 0 {
                observable[[i, i - 1]] = Complex64::new(0.5, 0.0);
                observable[[i - 1, i]] = Complex64::new(0.5, 0.0);
            }
        }

        Ok(observable)
    }

    /// Helper functions
    fn quantum_state_to_bytes(&self, state: &Array1<Complex64>) -> Result<Vec<u8>, QuantRS2Error> {
        let bits: Vec<bool> = state.iter().map(|amp| amp.norm_sqr() > 0.5).collect();
        Ok(self.bits_to_bytes(&bits))
    }

    fn bytes_to_bits(&self, bytes: &[u8]) -> Vec<bool> {
        bytes
            .iter()
            .flat_map(|&byte| (0..8).map(move |i| (byte >> i) & 1 == 1))
            .collect()
    }

    fn bits_to_bytes(&self, bits: &[bool]) -> Vec<u8> {
        bits.chunks(8)
            .map(|chunk| {
                chunk.iter().enumerate().fold(
                    0u8,
                    |acc, (i, &bit)| {
                        if bit {
                            acc | (1 << i)
                        } else {
                            acc
                        }
                    },
                )
            })
            .collect()
    }
}

/// Quantum signature structure
#[derive(Debug, Clone)]
pub struct QuantumSignature {
    pub signature_state: Array1<Complex64>,
    pub message_hash: Vec<u8>,
    pub timestamp: std::time::SystemTime,
}

/// Quantum Key Distribution protocol gates
#[derive(Debug, Clone)]
pub struct QuantumKeyDistribution {
    pub protocol_type: QKDProtocol,
    pub key_length: usize,
    pub security_parameter: f64,
    pub noise_threshold: f64,
}

#[derive(Debug, Clone)]
pub enum QKDProtocol {
    BB84,
    E91,
    SARG04,
    COW,
}

impl QuantumKeyDistribution {
    /// Create a new QKD protocol
    pub fn new(protocol_type: QKDProtocol, key_length: usize, security_parameter: f64) -> Self {
        Self {
            protocol_type,
            key_length,
            security_parameter,
            noise_threshold: 0.11, // Standard QBER threshold
        }
    }

    /// Execute QKD protocol
    pub fn distribute_key(&self) -> Result<QKDResult, QuantRS2Error> {
        match self.protocol_type {
            QKDProtocol::BB84 => self.execute_bb84(),
            QKDProtocol::E91 => self.execute_e91(),
            QKDProtocol::SARG04 => self.execute_sarg04(),
            QKDProtocol::COW => self.execute_cow(),
        }
    }

    /// Execute BB84 protocol
    fn execute_bb84(&self) -> Result<QKDResult, QuantRS2Error> {
        let mut rng = ChaCha20Rng::from_seed([0u8; 32]); // Use fixed seed for deterministic behavior

        // Alice's random bits and bases
        let alice_bits: Vec<bool> = (0..self.key_length * 2).map(|_| rng.random()).collect();
        let alice_bases: Vec<bool> = (0..self.key_length * 2).map(|_| rng.random()).collect();

        // Bob's random bases
        let bob_bases: Vec<bool> = (0..self.key_length * 2).map(|_| rng.random()).collect();

        // Quantum state preparation and measurement
        let mut sifted_key = Vec::new();
        let mut qber_errors = 0;
        let total_bits = alice_bits.len();

        for i in 0..total_bits {
            // Alice prepares quantum state
            let quantum_state = self.prepare_bb84_state(alice_bits[i], alice_bases[i])?;

            // Simulate quantum channel with noise
            let noisy_state = self.apply_channel_noise(&quantum_state)?;

            // Bob measures
            let (measurement_result, measurement_success) =
                self.measure_bb84_state(&noisy_state, bob_bases[i])?;

            // Basis reconciliation
            if alice_bases[i] == bob_bases[i] && measurement_success {
                sifted_key.push(measurement_result);

                // Check for errors (measurement should match Alice's bit when bases match)
                if measurement_result != alice_bits[i] {
                    qber_errors += 1;
                }
            }
        }

        let qber = if sifted_key.is_empty() {
            0.0
        } else {
            qber_errors as f64 / sifted_key.len() as f64
        };

        if sifted_key.is_empty() {
            return Err(QuantRS2Error::QKDFailure(
                "No sifted key bits available".to_string(),
            ));
        }

        if qber > self.noise_threshold {
            return Err(QuantRS2Error::QKDFailure(format!(
                "QBER {} exceeds threshold {}",
                qber, self.noise_threshold
            )));
        }

        // Privacy amplification
        let final_key = self.privacy_amplification(&sifted_key, qber)?;

        Ok(QKDResult {
            shared_key: final_key,
            qber,
            key_rate: sifted_key.len() as f64 / total_bits as f64,
            security_parameter: self.security_parameter,
        })
    }

    /// Prepare BB84 quantum state
    fn prepare_bb84_state(
        &self,
        bit: bool,
        basis: bool,
    ) -> Result<Array1<Complex64>, QuantRS2Error> {
        match (bit, basis) {
            (false, false) => Ok(scirs2_core::ndarray::array![
                Complex64::new(1.0, 0.0),
                Complex64::new(0.0, 0.0)
            ]), // |0⟩
            (true, false) => Ok(scirs2_core::ndarray::array![
                Complex64::new(0.0, 0.0),
                Complex64::new(1.0, 0.0)
            ]), // |1⟩
            (false, true) => Ok(scirs2_core::ndarray::array![
                Complex64::new(1.0 / 2.0_f64.sqrt(), 0.0),
                Complex64::new(1.0 / 2.0_f64.sqrt(), 0.0)
            ]), // |+⟩
            (true, true) => Ok(scirs2_core::ndarray::array![
                Complex64::new(1.0 / 2.0_f64.sqrt(), 0.0),
                Complex64::new(-1.0 / 2.0_f64.sqrt(), 0.0)
            ]), // |-⟩
        }
    }

    /// Measure BB84 quantum state
    fn measure_bb84_state(
        &self,
        state: &Array1<Complex64>,
        basis: bool,
    ) -> Result<(bool, bool), QuantRS2Error> {
        let mut rng = ChaCha20Rng::from_seed([42u8; 32]); // Use fixed seed for deterministic behavior

        if basis {
            // X basis measurement (+ or - basis)
            // |+⟩ = (|0⟩ + |1⟩)/√2, |-⟩ = (|0⟩ - |1⟩)/√2
            let x_plus_amplitude = (state[0] + state[1]) / 2.0_f64.sqrt();
            let prob_plus = x_plus_amplitude.norm_sqr();
            let measurement = rng.random::<f64>() < prob_plus;
            Ok((!measurement, true)) // Map: |+⟩ -> false, |-⟩ -> true
        } else {
            // Z basis measurement (|0⟩ or |1⟩ basis)
            let prob_zero = state[0].norm_sqr();
            let measurement = rng.random::<f64>() < prob_zero;
            Ok((!measurement, true)) // Map: |0⟩ -> false, |1⟩ -> true
        }
    }

    /// Execute E91 protocol
    fn execute_e91(&self) -> Result<QKDResult, QuantRS2Error> {
        // Simplified E91 implementation
        let mut rng = ChaCha20Rng::from_seed([0u8; 32]); // Use fixed seed for deterministic behavior
        let mut shared_key = Vec::new();

        for _ in 0..self.key_length {
            // Create entangled pair
            let entangled_state = self.create_bell_state()?;

            // Alice and Bob choose random measurement bases
            let alice_basis = rng.random_range(0..3);
            let bob_basis = rng.random_range(0..3);

            // Perform measurements
            let alice_result = self.measure_entangled_qubit(&entangled_state, alice_basis, 0)?;
            let bob_result = self.measure_entangled_qubit(&entangled_state, bob_basis, 1)?;

            // Check Bell inequality violation for security
            let _correlation = if alice_result == bob_result {
                1.0
            } else {
                -1.0
            };

            // Use results for key generation (simplified)
            if alice_basis == bob_basis {
                shared_key.push(if alice_result { 1 } else { 0 });
            }
        }

        Ok(QKDResult {
            shared_key,
            qber: 0.01, // Simplified
            key_rate: 0.5,
            security_parameter: self.security_parameter,
        })
    }

    /// Execute SARG04 protocol
    fn execute_sarg04(&self) -> Result<QKDResult, QuantRS2Error> {
        // SARG04 is similar to BB84 but with different information reconciliation
        let bb84_result = self.execute_bb84()?;

        // SARG04 specific post-processing would go here
        Ok(bb84_result)
    }

    /// Execute COW protocol
    fn execute_cow(&self) -> Result<QKDResult, QuantRS2Error> {
        // Coherent one-way protocol
        let mut rng = ChaCha20Rng::from_seed([0u8; 32]); // Use fixed seed for deterministic behavior
        let mut shared_key = Vec::new();

        for _ in 0..self.key_length {
            // Alice sends coherent states
            let bit = rng.random::<bool>();
            let coherent_state = self.prepare_coherent_state(bit)?;

            // Bob performs measurements
            let measurement_result = self.measure_coherent_state(&coherent_state)?;

            shared_key.push(if measurement_result { 1 } else { 0 });
        }

        Ok(QKDResult {
            shared_key,
            qber: 0.05,
            key_rate: 0.8,
            security_parameter: self.security_parameter,
        })
    }

    /// Create Bell state for E91
    fn create_bell_state(&self) -> Result<Array1<Complex64>, QuantRS2Error> {
        // |Φ+⟩ = (|00⟩ + |11⟩)/√2
        Ok(scirs2_core::ndarray::array![
            Complex64::new(1.0 / 2.0_f64.sqrt(), 0.0),
            Complex64::new(0.0, 0.0),
            Complex64::new(0.0, 0.0),
            Complex64::new(1.0 / 2.0_f64.sqrt(), 0.0)
        ])
    }

    /// Measure entangled qubit
    fn measure_entangled_qubit(
        &self,
        _state: &Array1<Complex64>,
        _basis: usize,
        _qubit: usize,
    ) -> Result<bool, QuantRS2Error> {
        let mut rng = ChaCha20Rng::from_seed([0u8; 32]); // Use fixed seed for deterministic behavior
        Ok(rng.random())
    }

    /// Prepare coherent state for COW
    fn prepare_coherent_state(&self, bit: bool) -> Result<Array1<Complex64>, QuantRS2Error> {
        let alpha: f64 = if bit { 1.0 } else { -1.0 };

        // Simplified coherent state |α⟩
        Ok(scirs2_core::ndarray::array![
            Complex64::new((-alpha.powi(2) / 2.0).exp() * alpha, 0.0),
            Complex64::new((-alpha.powi(2) / 2.0).exp(), 0.0)
        ])
    }

    /// Measure coherent state
    fn measure_coherent_state(&self, _state: &Array1<Complex64>) -> Result<bool, QuantRS2Error> {
        let mut rng = ChaCha20Rng::from_seed([0u8; 32]); // Use fixed seed for deterministic behavior
        Ok(rng.random())
    }

    /// Apply channel noise
    fn apply_channel_noise(
        &self,
        state: &Array1<Complex64>,
    ) -> Result<Array1<Complex64>, QuantRS2Error> {
        let mut rng = ChaCha20Rng::from_seed([0u8; 32]); // Use fixed seed for deterministic behavior
        let noise_level = 0.02; // 2% noise - realistic for QKD

        let mut noisy_state = state.clone();
        if rng.random::<f64>() < noise_level {
            // Apply bit flip
            noisy_state = scirs2_core::ndarray::array![state[1], state[0]];
        }

        Ok(noisy_state)
    }

    /// Privacy amplification
    fn privacy_amplification(&self, key: &[bool], qber: f64) -> Result<Vec<u8>, QuantRS2Error> {
        // Simplified privacy amplification using classical hash
        let entropy_loss = if qber > 0.0 && qber < 1.0 {
            2.0 * qber * (qber.log2() + (1.0 - qber).log2())
        } else {
            0.5 // Default entropy loss for edge cases
        };
        let final_length = ((key.len() as f64) * (1.0 - entropy_loss.abs())).max(1.0) as usize;

        let key_bytes = self.bits_to_bytes(key);
        // Simplified hash function (in production, use proper SHA3-256)
        let mut hash_result = Vec::new();
        for (i, &byte) in key_bytes.iter().enumerate() {
            hash_result.push(byte.wrapping_add(i as u8));
        }

        Ok(hash_result[..final_length.min(key_bytes.len())].to_vec())
    }

    fn bits_to_bytes(&self, bits: &[bool]) -> Vec<u8> {
        bits.chunks(8)
            .map(|chunk| {
                chunk.iter().enumerate().fold(
                    0u8,
                    |acc, (i, &bit)| {
                        if bit {
                            acc | (1 << i)
                        } else {
                            acc
                        }
                    },
                )
            })
            .collect()
    }
}

/// QKD result structure
#[derive(Debug, Clone)]
pub struct QKDResult {
    pub shared_key: Vec<u8>,
    pub qber: f64,
    pub key_rate: f64,
    pub security_parameter: f64,
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_quantum_hash_function() {
        let hash_function = QuantumHashFunction::new(
            4,
            8,
            CompressionFunction::QuantumSponge {
                rate: 2,
                capacity: 2,
            },
        );

        assert!(hash_function.is_ok());
        let qhf = hash_function.unwrap();

        let input = b"test message";
        let hash_result = qhf.hash(input);
        assert!(hash_result.is_ok());
        assert_eq!(hash_result.unwrap().len(), 8);
    }

    #[test]
    #[ignore]
    fn test_quantum_digital_signature() {
        let signature_scheme = QuantumDigitalSignature::new(16, 128);
        assert!(signature_scheme.is_ok());

        let qds = signature_scheme.unwrap();
        let message = scirs2_core::ndarray::array![Complex64::new(1.0, 0.0), Complex64::new(0.0, 1.0)];

        let signature = qds.sign(&message);
        assert!(signature.is_ok());

        let verification = qds.verify(&message, &signature.unwrap());
        assert!(verification.is_ok());
    }

    #[test]
    fn test_quantum_key_distribution() {
        let qkd = QuantumKeyDistribution::new(QKDProtocol::BB84, 100, 0.1); // 10% error threshold

        let result = qkd.distribute_key();
        match result {
            Ok(qkd_result) => {
                assert!(!qkd_result.shared_key.is_empty());
                assert!(qkd_result.qber >= 0.0);
                assert!(qkd_result.key_rate > 0.0);
            }
            Err(e) => {
                println!("QKD error: {:?}", e);
                panic!("QKD failed with error: {:?}", e);
            }
        }
    }

    #[test]
    fn test_bb84_state_preparation() {
        let qkd = QuantumKeyDistribution::new(QKDProtocol::BB84, 1, 1e-6);

        let state_0_z = qkd.prepare_bb84_state(false, false).unwrap();
        assert!((state_0_z[0].norm() - 1.0).abs() < 1e-10);

        let state_plus = qkd.prepare_bb84_state(false, true).unwrap();
        assert!((state_plus[0].norm() - 1.0 / 2.0_f64.sqrt()).abs() < 1e-10);
    }
}
