//! Enhanced Noise Characterization with Advanced SciRS2 Statistical Analysis
//!
//! This module provides state-of-the-art noise characterization for quantum devices
//! using ML-based analysis, statistical modeling, temporal correlation tracking,
//! and comprehensive error rate characterization powered by SciRS2's statistics tools.

use quantrs2_core::{
    error::{QuantRS2Error, QuantRS2Result},
    gate::GateOp,
    qubit::QubitId,
    buffer_pool::BufferPool,
};
use quantrs2_circuit::builder::Circuit;
use crate::scirs2_hardware_benchmarks_enhanced::StatisticalAnalysis;
// use scirs2_core::parallel_ops::*;
// use scirs2_core::memory::BufferPool;
// use scirs2_core::platform::PlatformCapabilities;
// use scirs2_optimize::statistics::{
//     StatisticalAnalyzer, DistributionFitter, CorrelationAnalyzer,
//     TimeSeriesAnalyzer, SpectralAnalyzer
// };
// use scirs2_linalg::{Matrix, Vector, SVD, Eigendecomposition};
// use scirs2_sparse::CSRMatrix;
use scirs2_core::ndarray::{Array1, Array2, Array3, Array4, ArrayView2};
use scirs2_core::Complex64;
use serde::{Deserialize, Serialize};
use std::collections::{HashMap, VecDeque, BTreeMap};
use std::sync::{Arc, Mutex};
use std::fmt;
use scirs2_core::random::{Distribution, Normal, Exp as Exponential};
// use statrs::statistics::{Statistics, OrderStatistics};
// use statrs::distribution::{Beta, Gamma, Weibull};

/// Enhanced noise characterization configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct EnhancedNoiseConfig {
    /// Base noise configuration
    pub base_config: NoiseCharacterizationConfig,

    /// Enable ML-based noise analysis
    pub enable_ml_analysis: bool,

    /// Enable temporal correlation tracking
    pub enable_temporal_tracking: bool,

    /// Enable spectral noise analysis
    pub enable_spectral_analysis: bool,

    /// Enable multi-qubit correlation analysis
    pub enable_correlation_analysis: bool,

    /// Enable predictive noise modeling
    pub enable_predictive_modeling: bool,

    /// Enable real-time monitoring
    pub enable_realtime_monitoring: bool,

    /// Noise models to characterize
    pub noise_models: Vec<NoiseModel>,

    /// Statistical methods
    pub statistical_methods: Vec<StatisticalMethod>,

    /// Analysis parameters
    pub analysis_parameters: AnalysisParameters,

    /// Reporting options
    pub reporting_options: ReportingOptions,
}

impl Default for EnhancedNoiseConfig {
    fn default() -> Self {
        Self {
            base_config: NoiseCharacterizationConfig::default(),
            enable_ml_analysis: true,
            enable_temporal_tracking: true,
            enable_spectral_analysis: true,
            enable_correlation_analysis: true,
            enable_predictive_modeling: true,
            enable_realtime_monitoring: true,
            noise_models: vec![
                NoiseModel::Depolarizing,
                NoiseModel::Dephasing,
                NoiseModel::AmplitudeDamping,
                NoiseModel::ThermalRelaxation,
                NoiseModel::CoherentError,
            ],
            statistical_methods: vec![
                StatisticalMethod::MaximumLikelihood,
                StatisticalMethod::BayesianInference,
                StatisticalMethod::SpectralDensity,
            ],
            analysis_parameters: AnalysisParameters::default(),
            reporting_options: ReportingOptions::default(),
        }
    }
}

/// Base noise characterization configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct NoiseCharacterizationConfig {
    /// Number of characterization sequences
    pub num_sequences: usize,

    /// Sequence lengths for RB
    pub sequence_lengths: Vec<usize>,

    /// Number of shots per sequence
    pub shots_per_sequence: usize,

    /// Confidence level for error bars
    pub confidence_level: f64,
}

impl Default for NoiseCharacterizationConfig {
    fn default() -> Self {
        Self {
            num_sequences: 100,
            sequence_lengths: vec![1, 2, 4, 8, 16, 32, 64, 128],
            shots_per_sequence: 1000,
            confidence_level: 0.95,
        }
    }
}

/// Noise model types
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash, Serialize, Deserialize)]
pub enum NoiseModel {
    Depolarizing,
    Dephasing,
    AmplitudeDamping,
    PhaseDamping,
    ThermalRelaxation,
    CoherentError,
    Leakage,
    Crosstalk,
    NonMarkovian,
    Correlated,
}

/// Statistical methods for noise analysis
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash, Serialize, Deserialize)]
pub enum StatisticalMethod {
    MaximumLikelihood,
    BayesianInference,
    SpectralDensity,
    ProcessTomography,
    RandomizedBenchmarking,
    InterlevedRB,
    PurityBenchmarking,
    CrossEntropyBenchmarking,
}

/// Analysis parameters
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct AnalysisParameters {
    /// Time window for temporal analysis (microseconds)
    pub temporal_window: f64,

    /// Frequency resolution for spectral analysis (Hz)
    pub frequency_resolution: f64,

    /// Correlation distance threshold
    pub correlation_threshold: f64,

    /// ML model update frequency
    pub ml_update_frequency: usize,

    /// Prediction horizon (microseconds)
    pub prediction_horizon: f64,
}

impl Default for AnalysisParameters {
    fn default() -> Self {
        Self {
            temporal_window: 1000.0,
            frequency_resolution: 1e3,
            correlation_threshold: 0.1,
            ml_update_frequency: 100,
            prediction_horizon: 100.0,
        }
    }
}

/// Reporting options
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ReportingOptions {
    /// Generate visual plots
    pub generate_plots: bool,

    /// Include raw data
    pub include_raw_data: bool,

    /// Include confidence intervals
    pub include_confidence_intervals: bool,

    /// Export format
    pub export_format: ExportFormat,
}

impl Default for ReportingOptions {
    fn default() -> Self {
        Self {
            generate_plots: true,
            include_raw_data: false,
            include_confidence_intervals: true,
            export_format: ExportFormat::JSON,
        }
    }
}

/// Export format for reports
#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
pub enum ExportFormat {
    JSON,
    CSV,
    HDF5,
    LaTeX,
}

/// Enhanced noise characterization system
pub struct EnhancedNoiseCharacterizer {
    config: EnhancedNoiseConfig,
    statistical_analyzer: Arc<StatisticalAnalysis>,
    ml_analyzer: Option<Arc<MLNoiseAnalyzer>>,
    temporal_tracker: Arc<TemporalNoiseTracker>,
    spectral_analyzer: Arc<SpectralNoiseAnalyzer>,
    correlation_analyzer: Arc<CorrelationAnalysis>,
    predictive_modeler: Arc<PredictiveNoiseModeler>,
    buffer_pool: BufferPool<f64>,
    cache: Arc<Mutex<NoiseCache>>,
}

impl EnhancedNoiseCharacterizer {
    /// Create new enhanced noise characterizer
    pub fn new(config: EnhancedNoiseConfig) -> Self {
        let buffer_pool = BufferPool::new(1000, 1024 * 1024);

        Self {
            config: config.clone(),
            statistical_analyzer: Arc::new(StatisticalAnalysis::default()),
            ml_analyzer: if config.enable_ml_analysis {
                Some(Arc::new(MLNoiseAnalyzer::new(config.clone())))
            } else {
                None
            },
            temporal_tracker: Arc::new(TemporalNoiseTracker::new(config.clone())),
            spectral_analyzer: Arc::new(SpectralNoiseAnalyzer::new(config.clone())),
            correlation_analyzer: Arc::new(CorrelationAnalysis::default()),
            predictive_modeler: Arc::new(PredictiveNoiseModeler::new(config.clone())),
            buffer_pool,
            cache: Arc::new(Mutex::new(NoiseCache::new())),
        }
    }

    /// Characterize noise for a quantum device
    pub fn characterize_noise(
        &self,
        device: &impl QuantumDevice,
        qubits: &[QubitId],
    ) -> QuantRS2Result<NoiseCharacterizationResult> {
        let mut result = NoiseCharacterizationResult::new();

        // Run different characterization protocols in parallel
        let tasks: Vec<_> = vec![
            self.run_randomized_benchmarking(device, qubits),
            self.run_process_tomography(device, qubits),
            self.run_spectral_analysis(device, qubits),
            self.run_correlation_analysis(device, qubits),
        ];

        let characterizations: Vec<_> = tasks.into_par_iter()
            .map(|task| task)
            .collect();

        // Combine results
        for char_result in characterizations {
            match char_result {
                Ok(data) => result.merge(data),
                Err(e) => return Err(e),
            }
        }

        // ML analysis if enabled
        if let Some(ml_analyzer) = &self.ml_analyzer {
            let ml_insights = ml_analyzer.analyze_noise_patterns(&result)?;
            result.ml_insights = Some(ml_insights);
        }

        // Predictive modeling
        if self.config.enable_predictive_modeling {
            let predictions = self.predictive_modeler.predict_noise_evolution(&result)?;
            result.noise_predictions = Some(predictions);
        }

        // Generate comprehensive report
        let report = self.generate_report(&result)?;
        result.report = Some(report);

        Ok(result)
    }

    /// Run randomized benchmarking
    fn run_randomized_benchmarking(
        &self,
        device: &impl QuantumDevice,
        qubits: &[QubitId],
    ) -> QuantRS2Result<CharacterizationData> {
        let mut rb_data = RBData::new();

        for &seq_length in &self.config.base_config.sequence_lengths {
            let sequences = self.generate_rb_sequences(qubits.len(), seq_length);

            let results: Vec<_> = sequences.par_iter()
                .map(|seq| self.execute_rb_sequence(device, qubits, seq))
                .collect();

            // Analyze results
            let survival_prob = self.calculate_survival_probability(&results)?;
            rb_data.add_point(seq_length, survival_prob);
        }

        // Fit exponential decay
        let fit_params = self.fit_rb_decay(&rb_data)?;

        Ok(CharacterizationData::RandomizedBenchmarking(rb_data, fit_params))
    }

    /// Run process tomography
    fn run_process_tomography(
        &self,
        device: &impl QuantumDevice,
        qubits: &[QubitId],
    ) -> QuantRS2Result<CharacterizationData> {
        let mut tomography_data = TomographyData::new();

        // Generate preparation and measurement bases
        let prep_states = self.generate_preparation_states(qubits.len());
        let meas_bases = self.generate_measurement_bases(qubits.len());

        // Run tomography experiments
        let experiments: Vec<_> = prep_states.par_iter()
            .flat_map(|prep| {
                meas_bases.par_iter()
                    .map(move |meas| (prep, meas))
            })
            .collect();

        let results: Vec<_> = experiments.par_iter()
            .map(|(prep, meas)| {
                self.execute_tomography_experiment(device, qubits, prep, meas)
            })
            .collect();

        // Reconstruct process matrix
        let process_matrix = self.reconstruct_process_matrix(&results)?;
        tomography_data.process_matrix = process_matrix;

        // Extract noise parameters
        let noise_params = self.extract_noise_parameters(&process_matrix)?;

        Ok(CharacterizationData::ProcessTomography(tomography_data, noise_params))
    }

    /// Run spectral noise analysis
    fn run_spectral_analysis(
        &self,
        device: &impl QuantumDevice,
        qubits: &[QubitId],
    ) -> QuantRS2Result<CharacterizationData> {
        let mut spectral_data = SpectralData::new();

        // Collect time series data
        let time_series = self.collect_noise_time_series(device, qubits)?;

        // Perform FFT analysis
        let spectrum = self.spectral_analyzer.compute_power_spectrum(&time_series)?;
        spectral_data.power_spectrum = spectrum;

        // Identify noise peaks
        let noise_peaks = self.identify_noise_peaks(&spectrum)?;
        spectral_data.noise_peaks = noise_peaks;

        // Analyze 1/f noise
        let one_over_f_params = self.analyze_one_over_f_noise(&spectrum)?;
        spectral_data.one_over_f_params = Some(one_over_f_params);

        Ok(CharacterizationData::SpectralAnalysis(spectral_data))
    }

    /// Run correlation analysis
    fn run_correlation_analysis(
        &self,
        device: &impl QuantumDevice,
        qubits: &[QubitId],
    ) -> QuantRS2Result<CharacterizationData> {
        let mut correlation_data = CorrelationData::new();

        // Measure simultaneous errors
        let error_data = self.measure_correlated_errors(device, qubits)?;

        // Compute correlation matrix
        let correlationmatrix = self.correlation_analyzer
            .compute_correlationmatrix(&error_data)?;
        correlation_data.correlationmatrix = correlationmatrix;

        // Identify correlated error clusters
        let clusters = self.identify_error_clusters(&correlationmatrix)?;
        correlation_data.error_clusters = clusters;

        // Analyze spatial correlations
        let spatial_corr = self.analyze_spatial_correlations(device, qubits)?;
        correlation_data.spatial_correlations = Some(spatial_corr);

        Ok(CharacterizationData::CorrelationAnalysis(correlation_data))
    }

    /// Generate comprehensive noise report
    fn generate_report(&self, result: &NoiseCharacterizationResult) -> QuantRS2Result<NoiseReport> {
        let mut report = NoiseReport::new();

        // Summary statistics
        report.summary = self.generate_summary_statistics(result)?;

        // Detailed analysis for each noise model
        for noise_model in &self.config.noise_models {
            let analysis = self.analyze_noise_model(result, *noise_model)?;
            report.model_analyses.insert(*noise_model, analysis);
        }

        // Temporal evolution
        if self.config.enable_temporal_tracking {
            report.temporal_analysis = Some(self.analyze_temporal_evolution(result)?);
        }

        // Spectral characteristics
        if self.config.enable_spectral_analysis {
            report.spectral_analysis = Some(self.analyze_spectral_characteristics(result)?);
        }

        // Correlation analysis
        if self.config.enable_correlation_analysis {
            report.correlation_analysis = Some(self.analyze_correlations(result)?);
        }

        // Recommendations
        report.recommendations = self.generate_recommendations(result)?;

        // Visualizations
        if self.config.reporting_options.generate_plots {
            report.visualizations = Some(self.generate_visualizations(result)?);
        }

        Ok(report)
    }

    /// Generate RB sequences
    fn generate_rb_sequences(&self, num_qubits: usize, length: usize) -> Vec<RBSequence> {
        let mut sequences = Vec::new();

        for _ in 0..self.config.base_config.num_sequences {
            let mut sequence = RBSequence::new();

            // Random Clifford gates
            for _ in 0..length {
                let clifford = self.random_clifford_gate(num_qubits);
                sequence.add_gate(clifford);
            }

            // Recovery gate
            let recovery = self.compute_recovery_gate(&sequence);
            sequence.add_gate(recovery);

            sequences.push(sequence);
        }

        sequences
    }

    /// Execute RB sequence
    fn execute_rb_sequence(
        &self,
        device: &impl QuantumDevice,
        qubits: &[QubitId],
        sequence: &RBSequence,
    ) -> QuantRS2Result<RBResult> {
        let circuit = sequence.to_circuit(qubits)?;
        let job = device.execute(circuit, self.config.base_config.shots_per_sequence)?;
        let counts = job.get_counts()?;

        // Calculate survival probability
        let total_shots = counts.values().sum::<usize>() as f64;
        let success_state = vec![false; qubits.len()]; // All zeros
        let success_count = counts.get(&success_state).unwrap_or(&0);
        let survival_prob = *success_count as f64 / total_shots;

        Ok(RBResult {
            sequence_length: sequence.length(),
            survival_probability: survival_prob,
            error_bars: self.calculate_error_bars(survival_prob, total_shots as usize),
        })
    }

    /// Fit RB decay curve
    fn fit_rb_decay(&self, rb_data: &RBData) -> QuantRS2Result<RBFitParameters> {
        let x: Vec<f64> = rb_data.sequence_lengths.iter()
            .map(|&l| l as f64)
            .collect();
        let y: Vec<f64> = rb_data.survival_probabilities.clone();

        // Fit: f(x) = A * p^x + B
        // where p is the decay parameter
        let (a, p, b) = self.statistical_analyzer.fit_exponential_decay(&x, &y)?;

        // Calculate average error rate
        let r = (1.0 - p) * (1.0 - 1.0 / 2.0_f64.powi(rb_data.num_qubits as i32));

        Ok(RBFitParameters {
            amplitude: a,
            decay_parameter: p,
            offset: b,
            average_error_rate: r,
            confidence_interval: self.calculate_fit_confidence_interval(&x, &y, a, p, b)?,
        })
    }
}

/// ML noise analyzer
struct MLNoiseAnalyzer {
    config: EnhancedNoiseConfig,
    model: Arc<Mutex<NoiseMLModel>>,
    feature_extractor: Arc<NoiseFeatureExtractor>,
}

impl MLNoiseAnalyzer {
    fn new(config: EnhancedNoiseConfig) -> Self {
        Self {
            config,
            model: Arc::new(Mutex::new(NoiseMLModel::new())),
            feature_extractor: Arc::new(NoiseFeatureExtractor::new()),
        }
    }

    fn analyze_noise_patterns(
        &self,
        result: &NoiseCharacterizationResult,
    ) -> QuantRS2Result<MLNoiseInsights> {
        let features = self.feature_extractor.extract_features(result)?;

        let mut model = self.model.lock().unwrap();
        let predictions = model.predict(&features)?;

        Ok(MLNoiseInsights {
            noise_classification: predictions.classification,
            anomaly_score: predictions.anomaly_score,
            predicted_evolution: predictions.evolution,
            confidence: predictions.confidence,
        })
    }
}

/// Temporal noise tracker
struct TemporalNoiseTracker {
    config: EnhancedNoiseConfig,
    time_series_analyzer: Arc<TimeSeriesAnalyzer>,
    history: Arc<Mutex<NoiseHistory>>,
}

impl TemporalNoiseTracker {
    fn new(config: EnhancedNoiseConfig) -> Self {
        Self {
            config,
            time_series_analyzer: Arc::new(TimeSeriesAnalyzer::new()),
            history: Arc::new(Mutex::new(NoiseHistory::new())),
        }
    }

    fn track_noise_evolution(
        &self,
        timestamp: f64,
        noise_data: &NoiseData,
    ) -> QuantRS2Result<()> {
        let mut history = self.history.lock().unwrap();
        history.add_measurement(timestamp, noise_data.clone());

        // Analyze trends
        if history.len() > 10 {
            let trend = self.time_series_analyzer.analyze_trend(&history.to_time_series())?;
            history.update_trend(trend);
        }

        Ok(())
    }
}

/// Spectral noise analyzer
struct SpectralNoiseAnalyzer {
    config: EnhancedNoiseConfig,
    spectral_analyzer: Arc<SpectralAnalyzer>,
}

impl SpectralNoiseAnalyzer {
    fn new(config: EnhancedNoiseConfig) -> Self {
        Self {
            config,
            spectral_analyzer: Arc::new(SpectralAnalyzer::new()),
        }
    }

    fn compute_power_spectrum(&self, time_series: &TimeSeries) -> QuantRS2Result<PowerSpectrum> {
        let spectrum = self.spectral_analyzer.compute_fft(time_series)?;
        let power = self.spectral_analyzer.compute_power_spectral_density(&spectrum)?;

        Ok(PowerSpectrum {
            frequencies: self.generate_frequency_bins(time_series.timestamps.len()),
            power_density: power,
            resolution: self.config.analysis_parameters.frequency_resolution,
        })
    }

    fn generate_frequency_bins(&self, n: usize) -> Vec<f64> {
        let nyquist = 1.0 / (2.0 * self.config.analysis_parameters.frequency_resolution);
        (0..n/2).map(|i| i as f64 * nyquist / (n/2) as f64).collect()
    }
}

/// Predictive noise modeler
struct PredictiveNoiseModeler {
    config: EnhancedNoiseConfig,
    predictor: Arc<Mutex<NoisePredictor>>,
}

impl PredictiveNoiseModeler {
    fn new(config: EnhancedNoiseConfig) -> Self {
        Self {
            config,
            predictor: Arc::new(Mutex::new(NoisePredictor::new())),
        }
    }

    fn predict_noise_evolution(
        &self,
        result: &NoiseCharacterizationResult,
    ) -> QuantRS2Result<NoisePredictions> {
        let mut predictor = self.predictor.lock().unwrap();

        // Update model with latest data
        predictor.update(result)?;

        // Generate predictions
        let horizon = self.config.analysis_parameters.prediction_horizon;
        let predictions = predictor.predict(horizon)?;

        Ok(predictions)
    }
}

/// Noise characterization result
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct NoiseCharacterizationResult {
    /// Timestamp
    pub timestamp: f64,

    /// Device identifier
    pub device_id: String,

    /// Characterized qubits
    pub qubits: Vec<QubitId>,

    /// RB results
    pub rb_results: Option<RBResults>,

    /// Process tomography results
    pub tomography_results: Option<TomographyResults>,

    /// Spectral analysis results
    pub spectral_results: Option<SpectralResults>,

    /// Correlation analysis results
    pub correlation_results: Option<CorrelationResults>,

    /// ML insights
    pub ml_insights: Option<MLNoiseInsights>,

    /// Noise predictions
    pub noise_predictions: Option<NoisePredictions>,

    /// Comprehensive report
    pub report: Option<NoiseReport>,
}

impl NoiseCharacterizationResult {
    fn new() -> Self {
        Self {
            timestamp: 0.0,
            device_id: String::new(),
            qubits: Vec::new(),
            rb_results: None,
            tomography_results: None,
            spectral_results: None,
            correlation_results: None,
            ml_insights: None,
            noise_predictions: None,
            report: None,
        }
    }

    fn merge(&mut self, data: CharacterizationData) {
        match data {
            CharacterizationData::RandomizedBenchmarking(rb_data, fit_params) => {
                self.rb_results = Some(RBResults { rb_data, fit_params });
            }
            CharacterizationData::ProcessTomography(tomo_data, noise_params) => {
                self.tomography_results = Some(TomographyResults { tomo_data, noise_params });
            }
            CharacterizationData::SpectralAnalysis(spectral_data) => {
                self.spectral_results = Some(SpectralResults { spectral_data });
            }
            CharacterizationData::CorrelationAnalysis(corr_data) => {
                self.correlation_results = Some(CorrelationResults { corr_data });
            }
        }
    }
}

/// Characterization data types
enum CharacterizationData {
    RandomizedBenchmarking(RBData, RBFitParameters),
    ProcessTomography(TomographyData, NoiseParameters),
    SpectralAnalysis(SpectralData),
    CorrelationAnalysis(CorrelationData),
}

/// RB data
#[derive(Debug, Clone, Serialize, Deserialize)]
struct RBData {
    sequence_lengths: Vec<usize>,
    survival_probabilities: Vec<f64>,
    error_bars: Vec<f64>,
    num_qubits: usize,
}

impl RBData {
    fn new() -> Self {
        Self {
            sequence_lengths: Vec::new(),
            survival_probabilities: Vec::new(),
            error_bars: Vec::new(),
            num_qubits: 0,
        }
    }

    fn add_point(&mut self, length: usize, probability: f64) {
        self.sequence_lengths.push(length);
        self.survival_probabilities.push(probability);
    }
}

/// RB fit parameters
#[derive(Debug, Clone, Serialize, Deserialize)]
struct RBFitParameters {
    amplitude: f64,
    decay_parameter: f64,
    offset: f64,
    average_error_rate: f64,
    confidence_interval: (f64, f64),
}

/// Tomography data
#[derive(Debug, Clone, Serialize, Deserialize)]
struct TomographyData {
    process_matrix: Array2<Complex64>,
    preparation_states: Vec<QuantumState>,
    measurement_bases: Vec<MeasurementBasis>,
    measurement_outcomes: HashMap<(usize, usize), Vec<f64>>,
}

impl TomographyData {
    fn new() -> Self {
        Self {
            process_matrix: Array2::zeros((0, 0)),
            preparation_states: Vec::new(),
            measurement_bases: Vec::new(),
            measurement_outcomes: HashMap::new(),
        }
    }
}

/// Noise parameters extracted from process tomography
#[derive(Debug, Clone, Serialize, Deserialize)]
struct NoiseParameters {
    depolarizing_rate: f64,
    dephasing_rate: f64,
    amplitude_damping_rate: f64,
    coherent_error_angle: f64,
    leakage_rate: Option<f64>,
}

/// Spectral data
#[derive(Debug, Clone, Serialize, Deserialize)]
struct SpectralData {
    power_spectrum: PowerSpectrum,
    noise_peaks: Vec<NoisePeak>,
    one_over_f_params: Option<OneOverFParameters>,
}

impl SpectralData {
    fn new() -> Self {
        Self {
            power_spectrum: PowerSpectrum::new(),
            noise_peaks: Vec::new(),
            one_over_f_params: None,
        }
    }
}

/// Power spectrum
#[derive(Debug, Clone, Serialize, Deserialize)]
struct PowerSpectrum {
    frequencies: Vec<f64>,
    power_density: Vec<f64>,
    resolution: f64,
}

impl PowerSpectrum {
    fn new() -> Self {
        Self {
            frequencies: Vec::new(),
            power_density: Vec::new(),
            resolution: 0.0,
        }
    }
}

/// Noise peak in spectrum
#[derive(Debug, Clone, Serialize, Deserialize)]
struct NoisePeak {
    frequency: f64,
    amplitude: f64,
    width: f64,
    source: Option<String>,
}

/// 1/f noise parameters
#[derive(Debug, Clone, Serialize, Deserialize)]
struct OneOverFParameters {
    amplitude: f64,
    exponent: f64,
    cutoff_frequency: f64,
}

/// Correlation data
#[derive(Debug, Clone, Serialize, Deserialize)]
struct CorrelationData {
    correlationmatrix: Array2<f64>,
    error_clusters: Vec<ErrorCluster>,
    spatial_correlations: Option<SpatialCorrelations>,
}

impl CorrelationData {
    fn new() -> Self {
        Self {
            correlationmatrix: Array2::zeros((0, 0)),
            error_clusters: Vec::new(),
            spatial_correlations: None,
        }
    }
}

/// Error cluster
#[derive(Debug, Clone, Serialize, Deserialize)]
struct ErrorCluster {
    qubits: Vec<QubitId>,
    correlation_strength: f64,
    cluster_type: ClusterType,
}

/// Cluster type
#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
enum ClusterType {
    NearestNeighbor,
    LongRange,
    AllToAll,
}

/// Spatial correlations
#[derive(Debug, Clone, Serialize, Deserialize)]
struct SpatialCorrelations {
    distance_correlations: Vec<(f64, f64)>, // (distance, correlation)
    decay_length: f64,
    correlation_type: SpatialCorrelationType,
}

/// Spatial correlation type
#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
enum SpatialCorrelationType {
    Exponential,
    PowerLaw,
    Mixed,
}

/// ML noise insights
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct MLNoiseInsights {
    /// Noise classification
    pub noise_classification: NoiseClassification,

    /// Anomaly score
    pub anomaly_score: f64,

    /// Predicted evolution
    pub predicted_evolution: Vec<PredictedNoisePoint>,

    /// Confidence level
    pub confidence: f64,
}

/// Noise classification
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct NoiseClassification {
    /// Primary noise type
    pub primary_type: NoiseModel,

    /// Secondary contributions
    pub secondary_types: Vec<(NoiseModel, f64)>,

    /// Classification confidence
    pub confidence: f64,
}

/// Predicted noise point
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct PredictedNoisePoint {
    /// Time offset from now
    pub time_offset: f64,

    /// Predicted noise rates
    pub noise_rates: HashMap<NoiseModel, f64>,

    /// Prediction uncertainty
    pub uncertainty: f64,
}

/// Noise predictions
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct NoisePredictions {
    /// Prediction horizon
    pub horizon: f64,

    /// Predicted noise evolution
    pub evolution: Vec<PredictedNoisePoint>,

    /// Trend analysis
    pub trend: NoiseTrend,

    /// Alert thresholds
    pub alerts: Vec<NoiseAlert>,
}

/// Noise trend
#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
pub enum NoiseTrend {
    Stable,
    Increasing,
    Decreasing,
    Oscillating,
    Chaotic,
}

/// Noise alert
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct NoiseAlert {
    /// Alert type
    pub alert_type: AlertType,

    /// Affected qubits
    pub qubits: Vec<QubitId>,

    /// Severity level
    pub severity: Severity,

    /// Recommended action
    pub recommendation: String,
}

/// Alert type
#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
pub enum AlertType {
    HighNoiseRate,
    RapidDegradation,
    CorrelatedErrors,
    AnomalousPattern,
}

/// Severity level
#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
pub enum Severity {
    Low,
    Medium,
    High,
    Critical,
}

/// Comprehensive noise report
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct NoiseReport {
    /// Summary statistics
    pub summary: NoiseSummary,

    /// Model-specific analyses
    pub model_analyses: HashMap<NoiseModel, ModelAnalysis>,

    /// Temporal analysis
    pub temporal_analysis: Option<TemporalAnalysis>,

    /// Spectral analysis
    pub spectral_analysis: Option<SpectralAnalysis>,

    /// Correlation analysis
    pub correlation_analysis: Option<CorrelationAnalysis>,

    /// Recommendations
    pub recommendations: Vec<Recommendation>,

    /// Visualizations
    pub visualizations: Option<NoiseVisualizations>,
}

impl NoiseReport {
    fn new() -> Self {
        Self {
            summary: NoiseSummary::new(),
            model_analyses: HashMap::new(),
            temporal_analysis: None,
            spectral_analysis: None,
            correlation_analysis: None,
            recommendations: Vec::new(),
            visualizations: None,
        }
    }
}

/// Noise summary
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct NoiseSummary {
    /// Overall noise level
    pub overall_noise_rate: f64,

    /// Dominant noise type
    pub dominant_noise: NoiseModel,

    /// Quality factor
    pub quality_factor: f64,

    /// Comparison to baseline
    pub baseline_comparison: Option<f64>,
}

impl NoiseSummary {
    fn new() -> Self {
        Self {
            overall_noise_rate: 0.0,
            dominant_noise: NoiseModel::Depolarizing,
            quality_factor: 0.0,
            baseline_comparison: None,
        }
    }
}

/// Model-specific analysis
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ModelAnalysis {
    /// Model parameters
    pub parameters: HashMap<String, f64>,

    /// Goodness of fit
    pub goodness_of_fit: f64,

    /// Confidence intervals
    pub confidence_intervals: HashMap<String, (f64, f64)>,

    /// Model-specific insights
    pub insights: Vec<String>,
}

/// Temporal analysis
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct TemporalAnalysis {
    /// Time series data
    pub time_series: TimeSeries,

    /// Trend analysis
    pub trend: TrendAnalysis,

    /// Periodicity analysis
    pub periodicity: Option<PeriodicityAnalysis>,

    /// Drift characterization
    pub drift: DriftCharacterization,
}

/// Spectral analysis results
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct SpectralAnalysis {
    /// Dominant frequencies
    pub dominant_frequencies: Vec<(f64, f64)>, // (frequency, amplitude)

    /// Spectral features
    pub spectral_features: SpectralFeatures,

    /// Noise color classification
    pub noise_color: NoiseColor,
}

/// Correlation analysis results
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct CorrelationAnalysis {
    /// Correlation summary
    pub correlation_summary: CorrelationSummary,

    /// Significant correlations
    pub significant_correlations: Vec<SignificantCorrelation>,

    /// Correlation network
    pub correlation_network: CorrelationNetwork,
}

/// Recommendation
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Recommendation {
    /// Recommendation type
    pub rec_type: RecommendationType,

    /// Priority level
    pub priority: Priority,

    /// Description
    pub description: String,

    /// Expected improvement
    pub expected_improvement: f64,
}

/// Recommendation type
#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
pub enum RecommendationType {
    Recalibration,
    DecouplingSequence,
    ErrorMitigation,
    HardwareMaintenance,
    AlgorithmOptimization,
}

/// Priority level
#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
pub enum Priority {
    Low,
    Medium,
    High,
    Urgent,
}

/// Noise visualizations
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct NoiseVisualizations {
    /// RB decay plot
    pub rb_decay_plot: PlotData,

    /// Noise spectrum plot
    pub spectrum_plot: PlotData,

    /// Correlation heatmap
    pub correlation_heatmap: HeatmapData,

    /// Temporal evolution plot
    pub temporal_plot: PlotData,

    /// 3D noise landscape
    pub noise_landscape: Landscape3D,
}

/// Plot data
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct PlotData {
    /// X-axis data
    pub x_data: Vec<f64>,

    /// Y-axis data
    pub y_data: Vec<f64>,

    /// Error bars
    pub error_bars: Option<Vec<f64>>,

    /// Plot metadata
    pub metadata: PlotMetadata,
}

/// Heatmap data
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct HeatmapData {
    /// 2D data matrix
    pub data: Array2<f64>,

    /// Row labels
    pub row_labels: Vec<String>,

    /// Column labels
    pub col_labels: Vec<String>,

    /// Colormap
    pub colormap: String,
}

/// 3D landscape
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Landscape3D {
    /// X coordinates
    pub x: Vec<f64>,

    /// Y coordinates
    pub y: Vec<f64>,

    /// Z values (noise rates)
    pub z: Array2<f64>,

    /// Visualization parameters
    pub viz_params: Visualization3DParams,
}

/// Plot metadata
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct PlotMetadata {
    /// Title
    pub title: String,

    /// X-axis label
    pub x_label: String,

    /// Y-axis label
    pub y_label: String,

    /// Plot type
    pub plot_type: PlotType,
}

/// Plot type
#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
pub enum PlotType {
    Line,
    Scatter,
    Bar,
    Histogram,
}

/// Visualization 3D parameters
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Visualization3DParams {
    /// View angle
    pub view_angle: (f64, f64),

    /// Color scheme
    pub color_scheme: String,

    /// Surface type
    pub surface_type: SurfaceType,
}

/// Surface type
#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
pub enum SurfaceType {
    Mesh,
    Contour,
    Surface,
}

/// Helper types for internal use

/// RB sequence
struct RBSequence {
    gates: Vec<CliffordGate>,
}

impl RBSequence {
    fn new() -> Self {
        Self { gates: Vec::new() }
    }

    fn add_gate(&mut self, gate: CliffordGate) {
        self.gates.push(gate);
    }

    fn length(&self) -> usize {
        self.gates.len() - 1 // Exclude recovery gate
    }

    fn to_circuit(&self, qubits: &[QubitId]) -> QuantRS2Result<Circuit> {
        // Convert to quantum circuit
        unimplemented!()
    }
}

/// Clifford gate
#[derive(Debug, Clone)]
struct CliffordGate {
    gate_type: CliffordType,
    target_qubits: Vec<usize>,
}

/// Clifford gate types
#[derive(Debug, Clone, Copy)]
enum CliffordType {
    Identity,
    PauliX,
    PauliY,
    PauliZ,
    Hadamard,
    Phase,
    CNOT,
    CZ,
}

/// RB result
struct RBResult {
    sequence_length: usize,
    survival_probability: f64,
    error_bars: f64,
}

/// Quantum state for tomography
#[derive(Debug, Clone, Serialize, Deserialize)]
struct QuantumState {
    state_vector: Array1<Complex64>,
    preparation_circuit: Circuit,
}

/// Measurement basis
#[derive(Debug, Clone, Serialize, Deserialize)]
struct MeasurementBasis {
    basis_name: String,
    measurement_circuit: Circuit,
}

/// Time series data
#[derive(Debug, Clone, Serialize, Deserialize)]
struct TimeSeries {
    timestamps: Vec<f64>,
    values: Vec<f64>,
}

/// Noise data point
#[derive(Debug, Clone)]
struct NoiseData {
    timestamp: f64,
    noise_rates: HashMap<NoiseModel, f64>,
    correlations: Option<Array2<f64>>,
}

/// Noise history
struct NoiseHistory {
    measurements: VecDeque<(f64, NoiseData)>,
    max_size: usize,
    current_trend: Option<NoiseTrend>,
}

impl NoiseHistory {
    fn new() -> Self {
        Self {
            measurements: VecDeque::new(),
            max_size: 1000,
            current_trend: None,
        }
    }

    fn add_measurement(&mut self, timestamp: f64, data: NoiseData) {
        if self.measurements.len() >= self.max_size {
            self.measurements.pop_front();
        }
        self.measurements.push_back((timestamp, data));
    }

    fn len(&self) -> usize {
        self.measurements.len()
    }

    fn to_time_series(&self) -> TimeSeries {
        let timestamps: Vec<f64> = self.measurements.iter().map(|(t, _)| *t).collect();
        let values: Vec<f64> = self.measurements.iter()
            .map(|(_, d)| d.noise_rates.values().sum::<f64>() / d.noise_rates.len() as f64)
            .collect();

        TimeSeries { timestamps, values }
    }

    fn update_trend(&mut self, trend: NoiseTrend) {
        self.current_trend = Some(trend);
    }
}

/// Noise ML model
struct NoiseMLModel {
    // Placeholder for ML model implementation
}

impl NoiseMLModel {
    fn new() -> Self {
        Self {}
    }

    fn predict(&self, features: &NoiseFeatures) -> QuantRS2Result<NoisePrediction> {
        // Placeholder implementation
        Ok(NoisePrediction {
            classification: NoiseClassification {
                primary_type: NoiseModel::Depolarizing,
                secondary_types: vec![],
                confidence: 0.9,
            },
            anomaly_score: 0.1,
            evolution: vec![],
            confidence: 0.85,
        })
    }
}

/// Noise feature extractor
struct NoiseFeatureExtractor {
    // Feature extraction logic
}

impl NoiseFeatureExtractor {
    fn new() -> Self {
        Self {}
    }

    fn extract_features(&self, result: &NoiseCharacterizationResult) -> QuantRS2Result<NoiseFeatures> {
        // Extract relevant features for ML analysis
        Ok(NoiseFeatures {
            statistical_features: vec![],
            spectral_features: vec![],
            temporal_features: vec![],
            correlation_features: vec![],
        })
    }
}

/// Noise features for ML
struct NoiseFeatures {
    statistical_features: Vec<f64>,
    spectral_features: Vec<f64>,
    temporal_features: Vec<f64>,
    correlation_features: Vec<f64>,
}

/// Noise prediction from ML
struct NoisePrediction {
    classification: NoiseClassification,
    anomaly_score: f64,
    evolution: Vec<PredictedNoisePoint>,
    confidence: f64,
}

/// Noise predictor
struct NoisePredictor {
    // Time series prediction model
}

impl NoisePredictor {
    fn new() -> Self {
        Self {}
    }

    fn update(&mut self, result: &NoiseCharacterizationResult) -> QuantRS2Result<()> {
        // Update prediction model with new data
        Ok(())
    }

    fn predict(&self, horizon: f64) -> QuantRS2Result<NoisePredictions> {
        // Generate predictions
        Ok(NoisePredictions {
            horizon,
            evolution: vec![],
            trend: NoiseTrend::Stable,
            alerts: vec![],
        })
    }
}

/// Noise cache
struct NoiseCache {
    characterization_results: HashMap<String, NoiseCharacterizationResult>,
    analysis_results: HashMap<String, NoiseReport>,
}

impl NoiseCache {
    fn new() -> Self {
        Self {
            characterization_results: HashMap::new(),
            analysis_results: HashMap::new(),
        }
    }
}

/// Helper trait for quantum devices
trait QuantumDevice {
    fn execute(&self, circuit: Circuit, shots: usize) -> QuantRS2Result<QuantumJob>;
    fn get_topology(&self) -> &DeviceTopology;
    fn get_calibration_data(&self) -> &CalibrationData;
}

/// Quantum job result
struct QuantumJob {
    job_id: String,
    status: JobStatus,
    results: Option<JobResults>,
}

impl QuantumJob {
    fn get_counts(&self) -> QuantRS2Result<HashMap<Vec<bool>, usize>> {
        // Get measurement counts
        unimplemented!()
    }
}

/// Job status
enum JobStatus {
    Queued,
    Running,
    Completed,
    Failed(String),
}

/// Job results
struct JobResults {
    counts: HashMap<Vec<bool>, usize>,
    metadata: HashMap<String, String>,
}

/// Device topology
struct DeviceTopology {
    num_qubits: usize,
    connectivity: Vec<(usize, usize)>,
}

/// Calibration data
struct CalibrationData {
    gate_errors: HashMap<String, f64>,
    readout_errors: Vec<f64>,
    coherence_times: Vec<(f64, f64)>, // (T1, T2)
}

/// Additional result types
#[derive(Debug, Clone, Serialize, Deserialize)]
struct RBResults {
    rb_data: RBData,
    fit_params: RBFitParameters,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
struct TomographyResults {
    tomo_data: TomographyData,
    noise_params: NoiseParameters,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
struct SpectralResults {
    spectral_data: SpectralData,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
struct CorrelationResults {
    corr_data: CorrelationData,
}

/// Analysis helper types
#[derive(Debug, Clone, Serialize, Deserialize)]
struct TrendAnalysis {
    trend_type: TrendType,
    slope: f64,
    confidence: f64,
}

#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
enum TrendType {
    Linear,
    Exponential,
    Logarithmic,
    Polynomial,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
struct PeriodicityAnalysis {
    periods: Vec<f64>,
    amplitudes: Vec<f64>,
    phases: Vec<f64>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
struct DriftCharacterization {
    drift_rate: f64,
    drift_type: DriftType,
    time_constant: f64,
}

#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
enum DriftType {
    Linear,
    Exponential,
    Oscillatory,
    Random,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
struct SpectralFeatures {
    peak_frequency: f64,
    bandwidth: f64,
    spectral_entropy: f64,
}

#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
enum NoiseColor {
    White,
    Pink,
    Brown,
    Blue,
    Violet,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
struct CorrelationSummary {
    max_correlation: f64,
    mean_correlation: f64,
    correlation_radius: f64,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
struct SignificantCorrelation {
    qubit_pair: (QubitId, QubitId),
    correlation_value: f64,
    p_value: f64,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
struct CorrelationNetwork {
    nodes: Vec<QubitId>,
    edges: Vec<(usize, usize, f64)>, // (node1, node2, weight)
}

impl EnhancedNoiseCharacterizer {
    /// Analyze temporal characteristics
    fn analyze_temporal_characteristics(&self, _result: &NoiseCharacterizationResult) -> QuantRS2Result<TemporalAnalysis> {
        // Stub implementation
        Ok(TemporalAnalysis {
            time_series: TimeSeries {
                timestamps: vec![],
                values: vec![],
            },
            trend: TrendAnalysis::default(),
            periodicity: None,
        })
    }

    /// Analyze spectral characteristics
    fn analyze_spectral_characteristics(&self, _result: &NoiseCharacterizationResult) -> QuantRS2Result<SpectralAnalysis> {
        // Stub implementation
        Ok(SpectralAnalysis {
            dominant_frequencies: vec![],
            spectral_features: SpectralFeatures::default(),
            noise_color: NoiseColor::White,
        })
    }

    /// Analyze correlations
    fn analyze_correlations(&self, _result: &NoiseCharacterizationResult) -> QuantRS2Result<CorrelationAnalysis> {
        // Stub implementation
        Ok(CorrelationAnalysis {
            correlation_summary: CorrelationSummary {
                max_correlation: 0.0,
                mean_correlation: 0.0,
                correlation_radius: 0.0,
            },
            significant_correlations: vec![],
            correlation_network: CorrelationNetwork {
                nodes: vec![],
                edges: vec![],
            },
        })
    }

    /// Generate recommendations
    fn generate_recommendations(&self, _result: &NoiseCharacterizationResult) -> QuantRS2Result<Vec<Recommendation>> {
        // Stub implementation
        Ok(vec![])
    }

    /// Generate visualizations
    fn generate_visualizations(&self, _result: &NoiseCharacterizationResult) -> QuantRS2Result<NoiseVisualizations> {
        // Stub implementation
        Ok(NoiseVisualizations {
            rb_decay_plot: PlotData {
                x_data: vec![],
                y_data: vec![],
                error_bars: None,
                metadata: PlotMetadata {
                    title: "RB Decay".to_string(),
                    x_label: "Sequence Length".to_string(),
                    y_label: "Survival Probability".to_string(),
                    plot_type: PlotType::Line,
                },
            },
            spectrum_plot: PlotData {
                x_data: vec![],
                y_data: vec![],
                error_bars: None,
                metadata: PlotMetadata {
                    title: "Noise Spectrum".to_string(),
                    x_label: "Frequency".to_string(),
                    y_label: "Power Density".to_string(),
                    plot_type: PlotType::Line,
                },
            },
            correlation_heatmap: HeatmapData {
                data: Array2::zeros((0, 0)),
                row_labels: vec![],
                col_labels: vec![],
                colormap: "viridis".to_string(),
            },
            temporal_plot: PlotData {
                x_data: vec![],
                y_data: vec![],
                error_bars: None,
                metadata: PlotMetadata {
                    title: "Temporal Evolution".to_string(),
                    x_label: "Time".to_string(),
                    y_label: "Noise Level".to_string(),
                    plot_type: PlotType::Line,
                },
            },
            noise_landscape: Landscape3D {
                x: vec![],
                y: vec![],
                z: Array2::zeros((0, 0)),
                viz_params: Visualization3DParams {
                    title: "Noise Landscape".to_string(),
                    colormap: "plasma".to_string(),
                    view_angle: (30.0, 45.0),
                },
            },
        })
    }

    /// Generate random Clifford gate
    fn random_clifford_gate(&self, _num_qubits: usize) -> Circuit {
        // Stub implementation
        Circuit::new()
    }

    /// Compute recovery gate
    fn compute_recovery_gate(&self, _sequence: &[Circuit]) -> Circuit {
        // Stub implementation
        Circuit::new()
    }

    /// Calculate error bars
    fn calculate_error_bars(&self, _survival_prob: f64, _shots: usize) -> f64 {
        // Stub implementation using standard error
        0.01 // placeholder
    }

    /// Calculate fit confidence interval
    fn calculate_fit_confidence_interval(&self, _x: &[f64], _y: &[f64], _a: f64, _p: f64, _b: f64) -> QuantRS2Result<(f64, f64)> {
        // Stub implementation
        Ok((0.0, 1.0)) // placeholder
    }
}


#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_noise_characterizer_creation() {
        let config = EnhancedNoiseConfig::default();
        let characterizer = EnhancedNoiseCharacterizer::new(config);

        // Basic test to ensure creation works
        assert!(characterizer.config.enable_ml_analysis);
    }

    #[test]
    fn test_rb_sequence_generation() {
        let config = EnhancedNoiseConfig::default();
        let characterizer = EnhancedNoiseCharacterizer::new(config);

        let sequences = characterizer.generate_rb_sequences(2, 10);
        assert_eq!(sequences.len(), characterizer.config.base_config.num_sequences);

        for seq in sequences {
            assert_eq!(seq.gates.len(), 11); // 10 + recovery gate
        }
    }
}