//! Comprehensive Benchmarking Framework Example
//!
//! This example demonstrates the benchmarking framework for comparing quantum ML models
//! across different algorithms, hardware backends, and problem sizes.

use quantrs2_ml::benchmarking::algorithm_benchmarks::*;
use quantrs2_ml::benchmarking::benchmark_utils::*;
use quantrs2_ml::benchmarking::*;
use quantrs2_ml::prelude::*;
use quantrs2_ml::simulator_backends::Backend;
use std::time::Duration;

// Placeholder type for missing BenchmarkContext
#[derive(Debug, Clone)]
pub struct BenchmarkContext {
    pub config: String,
}

impl BenchmarkContext {
    pub fn new() -> Self {
        Self {
            config: "default".to_string(),
        }
    }
}

fn main() -> Result<()> {
    println!("=== Comprehensive Quantum ML Benchmarking Demo ===\n");

    // Step 1: Initialize benchmarking framework
    println!("1. Initializing benchmarking framework...");

    let config = BenchmarkConfig {
        repetitions: 3,
        warmup_runs: 1,
        max_time_per_benchmark: 60.0, // 1 minute per benchmark
        profile_memory: true,
        analyze_convergence: true,
        confidence_level: 0.95,
        ..Default::default()
    };

    let mut framework = BenchmarkFramework::new().with_config(config);

    println!("   - Framework initialized");
    println!("   - Output directory: benchmark_results/");
    println!("   - Repetitions per benchmark: 3");

    // Step 2: Register benchmarks
    println!("\n2. Registering benchmarks...");

    // VQE benchmarks for different qubit counts
    framework.register_benchmark("vqe_4q", Box::new(VQEBenchmark::new(4, 8)));
    framework.register_benchmark("vqe_6q", Box::new(VQEBenchmark::new(6, 12)));
    framework.register_benchmark("vqe_8q", Box::new(VQEBenchmark::new(8, 16)));

    // QAOA benchmarks
    framework.register_benchmark("qaoa_4q", Box::new(QAOABenchmark::new(4, 2, 8)));
    framework.register_benchmark("qaoa_6q", Box::new(QAOABenchmark::new(6, 3, 12)));

    // QNN benchmarks
    framework.register_benchmark("qnn_4q", Box::new(QNNBenchmark::new(4, 2, 100)));
    framework.register_benchmark("qnn_6q", Box::new(QNNBenchmark::new(6, 3, 100)));

    println!("   - Registered 7 benchmarks total");

    // Step 3: Create backend configurations
    println!("\n3. Setting up backends...");

    let backends = create_benchmark_backends();
    let backend_refs: Vec<&Backend> = backends.iter().collect();

    println!("   - Created {} backends", backends.len());
    for backend in &backends {
        println!("     - {}", backend.name());
    }

    // Step 4: Run all benchmarks
    println!("\n4. Running all benchmarks...");

    framework.run_all_benchmarks(&backend_refs)?;

    println!("   - All benchmarks completed");

    // Step 5: Generate and display report
    println!("\n5. Generating benchmark report...");

    let report = framework.generate_report();
    println!("\n{}", report.to_string());

    // Step 6: Print detailed results
    println!("\n6. Detailed Results Analysis:");

    // Get results again for analysis since we can't hold onto the reference
    let results = framework.run_all_benchmarks(&backend_refs)?;
    print_performance_summary(results);
    print_scaling_analysis(results);
    print_memory_analysis(results);

    println!("\n=== Comprehensive Benchmarking Demo Complete ===");

    Ok(())
}

fn print_performance_summary(results: &BenchmarkResults) {
    println!("\n   Performance Summary:");
    println!("   ===================");

    // Print summaries for each benchmark
    for (name, summary) in results.summaries() {
        println!("   {}:", name);
        println!("     - Mean time: {:.3}s", summary.mean_time.as_secs_f64());
        println!("     - Min time:  {:.3}s", summary.min_time.as_secs_f64());
        println!("     - Max time:  {:.3}s", summary.max_time.as_secs_f64());
        println!("     - Success rate: {:.1}%", summary.success_rate * 100.0);
        if let Some(memory) = summary.mean_memory {
            println!(
                "     - Memory usage: {:.1} MB",
                memory as f64 / 1024.0 / 1024.0
            );
        }
        println!();
    }
}

fn print_scaling_analysis(results: &BenchmarkResults) {
    println!("   Scaling Analysis:");
    println!("   =================");

    // Group by algorithm type
    let mut vqe_results = Vec::new();
    let mut qaoa_results = Vec::new();
    let mut qnn_results = Vec::new();

    for (name, summary) in results.summaries() {
        if name.starts_with("vqe_") {
            vqe_results.push((name, summary));
        } else if name.starts_with("qaoa_") {
            qaoa_results.push((name, summary));
        } else if name.starts_with("qnn_") {
            qnn_results.push((name, summary));
        }
    }

    // Analyze VQE scaling
    if !vqe_results.is_empty() {
        println!("   VQE Algorithm Scaling:");
        vqe_results.sort_by_key(|(name, _)| name.clone());
        for (name, summary) in vqe_results {
            let qubits = extract_qubit_count(name);
            println!(
                "     - {} qubits: {:.3}s",
                qubits,
                summary.mean_time.as_secs_f64()
            );
        }
        println!("     - Scaling trend: Exponential (as expected for VQE)");
        println!();
    }

    // Analyze QAOA scaling
    if !qaoa_results.is_empty() {
        println!("   QAOA Algorithm Scaling:");
        qaoa_results.sort_by_key(|(name, _)| name.clone());
        for (name, summary) in qaoa_results {
            let qubits = extract_qubit_count(name);
            println!(
                "     - {} qubits: {:.3}s",
                qubits,
                summary.mean_time.as_secs_f64()
            );
        }
        println!("     - Scaling trend: Polynomial (as expected for QAOA)");
        println!();
    }

    // Analyze QNN scaling
    if !qnn_results.is_empty() {
        println!("   QNN Algorithm Scaling:");
        qnn_results.sort_by_key(|(name, _)| name.clone());
        for (name, summary) in qnn_results {
            let qubits = extract_qubit_count(name);
            println!(
                "     - {} qubits: {:.3}s",
                qubits,
                summary.mean_time.as_secs_f64()
            );
        }
        println!("     - Scaling trend: Polynomial (training overhead)");
        println!();
    }
}

fn print_memory_analysis(results: &BenchmarkResults) {
    println!("   Memory Usage Analysis:");
    println!("   =====================");

    let mut memory_data = Vec::new();
    for (name, summary) in results.summaries() {
        if let Some(memory) = summary.mean_memory {
            let qubits = extract_qubit_count(name);
            memory_data.push((qubits, memory, name));
        }
    }

    if !memory_data.is_empty() {
        memory_data.sort_by_key(|(qubits, _, _)| *qubits);

        println!("   Memory scaling by qubit count:");
        for (qubits, memory, name) in memory_data {
            println!(
                "     - {} qubits ({}): {:.1} MB",
                qubits,
                name,
                memory as f64 / 1024.0 / 1024.0
            );
        }
        println!("     - Expected scaling: O(2^n) for statevector simulation");
        println!();
    }

    // Print recommendations
    println!("   Recommendations:");
    println!("     - Use statevector backend for circuits ≤ 12 qubits");
    println!("     - Use MPS backend for larger circuits with limited entanglement");
    println!("     - Consider circuit optimization for memory-constrained environments");
}

fn extract_qubit_count(benchmark_name: &str) -> usize {
    // Extract number from strings like "vqe_4q_statevector", "qaoa_6q_mps", etc.
    for part in benchmark_name.split('_') {
        if part.ends_with('q') {
            if let Ok(num) = part.trim_end_matches('q').parse::<usize>() {
                return num;
            }
        }
    }
    0 // Default if not found
}

// Additional analysis functions
fn analyze_backend_performance(results: &BenchmarkResults) {
    println!("   Backend Performance Comparison:");
    println!("   ==============================");

    // Group results by backend type
    let mut backend_performance = std::collections::HashMap::new();

    for (name, summary) in results.summaries() {
        let backend_type = extract_backend_type(name);
        backend_performance
            .entry(backend_type)
            .or_insert_with(Vec::new)
            .push(summary.mean_time.as_secs_f64());
    }

    for (backend, times) in backend_performance {
        let avg_time = times.iter().sum::<f64>() / times.len() as f64;
        println!("     - {} backend: {:.3}s average", backend, avg_time);
    }
}

fn extract_backend_type(benchmark_name: &str) -> &str {
    if benchmark_name.contains("statevector") {
        "Statevector"
    } else if benchmark_name.contains("mps") {
        "MPS"
    } else if benchmark_name.contains("gpu") {
        "GPU"
    } else {
        "Unknown"
    }
}

// Test helper functions
#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_extract_qubit_count() {
        assert_eq!(extract_qubit_count("vqe_4q_statevector"), 4);
        assert_eq!(extract_qubit_count("qaoa_6q_mps"), 6);
        assert_eq!(extract_qubit_count("qnn_8q_gpu"), 8);
        assert_eq!(extract_qubit_count("unknown_format"), 0);
    }

    #[test]
    fn test_extract_backend_type() {
        assert_eq!(extract_backend_type("vqe_4q_statevector"), "Statevector");
        assert_eq!(extract_backend_type("qaoa_6q_mps"), "MPS");
        assert_eq!(extract_backend_type("qnn_8q_gpu"), "GPU");
        assert_eq!(extract_backend_type("unknown_backend"), "Unknown");
    }
}

// Placeholder functions to satisfy compilation errors
fn create_algorithm_comparison_benchmarks() -> Result<Vec<Box<dyn Benchmark>>> {
    let mut benchmarks = Vec::new();
    Ok(benchmarks)
}

fn create_scaling_benchmarks() -> Result<Vec<Box<dyn Benchmark>>> {
    let mut benchmarks = Vec::new();
    Ok(benchmarks)
}

fn create_hardware_benchmarks() -> Result<Vec<Box<dyn Benchmark>>> {
    let mut benchmarks = Vec::new();
    Ok(benchmarks)
}

fn create_framework_benchmarks() -> Result<Vec<Box<dyn Benchmark>>> {
    let mut benchmarks = Vec::new();
    Ok(benchmarks)
}
