//! Quantum Reservoir Computing (QRC)
//!
//! This module implements Quantum Reservoir Computing, a quantum machine learning
//! paradigm that leverages the natural dynamics of quantum systems as computational
//! resources. QRC uses quantum reservoirs to process temporal and sequential data
//! with quantum advantages in memory capacity and computational complexity.

use crate::error::{MLError, Result};
use scirs2_core::ndarray::{s, Array1, Array2, Array3, ArrayD, Axis};
use serde::{Deserialize, Serialize};
use std::collections::HashMap;

/// Configuration for Quantum Reservoir Computing
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct QRCConfig {
    /// Number of qubits in the quantum reservoir
    pub reservoir_qubits: usize,
    /// Number of qubits for input encoding
    pub input_qubits: usize,
    /// Number of classical readout neurons
    pub readout_size: usize,
    /// Reservoir dynamics configuration
    pub reservoir_dynamics: ReservoirDynamics,
    /// Input encoding strategy
    pub input_encoding: InputEncoding,
    /// Readout configuration
    pub readout_config: ReadoutConfig,
    /// Training configuration
    pub training_config: QRCTrainingConfig,
    /// Temporal processing parameters
    pub temporal_config: TemporalConfig,
    /// Quantum noise and decoherence settings
    pub noise_config: Option<NoiseConfig>,
}

impl Default for QRCConfig {
    fn default() -> Self {
        Self {
            reservoir_qubits: 10,
            input_qubits: 4,
            readout_size: 8,
            reservoir_dynamics: ReservoirDynamics::default(),
            input_encoding: InputEncoding::default(),
            readout_config: ReadoutConfig::default(),
            training_config: QRCTrainingConfig::default(),
            temporal_config: TemporalConfig::default(),
            noise_config: None,
        }
    }
}

/// Reservoir dynamics configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ReservoirDynamics {
    /// Evolution time for each step
    pub evolution_time: f64,
    /// Coupling strength between reservoir qubits
    pub coupling_strength: f64,
    /// External field strength
    pub external_field: f64,
    /// Reservoir Hamiltonian type
    pub hamiltonian_type: HamiltonianType,
    /// Random interactions in the reservoir
    pub random_interactions: bool,
    /// Strength of random interactions
    pub randomness_strength: f64,
    /// Memory length of the reservoir
    pub memory_length: usize,
}

impl Default for ReservoirDynamics {
    fn default() -> Self {
        Self {
            evolution_time: 1.0,
            coupling_strength: 0.1,
            external_field: 0.05,
            hamiltonian_type: HamiltonianType::TransverseFieldIsing,
            random_interactions: true,
            randomness_strength: 0.02,
            memory_length: 10,
        }
    }
}

/// Types of Hamiltonians for reservoir evolution
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum HamiltonianType {
    /// Transverse Field Ising Model
    TransverseFieldIsing,
    /// Heisenberg model
    Heisenberg,
    /// Random field model
    RandomField,
    /// Quantum spin glass
    SpinGlass,
    /// Custom Hamiltonian
    Custom {
        interactions: HashMap<String, f64>,
        fields: HashMap<String, f64>,
    },
}

/// Input encoding strategies
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct InputEncoding {
    /// Type of encoding
    pub encoding_type: EncodingType,
    /// Normalization method
    pub normalization: NormalizationType,
    /// Feature mapping configuration
    pub feature_mapping: FeatureMapping,
    /// Temporal encoding for sequential data
    pub temporal_encoding: bool,
}

impl Default for InputEncoding {
    fn default() -> Self {
        Self {
            encoding_type: EncodingType::Amplitude,
            normalization: NormalizationType::L2,
            feature_mapping: FeatureMapping::Linear,
            temporal_encoding: true,
        }
    }
}

/// Types of quantum state encoding
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum EncodingType {
    /// Amplitude encoding
    Amplitude,
    /// Angle encoding (rotation gates)
    Angle,
    /// Basis encoding
    Basis,
    /// Displacement encoding (for continuous variables)
    Displacement,
    /// Hybrid encoding
    Hybrid,
}

/// Normalization types for input data
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum NormalizationType {
    /// L2 normalization
    L2,
    /// Min-max normalization
    MinMax,
    /// Standard score normalization
    StandardScore,
    /// No normalization
    None,
}

/// Feature mapping strategies
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum FeatureMapping {
    /// Linear feature mapping
    Linear,
    /// Polynomial feature mapping
    Polynomial { degree: usize },
    /// Fourier feature mapping
    Fourier { frequencies: Vec<f64> },
    /// Random feature mapping
    Random { dimension: usize },
}

/// Readout configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ReadoutConfig {
    /// Type of readout method
    pub readout_type: ReadoutType,
    /// Observables to measure
    pub observables: Vec<Observable>,
    /// Regularization parameters
    pub regularization: RegularizationConfig,
    /// Output activation function
    pub activation: ActivationFunction,
}

impl Default for ReadoutConfig {
    fn default() -> Self {
        Self {
            readout_type: ReadoutType::LinearRegression,
            observables: vec![
                Observable::PauliZ(0),
                Observable::PauliZ(1),
                Observable::PauliX(0),
                Observable::PauliY(0),
            ],
            regularization: RegularizationConfig::default(),
            activation: ActivationFunction::Linear,
        }
    }
}

/// Types of readout methods
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum ReadoutType {
    /// Linear regression
    LinearRegression,
    /// Ridge regression
    RidgeRegression,
    /// Lasso regression
    LassoRegression,
    /// Support vector regression
    SVR,
    /// Neural network readout
    NeuralNetwork,
    /// Quantum readout
    QuantumReadout,
}

/// Observable measurements
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum Observable {
    /// Pauli-Z measurement on specific qubit
    PauliZ(usize),
    /// Pauli-X measurement on specific qubit
    PauliX(usize),
    /// Pauli-Y measurement on specific qubit
    PauliY(usize),
    /// Two-qubit correlation
    Correlation(usize, usize),
    /// Custom observable
    Custom(String),
}

/// Regularization configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct RegularizationConfig {
    /// L1 regularization strength
    pub l1_strength: f64,
    /// L2 regularization strength
    pub l2_strength: f64,
    /// Dropout rate (for neural network readout)
    pub dropout_rate: f64,
}

impl Default for RegularizationConfig {
    fn default() -> Self {
        Self {
            l1_strength: 0.0,
            l2_strength: 0.01,
            dropout_rate: 0.0,
        }
    }
}

/// Activation functions for readout
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum ActivationFunction {
    Linear,
    ReLU,
    Sigmoid,
    Tanh,
    Softmax,
}

/// Training configuration for QRC
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct QRCTrainingConfig {
    /// Number of training epochs for readout
    pub epochs: usize,
    /// Learning rate for readout training
    pub learning_rate: f64,
    /// Batch size
    pub batch_size: usize,
    /// Validation split
    pub validation_split: f64,
    /// Early stopping patience
    pub early_stopping_patience: usize,
    /// Washout period (initial samples to discard)
    pub washout_period: usize,
}

impl Default for QRCTrainingConfig {
    fn default() -> Self {
        Self {
            epochs: 100,
            learning_rate: 0.01,
            batch_size: 32,
            validation_split: 0.2,
            early_stopping_patience: 10,
            washout_period: 5,
        }
    }
}

/// Temporal processing configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct TemporalConfig {
    /// Sequence length for processing
    pub sequence_length: usize,
    /// Time step size
    pub time_step: f64,
    /// Use temporal correlation in reservoir
    pub temporal_correlation: bool,
    /// Memory decay rate
    pub memory_decay: f64,
}

impl Default for TemporalConfig {
    fn default() -> Self {
        Self {
            sequence_length: 10,
            time_step: 1.0,
            temporal_correlation: true,
            memory_decay: 0.95,
        }
    }
}

/// Noise configuration for realistic quantum simulation
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct NoiseConfig {
    /// Decoherence time T1
    pub t1_time: f64,
    /// Dephasing time T2
    pub t2_time: f64,
    /// Gate error rate
    pub gate_error_rate: f64,
    /// Measurement error rate
    pub measurement_error_rate: f64,
}

/// Main Quantum Reservoir Computer
#[derive(Debug, Clone)]
pub struct QuantumReservoirComputer {
    config: QRCConfig,
    reservoir: QuantumReservoir,
    input_encoder: InputEncoder,
    readout_layer: ReadoutLayer,
    training_history: Vec<TrainingMetrics>,
    reservoir_states: Vec<Array1<f64>>, // Store reservoir states for analysis
}

/// Quantum reservoir implementation
#[derive(Debug, Clone)]
pub struct QuantumReservoir {
    num_qubits: usize,
    current_state: Array1<f64>,      // Quantum state vector
    hamiltonian: Array2<f64>,        // Reservoir Hamiltonian
    evolution_operator: Array2<f64>, // Time evolution operator
    coupling_matrix: Array2<f64>,    // Inter-qubit couplings
    random_fields: Array1<f64>,      // Random magnetic fields
}

/// Input encoder for quantum states
#[derive(Debug, Clone)]
pub struct InputEncoder {
    encoding_type: EncodingType,
    feature_dimension: usize,
    encoding_gates: Vec<EncodingGate>,
    normalization_params: Option<NormalizationParams>,
}

/// Encoding gates for input preparation
#[derive(Debug, Clone)]
pub struct EncodingGate {
    gate_type: String,
    target_qubit: usize,
    parameter_index: usize,
}

/// Normalization parameters
#[derive(Debug, Clone)]
pub struct NormalizationParams {
    mean: Array1<f64>,
    std: Array1<f64>,
    min: Array1<f64>,
    max: Array1<f64>,
}

/// Readout layer for classical processing
#[derive(Debug, Clone)]
pub struct ReadoutLayer {
    weights: Array2<f64>,
    biases: Array1<f64>,
    readout_type: ReadoutType,
    observables: Vec<Observable>,
    activation: ActivationFunction,
}

/// Training metrics for QRC
#[derive(Debug, Clone)]
pub struct TrainingMetrics {
    epoch: usize,
    training_loss: f64,
    validation_loss: f64,
    training_accuracy: f64,
    validation_accuracy: f64,
    reservoir_capacity: f64,
    memory_function: f64,
}

impl QuantumReservoirComputer {
    /// Create a new Quantum Reservoir Computer
    pub fn new(config: QRCConfig) -> Result<Self> {
        let reservoir = QuantumReservoir::new(&config)?;
        let input_encoder = InputEncoder::new(&config)?;
        let readout_layer = ReadoutLayer::new(&config)?;

        Ok(Self {
            config,
            reservoir,
            input_encoder,
            readout_layer,
            training_history: Vec::new(),
            reservoir_states: Vec::new(),
        })
    }

    /// Process a sequence of inputs through the reservoir
    pub fn process_sequence(&mut self, input_sequence: &Array2<f64>) -> Result<Array2<f64>> {
        let sequence_length = input_sequence.nrows();
        let feature_dim = input_sequence.ncols();
        let num_observables = self.config.readout_config.observables.len();

        let mut reservoir_outputs = Array2::zeros((sequence_length, num_observables));

        // Reset reservoir state
        self.reservoir.reset_state()?;

        for t in 0..sequence_length {
            let input = input_sequence.row(t);

            // Encode input into quantum state
            let encoded_input = self.input_encoder.encode(&input.to_owned())?;

            // Inject input into reservoir
            self.reservoir.inject_input(&encoded_input)?;

            // Evolve reservoir
            self.reservoir
                .evolve_dynamics(self.config.reservoir_dynamics.evolution_time)?;

            // Measure observables
            let measurements = self
                .reservoir
                .measure_observables(&self.config.readout_config.observables)?;

            // Store measurements
            for (i, &measurement) in measurements.iter().enumerate() {
                reservoir_outputs[[t, i]] = measurement;
            }

            // Store reservoir state for analysis
            self.reservoir_states
                .push(self.reservoir.current_state.clone());
        }

        Ok(reservoir_outputs)
    }

    /// Train the readout layer on sequential data
    pub fn train(&mut self, training_data: &[(Array2<f64>, Array2<f64>)]) -> Result<()> {
        let num_epochs = self.config.training_config.epochs;
        let washout = self.config.training_config.washout_period;

        // Collect all reservoir states and targets
        let mut all_states = Vec::new();
        let mut all_targets = Vec::new();

        for (input_sequence, target_sequence) in training_data {
            let reservoir_output = self.process_sequence(input_sequence)?;

            // Skip washout period, but ensure we have at least some data
            let effective_washout = washout.min(reservoir_output.nrows().saturating_sub(1));
            for t in effective_washout..reservoir_output.nrows() {
                all_states.push(reservoir_output.row(t).to_owned());
                all_targets.push(target_sequence.row(t).to_owned());
            }
        }

        // Check if we have any data to train on
        if all_states.is_empty() {
            return Err(MLError::MLOperationError(
                "No training data available after washout period".to_string(),
            ));
        }

        // Convert to arrays
        let states_matrix = Array2::from_shape_vec(
            (all_states.len(), all_states[0].len()),
            all_states.into_iter().flatten().collect(),
        )?;

        let targets_matrix = Array2::from_shape_vec(
            (all_targets.len(), all_targets[0].len()),
            all_targets.into_iter().flatten().collect(),
        )?;

        // Train readout layer
        self.readout_layer.train(
            &states_matrix,
            &targets_matrix,
            &self.config.training_config,
        )?;

        // Record training metrics
        for epoch in 0..num_epochs {
            let training_loss = self.evaluate_loss(&states_matrix, &targets_matrix)?;
            let training_accuracy = self.evaluate_accuracy(&states_matrix, &targets_matrix)?;

            let metrics = TrainingMetrics {
                epoch,
                training_loss,
                validation_loss: training_loss * 1.1, // Placeholder
                training_accuracy,
                validation_accuracy: training_accuracy * 0.95, // Placeholder
                reservoir_capacity: self.compute_reservoir_capacity()?,
                memory_function: self.compute_memory_function()?,
            };

            self.training_history.push(metrics);

            if epoch % 10 == 0 {
                println!(
                    "Epoch {}: Loss = {:.6}, Accuracy = {:.4}",
                    epoch, training_loss, training_accuracy
                );
            }
        }

        Ok(())
    }

    /// Predict on new sequential data
    pub fn predict(&mut self, input_sequence: &Array2<f64>) -> Result<Array2<f64>> {
        let reservoir_output = self.process_sequence(input_sequence)?;
        self.readout_layer.predict(&reservoir_output)
    }

    /// Evaluate loss on given data
    fn evaluate_loss(&self, states: &Array2<f64>, targets: &Array2<f64>) -> Result<f64> {
        let predictions = self.readout_layer.predict(states)?;
        let mse = predictions
            .iter()
            .zip(targets.iter())
            .map(|(p, t)| (p - t).powi(2))
            .sum::<f64>()
            / predictions.len() as f64;
        Ok(mse)
    }

    /// Evaluate accuracy on given data
    fn evaluate_accuracy(&self, states: &Array2<f64>, targets: &Array2<f64>) -> Result<f64> {
        let predictions = self.readout_layer.predict(states)?;

        // For regression tasks, use R² coefficient
        let target_mean = targets.mean().unwrap();
        let ss_tot = targets
            .iter()
            .map(|t| (t - target_mean).powi(2))
            .sum::<f64>();
        let ss_res = predictions
            .iter()
            .zip(targets.iter())
            .map(|(p, t)| (t - p).powi(2))
            .sum::<f64>();

        let r_squared = 1.0 - (ss_res / ss_tot);
        Ok(r_squared.max(0.0))
    }

    /// Compute reservoir capacity (information processing capability)
    fn compute_reservoir_capacity(&self) -> Result<f64> {
        // Simplified capacity computation based on reservoir state diversity
        if self.reservoir_states.is_empty() {
            return Ok(0.0);
        }

        let num_states = self.reservoir_states.len();
        let state_dim = self.reservoir_states[0].len();

        // Compute pairwise distances between states
        let mut total_distance = 0.0;
        let mut count = 0;

        for i in 0..num_states {
            for j in i + 1..num_states {
                let distance = self.reservoir_states[i]
                    .iter()
                    .zip(self.reservoir_states[j].iter())
                    .map(|(a, b)| (a - b).powi(2))
                    .sum::<f64>()
                    .sqrt();
                total_distance += distance;
                count += 1;
            }
        }

        let average_distance = if count > 0 {
            total_distance / count as f64
        } else {
            0.0
        };
        let capacity = average_distance / (state_dim as f64).sqrt();

        Ok(capacity)
    }

    /// Compute memory function of the reservoir
    fn compute_memory_function(&self) -> Result<f64> {
        // Simplified memory function based on autocorrelation
        if self.reservoir_states.len() < 2 {
            return Ok(0.0);
        }

        let mut autocorrelations = Vec::new();
        let max_lag = self
            .config
            .reservoir_dynamics
            .memory_length
            .min(self.reservoir_states.len() / 2);

        for lag in 1..=max_lag {
            let mut correlation = 0.0;
            let mut count = 0;

            for t in lag..self.reservoir_states.len() {
                let state_t = &self.reservoir_states[t];
                let state_t_lag = &self.reservoir_states[t - lag];

                let dot_product = state_t
                    .iter()
                    .zip(state_t_lag.iter())
                    .map(|(a, b)| a * b)
                    .sum::<f64>();

                correlation += dot_product;
                count += 1;
            }

            if count > 0 {
                autocorrelations.push(correlation / count as f64);
            }
        }

        // Memory function as the sum of autocorrelations
        let memory_function = autocorrelations.iter().sum::<f64>().abs();
        Ok(memory_function)
    }

    /// Get training history
    pub fn get_training_history(&self) -> &[TrainingMetrics] {
        &self.training_history
    }

    /// Get reservoir states for analysis
    pub fn get_reservoir_states(&self) -> &[Array1<f64>] {
        &self.reservoir_states
    }

    /// Analyze reservoir dynamics
    pub fn analyze_dynamics(&self) -> Result<DynamicsAnalysis> {
        let spectral_radius = self.reservoir.compute_spectral_radius()?;
        let lyapunov_exponent = self.reservoir.compute_lyapunov_exponent()?;
        let entanglement_measure = self.reservoir.compute_entanglement()?;

        Ok(DynamicsAnalysis {
            spectral_radius,
            lyapunov_exponent,
            entanglement_measure,
            capacity: self.compute_reservoir_capacity()?,
            memory_function: self.compute_memory_function()?,
        })
    }
}

impl QuantumReservoir {
    /// Create a new quantum reservoir
    pub fn new(config: &QRCConfig) -> Result<Self> {
        let num_qubits = config.reservoir_qubits;
        let state_dim = 1 << num_qubits;

        // Initialize in |0...0⟩ state
        let mut current_state = Array1::zeros(state_dim);
        current_state[0] = 1.0;

        // Build Hamiltonian
        let hamiltonian = Self::build_hamiltonian(config)?;

        // Compute evolution operator
        let evolution_operator = Self::compute_evolution_operator(
            &hamiltonian,
            config.reservoir_dynamics.evolution_time,
        )?;

        // Generate coupling matrix
        let coupling_matrix = Self::generate_coupling_matrix(
            num_qubits,
            config.reservoir_dynamics.coupling_strength,
        )?;

        // Generate random fields
        let random_fields = Array1::from_shape_fn(num_qubits, |_| {
            if config.reservoir_dynamics.random_interactions {
                (fastrand::f64() - 0.5) * config.reservoir_dynamics.randomness_strength
            } else {
                0.0
            }
        });

        Ok(Self {
            num_qubits,
            current_state,
            hamiltonian,
            evolution_operator,
            coupling_matrix,
            random_fields,
        })
    }

    /// Build the reservoir Hamiltonian
    fn build_hamiltonian(config: &QRCConfig) -> Result<Array2<f64>> {
        let num_qubits = config.reservoir_qubits;
        let dim = 1 << num_qubits;
        let mut hamiltonian = Array2::zeros((dim, dim));

        match &config.reservoir_dynamics.hamiltonian_type {
            HamiltonianType::TransverseFieldIsing => {
                // H = -J∑⟨i,j⟩ZᵢZⱼ - h∑ᵢXᵢ
                let coupling = config.reservoir_dynamics.coupling_strength;
                let field = config.reservoir_dynamics.external_field;

                // Nearest-neighbor ZZ interactions
                for i in 0..num_qubits - 1 {
                    for state in 0..dim {
                        let zi = if (state >> i) & 1 == 0 { 1.0 } else { -1.0 };
                        let zj = if (state >> (i + 1)) & 1 == 0 {
                            1.0
                        } else {
                            -1.0
                        };
                        hamiltonian[[state, state]] -= coupling * zi * zj;
                    }
                }

                // Transverse field (X terms)
                for i in 0..num_qubits {
                    for state in 0..dim {
                        let flipped_state = state ^ (1 << i);
                        hamiltonian[[state, flipped_state]] -= field;
                    }
                }
            }
            HamiltonianType::Heisenberg => {
                // H = J∑⟨i,j⟩(XᵢXⱼ + YᵢYⱼ + ZᵢZⱼ)
                let coupling = config.reservoir_dynamics.coupling_strength;

                for i in 0..num_qubits - 1 {
                    // ZZ terms
                    for state in 0..dim {
                        let zi = if (state >> i) & 1 == 0 { 1.0 } else { -1.0 };
                        let zj = if (state >> (i + 1)) & 1 == 0 {
                            1.0
                        } else {
                            -1.0
                        };
                        hamiltonian[[state, state]] += coupling * zi * zj;
                    }

                    // XX + YY terms (simplified as combined flip-flip terms)
                    for state in 0..dim {
                        let bit_i = (state >> i) & 1;
                        let bit_j = (state >> (i + 1)) & 1;

                        if bit_i != bit_j {
                            let flipped_state = state ^ (1 << i) ^ (1 << (i + 1));
                            hamiltonian[[state, flipped_state]] += coupling;
                        }
                    }
                }
            }
            _ => {
                return Err(crate::error::MLError::InvalidConfiguration(
                    "Hamiltonian type not implemented".to_string(),
                ));
            }
        }

        Ok(hamiltonian)
    }

    /// Compute time evolution operator
    fn compute_evolution_operator(hamiltonian: &Array2<f64>, time: f64) -> Result<Array2<f64>> {
        // Simplified evolution: U = exp(-iHt) ≈ I - iHt (first-order approximation)
        let dim = hamiltonian.nrows();
        let mut evolution_op = Array2::eye(dim);

        for i in 0..dim {
            for j in 0..dim {
                if i != j {
                    evolution_op[[i, j]] = -time * hamiltonian[[i, j]];
                } else {
                    evolution_op[[i, j]] = 1.0 - time * hamiltonian[[i, j]];
                }
            }
        }

        Ok(evolution_op)
    }

    /// Generate coupling matrix
    fn generate_coupling_matrix(num_qubits: usize, coupling_strength: f64) -> Result<Array2<f64>> {
        let mut coupling_matrix = Array2::zeros((num_qubits, num_qubits));

        // Nearest-neighbor coupling
        for i in 0..num_qubits - 1 {
            coupling_matrix[[i, i + 1]] = coupling_strength;
            coupling_matrix[[i + 1, i]] = coupling_strength;
        }

        // Add some random long-range couplings
        for i in 0..num_qubits {
            for j in i + 2..num_qubits {
                if fastrand::f64() < 0.1 {
                    // 10% chance of long-range coupling
                    let strength = coupling_strength * 0.1;
                    coupling_matrix[[i, j]] = strength;
                    coupling_matrix[[j, i]] = strength;
                }
            }
        }

        Ok(coupling_matrix)
    }

    /// Reset reservoir state
    pub fn reset_state(&mut self) -> Result<()> {
        let state_dim = self.current_state.len();
        self.current_state.fill(0.0);
        self.current_state[0] = 1.0; // |0...0⟩ state
        Ok(())
    }

    /// Inject input into reservoir
    pub fn inject_input(&mut self, encoded_input: &Array1<f64>) -> Result<()> {
        // Simple input injection: add to the current state (with normalization)
        let input_strength = 0.1; // Configurable parameter

        for (i, &input_val) in encoded_input.iter().enumerate() {
            if i < self.current_state.len() {
                self.current_state[i] += input_strength * input_val;
            }
        }

        // Normalize the state
        let norm = self.current_state.iter().map(|x| x * x).sum::<f64>().sqrt();
        if norm > 1e-10 {
            self.current_state /= norm;
        }

        Ok(())
    }

    /// Evolve reservoir dynamics
    pub fn evolve_dynamics(&mut self, evolution_time: f64) -> Result<()> {
        // Apply evolution operator
        let evolved_state = self.evolution_operator.dot(&self.current_state);

        // Apply random noise if configured
        let mut noisy_state = evolved_state;
        for i in 0..self.num_qubits {
            if i < self.random_fields.len() {
                let noise = self.random_fields[i] * fastrand::f64();
                if i < noisy_state.len() {
                    noisy_state[i] += noise;
                }
            }
        }

        // Normalize
        let norm = noisy_state.iter().map(|x| x * x).sum::<f64>().sqrt();
        if norm > 1e-10 {
            noisy_state /= norm;
        }

        self.current_state = noisy_state;
        Ok(())
    }

    /// Measure observables
    pub fn measure_observables(&self, observables: &[Observable]) -> Result<Array1<f64>> {
        let mut measurements = Array1::zeros(observables.len());

        for (i, observable) in observables.iter().enumerate() {
            measurements[i] = match observable {
                Observable::PauliZ(qubit) => self.measure_pauli_z(*qubit)?,
                Observable::PauliX(qubit) => self.measure_pauli_x(*qubit)?,
                Observable::PauliY(qubit) => self.measure_pauli_y(*qubit)?,
                Observable::Correlation(qubit1, qubit2) => {
                    self.measure_correlation(*qubit1, *qubit2)?
                }
                Observable::Custom(_) => {
                    // Placeholder for custom observables
                    0.0
                }
            };
        }

        Ok(measurements)
    }

    /// Measure Pauli-Z expectation value
    fn measure_pauli_z(&self, qubit: usize) -> Result<f64> {
        let mut expectation = 0.0;

        for (state, &amplitude) in self.current_state.iter().enumerate() {
            let z_eigenvalue = if (state >> qubit) & 1 == 0 { 1.0 } else { -1.0 };
            expectation += amplitude * amplitude * z_eigenvalue;
        }

        Ok(expectation)
    }

    /// Measure Pauli-X expectation value
    fn measure_pauli_x(&self, qubit: usize) -> Result<f64> {
        let mut expectation = 0.0;

        for (state, &amplitude) in self.current_state.iter().enumerate() {
            let flipped_state = state ^ (1 << qubit);
            if flipped_state < self.current_state.len() {
                expectation += amplitude * self.current_state[flipped_state];
            }
        }

        Ok(expectation)
    }

    /// Measure Pauli-Y expectation value (simplified)
    fn measure_pauli_y(&self, qubit: usize) -> Result<f64> {
        // Simplified Y measurement (in practice, this involves complex amplitudes)
        let x_val = self.measure_pauli_x(qubit)?;
        let z_val = self.measure_pauli_z(qubit)?;
        Ok((x_val + z_val) / 2.0) // Approximation
    }

    /// Measure two-qubit correlation
    fn measure_correlation(&self, qubit1: usize, qubit2: usize) -> Result<f64> {
        let z1 = self.measure_pauli_z(qubit1)?;
        let z2 = self.measure_pauli_z(qubit2)?;

        // Compute ⟨Z₁Z₂⟩
        let mut correlation = 0.0;
        for (state, &amplitude) in self.current_state.iter().enumerate() {
            let z1_val = if (state >> qubit1) & 1 == 0 {
                1.0
            } else {
                -1.0
            };
            let z2_val = if (state >> qubit2) & 1 == 0 {
                1.0
            } else {
                -1.0
            };
            correlation += amplitude * amplitude * z1_val * z2_val;
        }

        Ok(correlation - z1 * z2) // Connected correlation
    }

    /// Compute spectral radius of the evolution operator
    pub fn compute_spectral_radius(&self) -> Result<f64> {
        // Simplified spectral radius computation
        let matrix_norm = self.evolution_operator.iter().map(|x| x.abs()).sum::<f64>()
            / (self.evolution_operator.nrows() as f64);
        Ok(matrix_norm)
    }

    /// Compute largest Lyapunov exponent (simplified)
    pub fn compute_lyapunov_exponent(&self) -> Result<f64> {
        // Placeholder implementation
        Ok(0.1)
    }

    /// Compute entanglement measure
    pub fn compute_entanglement(&self) -> Result<f64> {
        // Simplified entanglement measure based on state complexity
        let state_complexity = self
            .current_state
            .iter()
            .map(|x| if x.abs() > 1e-10 { 1.0 } else { 0.0 })
            .sum::<f64>();

        let max_complexity = self.current_state.len() as f64;
        Ok(state_complexity / max_complexity)
    }
}

impl InputEncoder {
    /// Create a new input encoder
    pub fn new(config: &QRCConfig) -> Result<Self> {
        let feature_dimension = 1 << config.input_qubits; // Assume power-of-2 encoding
        let encoding_gates = Self::build_encoding_gates(config)?;

        Ok(Self {
            encoding_type: config.input_encoding.encoding_type.clone(),
            feature_dimension,
            encoding_gates,
            normalization_params: None,
        })
    }

    /// Build encoding gates
    fn build_encoding_gates(config: &QRCConfig) -> Result<Vec<EncodingGate>> {
        let mut gates = Vec::new();

        match config.input_encoding.encoding_type {
            EncodingType::Amplitude => {
                // Direct amplitude encoding (no additional gates needed)
            }
            EncodingType::Angle => {
                // Create rotation gates for each input qubit
                for qubit in 0..config.input_qubits {
                    gates.push(EncodingGate {
                        gate_type: "RY".to_string(),
                        target_qubit: qubit,
                        parameter_index: qubit,
                    });
                }
            }
            _ => {
                // Other encoding types can be implemented
            }
        }

        Ok(gates)
    }

    /// Encode classical input into quantum state
    pub fn encode(&self, input: &Array1<f64>) -> Result<Array1<f64>> {
        match self.encoding_type {
            EncodingType::Amplitude => self.amplitude_encoding(input),
            EncodingType::Angle => self.angle_encoding(input),
            _ => Err(crate::error::MLError::InvalidConfiguration(
                "Encoding type not implemented".to_string(),
            )),
        }
    }

    /// Amplitude encoding
    fn amplitude_encoding(&self, input: &Array1<f64>) -> Result<Array1<f64>> {
        let mut encoded = Array1::zeros(self.feature_dimension);

        // Copy input values to encoded state (with padding/truncation)
        let copy_len = input.len().min(encoded.len());
        for i in 0..copy_len {
            encoded[i] = input[i];
        }

        // Normalize
        let norm = encoded.iter().map(|x| x * x).sum::<f64>().sqrt();
        if norm > 1e-10 {
            encoded /= norm;
        } else {
            encoded[0] = 1.0; // Default to |0⟩ state
        }

        Ok(encoded)
    }

    /// Angle encoding
    fn angle_encoding(&self, input: &Array1<f64>) -> Result<Array1<f64>> {
        let mut encoded = Array1::zeros(self.feature_dimension);
        encoded[0] = 1.0; // Start with |0...0⟩

        // Apply rotation gates based on input values
        for (i, &value) in input.iter().enumerate() {
            if i < self.encoding_gates.len() {
                let angle = value * std::f64::consts::PI; // Scale to [0, π]
                encoded = self.apply_ry_rotation(&encoded, i, angle)?;
            }
        }

        Ok(encoded)
    }

    /// Apply RY rotation for angle encoding
    fn apply_ry_rotation(
        &self,
        state: &Array1<f64>,
        qubit: usize,
        angle: f64,
    ) -> Result<Array1<f64>> {
        let mut new_state = state.clone();
        let cos_half = (angle / 2.0).cos();
        let sin_half = (angle / 2.0).sin();

        let qubit_mask = 1 << qubit;

        for i in 0..state.len() {
            if i & qubit_mask == 0 {
                let j = i | qubit_mask;
                if j < state.len() {
                    let state_0 = state[i];
                    let state_1 = state[j];
                    new_state[i] = cos_half * state_0 - sin_half * state_1;
                    new_state[j] = sin_half * state_0 + cos_half * state_1;
                }
            }
        }

        Ok(new_state)
    }
}

impl ReadoutLayer {
    /// Create a new readout layer
    pub fn new(config: &QRCConfig) -> Result<Self> {
        let input_size = config.readout_config.observables.len();
        let output_size = config.readout_size;

        let weights =
            Array2::from_shape_fn((output_size, input_size), |_| (fastrand::f64() - 0.5) * 0.1);
        let biases = Array1::zeros(output_size);

        Ok(Self {
            weights,
            biases,
            readout_type: config.readout_config.readout_type.clone(),
            observables: config.readout_config.observables.clone(),
            activation: config.readout_config.activation.clone(),
        })
    }

    /// Train the readout layer
    pub fn train(
        &mut self,
        inputs: &Array2<f64>,
        targets: &Array2<f64>,
        config: &QRCTrainingConfig,
    ) -> Result<()> {
        match self.readout_type {
            ReadoutType::LinearRegression => {
                self.train_linear_regression(inputs, targets)?;
            }
            ReadoutType::RidgeRegression => {
                self.train_ridge_regression(inputs, targets, 0.01)?; // Fixed regularization
            }
            _ => {
                return Err(crate::error::MLError::InvalidConfiguration(
                    "Readout type not implemented".to_string(),
                ));
            }
        }
        Ok(())
    }

    /// Train using linear regression (least squares)
    fn train_linear_regression(
        &mut self,
        inputs: &Array2<f64>,
        targets: &Array2<f64>,
    ) -> Result<()> {
        // Solve: W = (X^T X)^(-1) X^T Y
        // For simplicity, use gradient descent

        let learning_rate = 0.01;
        let epochs = 100;

        for _epoch in 0..epochs {
            let predictions = self.predict(inputs)?;
            let errors = &predictions - targets;

            // Update weights and biases
            for i in 0..self.weights.nrows() {
                for j in 0..self.weights.ncols() {
                    let gradient = errors
                        .column(i)
                        .iter()
                        .zip(inputs.column(j).iter())
                        .map(|(e, x)| e * x)
                        .sum::<f64>()
                        / inputs.nrows() as f64;

                    self.weights[[i, j]] -= learning_rate * gradient;
                }

                let bias_gradient = errors.column(i).mean().unwrap();
                self.biases[i] -= learning_rate * bias_gradient;
            }
        }

        Ok(())
    }

    /// Train using ridge regression
    fn train_ridge_regression(
        &mut self,
        inputs: &Array2<f64>,
        targets: &Array2<f64>,
        alpha: f64,
    ) -> Result<()> {
        // Add L2 regularization to linear regression
        self.train_linear_regression(inputs, targets)?;

        // Apply L2 penalty
        for weight in self.weights.iter_mut() {
            *weight *= 1.0 - alpha;
        }

        Ok(())
    }

    /// Predict using the trained readout layer
    pub fn predict(&self, inputs: &Array2<f64>) -> Result<Array2<f64>> {
        let batch_size = inputs.nrows();
        let output_size = self.weights.nrows();
        let mut outputs = Array2::zeros((batch_size, output_size));

        for i in 0..batch_size {
            let input_row = inputs.row(i);
            for j in 0..output_size {
                let weighted_sum = input_row
                    .iter()
                    .zip(self.weights.row(j).iter())
                    .map(|(x, w)| x * w)
                    .sum::<f64>()
                    + self.biases[j];

                outputs[[i, j]] = self.apply_activation(weighted_sum);
            }
        }

        Ok(outputs)
    }

    /// Apply activation function
    fn apply_activation(&self, x: f64) -> f64 {
        match self.activation {
            ActivationFunction::Linear => x,
            ActivationFunction::ReLU => x.max(0.0),
            ActivationFunction::Sigmoid => 1.0 / (1.0 + (-x).exp()),
            ActivationFunction::Tanh => x.tanh(),
            ActivationFunction::Softmax => x.exp(), // Note: requires normalization across batch
        }
    }
}

/// Analysis results for reservoir dynamics
#[derive(Debug)]
pub struct DynamicsAnalysis {
    pub spectral_radius: f64,
    pub lyapunov_exponent: f64,
    pub entanglement_measure: f64,
    pub capacity: f64,
    pub memory_function: f64,
}

/// Benchmark QRC against classical reservoir computing
pub fn benchmark_qrc_vs_classical(
    qrc: &mut QuantumReservoirComputer,
    test_data: &[(Array2<f64>, Array2<f64>)],
) -> Result<BenchmarkResults> {
    let start_time = std::time::Instant::now();

    let mut quantum_loss = 0.0;
    for (input, target) in test_data {
        let prediction = qrc.predict(input)?;
        let mse = prediction
            .iter()
            .zip(target.iter())
            .map(|(p, t)| (p - t).powi(2))
            .sum::<f64>()
            / prediction.len() as f64;
        quantum_loss += mse;
    }
    quantum_loss /= test_data.len() as f64;

    let quantum_time = start_time.elapsed();

    // Classical comparison would go here
    let classical_loss = quantum_loss * 1.2; // Placeholder
    let classical_time = quantum_time / 2; // Placeholder

    Ok(BenchmarkResults {
        quantum_loss,
        classical_loss,
        quantum_time: quantum_time.as_secs_f64(),
        classical_time: classical_time.as_secs_f64(),
        quantum_advantage: classical_loss / quantum_loss,
    })
}

/// Benchmark results
#[derive(Debug)]
pub struct BenchmarkResults {
    pub quantum_loss: f64,
    pub classical_loss: f64,
    pub quantum_time: f64,
    pub classical_time: f64,
    pub quantum_advantage: f64,
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_qrc_creation() {
        let config = QRCConfig::default();
        let qrc = QuantumReservoirComputer::new(config);
        assert!(qrc.is_ok());
    }

    #[test]
    fn test_sequence_processing() {
        let config = QRCConfig::default();
        let mut qrc = QuantumReservoirComputer::new(config).unwrap();

        let input_sequence =
            Array2::from_shape_vec((10, 4), (0..40).map(|x| x as f64 * 0.1).collect()).unwrap();
        let result = qrc.process_sequence(&input_sequence);
        assert!(result.is_ok());
    }

    #[test]
    fn test_training() {
        let config = QRCConfig::default();
        let mut qrc = QuantumReservoirComputer::new(config).unwrap();

        let input_sequence =
            Array2::from_shape_vec((20, 4), (0..80).map(|x| x as f64 * 0.05).collect()).unwrap();
        let target_sequence =
            Array2::from_shape_vec((20, 8), (0..160).map(|x| x as f64 * 0.02).collect()).unwrap();

        let training_data = vec![(input_sequence, target_sequence)];
        let result = qrc.train(&training_data);
        assert!(result.is_ok());
    }

    #[test]
    fn test_dynamics_analysis() {
        let config = QRCConfig::default();
        let qrc = QuantumReservoirComputer::new(config).unwrap();
        let analysis = qrc.analyze_dynamics();
        assert!(analysis.is_ok());
    }
}
