//! Quantum Transfer Learning
//!
//! This module implements transfer learning techniques for quantum machine learning models.
//! It provides methods to leverage pre-trained quantum circuits for new tasks, enabling
//! efficient learning with limited data and quantum resources.

use crate::autodiff::optimizers::Optimizer;
use scirs2_core::random::prelude::*;
use crate::error::{MLError, Result};
use crate::optimization::OptimizationMethod;
use crate::qnn::{QNNLayerType, QuantumNeuralNetwork, TrainingResult};
use scirs2_core::ndarray::{Array1, Array2};
use quantrs2_circuit::builder::{Circuit, Simulator};
use quantrs2_core::gate::{
    single::{RotationX, RotationY, RotationZ},
    GateOp,
};
use quantrs2_sim::statevector::StateVectorSimulator;
use std::collections::HashMap;

/// Transfer learning strategies
#[derive(Debug, Clone, Copy, PartialEq)]
pub enum TransferStrategy {
    /// Freeze all layers except the last few
    FineTuning { num_trainable_layers: usize },

    /// Use pre-trained model as feature extractor
    FeatureExtraction,

    /// Adapt specific layers while keeping others frozen
    SelectiveAdaptation,

    /// Progressive unfreezing of layers during training
    ProgressiveUnfreezing { unfreeze_rate: usize },
}

/// Layer freezing configuration
#[derive(Debug, Clone)]
pub struct LayerConfig {
    /// Whether the layer is frozen (non-trainable)
    pub frozen: bool,

    /// Learning rate multiplier for this layer
    pub learning_rate_multiplier: f64,

    /// Parameter indices for this layer
    pub parameter_indices: Vec<usize>,
}

/// Pre-trained quantum model for transfer learning
#[derive(Debug, Clone)]
pub struct PretrainedModel {
    /// The underlying quantum neural network
    pub qnn: QuantumNeuralNetwork,

    /// Training task description
    pub task_description: String,

    /// Performance metrics on original task
    pub performance_metrics: HashMap<String, f64>,

    /// Model metadata
    pub metadata: HashMap<String, String>,
}

/// Quantum transfer learning framework
pub struct QuantumTransferLearning {
    /// The pre-trained source model
    source_model: PretrainedModel,

    /// Target model being trained
    target_model: QuantumNeuralNetwork,

    /// Transfer strategy being used
    strategy: TransferStrategy,

    /// Layer-wise configuration
    layer_configs: Vec<LayerConfig>,

    /// Current training epoch
    current_epoch: usize,
}

impl QuantumTransferLearning {
    /// Create a new transfer learning instance
    pub fn new(
        source_model: PretrainedModel,
        target_layers: Vec<QNNLayerType>,
        strategy: TransferStrategy,
    ) -> Result<Self> {
        // Note: In a real implementation, we would validate qubit compatibility
        // For this example, we'll simply use the source model's qubit count

        // Create target model by combining source and new layers
        let mut all_layers = source_model.qnn.layers.clone();

        // Add new layers based on strategy
        match strategy {
            TransferStrategy::FineTuning { .. } => {
                // Keep existing layers and potentially add new output layers
                all_layers.extend(target_layers);
            }
            TransferStrategy::FeatureExtraction => {
                // Remove output layers and add new ones
                if all_layers.len() > 2 {
                    all_layers.truncate(all_layers.len() - 2);
                }
                all_layers.extend(target_layers);
            }
            _ => {
                // For other strategies, combine appropriately
                all_layers.extend(target_layers);
            }
        }

        // Initialize target model
        let target_model = QuantumNeuralNetwork::new(
            all_layers,
            source_model.qnn.num_qubits,
            source_model.qnn.input_dim,
            source_model.qnn.output_dim,
        )?;

        // Configure layers based on strategy
        let layer_configs = Self::configure_layers(&target_model, &strategy);

        Ok(Self {
            source_model,
            target_model,
            strategy,
            layer_configs,
            current_epoch: 0,
        })
    }

    /// Configure layer freezing based on transfer strategy
    fn configure_layers(
        model: &QuantumNeuralNetwork,
        strategy: &TransferStrategy,
    ) -> Vec<LayerConfig> {
        let mut configs = Vec::new();
        let num_layers = model.layers.len();

        match strategy {
            TransferStrategy::FineTuning {
                num_trainable_layers,
            } => {
                // Freeze all layers except the last few
                for i in 0..num_layers {
                    configs.push(LayerConfig {
                        frozen: i < num_layers - num_trainable_layers,
                        learning_rate_multiplier: if i < num_layers - num_trainable_layers {
                            0.0
                        } else {
                            1.0
                        },
                        parameter_indices: Self::get_layer_parameters(model, i),
                    });
                }
            }
            TransferStrategy::FeatureExtraction => {
                // Freeze all pre-trained layers
                for i in 0..num_layers {
                    let is_new_layer = i >= num_layers - 2; // Assume last 2 layers are new
                    configs.push(LayerConfig {
                        frozen: !is_new_layer,
                        learning_rate_multiplier: if is_new_layer { 1.0 } else { 0.0 },
                        parameter_indices: Self::get_layer_parameters(model, i),
                    });
                }
            }
            TransferStrategy::SelectiveAdaptation => {
                // Freeze specific layers (e.g., encoding layers)
                for (i, layer) in model.layers.iter().enumerate() {
                    let frozen = matches!(layer, QNNLayerType::EncodingLayer { .. });
                    configs.push(LayerConfig {
                        frozen,
                        learning_rate_multiplier: if frozen { 0.0 } else { 0.5 },
                        parameter_indices: Self::get_layer_parameters(model, i),
                    });
                }
            }
            TransferStrategy::ProgressiveUnfreezing { .. } => {
                // Initially freeze all but last layer
                for i in 0..num_layers {
                    configs.push(LayerConfig {
                        frozen: i < num_layers - 1,
                        learning_rate_multiplier: if i == num_layers - 1 { 1.0 } else { 0.0 },
                        parameter_indices: Self::get_layer_parameters(model, i),
                    });
                }
            }
        }

        configs
    }

    /// Get parameter indices for a specific layer
    fn get_layer_parameters(model: &QuantumNeuralNetwork, layer_idx: usize) -> Vec<usize> {
        // This is a simplified implementation
        // In practice, would need to track actual parameter mapping
        let params_per_layer = model.parameters.len() / model.layers.len();
        let start = layer_idx * params_per_layer;
        let end = start + params_per_layer;
        (start..end).collect()
    }

    /// Train the target model on new data
    pub fn train(
        &mut self,
        training_data: &Array2<f64>,
        labels: &Array1<f64>,
        optimizer: &mut dyn Optimizer,
        epochs: usize,
        batch_size: usize,
    ) -> Result<TrainingResult> {
        let mut loss_history = Vec::new();
        let mut best_loss = f64::INFINITY;
        let mut best_params = self.target_model.parameters.clone();

        // Convert parameters to HashMap for optimizer
        let mut params_map = HashMap::new();
        for (i, value) in self.target_model.parameters.iter().enumerate() {
            params_map.insert(format!("param_{}", i), *value);
        }

        for epoch in 0..epochs {
            self.current_epoch = epoch;

            // Update layer configurations for progressive unfreezing
            if let TransferStrategy::ProgressiveUnfreezing { unfreeze_rate } = self.strategy {
                if epoch > 0 && epoch % unfreeze_rate == 0 {
                    self.unfreeze_next_layer();
                }
            }

            // Compute gradients with frozen layer handling
            let gradients = self.compute_gradients(training_data, labels)?;

            // Apply layer-specific learning rates
            let scaled_gradients = self.scale_gradients(&gradients);

            // Convert gradients to HashMap
            let mut grads_map = HashMap::new();
            for (i, grad) in scaled_gradients.iter().enumerate() {
                grads_map.insert(format!("param_{}", i), *grad);
            }

            // Update parameters using optimizer
            optimizer.step(&mut params_map, &grads_map);

            // Convert back to Array1
            for (i, value) in self.target_model.parameters.iter_mut().enumerate() {
                if let Some(new_val) = params_map.get(&format!("param_{}", i)) {
                    *value = *new_val;
                }
            }

            // Compute loss
            let loss = self.compute_loss(training_data, labels)?;
            loss_history.push(loss);

            if loss < best_loss {
                best_loss = loss;
                best_params = self.target_model.parameters.clone();
            }
        }

        // Compute final accuracy
        let predictions = self.predict(training_data)?;
        let accuracy = Self::compute_accuracy(&predictions, labels);

        Ok(TrainingResult {
            final_loss: best_loss,
            accuracy,
            loss_history,
            optimal_parameters: best_params,
        })
    }

    /// Progressively unfreeze layers
    fn unfreeze_next_layer(&mut self) {
        // Find the last frozen layer and unfreeze it
        let num_layers = self.layer_configs.len();
        for (i, config) in self.layer_configs.iter_mut().enumerate().rev() {
            if config.frozen {
                config.frozen = false;
                config.learning_rate_multiplier = 0.1 * (i as f64 / num_layers as f64);
                break;
            }
        }
    }

    /// Compute gradients with frozen layer handling
    fn compute_gradients(&self, data: &Array2<f64>, labels: &Array1<f64>) -> Result<Array1<f64>> {
        // Placeholder implementation
        // In practice, would compute actual quantum gradients
        let mut gradients = Array1::zeros(self.target_model.parameters.len());

        // Only compute gradients for non-frozen layers
        for config in &self.layer_configs {
            if !config.frozen {
                for &idx in &config.parameter_indices {
                    if idx < gradients.len() {
                        gradients[idx] = 0.1 * (2.0 * thread_rng().gen::<f64>() - 1.0);
                    }
                }
            }
        }

        Ok(gradients)
    }

    /// Scale gradients based on layer configuration
    fn scale_gradients(&self, gradients: &Array1<f64>) -> Array1<f64> {
        let mut scaled = gradients.clone();

        for config in &self.layer_configs {
            for &idx in &config.parameter_indices {
                if idx < scaled.len() {
                    scaled[idx] *= config.learning_rate_multiplier;
                }
            }
        }

        scaled
    }

    /// Compute loss on the target task
    fn compute_loss(&self, data: &Array2<f64>, labels: &Array1<f64>) -> Result<f64> {
        let predictions = self.predict(data)?;

        // Mean squared error
        let mut loss = 0.0;
        for (pred, label) in predictions.iter().zip(labels.iter()) {
            loss += (pred - label).powi(2);
        }

        Ok(loss / labels.len() as f64)
    }

    /// Make predictions using the target model
    pub fn predict(&self, data: &Array2<f64>) -> Result<Array1<f64>> {
        // Placeholder implementation
        // In practice, would run quantum circuit and measure
        let num_samples = data.nrows();
        Ok(Array1::from_vec(vec![0.5; num_samples]))
    }

    /// Compute classification accuracy
    fn compute_accuracy(predictions: &Array1<f64>, labels: &Array1<f64>) -> f64 {
        let correct = predictions
            .iter()
            .zip(labels.iter())
            .filter(|(p, l)| (p.round() - l.round()).abs() < 0.1)
            .count();

        correct as f64 / labels.len() as f64
    }

    /// Extract features using the pre-trained layers
    pub fn extract_features(&self, data: &Array2<f64>) -> Result<Array2<f64>> {
        // Use only the frozen (pre-trained) layers for feature extraction
        let feature_dim = self
            .layer_configs
            .iter()
            .filter(|c| c.frozen)
            .map(|c| c.parameter_indices.len())
            .sum();

        let num_samples = data.nrows();
        let features = Array2::zeros((num_samples, feature_dim));

        // Placeholder - in practice would run partial circuit
        Ok(features)
    }

    /// Save the fine-tuned model
    pub fn save_model(&self, path: &str) -> Result<()> {
        // Placeholder - would serialize model to file
        Ok(())
    }

    /// Load a pre-trained model for transfer learning
    pub fn load_pretrained(path: &str) -> Result<PretrainedModel> {
        // Placeholder - would deserialize model from file
        Err(MLError::ModelCreationError("Not implemented".to_string()))
    }
}

/// Model zoo for pre-trained quantum models
pub struct QuantumModelZoo;

impl QuantumModelZoo {
    /// Get a pre-trained model for image classification
    pub fn get_image_classifier() -> Result<PretrainedModel> {
        // Create a simple pre-trained model
        let layers = vec![
            QNNLayerType::EncodingLayer { num_features: 4 },
            QNNLayerType::VariationalLayer { num_params: 8 },
            QNNLayerType::EntanglementLayer {
                connectivity: "linear".to_string(),
            },
            QNNLayerType::MeasurementLayer {
                measurement_basis: "computational".to_string(),
            },
        ];

        let qnn = QuantumNeuralNetwork::new(layers, 4, 4, 2)?;

        let mut metadata = HashMap::new();
        metadata.insert("task".to_string(), "image_classification".to_string());
        metadata.insert("dataset".to_string(), "mnist_subset".to_string());

        let mut performance = HashMap::new();
        performance.insert("accuracy".to_string(), 0.85);
        performance.insert("loss".to_string(), 0.32);

        Ok(PretrainedModel {
            qnn,
            task_description: "Pre-trained on MNIST subset for binary classification".to_string(),
            performance_metrics: performance,
            metadata,
        })
    }

    /// Get a pre-trained model for quantum chemistry
    pub fn get_chemistry_model() -> Result<PretrainedModel> {
        let layers = vec![
            QNNLayerType::EncodingLayer { num_features: 6 },
            QNNLayerType::VariationalLayer { num_params: 12 },
            QNNLayerType::EntanglementLayer {
                connectivity: "full".to_string(),
            },
            QNNLayerType::VariationalLayer { num_params: 12 },
            QNNLayerType::MeasurementLayer {
                measurement_basis: "Pauli-Z".to_string(),
            },
        ];

        let qnn = QuantumNeuralNetwork::new(layers, 6, 6, 1)?;

        let mut metadata = HashMap::new();
        metadata.insert("task".to_string(), "molecular_energy".to_string());
        metadata.insert("dataset".to_string(), "h2_h4_molecules".to_string());

        let mut performance = HashMap::new();
        performance.insert("mae".to_string(), 0.001);
        performance.insert("r2_score".to_string(), 0.98);

        Ok(PretrainedModel {
            qnn,
            task_description: "Pre-trained on molecular energy prediction".to_string(),
            performance_metrics: performance,
            metadata,
        })
    }

    /// Get a VQE feature extractor model
    pub fn vqe_feature_extractor(n_qubits: usize) -> Result<PretrainedModel> {
        let layers = vec![
            QNNLayerType::EncodingLayer {
                num_features: n_qubits,
            },
            QNNLayerType::VariationalLayer {
                num_params: n_qubits * 2,
            },
            QNNLayerType::EntanglementLayer {
                connectivity: "linear".to_string(),
            },
            QNNLayerType::VariationalLayer {
                num_params: n_qubits,
            },
            QNNLayerType::MeasurementLayer {
                measurement_basis: "Pauli-Z".to_string(),
            },
        ];

        let qnn = QuantumNeuralNetwork::new(layers, n_qubits, n_qubits, n_qubits / 2)?;

        let mut metadata = HashMap::new();
        metadata.insert("task".to_string(), "feature_extraction".to_string());
        metadata.insert("algorithm".to_string(), "VQE".to_string());

        let mut performance = HashMap::new();
        performance.insert("fidelity".to_string(), 0.92);
        performance.insert("feature_quality".to_string(), 0.88);

        Ok(PretrainedModel {
            qnn,
            task_description: format!("Pre-trained VQE feature extractor for {} qubits", n_qubits),
            performance_metrics: performance,
            metadata,
        })
    }

    /// Get a QAOA classifier model
    pub fn qaoa_classifier(n_qubits: usize, n_layers: usize) -> Result<PretrainedModel> {
        let mut layers = vec![QNNLayerType::EncodingLayer {
            num_features: n_qubits,
        }];

        // Add QAOA layers
        for _ in 0..n_layers {
            layers.push(QNNLayerType::VariationalLayer {
                num_params: n_qubits,
            });
            layers.push(QNNLayerType::EntanglementLayer {
                connectivity: "circular".to_string(),
            });
        }

        layers.push(QNNLayerType::MeasurementLayer {
            measurement_basis: "computational".to_string(),
        });

        let qnn = QuantumNeuralNetwork::new(layers, n_qubits, n_qubits, 2)?;

        let mut metadata = HashMap::new();
        metadata.insert("task".to_string(), "classification".to_string());
        metadata.insert("algorithm".to_string(), "QAOA".to_string());
        metadata.insert("layers".to_string(), n_layers.to_string());

        let mut performance = HashMap::new();
        performance.insert("accuracy".to_string(), 0.86);
        performance.insert("f1_score".to_string(), 0.84);

        Ok(PretrainedModel {
            qnn,
            task_description: format!(
                "Pre-trained QAOA classifier with {} qubits and {} layers",
                n_qubits, n_layers
            ),
            performance_metrics: performance,
            metadata,
        })
    }

    /// Get a quantum autoencoder model
    pub fn quantum_autoencoder(n_qubits: usize, latent_dim: usize) -> Result<PretrainedModel> {
        let layers = vec![
            QNNLayerType::EncodingLayer {
                num_features: n_qubits,
            },
            QNNLayerType::VariationalLayer {
                num_params: n_qubits * 2,
            },
            QNNLayerType::EntanglementLayer {
                connectivity: "linear".to_string(),
            },
            // Compression layer
            QNNLayerType::VariationalLayer {
                num_params: latent_dim * 2,
            },
            // Decompression layer
            QNNLayerType::VariationalLayer {
                num_params: n_qubits,
            },
            QNNLayerType::EntanglementLayer {
                connectivity: "full".to_string(),
            },
            QNNLayerType::MeasurementLayer {
                measurement_basis: "computational".to_string(),
            },
        ];

        let qnn = QuantumNeuralNetwork::new(layers, n_qubits, n_qubits, n_qubits)?;

        let mut metadata = HashMap::new();
        metadata.insert("task".to_string(), "autoencoding".to_string());
        metadata.insert("latent_dimension".to_string(), latent_dim.to_string());

        let mut performance = HashMap::new();
        performance.insert("reconstruction_fidelity".to_string(), 0.94);
        performance.insert(
            "compression_ratio".to_string(),
            n_qubits as f64 / latent_dim as f64,
        );

        Ok(PretrainedModel {
            qnn,
            task_description: format!(
                "Pre-trained quantum autoencoder with {} qubits and {} latent dimensions",
                n_qubits, latent_dim
            ),
            performance_metrics: performance,
            metadata,
        })
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::autodiff::optimizers::Adam;

    #[test]
    fn test_transfer_learning_creation() {
        let source = QuantumModelZoo::get_image_classifier().unwrap();
        let target_layers = vec![
            QNNLayerType::VariationalLayer { num_params: 4 },
            QNNLayerType::MeasurementLayer {
                measurement_basis: "computational".to_string(),
            },
        ];

        let transfer = QuantumTransferLearning::new(
            source,
            target_layers,
            TransferStrategy::FineTuning {
                num_trainable_layers: 2,
            },
        )
        .unwrap();

        assert_eq!(transfer.current_epoch, 0);
        assert!(transfer.layer_configs.len() > 0);
    }

    #[test]
    fn test_layer_freezing() {
        let source = QuantumModelZoo::get_chemistry_model().unwrap();
        let target_layers = vec![];

        let transfer = QuantumTransferLearning::new(
            source,
            target_layers,
            TransferStrategy::FeatureExtraction,
        )
        .unwrap();

        // Check that early layers are frozen
        assert!(transfer.layer_configs[0].frozen);
        assert_eq!(transfer.layer_configs[0].learning_rate_multiplier, 0.0);
    }

    #[test]
    fn test_progressive_unfreezing() {
        let source = QuantumModelZoo::get_image_classifier().unwrap();
        let target_layers = vec![];

        let mut transfer = QuantumTransferLearning::new(
            source,
            target_layers,
            TransferStrategy::ProgressiveUnfreezing { unfreeze_rate: 5 },
        )
        .unwrap();

        // Initially most layers should be frozen
        let frozen_count = transfer.layer_configs.iter().filter(|c| c.frozen).count();
        assert!(frozen_count > 0);

        // Simulate training epochs
        transfer.current_epoch = 5;
        transfer.unfreeze_next_layer();

        // Check that a layer was unfrozen
        let new_frozen_count = transfer.layer_configs.iter().filter(|c| c.frozen).count();
        assert_eq!(new_frozen_count, frozen_count - 1);
    }
}
