"""
Resilient Quantum Circuit Execution for QuantRS2

This module provides production-grade quantum circuit execution with
comprehensive error handling, recovery mechanisms, resilience patterns,
and advanced resource management.
"""

import logging
import time
import asyncio
import threading
from typing import Any, Dict, List, Optional, Union, Callable, Type, Tuple
from dataclasses import dataclass, field
from enum import Enum
from concurrent.futures import ThreadPoolExecutor, TimeoutError
import json

# Import error handling components
from .error_handling import (
    ErrorSeverity,
    ErrorCategory, 
    RecoveryStrategy,
    ErrorContext,
    QuantumError,
    QuantumHardwareError,
    CircuitCompilationError,
    SimulationError,
    ResourceError,
    ErrorRecoveryManager,
    quantum_error_handler,
    quantum_error_context,
    get_error_manager,
    create_error_context
)

# Import resource management components
from .resource_management import (
    ResourceType,
    ResourceStatus,
    ResourceConfig,
    ResourceMonitor,
    ResourcePool,
    ResourceException,
    resource_context,
    analyze_circuit_resources
)

logger = logging.getLogger(__name__)

class ExecutionMode(Enum):
    """Execution modes for quantum circuits."""
    SIMULATION = "simulation"
    HARDWARE = "hardware"
    HYBRID = "hybrid"
    AUTO = "auto"

class ExecutionStatus(Enum):
    """Status of circuit execution."""
    PENDING = "pending"
    RUNNING = "running"
    COMPLETED = "completed"
    FAILED = "failed"
    RECOVERED = "recovered"
    CANCELLED = "cancelled"

@dataclass
class ExecutionConfig:
    """Configuration for resilient circuit execution."""
    
    mode: ExecutionMode = ExecutionMode.AUTO
    timeout_seconds: float = 300.0
    max_retries: int = 3
    retry_delay: float = 1.0
    enable_circuit_optimization: bool = True
    enable_noise_mitigation: bool = True
    fallback_to_simulation: bool = True
    preferred_backends: List[str] = field(default_factory=list)
    resource_limits: Dict[str, Any] = field(default_factory=dict)
    enable_parallel_execution: bool = False
    max_parallel_jobs: int = 4
    
    # Resource management configuration
    resource_config: Optional[ResourceConfig] = None
    enable_resource_monitoring: bool = True
    enable_resource_enforcement: bool = True
    execution_priority: str = "normal"  # "high", "normal", "low"
    
    def get_resource_config(self) -> ResourceConfig:
        """Get resource configuration, creating default if needed."""
        if self.resource_config is None:
            self.resource_config = ResourceConfig(
                max_memory_mb=4096.0,
                max_cpu_percent=80.0,
                max_execution_time=self.timeout_seconds,
                max_concurrent_executions=self.max_parallel_jobs
            )
        return self.resource_config

@dataclass
class ExecutionResult:
    """Result of quantum circuit execution."""
    
    execution_id: str
    status: ExecutionStatus
    result_data: Optional[Dict[str, Any]] = None
    error_details: Optional[Dict[str, Any]] = None
    execution_time: float = 0.0
    backend_used: Optional[str] = None
    recovery_applied: List[str] = field(default_factory=list)
    metadata: Dict[str, Any] = field(default_factory=dict)
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for serialization."""
        return {
            'execution_id': self.execution_id,
            'status': self.status.value,
            'result_data': self.result_data,
            'error_details': self.error_details,
            'execution_time': self.execution_time,
            'backend_used': self.backend_used,
            'recovery_applied': self.recovery_applied,
            'metadata': self.metadata
        }

class CircuitExecutionEngine:
    """Core engine for resilient quantum circuit execution with resource management."""
    
    def __init__(self, config: Optional[ExecutionConfig] = None):
        self.config = config or ExecutionConfig()
        self.error_manager = get_error_manager()
        
        # Resource management
        if self.config.enable_resource_monitoring:
            self.resource_config = self.config.get_resource_config()
            self.resource_monitor = ResourceMonitor(self.resource_config)
            self.resource_pool = ResourcePool(self.resource_config, self.resource_monitor)
            self.resource_monitor.start_monitoring()
            logger.info("Resource monitoring enabled")
        else:
            self.resource_monitor = None
            self.resource_pool = None
            # Fallback to basic executor
            self.executor = ThreadPoolExecutor(max_workers=self.config.max_parallel_jobs)
        
        # Execution tracking
        self.active_executions: Dict[str, ExecutionResult] = {}
        self.execution_history: List[ExecutionResult] = []
        
        # Backend availability tracking
        self.backend_status: Dict[str, Dict[str, Any]] = {}
        
        # Lock for thread safety
        self._lock = threading.RLock()
        
        logger.info("Circuit execution engine initialized with advanced resource management")
    
    @quantum_error_handler("circuit_execution")
    def execute_circuit(self, circuit: Any, execution_id: Optional[str] = None,
                       config: Optional[ExecutionConfig] = None) -> ExecutionResult:
        """
        Execute quantum circuit with resilience, error recovery, and resource management.
        
        Args:
            circuit: Quantum circuit to execute
            execution_id: Optional execution identifier
            config: Optional execution configuration
            
        Returns:
            ExecutionResult with status and results
        """
        import uuid
        
        # Use provided config or default
        exec_config = config or self.config
        
        # Generate execution ID if not provided
        if execution_id is None:
            execution_id = str(uuid.uuid4())
        
        # Analyze circuit resources
        circuit_info = self._get_circuit_info(circuit)
        resource_analysis = analyze_circuit_resources(circuit)
        circuit_info.update(resource_analysis)
        
        # Check resource requirements before execution
        if self.resource_monitor and exec_config.enable_resource_enforcement:
            try:
                self._check_circuit_resource_requirements(circuit_info)
                self.resource_monitor.enforce_limits()
            except ResourceException as e:
                logger.error(f"Circuit {execution_id} exceeds resource limits: {e}")
                return ExecutionResult(
                    execution_id=execution_id,
                    status=ExecutionStatus.FAILED,
                    error_details={
                        'error_type': 'ResourceException',
                        'error_message': str(e),
                        'timestamp': time.time()
                    }
                )
        
        # Create execution result
        result = ExecutionResult(
            execution_id=execution_id,
            status=ExecutionStatus.PENDING,
            metadata={'circuit_info': circuit_info}
        )
        
        # Add to active executions
        with self._lock:
            self.active_executions[execution_id] = result
        
        start_time = time.time()
        
        try:
            with quantum_error_context(
                "circuit_execution",
                session_id=execution_id,
                circuit_info=circuit_info
            ) as context:
                
                result.status = ExecutionStatus.RUNNING
                
                # Execute with resource management
                if self.resource_pool and exec_config.enable_resource_monitoring:
                    execution_result = self._execute_with_resource_management(
                        circuit, exec_config, context, execution_id, circuit_info
                    )
                elif exec_config.timeout_seconds > 0:
                    execution_result = self._execute_with_timeout(
                        circuit, exec_config, context
                    )
                else:
                    execution_result = self._execute_circuit_internal(
                        circuit, exec_config, context
                    )
                
                # Update result
                result.status = ExecutionStatus.COMPLETED
                result.result_data = execution_result
                result.execution_time = time.time() - start_time
                
                # Add resource usage to metadata
                if self.resource_monitor:
                    result.metadata['resource_usage'] = {
                        rt.value: usage.__dict__ for rt, usage in 
                        self.resource_monitor.get_all_usage().items()
                    }
                
                logger.info(f"Circuit execution {execution_id} completed successfully")
                
        except ResourceException as e:
            result.status = ExecutionStatus.FAILED
            result.error_details = {
                'error_type': 'ResourceException',
                'error_message': str(e),
                'timestamp': time.time(),
                'resource_type': e.resource_type.value,
                'current_usage': e.current_usage,
                'limit': e.limit
            }
            result.execution_time = time.time() - start_time
            
            logger.error(f"Circuit execution {execution_id} failed due to resource limits: {e}")
            raise e
            
        except Exception as e:
            result.status = ExecutionStatus.FAILED
            result.error_details = {
                'error_type': type(e).__name__,
                'error_message': str(e),
                'timestamp': time.time()
            }
            result.execution_time = time.time() - start_time
            
            logger.error(f"Circuit execution {execution_id} failed: {e}")
            
            # Attempt recovery
            recovery_result = self._attempt_execution_recovery(
                circuit, e, exec_config, context
            )
            
            if recovery_result:
                result.status = ExecutionStatus.RECOVERED
                result.result_data = recovery_result
                result.recovery_applied = ["execution_recovery"]
                logger.info(f"Circuit execution {execution_id} recovered")
            else:
                raise e
        
        finally:
            # Move to history and clean up
            with self._lock:
                if execution_id in self.active_executions:
                    del self.active_executions[execution_id]
                self.execution_history.append(result)
        
        return result
    
    def _check_circuit_resource_requirements(self, circuit_info: Dict[str, Any]) -> None:
        """Check if circuit meets resource requirements."""
        if not self.resource_monitor:
            return
        
        # Check circuit complexity limits
        qubits = circuit_info.get('qubits', 0)
        gates = circuit_info.get('gates', 0)
        depth = circuit_info.get('depth', 0)
        estimated_memory = circuit_info.get('estimated_memory_mb', 0)
        
        # Check against limits
        limits = self.resource_monitor.limits
        
        if ResourceType.QUBITS in limits and qubits > limits[ResourceType.QUBITS].hard_limit:
            raise ResourceException(
                f"Circuit requires {qubits} qubits, exceeds limit of {limits[ResourceType.QUBITS].hard_limit}",
                ResourceType.QUBITS, qubits, limits[ResourceType.QUBITS].hard_limit
            )
        
        if ResourceType.GATES in limits and gates > limits[ResourceType.GATES].hard_limit:
            raise ResourceException(
                f"Circuit has {gates} gates, exceeds limit of {limits[ResourceType.GATES].hard_limit}",
                ResourceType.GATES, gates, limits[ResourceType.GATES].hard_limit
            )
        
        if ResourceType.DEPTH in limits and depth > limits[ResourceType.DEPTH].hard_limit:
            raise ResourceException(
                f"Circuit depth {depth} exceeds limit of {limits[ResourceType.DEPTH].hard_limit}",
                ResourceType.DEPTH, depth, limits[ResourceType.DEPTH].hard_limit
            )
        
        if ResourceType.MEMORY in limits and estimated_memory > limits[ResourceType.MEMORY].hard_limit:
            raise ResourceException(
                f"Circuit requires {estimated_memory}MB memory, exceeds limit of {limits[ResourceType.MEMORY].hard_limit}MB",
                ResourceType.MEMORY, estimated_memory, limits[ResourceType.MEMORY].hard_limit
            )
    
    def _execute_with_resource_management(self, circuit: Any, config: ExecutionConfig,
                                        context: ErrorContext, execution_id: str,
                                        circuit_info: Dict[str, Any]) -> Dict[str, Any]:
        """Execute circuit using the resource management system."""
        
        def execution_func():
            return self._execute_circuit_internal(circuit, config, context)
        
        # Submit to resource pool
        future = self.resource_pool.submit_execution(
            execution_id, circuit_info, execution_func, config.execution_priority
        )
        
        try:
            # Wait for completion with timeout
            result = future.result(timeout=config.timeout_seconds if config.timeout_seconds > 0 else None)
            return result
        
        except TimeoutError:
            # Cancel the execution
            self.resource_pool.cancel_execution(execution_id)
            raise ResourceError(
                f"Circuit execution timed out after {config.timeout_seconds} seconds",
                resource_type="time",
                required=int(config.timeout_seconds + 1)
            )
        
        except Exception as e:
            # Cancel the execution
            self.resource_pool.cancel_execution(execution_id)
            raise e
    
    def _execute_with_timeout(self, circuit: Any, config: ExecutionConfig,
                            context: ErrorContext) -> Dict[str, Any]:
        """Execute circuit with timeout protection."""
        future = self.executor.submit(
            self._execute_circuit_internal, circuit, config, context
        )
        
        try:
            return future.result(timeout=config.timeout_seconds)
        except TimeoutError:
            future.cancel()
            raise ResourceError(
                f"Circuit execution timed out after {config.timeout_seconds} seconds",
                resource_type="time",
                required=int(config.timeout_seconds + 1)
            )
    
    def _execute_circuit_internal(self, circuit: Any, config: ExecutionConfig,
                                context: ErrorContext) -> Dict[str, Any]:
        """Internal circuit execution logic."""
        
        # Validate circuit
        self._validate_circuit(circuit)
        
        # Optimize circuit if enabled
        if config.enable_circuit_optimization:
            circuit = self._optimize_circuit(circuit, context)
        
        # Apply noise mitigation if enabled
        if config.enable_noise_mitigation:
            circuit = self._apply_noise_mitigation(circuit, context)
        
        # Select backend based on mode
        backend = self._select_backend(config)
        
        # Execute on selected backend
        try:
            result = self._execute_on_backend(circuit, backend, context)
            
            # Post-process results
            processed_result = self._post_process_results(result, config)
            
            return processed_result
            
        except QuantumHardwareError as e:
            # Try fallback to simulation if allowed
            if config.fallback_to_simulation and backend != "simulation":
                logger.warning(f"Hardware execution failed, falling back to simulation: {e}")
                return self._execute_on_backend(circuit, "simulation", context)
            else:
                raise e
    
    def _validate_circuit(self, circuit: Any) -> None:
        """Validate circuit before execution."""
        # Placeholder for circuit validation
        # In real implementation, would check:
        # - Circuit structure
        # - Gate validity
        # - Qubit connectivity
        # - Resource requirements
        
        if circuit is None:
            raise CircuitCompilationError("Circuit cannot be None")
        
        # Check if circuit has required attributes/methods
        if not hasattr(circuit, 'gates') and not hasattr(circuit, 'run'):
            logger.warning("Circuit object may not be valid - missing expected attributes")
    
    def _optimize_circuit(self, circuit: Any, context: ErrorContext) -> Any:
        """Optimize circuit for execution."""
        try:
            # Placeholder for circuit optimization
            # In real implementation, would apply:
            # - Gate fusion
            # - Dead code elimination
            # - Qubit routing optimization
            # - Depth reduction
            
            logger.debug("Circuit optimization applied")
            return circuit
            
        except Exception as e:
            logger.warning(f"Circuit optimization failed: {e}")
            return circuit  # Return original circuit if optimization fails
    
    def _apply_noise_mitigation(self, circuit: Any, context: ErrorContext) -> Any:
        """Apply noise mitigation techniques."""
        try:
            # Placeholder for noise mitigation
            # In real implementation, would apply:
            # - Error correction codes
            # - Zero-noise extrapolation
            # - Symmetry verification
            # - Readout error mitigation
            
            logger.debug("Noise mitigation applied")
            return circuit
            
        except Exception as e:
            logger.warning(f"Noise mitigation failed: {e}")
            return circuit  # Return original circuit if mitigation fails
    
    def _select_backend(self, config: ExecutionConfig) -> str:
        """Select appropriate backend for execution."""
        if config.mode == ExecutionMode.SIMULATION:
            return "simulation"
        elif config.mode == ExecutionMode.HARDWARE:
            return self._select_hardware_backend(config)
        elif config.mode == ExecutionMode.HYBRID:
            return self._select_hybrid_backend(config)
        else:  # AUTO mode
            return self._select_auto_backend(config)
    
    def _select_hardware_backend(self, config: ExecutionConfig) -> str:
        """Select hardware backend."""
        # Check preferred backends first
        for backend in config.preferred_backends:
            if self._is_backend_available(backend):
                return backend
        
        # Fall back to any available hardware backend
        available_backends = self._get_available_hardware_backends()
        if available_backends:
            return available_backends[0]
        
        raise QuantumHardwareError("No hardware backends available")
    
    def _select_hybrid_backend(self, config: ExecutionConfig) -> str:
        """Select backend for hybrid execution."""
        # For hybrid mode, prefer hardware but allow simulation fallback
        try:
            return self._select_hardware_backend(config)
        except QuantumHardwareError:
            if config.fallback_to_simulation:
                return "simulation"
            else:
                raise
    
    def _select_auto_backend(self, config: ExecutionConfig) -> str:
        """Automatically select best backend."""
        # Auto mode: try hardware first, fall back to simulation
        try:
            return self._select_hardware_backend(config)
        except QuantumHardwareError:
            return "simulation"
    
    def _execute_on_backend(self, circuit: Any, backend: str,
                          context: ErrorContext) -> Dict[str, Any]:
        """Execute circuit on specified backend."""
        logger.info(f"Executing circuit on backend: {backend}")
        
        if backend == "simulation":
            return self._execute_simulation(circuit, context)
        else:
            return self._execute_hardware(circuit, backend, context)
    
    def _execute_simulation(self, circuit: Any, context: ErrorContext) -> Dict[str, Any]:
        """Execute circuit on simulator."""
        try:
            # Placeholder for simulation execution
            # In real implementation, would:
            # - Run circuit on state vector simulator
            # - Handle memory limitations
            # - Apply noise models if specified
            
            # Simulate some execution time
            time.sleep(0.1)
            
            # Return mock results
            return {
                "backend": "simulation",
                "execution_time": 0.1,
                "results": {
                    "00": 0.5,
                    "11": 0.5
                },
                "metadata": {
                    "shots": 1000,
                    "success": True
                }
            }
            
        except Exception as e:
            raise SimulationError(f"Simulation failed: {e}")
    
    def _execute_hardware(self, circuit: Any, backend: str,
                         context: ErrorContext) -> Dict[str, Any]:
        """Execute circuit on quantum hardware."""
        try:
            # Check backend availability
            if not self._is_backend_available(backend):
                raise QuantumHardwareError(f"Backend {backend} is not available")
            
            # Placeholder for hardware execution
            # In real implementation, would:
            # - Submit job to quantum cloud service
            # - Monitor job status
            # - Retrieve results
            # - Handle queue times and failures
            
            # Simulate hardware execution
            time.sleep(0.5)  # Simulate queue time
            
            # Simulate occasional hardware failures
            import random
            if random.random() < 0.1:  # 10% failure rate
                raise QuantumHardwareError(f"Hardware backend {backend} encountered an error")
            
            return {
                "backend": backend,
                "execution_time": 0.5,
                "results": {
                    "00": 0.48,
                    "01": 0.02,
                    "10": 0.03, 
                    "11": 0.47
                },
                "metadata": {
                    "shots": 1000,
                    "success": True,
                    "calibration_data": {"fidelity": 0.95}
                }
            }
            
        except Exception as e:
            if isinstance(e, QuantumHardwareError):
                raise e
            else:
                raise QuantumHardwareError(f"Hardware execution failed: {e}")
    
    def _post_process_results(self, result: Dict[str, Any],
                            config: ExecutionConfig) -> Dict[str, Any]:
        """Post-process execution results."""
        try:
            # Placeholder for result post-processing
            # In real implementation, would:
            # - Apply readout error correction
            # - Normalize probabilities
            # - Add statistical analysis
            # - Format output
            
            processed_result = result.copy()
            processed_result["post_processed"] = True
            processed_result["processing_timestamp"] = time.time()
            
            return processed_result
            
        except Exception as e:
            logger.warning(f"Result post-processing failed: {e}")
            return result  # Return original results if post-processing fails
    
    def _attempt_execution_recovery(self, circuit: Any, error: Exception,
                                  config: ExecutionConfig, context: ErrorContext) -> Optional[Dict[str, Any]]:
        """Attempt to recover from execution failure."""
        logger.info("Attempting execution recovery")
        
        try:
            # Circuit simplification recovery
            if isinstance(error, (ResourceError, SimulationError)):
                simplified_circuit = self._simplify_circuit_for_recovery(circuit)
                return self._execute_circuit_internal(simplified_circuit, config, context)
            
            # Backend switching recovery
            if isinstance(error, QuantumHardwareError):
                if config.fallback_to_simulation:
                    return self._execute_on_backend(circuit, "simulation", context)
                else:
                    # Try alternative hardware backend
                    alternative_backend = self._get_alternative_backend(config)
                    if alternative_backend:
                        return self._execute_on_backend(circuit, alternative_backend, context)
            
            # No recovery possible
            return None
            
        except Exception as recovery_error:
            logger.error(f"Recovery attempt failed: {recovery_error}")
            return None
    
    def _simplify_circuit_for_recovery(self, circuit: Any) -> Any:
        """Simplify circuit for recovery execution."""
        # Placeholder for circuit simplification
        # In real implementation, would:
        # - Remove non-essential gates
        # - Reduce circuit depth
        # - Approximate complex operations
        
        logger.info("Simplifying circuit for recovery")
        return circuit
    
    def _get_alternative_backend(self, config: ExecutionConfig) -> Optional[str]:
        """Get alternative hardware backend."""
        available_backends = self._get_available_hardware_backends()
        
        # Filter out backends already tried
        alternative_backends = [
            backend for backend in available_backends
            if backend not in config.preferred_backends
        ]
        
        return alternative_backends[0] if alternative_backends else None
    
    def _is_backend_available(self, backend: str) -> bool:
        """Check if backend is available."""
        # Placeholder for backend availability check
        # In real implementation, would check:
        # - Backend status
        # - Queue length
        # - Maintenance schedules
        # - Authentication
        
        # Simulate backend availability
        import random
        if backend == "simulation":
            return True
        else:
            return random.random() > 0.2  # 80% availability for hardware
    
    def _get_available_hardware_backends(self) -> List[str]:
        """Get list of available hardware backends."""
        # Placeholder for backend discovery
        # In real implementation, would query:
        # - IBM Quantum
        # - Google Quantum AI
        # - AWS Braket
        # - Local hardware
        
        all_backends = ["ibm_quantum", "google_quantum_ai", "aws_braket"]
        return [backend for backend in all_backends if self._is_backend_available(backend)]
    
    def _get_circuit_info(self, circuit: Any) -> Dict[str, Any]:
        """Extract circuit information."""
        # Placeholder for circuit analysis
        # In real implementation, would extract:
        # - Number of qubits
        # - Number of gates
        # - Circuit depth
        # - Gate types used
        
        return {
            "type": type(circuit).__name__,
            "estimated_qubits": "unknown",
            "estimated_gates": "unknown",
            "estimated_depth": "unknown"
        }
    
    def get_execution_status(self, execution_id: str) -> Optional[ExecutionResult]:
        """Get status of a specific execution."""
        with self._lock:
            if execution_id in self.active_executions:
                return self.active_executions[execution_id]
            
            # Check history
            for result in self.execution_history:
                if result.execution_id == execution_id:
                    return result
        
        return None
    
    def cancel_execution(self, execution_id: str) -> bool:
        """Cancel an active execution."""
        with self._lock:
            if execution_id in self.active_executions:
                result = self.active_executions[execution_id]
                result.status = ExecutionStatus.CANCELLED
                return True
        
        return False
    
    def get_execution_statistics(self) -> Dict[str, Any]:
        """Get execution statistics."""
        with self._lock:
            total_executions = len(self.execution_history)
            
            if total_executions == 0:
                return {"total_executions": 0}
            
            status_counts = {}
            total_time = 0
            recovery_count = 0
            
            for result in self.execution_history:
                status_counts[result.status.value] = status_counts.get(result.status.value, 0) + 1
                total_time += result.execution_time
                if result.recovery_applied:
                    recovery_count += 1
            
            success_rate = (status_counts.get("completed", 0) + status_counts.get("recovered", 0)) / total_executions
            average_time = total_time / total_executions
            recovery_rate = recovery_count / total_executions
            
            return {
                "total_executions": total_executions,
                "status_distribution": status_counts,
                "success_rate": success_rate,
                "recovery_rate": recovery_rate,
                "average_execution_time": average_time,
                "active_executions": len(self.active_executions)
            }
    
    def cleanup(self) -> None:
        """Cleanup resources."""
        try:
            # Cancel all active executions
            with self._lock:
                for execution_id in list(self.active_executions.keys()):
                    self.cancel_execution(execution_id)
            
            # Cleanup resource management components
            if self.resource_monitor:
                self.resource_monitor.stop_monitoring()
            
            if self.resource_pool:
                self.resource_pool.shutdown(wait=True)
            elif hasattr(self, 'executor'):
                # Cleanup fallback executor
                self.executor.shutdown(wait=True)
            
            logger.info("Circuit execution engine cleanup completed")
            
        except Exception as e:
            logger.error(f"Cleanup failed: {e}")
    
    def get_resource_status(self) -> Dict[str, Any]:
        """Get current resource status and statistics."""
        if not self.resource_monitor:
            return {'resource_monitoring': 'disabled'}
        
        status = {
            'resource_monitoring': 'enabled',
            'current_usage': {
                rt.value: usage.__dict__ for rt, usage in 
                self.resource_monitor.get_all_usage().items()
            },
            'resource_limits': {
                rt.value: {
                    'soft_limit': limit.soft_limit,
                    'hard_limit': limit.hard_limit,
                    'unit': limit.unit,
                    'enabled': limit.enabled
                } for rt, limit in self.resource_monitor.limits.items()
            },
            'system_resources': self.resource_monitor.get_system_resources()
        }
        
        if self.resource_pool:
            status['queue_status'] = self.resource_pool.get_queue_status()
        
        return status
    
    def force_resource_cleanup(self) -> Dict[str, Any]:
        """Force garbage collection and resource cleanup."""
        if not self.resource_monitor:
            return {'resource_monitoring': 'disabled'}
        
        return self.resource_monitor.force_garbage_collection()

# Global execution engine
_execution_engine = CircuitExecutionEngine()

def get_execution_engine(config: Optional[ExecutionConfig] = None) -> CircuitExecutionEngine:
    """Get the global execution engine."""
    global _execution_engine
    if config:
        _execution_engine = CircuitExecutionEngine(config)
    return _execution_engine

def execute_circuit_resilient(circuit: Any, execution_id: Optional[str] = None,
                            config: Optional[ExecutionConfig] = None) -> ExecutionResult:
    """Execute circuit with resilience and error recovery."""
    engine = get_execution_engine()
    return engine.execute_circuit(circuit, execution_id, config)

def execute_circuits_batch(circuits: List[Any], config: Optional[ExecutionConfig] = None) -> List[ExecutionResult]:
    """Execute multiple circuits with resilience."""
    engine = get_execution_engine()
    results = []
    
    for i, circuit in enumerate(circuits):
        execution_id = f"batch_{int(time.time())}_{i}"
        result = engine.execute_circuit(circuit, execution_id, config)
        results.append(result)
    
    return results

async def execute_circuit_async(circuit: Any, execution_id: Optional[str] = None,
                               config: Optional[ExecutionConfig] = None) -> ExecutionResult:
    """Execute circuit asynchronously with resilience."""
    loop = asyncio.get_event_loop()
    engine = get_execution_engine()
    
    # Run in thread pool to avoid blocking
    return await loop.run_in_executor(
        None, engine.execute_circuit, circuit, execution_id, config
    )

def configure_resilient_execution(config: ExecutionConfig) -> None:
    """Configure global resilient execution settings."""
    global _execution_engine
    _execution_engine = CircuitExecutionEngine(config)

# Export resilient execution components
__all__ = [
    # Enums
    "ExecutionMode",
    "ExecutionStatus",
    
    # Data classes
    "ExecutionConfig",
    "ExecutionResult",
    
    # Main classes
    "CircuitExecutionEngine",
    
    # Utilities
    "get_execution_engine",
    "execute_circuit_resilient",
    "execute_circuits_batch",
    "execute_circuit_async",
    "configure_resilient_execution",
    
    # Resource management (re-exported)
    "ResourceType",
    "ResourceStatus",
    "ResourceConfig",
    "ResourceException",
    "analyze_circuit_resources",
]