# QuantRS2-Sim: Advanced Quantum Simulation Suite

[![Crates.io](https://img.shields.io/crates/v/quantrs2-sim.svg)](https://crates.io/crates/quantrs2-sim)
[![Documentation](https://docs.rs/quantrs2-sim/badge.svg)](https://docs.rs/quantrs2-sim)
[![License](https://img.shields.io/badge/license-MIT%2FApache--2.0-blue.svg)](https://github.com/cool-japan/quantrs)

QuantRS2-Sim is the comprehensive simulation engine of the [QuantRS2](https://github.com/cool-japan/quantrs) quantum computing framework, providing state-of-the-art quantum simulation algorithms, error correction codes, and performance optimization techniques for simulating quantum systems up to 30+ qubits on standard hardware.

## Version 0.1.0-beta.2

This release features refined integration with [SciRS2](https://github.com/cool-japan/scirs2) v0.1.0-beta.3 for unprecedented performance:
- **Parallel Operations**: All simulators use `scirs2_core::parallel_ops` for automatic parallelization
- **SIMD Acceleration**: Vectorized quantum operations via SciRS2 SIMD abstractions
- **Advanced Linear Algebra**: Leveraging SciRS2's optimized BLAS/LAPACK bindings
- **Memory Efficiency**: SciRS2's memory management for large-scale quantum simulations

## Core Features

### Multi-Backend Simulation Architecture
- **State Vector Simulators**: Classical dense state vector simulation with SIMD acceleration
- **Matrix Product States (MPS)**: Memory-efficient simulation for low-entanglement circuits
- **Stabilizer Simulators**: Exponentially fast simulation for Clifford circuits
- **Decision Diagram (DD)**: Symbolic simulation using quantum decision diagrams
- **Tensor Network Simulators**: Advanced contraction algorithms for arbitrary network topologies
- **Path Integral Methods**: Feynman path integral approach for quantum evolution

### Advanced Quantum Simulation
- **Quantum Monte Carlo**: Variational (VMC), diffusion (DMC), and path integral (PIMC) methods
- **Trotter-Suzuki Decomposition**: Time evolution of quantum systems with optimized decompositions
- **Automatic Differentiation VQE**: Gradient-based optimization for variational quantum eigensolvers
- **Photonic Quantum Computing**: Continuous variable and discrete photonic system simulation
- **Fermionic Simulation**: Second quantization with Jordan-Wigner and Bravyi-Kitaev transformations
- **Open Quantum Systems**: Lindblad master equation integration for realistic quantum dynamics

### Error Correction & Fault Tolerance
- **Quantum Error Correction**: Surface, color, and concatenated codes
- **Noise Models**: Comprehensive realistic noise modeling including correlated errors
- **Error Mitigation**: Zero-noise extrapolation, virtual distillation, and symmetry verification
- **Process Tomography**: Quantum channel characterization and benchmarking
- **Quantum Volume**: IBM's quantum volume protocol

### Performance & Verification
- **Quantum Supremacy**: Cross-entropy benchmarking and Porter-Thomas verification
- **Hardware Optimization**: SIMD acceleration, memory-efficient algorithms, and GPU computing
- **SciRS2 Integration**: Advanced linear algebra operations using optimized BLAS/LAPACK
  - `scirs2_core::parallel_ops` for all parallel simulations
  - SciRS2 sparse matrix operations for efficient quantum state representation
  - SciRS2 eigensolvers for spectral analysis and VQE optimization
  - SciRS2-accelerated quantum Fourier transform implementation
- **Quantum Debugging**: Interactive debugging tools with breakpoints and state inspection
- **Performance Profiling**: Comprehensive benchmarking and optimization analysis

## Usage Examples

### Basic State Vector Simulation

```rust
use quantrs2_sim::prelude::*;

fn basic_simulation() -> Result<()> {
    // Create a Bell state circuit
    let mut simulator = StateVectorSimulator::new();
    
    // Apply gates directly
    simulator.h(0)?;
    simulator.cnot(0, 1)?;
    
    // Measure and get probabilities
    let probabilities = simulator.probabilities();
    for (state, prob) in probabilities.iter().enumerate() {
        println!("|{:02b}⟩: {:.6}", state, prob);
    }
    
    Ok(())
}
```

### Matrix Product State (MPS) Simulation

```rust
use quantrs2_sim::prelude::*;

fn mps_simulation() -> Result<()> {
    // MPS is memory-efficient for low-entanglement circuits
    let config = MPSConfig {
        max_bond_dimension: 64,
        compression_threshold: 1e-12,
    };
    
    let mut simulator = EnhancedMPSSimulator::new(10, config);
    
    // Create a low-entanglement circuit
    for i in 0..10 {
        simulator.h(i)?;
        if i > 0 {
            simulator.cnot(i-1, i)?;
        }
    }
    
    let result = simulator.measure_all(1000)?;
    println!("MPS simulation completed with bond dimension: {}", 
             simulator.max_bond_dimension());
    
    Ok(())
}
```

### Stabilizer Simulation for Clifford Circuits

```rust
use quantrs2_sim::prelude::*;

fn stabilizer_simulation() -> Result<()> {
    // Exponentially fast simulation for Clifford circuits
    let mut simulator = StabilizerSimulator::new(10);
    
    // Apply Clifford gates
    simulator.h(0)?;
    simulator.cnot(0, 1)?;
    simulator.s(1)?;
    
    // Check if circuit is still Clifford
    if simulator.is_clifford() {
        let measurements = simulator.measure_all(1000)?;
        println!("Clifford simulation: {} measurements", measurements.len());
    }
    
    Ok(())
}
```

### Quantum Monte Carlo Simulation

```rust
use quantrs2_sim::prelude::*;

fn quantum_monte_carlo() -> Result<()> {
    // Variational Monte Carlo for ground state estimation
    let hamiltonian = HamiltonianLibrary::heisenberg_chain(8, 1.0, 1.0, 1.0);
    let wave_function = WaveFunction::jastrow_gutzwiller(8);
    
    let mut vmc = VMC::new(hamiltonian, wave_function);
    let result = vmc.run(10000)?;
    
    println!("Ground state energy: {:.6} ± {:.6}", 
             result.energy, result.energy_error);
    
    Ok(())
}
```

### Automatic Differentiation VQE

```rust
use quantrs2_sim::prelude::*;

fn autodiff_vqe() -> Result<()> {
    // VQE with automatic gradient computation
    let hamiltonian = HamiltonianLibrary::hydrogen_molecule(1.4);
    let ansatz = ansatze::hardware_efficient_ansatz(4, 2);
    
    let mut vqe = VQEWithAutodiff::new(hamiltonian, ansatz);
    vqe.set_gradient_method(GradientMethod::ParameterShift);
    
    let result = vqe.optimize(100)?;
    println!("VQE converged to energy: {:.6} in {} iterations", 
             result.final_energy, result.iterations.len());
    
    Ok(())
}
```

### Noise Simulation and Error Mitigation

```rust
use quantrs2_sim::prelude::*;

fn noise_and_mitigation() -> Result<()> {
    // Create realistic noise model
    let noise_model = RealisticNoiseModelBuilder::new()
        .add_depolarizing_error(0.001)
        .add_thermal_relaxation(50e-6, 70e-6, 20e-9)
        .add_readout_error(0.02, 0.03)
        .build();
    
    let mut simulator = StateVectorSimulator::new_with_noise(noise_model);
    
    // Run circuit multiple times for statistics
    let mut results = Vec::new();
    for _ in 0..1000 {
        simulator.reset();
        simulator.h(0)?;
        simulator.cnot(0, 1)?;
        results.push(simulator.measure_all(1)?);
    }
    
    // Apply zero-noise extrapolation
    let extrapolator = ZeroNoiseExtrapolator::new();
    let mitigated_result = extrapolator.extrapolate(&results)?;
    
    println!("Mitigated expectation value: {:.6} ± {:.6}",
             mitigated_result.value, mitigated_result.error);
    
    Ok(())
}
```

### Quantum Supremacy Verification

```rust
use quantrs2_sim::prelude::*;

fn quantum_supremacy_verification() -> Result<()> {
    // Generate random quantum circuit for supremacy testing
    let params = VerificationParams {
        width: 20,
        depth: 20,
        gate_set: GateSet::GoogleSycamore,
    };
    
    let verifier = QuantumSupremacyVerifier::new(params);
    let circuit = verifier.generate_random_circuit()?;
    
    // Verify using cross-entropy benchmarking
    let ce_result = verifier.cross_entropy_benchmark(&circuit, 1000000)?;
    
    if ce_result.is_quantum_advantage() {
        println!("Quantum advantage detected! XEB = {:.6}", ce_result.xeb_value);
    }
    
    Ok(())
}
```

### Photonic Quantum Computing

```rust
use quantrs2_sim::prelude::*;

fn photonic_simulation() -> Result<()> {
    // Continuous variable photonic simulation
    let config = PhotonicConfig {
        cutoff_dimension: 10,
        method: PhotonicMethod::FockBasis,
    };
    
    let mut simulator = PhotonicSimulator::new(4, config);
    
    // Apply photonic operations
    simulator.displacement(0, Complex64::new(1.0, 0.5))?;
    simulator.squeezing(1, 0.5)?;
    simulator.beam_splitter(0, 1, 0.5)?;
    
    let result = simulator.measure_photon_number_distribution()?;
    println!("Photonic simulation completed with {} modes", result.modes);
    
    Ok(())
}
```

## Comprehensive Module Structure

### Core Simulation Engines
- **statevector.rs**: Dense state vector simulation with SIMD optimizations
- **enhanced_statevector.rs**: Enhanced state vector with lazy evaluation and memory optimization
- **mps_simulator.rs**: Basic matrix product state simulator
- **mps_basic.rs**: Lightweight MPS simulator for low-entanglement circuits
- **mps_enhanced.rs**: Advanced MPS with optimized contraction algorithms
- **stabilizer.rs**: Stabilizer formalism for exponentially fast Clifford simulation
- **clifford_sparse.rs**: Sparse representation of Clifford operations
- **decision_diagram.rs**: Quantum decision diagram symbolic simulation

### Advanced Simulation Methods
- **qmc.rs**: Quantum Monte Carlo methods (VMC, DMC, PIMC)
- **path_integral.rs**: Feynman path integral simulation techniques
- **trotter.rs**: Trotter-Suzuki decomposition for time evolution
- **photonic.rs**: Photonic quantum computing simulation (continuous variables)
- **fermionic_simulation.rs**: Second quantization with Jordan-Wigner transforms
- **open_quantum_systems.rs**: Lindblad master equation integration

### Error Correction & Noise
- **error_correction/**: Comprehensive quantum error correction codes
  - **codes.rs**: Surface, color, and concatenated codes
  - **mod.rs**: Error correction framework and utilities
- **noise.rs**: Basic noise models (bit-flip, phase-flip, depolarizing)
- **noise_advanced.rs**: Realistic device noise models with correlations
- **noise_extrapolation.rs**: Zero-noise extrapolation and error mitigation

### Optimization & Performance
- **optimized_simd.rs**: SIMD-accelerated quantum operations
- **optimized_chunked.rs**: Memory-efficient chunked processing
- **optimized_simple.rs**: Simplified optimized simulators
- **specialized_gates.rs**: Hardware-optimized gates
- **specialized_simulator.rs**: Simulator with specialized gate optimizations
- **fusion.rs**: Gate fusion optimization for reduced circuit depth
- **precision.rs**: Adaptive precision control for state vectors

### Variational & Machine Learning
- **autodiff_vqe.rs**: Automatic differentiation for variational quantum eigensolvers
- **pauli.rs**: Pauli string operations and expectation value computation

### Hardware Integration
- **gpu.rs**: GPU-accelerated simulation using WGPU compute shaders
- **gpu_linalg.rs**: GPU linear algebra operations for quantum simulation
- **scirs2_integration.rs**: Comprehensive SciRS2 backend integration
  - Unified parallel operations via `scirs2_core::parallel_ops`
  - SIMD vectorization through SciRS2 abstractions
  - Memory-efficient algorithms from `scirs2_core::memory_efficient`
  - Platform-aware optimization using `PlatformCapabilities`
- **scirs2_qft.rs**: SciRS2-accelerated quantum Fourier transform
- **scirs2_sparse.rs**: Sparse matrix operations using SciRS2
- **scirs2_eigensolvers.rs**: Spectral analysis and eigenvalue computations

### Verification & Benchmarking
- **quantum_supremacy.rs**: Cross-entropy benchmarking and Porter-Thomas verification
- **quantum_volume.rs**: IBM quantum volume protocol
- **benchmark.rs**: Performance benchmarking across different simulation methods
- **debugger.rs**: Interactive quantum debugging with breakpoints and state inspection

### Utility & Analysis
- **shot_sampling.rs**: Statistical sampling and measurement simulation
- **sparse.rs**: Sparse matrix representations for large quantum systems
- **linalg_ops.rs**: Linear algebra operations optimized for quantum simulation
- **utils.rs**: Common utilities and helper functions
- **tensor.rs**: Tensor manipulation utilities
- **tensor_network/**: Advanced tensor network contraction algorithms
  - **contraction.rs**: Optimal contraction order algorithms
  - **opt_contraction.rs**: Optimized contraction algorithms
  - **tensor.rs**: Tensor data structures and operations
- **dynamic.rs**: Dynamic qubit allocation and circuit optimization

## Feature Flags

- **default**: Optimized simulators with SIMD acceleration
- **gpu**: GPU acceleration using WGPU compute shaders (CUDA/OpenCL/Vulkan)
- **simd**: Platform-specific SIMD instructions for vectorized operations
- **optimize**: Advanced optimization algorithms and memory management
- **memory_efficient**: Large state vector optimizations for 25+ qubit simulation
- **advanced_math**: Full SciRS2 integration with optimized BLAS/LAPACK operations, parallel algorithms, and SIMD acceleration
- **mps**: Matrix Product State simulation with linear algebra support

## Performance Characteristics

### Simulation Capabilities
- **State Vector**: Up to 30+ qubits on standard hardware (16GB RAM)
- **MPS**: 50+ qubits for low-entanglement circuits with bond dimension control
- **Stabilizer**: Unlimited qubits for Clifford circuits (exponential speedup)
- **Decision Diagram**: Symbolic simulation with compact representations
- **GPU Acceleration**: 10-100x speedups for 20+ qubit circuits on compatible hardware

### Optimization Features
- **SIMD Vectorization**: AVX2/AVX-512 support for parallel complex arithmetic
- **Multi-threading**: Rayon-based parallelization across CPU cores
- **Memory Efficiency**: Chunked processing and lazy evaluation for large circuits
- **Gate Specialization**: Hardware-optimized processing for common gates
- **SciRS2 Integration**: Professional-grade linear algebra with Intel MKL/OpenBLAS

### Benchmarking Results
- Single-qubit gates: ~10ns per operation (SIMD optimized)
- Two-qubit gates: ~50ns per operation (depending on entanglement)
- GPU acceleration: 20-50x speedup for 25+ qubit state vector simulation
- MPS simulation: 100x memory reduction for product states

## Advanced Simulation Features

### Quantum Algorithm Support
- **VQE**: Automatic differentiation with gradient-based optimization
- **QAOA**: Optimized Hamiltonian evolution with Trotter decomposition
- **Quantum Monte Carlo**: Ground state estimation with statistical analysis
- **Quantum Supremacy**: Cross-entropy benchmarking with statistical verification

### Error Correction & Mitigation
- **Surface Codes**: Topological error correction with syndrome decoding
- **Zero-Noise Extrapolation**: Richardson extrapolation for error mitigation
- **Virtual Distillation**: Quantum error mitigation using symmetry verification
- **Process Tomography**: Complete characterization of quantum channels

### Hardware-Aware Simulation
- **Device Noise Models**: Realistic simulation matching hardware specifications
- **Calibration Integration**: Real-time hardware parameter updates
- **Cross-Platform GPU**: WGPU backend supporting CUDA, OpenCL, and Vulkan
- **Memory Hierarchy**: Cache-aware algorithms for optimal performance

## Integration with QuantRS2 Ecosystem

### Core Module Integration
- Leverages advanced gate decomposition algorithms for simulation optimization
- Uses optimized matrix representations from core for maximum performance
- Integrates quantum error correction codes for fault-tolerant simulation

### Circuit Module Integration
- Simulates circuits with automatic optimization pass selection
- Supports circuit compilation with hardware-aware gate translation
- Provides feedback for circuit optimization based on simulation performance

### Device Module Integration
- Accurately simulates real quantum hardware with calibrated noise models
- Provides device characterization through quantum volume and benchmarking
- Supports cloud quantum computer simulation with realistic latency models

### Machine Learning Module Integration
- Optimized simulation for variational quantum algorithms
- Automatic differentiation support for quantum neural network training
- Quantum kernel methods with efficient expectation value computation

## Research & Development Applications

### Quantum Computing Research
- Novel algorithm development with comprehensive simulation backends
- Quantum advantage verification with statistical confidence intervals
- Error correction threshold estimation with realistic noise models

### Quantum Chemistry & Physics
- Molecular simulation using fermionic operators and transformations
- Condensed matter physics with many-body quantum systems
- Quantum field theory simulation using path integral methods

### Quantum Machine Learning
- Quantum neural network training with automatic differentiation
- Quantum kernel methods for classical machine learning
- Variational quantum algorithm optimization and benchmarking

## License

This project is licensed under either:

- [Apache License, Version 2.0](../LICENSE-APACHE)
- [MIT License](../LICENSE-MIT)

at your option.