//! Quantum algorithm debugger interface.
//!
//! This module provides comprehensive debugging capabilities for quantum algorithms,
//! including step-by-step execution, state inspection, breakpoints, and analysis tools.

use scirs2_core::ndarray::{Array1, Array2};
use scirs2_core::Complex64;
use serde::{Deserialize, Serialize};
use std::collections::{HashMap, HashSet, VecDeque};
use std::sync::Arc;
use std::time::{Duration, Instant};

use crate::error::{Result, SimulatorError};
#[cfg(feature = "mps")]
use crate::mps_enhanced::{EnhancedMPS, MPSConfig};
use crate::statevector::StateVectorSimulator;
use quantrs2_circuit::builder::Circuit;
use quantrs2_core::gate::GateOp;

// Placeholder for MPSConfig when MPS feature is disabled
#[cfg(not(feature = "mps"))]
#[derive(Debug, Clone, Default)]
pub struct MPSConfig {
    pub max_bond_dim: usize,
    pub tolerance: f64,
}

/// Breakpoint condition types
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum BreakCondition {
    /// Break at specific gate index
    GateIndex(usize),
    /// Break when a qubit reaches a certain state
    QubitState { qubit: usize, state: bool },
    /// Break when entanglement entropy exceeds threshold
    EntanglementThreshold { cut: usize, threshold: f64 },
    /// Break when fidelity with target state drops below threshold
    FidelityThreshold {
        target_state: Vec<Complex64>,
        threshold: f64,
    },
    /// Break when a Pauli observable expectation value crosses threshold
    ObservableThreshold {
        observable: String,
        threshold: f64,
        direction: ThresholdDirection,
    },
    /// Break when circuit depth exceeds limit
    CircuitDepth(usize),
    /// Break when execution time exceeds limit
    ExecutionTime(Duration),
}

/// Threshold crossing direction
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum ThresholdDirection {
    Above,
    Below,
    Either,
}

/// Execution snapshot at a specific point
#[derive(Debug, Clone)]
pub struct ExecutionSnapshot {
    /// Gate index in the circuit
    pub gate_index: usize,
    /// Current quantum state
    pub state: Array1<Complex64>,
    /// Timestamp
    pub timestamp: Instant,
    /// Gate that was just executed (None for initial state)
    pub last_gate: Option<Arc<dyn GateOp + Send + Sync>>,
    /// Cumulative gate count by type
    pub gate_counts: HashMap<String, usize>,
    /// Entanglement entropies at different cuts
    pub entanglement_entropies: Vec<f64>,
    /// Circuit depth so far
    pub circuit_depth: usize,
}

/// Performance metrics during execution
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct PerformanceMetrics {
    /// Total execution time
    pub total_time: Duration,
    /// Time per gate type
    pub gate_times: HashMap<String, Duration>,
    /// Memory usage statistics
    pub memory_usage: MemoryUsage,
    /// Gate execution counts
    pub gate_counts: HashMap<String, usize>,
    /// Average entanglement entropy
    pub avg_entanglement: f64,
    /// Maximum entanglement entropy reached
    pub max_entanglement: f64,
    /// Number of snapshots taken
    pub snapshot_count: usize,
}

/// Memory usage tracking
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct MemoryUsage {
    /// Peak state vector memory (bytes)
    pub peak_statevector_memory: usize,
    /// Current MPS bond dimensions
    pub mps_bond_dims: Vec<usize>,
    /// Peak MPS memory (bytes)
    pub peak_mps_memory: usize,
    /// Debugger overhead (bytes)
    pub debugger_overhead: usize,
}

/// Watchpoint for monitoring specific properties
#[derive(Debug, Clone)]
pub struct Watchpoint {
    /// Unique identifier
    pub id: String,
    /// Description
    pub description: String,
    /// Property to watch
    pub property: WatchProperty,
    /// Logging frequency
    pub frequency: WatchFrequency,
    /// History of watched values
    pub history: VecDeque<(usize, f64)>, // (gate_index, value)
}

/// Properties that can be watched
#[derive(Debug, Clone)]
pub enum WatchProperty {
    /// Total probability (should be 1)
    Normalization,
    /// Entanglement entropy at specific cut
    EntanglementEntropy(usize),
    /// Expectation value of Pauli observable
    PauliExpectation(String),
    /// Fidelity with reference state
    Fidelity(Array1<Complex64>),
    /// Average gate fidelity
    GateFidelity,
    /// Circuit depth
    CircuitDepth,
    /// MPS bond dimension
    MPSBondDimension,
}

/// Watch frequency
#[derive(Debug, Clone)]
pub enum WatchFrequency {
    /// Watch at every gate
    EveryGate,
    /// Watch every N gates
    EveryNGates(usize),
    /// Watch at specific gate indices
    AtGates(HashSet<usize>),
}

/// Debugging session configuration
#[derive(Debug, Clone)]
pub struct DebugConfig {
    /// Whether to store full state snapshots
    pub store_snapshots: bool,
    /// Maximum number of snapshots to keep
    pub max_snapshots: usize,
    /// Whether to track performance metrics
    pub track_performance: bool,
    /// Whether to enable automatic state validation
    pub validate_state: bool,
    /// Entanglement entropy cut positions to monitor
    pub entropy_cuts: Vec<usize>,
    /// Use MPS representation for large systems
    pub use_mps: bool,
    /// MPS configuration if used
    pub mps_config: Option<MPSConfig>,
}

impl Default for DebugConfig {
    fn default() -> Self {
        Self {
            store_snapshots: true,
            max_snapshots: 100,
            track_performance: true,
            validate_state: true,
            entropy_cuts: vec![],
            use_mps: false,
            mps_config: None,
        }
    }
}

/// Main quantum algorithm debugger
pub struct QuantumDebugger<const N: usize> {
    /// Configuration
    config: DebugConfig,
    /// Current circuit being debugged
    circuit: Option<Circuit<N>>,
    /// Active breakpoints
    breakpoints: Vec<BreakCondition>,
    /// Active watchpoints
    watchpoints: HashMap<String, Watchpoint>,
    /// Execution snapshots
    snapshots: VecDeque<ExecutionSnapshot>,
    /// Performance metrics
    metrics: PerformanceMetrics,
    /// Current execution state
    execution_state: ExecutionState,
    /// State vector simulator
    simulator: StateVectorSimulator,
    /// MPS simulator (if enabled)
    #[cfg(feature = "mps")]
    mps_simulator: Option<EnhancedMPS>,
    /// Current gate index
    current_gate: usize,
    /// Execution start time
    start_time: Option<Instant>,
}

/// Current execution state
#[derive(Debug, Clone)]
enum ExecutionState {
    /// Not running
    Idle,
    /// Running normally
    Running,
    /// Paused at breakpoint
    Paused { reason: String },
    /// Finished execution
    Finished,
    /// Error occurred
    Error { message: String },
}

impl<const N: usize> QuantumDebugger<N> {
    /// Create a new quantum debugger
    pub fn new(config: DebugConfig) -> Result<Self> {
        let simulator = StateVectorSimulator::new();

        #[cfg(feature = "mps")]
        let mps_simulator = if config.use_mps {
            Some(EnhancedMPS::new(
                N,
                config.mps_config.clone().unwrap_or_default(),
            ))
        } else {
            None
        };

        Ok(Self {
            config,
            circuit: None,
            breakpoints: Vec::new(),
            watchpoints: HashMap::new(),
            snapshots: VecDeque::new(),
            metrics: PerformanceMetrics {
                total_time: Duration::new(0, 0),
                gate_times: HashMap::new(),
                memory_usage: MemoryUsage {
                    peak_statevector_memory: 0,
                    mps_bond_dims: vec![],
                    peak_mps_memory: 0,
                    debugger_overhead: 0,
                },
                gate_counts: HashMap::new(),
                avg_entanglement: 0.0,
                max_entanglement: 0.0,
                snapshot_count: 0,
            },
            execution_state: ExecutionState::Idle,
            simulator,
            #[cfg(feature = "mps")]
            mps_simulator,
            current_gate: 0,
            start_time: None,
        })
    }

    /// Load a circuit for debugging
    pub fn load_circuit(&mut self, circuit: Circuit<N>) -> Result<()> {
        self.circuit = Some(circuit);
        self.reset();
        Ok(())
    }

    /// Reset debugger state
    pub fn reset(&mut self) {
        self.snapshots.clear();
        self.metrics = PerformanceMetrics {
            total_time: Duration::new(0, 0),
            gate_times: HashMap::new(),
            memory_usage: MemoryUsage {
                peak_statevector_memory: 0,
                mps_bond_dims: vec![],
                peak_mps_memory: 0,
                debugger_overhead: 0,
            },
            gate_counts: HashMap::new(),
            avg_entanglement: 0.0,
            max_entanglement: 0.0,
            snapshot_count: 0,
        };
        self.execution_state = ExecutionState::Idle;
        self.current_gate = 0;
        self.start_time = None;

        // Reset simulator to |0...0> state
        self.simulator = StateVectorSimulator::new();
        #[cfg(feature = "mps")]
        if let Some(ref mut mps) = self.mps_simulator {
            *mps = EnhancedMPS::new(N, self.config.mps_config.clone().unwrap_or_default());
        }

        // Clear watchpoint histories
        for watchpoint in self.watchpoints.values_mut() {
            watchpoint.history.clear();
        }
    }

    /// Add a breakpoint
    pub fn add_breakpoint(&mut self, condition: BreakCondition) {
        self.breakpoints.push(condition);
    }

    /// Remove a breakpoint
    pub fn remove_breakpoint(&mut self, index: usize) -> Result<()> {
        if index >= self.breakpoints.len() {
            return Err(SimulatorError::IndexOutOfBounds(index));
        }
        self.breakpoints.remove(index);
        Ok(())
    }

    /// Add a watchpoint
    pub fn add_watchpoint(&mut self, watchpoint: Watchpoint) {
        self.watchpoints.insert(watchpoint.id.clone(), watchpoint);
    }

    /// Remove a watchpoint
    pub fn remove_watchpoint(&mut self, id: &str) -> Result<()> {
        if self.watchpoints.remove(id).is_none() {
            return Err(SimulatorError::InvalidInput(format!(
                "Watchpoint '{}' not found",
                id
            )));
        }
        Ok(())
    }

    /// Execute the circuit step by step
    pub fn step(&mut self) -> Result<StepResult> {
        let circuit = self
            .circuit
            .as_ref()
            .ok_or_else(|| SimulatorError::InvalidOperation("No circuit loaded".to_string()))?;

        if self.current_gate >= circuit.gates().len() {
            self.execution_state = ExecutionState::Finished;
            return Ok(StepResult::Finished);
        }

        // Check if we're paused
        if let ExecutionState::Paused { .. } = self.execution_state {
            // Continue from pause
            self.execution_state = ExecutionState::Running;
        }

        // Start timing if first step
        if self.start_time.is_none() {
            self.start_time = Some(Instant::now());
            self.execution_state = ExecutionState::Running;
        }

        // Get gate information before borrowing mutably
        let gate_name = circuit.gates()[self.current_gate].name().to_string();
        let total_gates = circuit.gates().len();

        // Execute the current gate
        let gate_start = Instant::now();

        // Apply gate to appropriate simulator
        #[cfg(feature = "mps")]
        if let Some(ref mut mps) = self.mps_simulator {
            mps.apply_gate(circuit.gates()[self.current_gate].as_ref())?;
        } else {
            // For now, we'll use a simplified approach for the state vector simulator
            // In practice, this would need proper integration with the actual simulator
        }

        #[cfg(not(feature = "mps"))]
        {
            // For now, we'll use a simplified approach for the state vector simulator
            // In practice, this would need proper integration with the actual simulator
        }

        let gate_time = gate_start.elapsed();

        // Update metrics
        *self
            .metrics
            .gate_times
            .entry(gate_name.clone())
            .or_insert(Duration::new(0, 0)) += gate_time;
        *self.metrics.gate_counts.entry(gate_name).or_insert(0) += 1;

        // Check watchpoints
        self.update_watchpoints()?;

        // Take snapshot if configured
        if self.config.store_snapshots {
            self.take_snapshot()?;
        }

        // Check breakpoints
        if let Some(reason) = self.check_breakpoints()? {
            self.execution_state = ExecutionState::Paused {
                reason: reason.clone(),
            };
            return Ok(StepResult::BreakpointHit { reason });
        }

        self.current_gate += 1;

        if self.current_gate >= total_gates {
            self.execution_state = ExecutionState::Finished;
            if let Some(start) = self.start_time {
                self.metrics.total_time = start.elapsed();
            }
            Ok(StepResult::Finished)
        } else {
            Ok(StepResult::Continue)
        }
    }

    /// Run until next breakpoint or completion
    pub fn run(&mut self) -> Result<StepResult> {
        loop {
            match self.step()? {
                StepResult::Continue => continue,
                result => return Ok(result),
            }
        }
    }

    /// Get current quantum state
    pub fn get_current_state(&self) -> Result<Array1<Complex64>> {
        #[cfg(feature = "mps")]
        if let Some(ref mps) = self.mps_simulator {
            return mps
                .to_statevector()
                .map_err(|e| SimulatorError::UnsupportedOperation(format!("MPS error: {}", e)));
        }

        // Return the state from the state vector simulator
        // For now, return a dummy state
        Ok(Array1::zeros(1 << N))
    }

    /// Get entanglement entropy at specified cut
    pub fn get_entanglement_entropy(&self, cut: usize) -> Result<f64> {
        #[cfg(feature = "mps")]
        if self.mps_simulator.is_some() {
            // For now, return dummy value due to borrow checker issues
            // In a real implementation, would need proper state management
            return Ok(0.0);
        }

        // Compute entanglement entropy from state vector
        let state = self.get_current_state()?;
        compute_entanglement_entropy(&state, cut, N)
    }

    /// Get expectation value of Pauli observable
    pub fn get_pauli_expectation(&self, pauli_string: &str) -> Result<Complex64> {
        #[cfg(feature = "mps")]
        if let Some(ref mps) = self.mps_simulator {
            return mps
                .expectation_value_pauli(pauli_string)
                .map_err(|e| SimulatorError::UnsupportedOperation(format!("MPS error: {}", e)));
        }

        let state = self.get_current_state()?;
        compute_pauli_expectation(&state, pauli_string)
    }

    /// Get performance metrics
    pub fn get_metrics(&self) -> &PerformanceMetrics {
        &self.metrics
    }

    /// Get all snapshots
    pub fn get_snapshots(&self) -> &VecDeque<ExecutionSnapshot> {
        &self.snapshots
    }

    /// Get watchpoint by ID
    pub fn get_watchpoint(&self, id: &str) -> Option<&Watchpoint> {
        self.watchpoints.get(id)
    }

    /// Get all watchpoints
    pub fn get_watchpoints(&self) -> &HashMap<String, Watchpoint> {
        &self.watchpoints
    }

    /// Check if execution is finished
    pub fn is_finished(&self) -> bool {
        matches!(self.execution_state, ExecutionState::Finished)
    }

    /// Check if execution is paused
    pub fn is_paused(&self) -> bool {
        matches!(self.execution_state, ExecutionState::Paused { .. })
    }

    /// Get current execution state
    pub fn get_execution_state(&self) -> &ExecutionState {
        &self.execution_state
    }

    /// Generate debugging report
    pub fn generate_report(&self) -> DebugReport {
        DebugReport {
            circuit_summary: self.circuit.as_ref().map(|c| CircuitSummary {
                total_gates: c.gates().len(),
                gate_types: self.metrics.gate_counts.clone(),
                estimated_depth: estimate_circuit_depth(c),
            }),
            performance: self.metrics.clone(),
            entanglement_analysis: self.analyze_entanglement(),
            state_analysis: self.analyze_state(),
            recommendations: self.generate_recommendations(),
        }
    }

    // Private helper methods

    fn take_snapshot(&mut self) -> Result<()> {
        if self.snapshots.len() >= self.config.max_snapshots {
            self.snapshots.pop_front();
        }

        let circuit = self.circuit.as_ref().unwrap();
        let state = self.get_current_state()?;

        let snapshot = ExecutionSnapshot {
            gate_index: self.current_gate,
            state,
            timestamp: Instant::now(),
            last_gate: if self.current_gate > 0 {
                Some(circuit.gates()[self.current_gate - 1].clone())
            } else {
                None
            },
            gate_counts: self.metrics.gate_counts.clone(),
            entanglement_entropies: self.compute_all_entanglement_entropies()?,
            circuit_depth: self.current_gate, // Simplified
        };

        self.snapshots.push_back(snapshot);
        self.metrics.snapshot_count += 1;
        Ok(())
    }

    fn check_breakpoints(&self) -> Result<Option<String>> {
        for breakpoint in &self.breakpoints {
            match breakpoint {
                BreakCondition::GateIndex(target) => {
                    if self.current_gate == *target {
                        return Ok(Some(format!("Reached gate index {}", target)));
                    }
                }
                BreakCondition::EntanglementThreshold { cut, threshold } => {
                    let entropy = self.get_entanglement_entropy(*cut)?;
                    if entropy > *threshold {
                        return Ok(Some(format!(
                            "Entanglement entropy {:.4} > {:.4} at cut {}",
                            entropy, threshold, cut
                        )));
                    }
                }
                BreakCondition::ObservableThreshold {
                    observable,
                    threshold,
                    direction,
                } => {
                    let expectation = self.get_pauli_expectation(observable)?.re;
                    let hit = match direction {
                        ThresholdDirection::Above => expectation > *threshold,
                        ThresholdDirection::Below => expectation < *threshold,
                        ThresholdDirection::Either => (expectation - threshold).abs() < 1e-10,
                    };
                    if hit {
                        return Ok(Some(format!(
                            "Observable {} = {:.4} crossed threshold {:.4}",
                            observable, expectation, threshold
                        )));
                    }
                }
                _ => {
                    // Other breakpoint types would be implemented here
                }
            }
        }
        Ok(None)
    }

    fn update_watchpoints(&mut self) -> Result<()> {
        let current_gate = self.current_gate;

        // Collect watchpoint updates to avoid borrowing issues
        let mut updates = Vec::new();

        for (id, watchpoint) in &self.watchpoints {
            let should_update = match &watchpoint.frequency {
                WatchFrequency::EveryGate => true,
                WatchFrequency::EveryNGates(n) => current_gate % n == 0,
                WatchFrequency::AtGates(gates) => gates.contains(&current_gate),
            };

            if should_update {
                let value = match &watchpoint.property {
                    WatchProperty::EntanglementEntropy(cut) => {
                        self.get_entanglement_entropy(*cut)?
                    }
                    WatchProperty::PauliExpectation(observable) => {
                        self.get_pauli_expectation(observable)?.re
                    }
                    WatchProperty::Normalization => {
                        let state = self.get_current_state()?;
                        state.iter().map(|x| x.norm_sqr()).sum::<f64>()
                    }
                    _ => 0.0, // Other properties would be implemented
                };

                updates.push((id.clone(), current_gate, value));
            }
        }

        // Apply updates
        for (id, gate, value) in updates {
            if let Some(watchpoint) = self.watchpoints.get_mut(&id) {
                watchpoint.history.push_back((gate, value));

                // Keep history size manageable
                if watchpoint.history.len() > 1000 {
                    watchpoint.history.pop_front();
                }
            }
        }

        Ok(())
    }

    fn compute_all_entanglement_entropies(&self) -> Result<Vec<f64>> {
        let mut entropies = Vec::new();
        for &cut in &self.config.entropy_cuts {
            if cut < N - 1 {
                entropies.push(self.get_entanglement_entropy(cut)?);
            }
        }
        Ok(entropies)
    }

    fn analyze_entanglement(&self) -> EntanglementAnalysis {
        // Analyze entanglement patterns from snapshots and watchpoints
        EntanglementAnalysis {
            max_entropy: self.metrics.max_entanglement,
            avg_entropy: self.metrics.avg_entanglement,
            entropy_evolution: Vec::new(), // Would be filled from watchpoint histories
        }
    }

    fn analyze_state(&self) -> StateAnalysis {
        // Analyze quantum state properties
        StateAnalysis {
            is_separable: false,      // Would compute this
            schmidt_rank: 1,          // Would compute this
            participation_ratio: 1.0, // Would compute this
        }
    }

    fn generate_recommendations(&self) -> Vec<String> {
        let mut recommendations = Vec::new();

        // Analyze performance and suggest optimizations
        if self.metrics.max_entanglement > 3.0 {
            recommendations.push(
                "High entanglement detected. Consider using MPS simulation for better scaling."
                    .to_string(),
            );
        }

        if self.metrics.gate_counts.get("CNOT").unwrap_or(&0) > &50 {
            recommendations
                .push("Many CNOT gates detected. Consider gate optimization.".to_string());
        }

        recommendations
    }
}

/// Result of a debugging step
#[derive(Debug, Clone)]
pub enum StepResult {
    /// Continue execution
    Continue,
    /// Breakpoint was hit
    BreakpointHit { reason: String },
    /// Execution finished
    Finished,
}

/// Circuit summary for debugging
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct CircuitSummary {
    pub total_gates: usize,
    pub gate_types: HashMap<String, usize>,
    pub estimated_depth: usize,
}

/// Entanglement analysis results
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct EntanglementAnalysis {
    pub max_entropy: f64,
    pub avg_entropy: f64,
    pub entropy_evolution: Vec<(usize, f64)>,
}

/// State analysis results
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct StateAnalysis {
    pub is_separable: bool,
    pub schmidt_rank: usize,
    pub participation_ratio: f64,
}

/// Complete debugging report
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct DebugReport {
    pub circuit_summary: Option<CircuitSummary>,
    pub performance: PerformanceMetrics,
    pub entanglement_analysis: EntanglementAnalysis,
    pub state_analysis: StateAnalysis,
    pub recommendations: Vec<String>,
}

// Helper functions

/// Compute entanglement entropy from state vector
fn compute_entanglement_entropy(
    state: &Array1<Complex64>,
    cut: usize,
    num_qubits: usize,
) -> Result<f64> {
    if cut >= num_qubits - 1 {
        return Err(SimulatorError::IndexOutOfBounds(cut));
    }

    let left_dim = 1 << cut;
    let right_dim = 1 << (num_qubits - cut);

    // Reshape state into matrix
    let state_matrix =
        Array2::from_shape_vec((left_dim, right_dim), state.to_vec()).map_err(|_| {
            SimulatorError::DimensionMismatch("Invalid state vector dimension".to_string())
        })?;

    // Compute SVD
    // For now, return dummy value - proper implementation would use ndarray-linalg
    Ok(0.0)
}

/// Compute Pauli expectation value from state vector
fn compute_pauli_expectation(state: &Array1<Complex64>, pauli_string: &str) -> Result<Complex64> {
    // Simplified implementation - would need proper Pauli string evaluation
    Ok(Complex64::new(0.0, 0.0))
}

/// Estimate circuit depth
fn estimate_circuit_depth<const N: usize>(circuit: &Circuit<N>) -> usize {
    // Simplified depth estimation - would need proper dependency analysis
    circuit.gates().len()
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_debugger_creation() {
        let config = DebugConfig::default();
        let debugger: QuantumDebugger<3> = QuantumDebugger::new(config).unwrap();
        assert!(matches!(debugger.execution_state, ExecutionState::Idle));
    }

    #[test]
    fn test_breakpoint_management() {
        let config = DebugConfig::default();
        let mut debugger: QuantumDebugger<3> = QuantumDebugger::new(config).unwrap();

        debugger.add_breakpoint(BreakCondition::GateIndex(5));
        assert_eq!(debugger.breakpoints.len(), 1);

        debugger.remove_breakpoint(0).unwrap();
        assert_eq!(debugger.breakpoints.len(), 0);
    }

    #[test]
    fn test_watchpoint_management() {
        let config = DebugConfig::default();
        let mut debugger: QuantumDebugger<3> = QuantumDebugger::new(config).unwrap();

        let watchpoint = Watchpoint {
            id: "test".to_string(),
            description: "Test watchpoint".to_string(),
            property: WatchProperty::Normalization,
            frequency: WatchFrequency::EveryGate,
            history: VecDeque::new(),
        };

        debugger.add_watchpoint(watchpoint);
        assert!(debugger.get_watchpoint("test").is_some());

        debugger.remove_watchpoint("test").unwrap();
        assert!(debugger.get_watchpoint("test").is_none());
    }
}
