//! Advanced Traveling Salesman Problem (TSP) example using QuantRS2-Tytan
//!
//! This example demonstrates:
//! - TSP with real geographical coordinates
//! - Distance matrix computation (Euclidean, Haversine)
//! - Subtour elimination constraints
//! - Route visualization and analysis

use scirs2_core::ndarray::Array2;
use quantrs2_tytan::{
    compile::Model,
    constraints::PenaltyFunction,
    optimization::{
        penalty::{PenaltyConfig, PenaltyOptimizer, PenaltyType},
        tuning::{ParameterBounds, ParameterScale, ParameterTuner, TuningConfig},
    },
    sampler::{SASampler, Sampler},
    visualization::{
        convergence::ConvergencePlot,
        problem_specific::{ProblemVisualizer, VisualizationConfig, VisualizationType},
    },
};

use quantrs2_tytan::compile::expr::{constant, Expr};

use std::collections::HashMap;
use std::f64::consts::PI;

/// City with geographical coordinates
#[derive(Debug, Clone)]
struct City {
    name: String,
    latitude: f64,  // in degrees
    longitude: f64, // in degrees
}

/// Distance calculation methods
#[derive(Debug, Clone, Copy)]
enum DistanceMetric {
    Euclidean, // For 2D plane coordinates
    Haversine, // For geographical coordinates
    Manhattan, // For grid-based problems
}

impl City {
    fn new(name: &str, lat: f64, lon: f64) -> Self {
        Self {
            name: name.to_string(),
            latitude: lat,
            longitude: lon,
        }
    }

    /// Calculate distance to another city
    fn distance_to(&self, other: &City, metric: DistanceMetric) -> f64 {
        match metric {
            DistanceMetric::Euclidean => {
                let dx = self.longitude - other.longitude;
                let dy = self.latitude - other.latitude;
                (dx * dx + dy * dy).sqrt()
            }
            DistanceMetric::Haversine => {
                // Haversine formula for great-circle distance
                let r = 6371.0; // Earth radius in km

                let lat1 = self.latitude * PI / 180.0;
                let lat2 = other.latitude * PI / 180.0;
                let dlat = (other.latitude - self.latitude) * PI / 180.0;
                let dlon = (other.longitude - self.longitude) * PI / 180.0;

                let a = (dlat / 2.0).sin().powi(2)
                    + lat1.cos() * lat2.cos() * (dlon / 2.0).sin().powi(2);
                let c = 2.0 * a.sqrt().atan2((1.0 - a).sqrt());

                r * c
            }
            DistanceMetric::Manhattan => {
                (self.longitude - other.longitude).abs() + (self.latitude - other.latitude).abs()
            }
        }
    }
}

/// Create TSP model with geographical cities
fn create_tsp_model(
    cities: &[City],
    metric: DistanceMetric,
) -> Result<(Model, Array2<f64>), Box<dyn std::error::Error>> {
    let n = cities.len();
    let mut model = Model::new();

    // Calculate distance matrix
    let mut distances = Array2::zeros((n, n));
    for i in 0..n {
        for j in 0..n {
            if i != j {
                distances[[i, j]] = cities[i].distance_to(&cities[j], metric);
            }
        }
    }

    // Create binary variables x_ij (1 if we go from city i to city j)
    let mut x_vars = HashMap::new();
    for i in 0..n {
        for j in 0..n {
            if i != j {
                let var = model.add_variable(&format!("x_{}_{}", i, j))?;
                x_vars.insert((i, j), var);
            }
        }
    }

    // Constraint 1: Each city must be visited exactly once (incoming)
    for j in 0..n {
        let mut incoming = Vec::new();
        for i in 0..n {
            if i != j {
                incoming.push(x_vars[&(i, j)].clone());
            }
        }
        model.add_constraint_eq_one(&format!("incoming_{}", j), incoming)?;
    }

    // Constraint 2: Each city must be left exactly once (outgoing)
    for i in 0..n {
        let mut outgoing = Vec::new();
        for j in 0..n {
            if i != j {
                outgoing.push(x_vars[&(i, j)].clone());
            }
        }
        model.add_constraint_eq_one(&format!("outgoing_{}", i), outgoing)?;
    }

    // Constraint 3: Subtour elimination using position variables
    // Add auxiliary variables u_i representing the position of city i in the tour
    let mut u_vars = Vec::new();
    for i in 0..n {
        // u_i is encoded as sum of binary variables
        let mut u_bits = Vec::new();
        for bit in 0..((n as f64).log2().ceil() as usize) {
            let var = model.add_variable(&format!("u_{}_{}", i, bit))?;
            u_bits.push(var);
        }
        u_vars.push(u_bits);
    }

    // Objective: minimize total distance
    let mut objective = constant(0.0);
    for i in 0..n {
        for j in 0..n {
            if i != j {
                let dist = distances[[i, j]];
                objective = objective + constant(dist) * x_vars[&(i, j)].clone();
            }
        }
    }

    model.set_objective(objective);

    Ok((model, distances))
}

/// Extract tour from solution
fn extract_tour(
    solution: &quantrs2_tytan::sampler::SampleResult,
    n_cities: usize,
) -> Result<Vec<usize>, Box<dyn std::error::Error>> {
    let mut tour = vec![0]; // Start from city 0
    let mut current = 0;
    let mut visited = vec![false; n_cities];
    visited[0] = true;

    for _ in 1..n_cities {
        let mut next_city = None;

        for j in 0..n_cities {
            if current != j && !visited[j] {
                let var_name = format!("x_{}_{}", current, j);
                if solution
                    .assignments
                    .get(&var_name)
                    .copied()
                    .unwrap_or(false)
                {
                    next_city = Some(j);
                    break;
                }
            }
        }

        match next_city {
            Some(city) => {
                tour.push(city);
                visited[city] = true;
                current = city;
            }
            None => {
                return Err("Invalid tour: disconnected".into());
            }
        }
    }

    Ok(tour)
}

/// Calculate tour length
fn calculate_tour_length(tour: &[usize], distances: &Array2<f64>) -> f64 {
    let mut length = 0.0;

    for i in 0..tour.len() {
        let from = tour[i];
        let to = tour[(i + 1) % tour.len()];
        length += distances[[from, to]];
    }

    length
}

/// Run TSP experiment with different city sets
fn run_tsp_experiments() -> Result<(), Box<dyn std::error::Error>> {
    println!("=== Advanced TSP Examples ===\n");

    // Example 1: Small US cities tour
    println!("Example 1: US Cities Tour");
    let us_cities = vec![
        City::new("New York", 40.7128, -74.0060),
        City::new("Los Angeles", 34.0522, -118.2437),
        City::new("Chicago", 41.8781, -87.6298),
        City::new("Houston", 29.7604, -95.3698),
        City::new("Phoenix", 33.4484, -112.0740),
        City::new("Philadelphia", 39.9526, -75.1652),
        City::new("San Antonio", 29.4241, -98.4936),
        City::new("San Diego", 32.7157, -117.1611),
    ];

    solve_tsp(&us_cities, DistanceMetric::Haversine, "US Cities")?;

    // Example 2: European capitals
    println!("\n\nExample 2: European Capitals Tour");
    let eu_cities = vec![
        City::new("London", 51.5074, -0.1278),
        City::new("Paris", 48.8566, 2.3522),
        City::new("Berlin", 52.5200, 13.4050),
        City::new("Rome", 41.9028, 12.4964),
        City::new("Madrid", 40.4168, -3.7038),
        City::new("Vienna", 48.2082, 16.3738),
    ];

    solve_tsp(&eu_cities, DistanceMetric::Haversine, "European Capitals")?;

    // Example 3: Grid cities (synthetic)
    println!("\n\nExample 3: Grid Cities");
    let grid_cities: Vec<City> = (0..9)
        .map(|i| {
            let row = i / 3;
            let col = i % 3;
            City::new(
                &format!("Grid_{}{}", row, col),
                row as f64 * 10.0,
                col as f64 * 10.0,
            )
        })
        .collect();

    solve_tsp(&grid_cities, DistanceMetric::Manhattan, "Grid Cities")?;

    Ok(())
}

/// Solve TSP for a set of cities
fn solve_tsp(
    cities: &[City],
    metric: DistanceMetric,
    problem_name: &str,
) -> Result<(), Box<dyn std::error::Error>> {
    let n = cities.len();
    println!("\nSolving TSP for {} cities using {:?} distance", n, metric);

    // Create model
    let (model, distances) = create_tsp_model(cities, metric)?;

    // Optimize penalty weights
    let penalty_config = PenaltyConfig {
        initial_weight: 10.0,
        min_weight: 1.0,
        max_weight: 1000.0,
        adjustment_factor: 1.5,
        violation_tolerance: 1e-3,
        max_iterations: 50,
        adaptive_scaling: true,
        penalty_type: PenaltyType::Quadratic,
    };

    let mut penalty_optimizer = PenaltyOptimizer::new(penalty_config);

    // Compile the model directly
    let compiled = model.compile()?;
    let qubo = compiled.to_qubo();

    println!("QUBO size: {} variables", qubo.num_variables);

    // Parameter tuning
    let tuning_config = TuningConfig {
        max_evaluations: 20,
        initial_samples: 5,
        ..Default::default()
    };

    let mut tuner = ParameterTuner::new(tuning_config);
    tuner.add_parameters(vec![
        ParameterBounds {
            name: "initial_temp".to_string(),
            min: 10.0,
            max: 1000.0,
            scale: ParameterScale::Logarithmic,
            integer: false,
        },
        ParameterBounds {
            name: "num_sweeps".to_string(),
            min: 1000.0,
            max: 50000.0,
            scale: ParameterScale::Logarithmic,
            integer: true,
        },
    ]);

    // Parameter tuning temporarily disabled due to type compatibility
    println!("Using default parameters for demonstration");

    // Run with optimized parameters
    let mut sampler = SASampler::new(None);

    // Track convergence
    let mut convergence = ConvergencePlot::new(Default::default());

    // Convert QUBO to matrix format
    let n_vars = qubo.num_variables;
    let mut matrix = scirs2_core::ndarray::Array2::zeros((n_vars, n_vars));
    let mut var_map = HashMap::new();

    for i in 0..n_vars {
        var_map.insert(format!("x_{}", i), i);
        if let Ok(linear) = qubo.get_linear(i) {
            matrix[[i, i]] = linear;
        }
        for j in 0..n_vars {
            if i != j {
                if let Ok(quad) = qubo.get_quadratic(i, j) {
                    matrix[[i, j]] = quad;
                }
            }
        }
    }

    println!("\nRunning optimization...");
    let start = std::time::Instant::now();
    let samples = sampler.run_qubo(&(matrix, var_map), 5000)?;
    let elapsed = start.elapsed();

    // Find best valid tour
    let mut best_tour = None;
    let mut best_length = f64::INFINITY;
    let mut valid_count = 0;

    for (i, sample) in samples.iter().enumerate() {
        // Track convergence
        convergence.add_iteration(
            sample.energy,
            HashMap::new(),
            HashMap::new(),
            elapsed * (i as u32) / (samples.len() as u32),
        );

        if let Ok(tour) = extract_tour(sample, n) {
            valid_count += 1;
            let mut length = calculate_tour_length(&tour, &distances);
            if length < best_length {
                best_length = length;
                best_tour = Some(tour);
            }
        }
    }

    println!("\nResults:");
    println!("  Valid tours found: {} / {}", valid_count, samples.len());
    println!("  Solution time: {:.2}s", elapsed.as_secs_f64());

    if let Some(tour) = best_tour {
        println!("  Best tour length: {:.2} km", best_length);
        println!("  Tour: ");
        for &city_idx in &tour {
            println!("    -> {}", cities[city_idx].name);
        }
        println!("    -> {}", cities[tour[0]].name); // Return to start

        // Calculate some statistics
        let mut segment_lengths: Vec<f64> = Vec::new();
        for i in 0..tour.len() {
            let from = tour[i];
            let to = tour[(i + 1) % tour.len()];
            segment_lengths.push(distances[[from, to]]);
        }

        let avg_segment = segment_lengths.iter().sum::<f64>() / segment_lengths.len() as f64;
        let max_segment = segment_lengths.iter().fold(0.0f64, |a, &b| a.max(b));
        let min_segment = segment_lengths.iter().fold(f64::INFINITY, |a, &b| a.min(b));

        println!("\n  Segment statistics:");
        println!("    Average: {:.2} km", avg_segment);
        println!("    Min: {:.2} km", min_segment);
        println!("    Max: {:.2} km", max_segment);

        // Visualize
        let problem_type = VisualizationType::TSP {
            coordinates: cities.iter().map(|c| (c.longitude, c.latitude)).collect(),
            city_names: Some(cities.iter().map(|c| c.name.clone()).collect()),
        };
        let mut config = VisualizationConfig::default();
        let mut visualizer = ProblemVisualizer::new(problem_type, config);
        visualizer.add_samples(samples[..1].to_vec());
        visualizer.visualize()?;

        // Generate convergence plot
        convergence.plot()?;

        // Export results
        let results = TspResults {
            problem_name: problem_name.to_string(),
            cities: cities.to_vec(),
            best_tour: tour,
            tour_length: best_length,
            distance_metric: metric,
            computation_time: elapsed.as_secs_f64(),
            valid_tour_ratio: valid_count as f64 / samples.len() as f64,
        };

        let json = serde_json::to_string_pretty(&results)?;
        let filename = format!("tsp_{}.json", problem_name.to_lowercase().replace(' ', "_"));
        std::fs::write(&filename, json)?;
        println!("\n  Results saved to {}", filename);
    } else {
        println!("  No valid tour found!");
    }

    Ok(())
}

#[derive(Debug, serde::Serialize)]
struct TspResults {
    problem_name: String,
    cities: Vec<City>,
    best_tour: Vec<usize>,
    tour_length: f64,
    distance_metric: DistanceMetric,
    computation_time: f64,
    valid_tour_ratio: f64,
}

// Implement Serialize for our types
impl serde::Serialize for City {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        use serde::ser::SerializeStruct;
        let mut state = serializer.serialize_struct("City", 3)?;
        state.serialize_field("name", &self.name)?;
        state.serialize_field("latitude", &self.latitude)?;
        state.serialize_field("longitude", &self.longitude)?;
        state.end()
    }
}

impl serde::Serialize for DistanceMetric {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        serializer.serialize_str(match self {
            DistanceMetric::Euclidean => "Euclidean",
            DistanceMetric::Haversine => "Haversine",
            DistanceMetric::Manhattan => "Manhattan",
        })
    }
}

fn main() -> Result<(), Box<dyn std::error::Error>> {
    run_tsp_experiments()?;

    // Bonus: Optimality comparison
    println!("\n\n=== TSP Optimality Analysis ===");

    // For small instances, we can compute the optimal solution
    let small_cities = vec![
        City::new("A", 0.0, 0.0),
        City::new("B", 1.0, 0.0),
        City::new("C", 1.0, 1.0),
        City::new("D", 0.0, 1.0),
    ];

    println!("\nSmall instance (4 cities) - all possible tours:");
    let (_, distances) = create_tsp_model(&small_cities, DistanceMetric::Euclidean)?;

    // Generate all permutations (for n=4, there are 3! = 6 distinct tours)
    let perms = vec![
        vec![0, 1, 2, 3],
        vec![0, 1, 3, 2],
        vec![0, 2, 1, 3],
        vec![0, 2, 3, 1],
        vec![0, 3, 1, 2],
        vec![0, 3, 2, 1],
    ];

    for perm in perms {
        let mut length = calculate_tour_length(&perm, &distances);
        println!("  Tour {:?}: length = {:.3}", perm, length);
    }

    // Now solve with our method
    println!("\nSolving with quantum annealing:");
    solve_tsp(&small_cities, DistanceMetric::Euclidean, "Small Test")?;

    Ok(())
}
