import abc
import datetime
import typing

import QuantConnect
import QuantConnect.Algorithm.Framework.Portfolio
import QuantConnect.Brokerages
import QuantConnect.Data
import QuantConnect.Data.Fundamental
import QuantConnect.Data.Market
import QuantConnect.Data.UniverseSelection
import QuantConnect.Indicators
import QuantConnect.Interfaces
import QuantConnect.Orders
import QuantConnect.Orders.Fees
import QuantConnect.Orders.Fills
import QuantConnect.Orders.Slippage
import QuantConnect.Securities
import QuantConnect.Securities.Interfaces
import QuantConnect.Securities.Volatility
import System
import System.Collections
import System.Collections.Concurrent
import System.Collections.Generic

QuantConnect_Securities_MarketHoursDatabase = typing.Any
System_EventHandler = typing.Any
IDynamicMetaObjectProvider = typing.Any
INotifyCollectionChanged = typing.Any
QuantConnect_Securities_SecurityDatabaseKey = typing.Any
QuantConnect_Securities_OptionFilterUniverse = typing.Any
QuantConnect_Securities_FutureFilterUniverse = typing.Any

QuantConnect_Securities_DynamicSecurityData_Get_T = typing.TypeVar("QuantConnect_Securities_DynamicSecurityData_Get_T")
QuantConnect_Securities_DynamicSecurityData_GetAll_T = typing.TypeVar("QuantConnect_Securities_DynamicSecurityData_GetAll_T")
QuantConnect_Securities_SecurityCache_GetData_T = typing.TypeVar("QuantConnect_Securities_SecurityCache_GetData_T")
QuantConnect_Securities_SecurityCache_GetAll_T = typing.TypeVar("QuantConnect_Securities_SecurityCache_GetAll_T")
QuantConnect_Securities_ContractSecurityFilterUniverse_T = typing.TypeVar("QuantConnect_Securities_ContractSecurityFilterUniverse_T")
QuantConnect_Securities_IndicatorVolatilityModel_T = typing.TypeVar("QuantConnect_Securities_IndicatorVolatilityModel_T")


class Cash(System.Object):
    """Represents a holding of a currency in cash."""

    @property
    def Updated(self) -> typing.List[System_EventHandler]:
        """
        Event fired when this instance is updated
        AddAmount, SetAmount, Update
        """
        ...

    @Updated.setter
    def Updated(self, value: typing.List[System_EventHandler]):
        """
        Event fired when this instance is updated
        AddAmount, SetAmount, Update
        """
        ...

    @property
    def SecuritySymbol(self) -> QuantConnect.Symbol:
        """
        Gets the symbol of the security required to provide conversion rates.
        If this cash represents the account currency, then QuantConnect.Symbol.Empty
        is returned
        """
        ...

    @property
    def ConversionRateSecurity(self) -> QuantConnect.Securities.Security:
        """
        Gets the security used to apply conversion rates.
        If this cash represents the account currency, then null is returned.
        """
        ...

    @ConversionRateSecurity.setter
    def ConversionRateSecurity(self, value: QuantConnect.Securities.Security):
        """
        Gets the security used to apply conversion rates.
        If this cash represents the account currency, then null is returned.
        """
        ...

    @property
    def Symbol(self) -> str:
        """Gets the symbol used to represent this cash"""
        ...

    @property
    def Amount(self) -> float:
        """Gets or sets the amount of cash held"""
        ...

    @Amount.setter
    def Amount(self, value: float):
        """Gets or sets the amount of cash held"""
        ...

    @property
    def ConversionRate(self) -> float:
        """Gets the conversion rate into account currency"""
        ...

    @ConversionRate.setter
    def ConversionRate(self, value: float):
        """Gets the conversion rate into account currency"""
        ...

    @property
    def CurrencySymbol(self) -> str:
        """The symbol of the currency, such as $"""
        ...

    @property
    def ValueInAccountCurrency(self) -> float:
        """Gets the value of this cash in the account currency"""
        ...

    def __init__(self, symbol: str, amount: float, conversionRate: float) -> None:
        """
        Initializes a new instance of the Cash class
        
        :param symbol: The symbol used to represent this cash
        :param amount: The amount of this currency held
        :param conversionRate: The initial conversion rate of this currency into the CashBook.AccountCurrency
        """
        ...

    def Update(self, data: QuantConnect.Data.BaseData) -> None:
        """
        Updates this cash object with the specified data
        
        :param data: The new data for this cash object
        """
        ...

    def AddAmount(self, amount: float) -> float:
        """
        Adds the specified amount of currency to this Cash instance and returns the new total.
        This operation is thread-safe
        
        :param amount: The amount of currency to be added
        :returns: The amount of currency directly after the addition.
        """
        ...

    def SetAmount(self, amount: float) -> None:
        """
        Sets the Quantity to the specified amount
        
        :param amount: The amount to set the quantity to
        """
        ...

    def EnsureCurrencyDataFeed(self, securities: QuantConnect.Securities.SecurityManager, subscriptions: QuantConnect.Data.SubscriptionManager, marketMap: System.Collections.Generic.IReadOnlyDictionary[QuantConnect.SecurityType, str], changes: QuantConnect.Data.UniverseSelection.SecurityChanges, securityService: QuantConnect.Interfaces.ISecurityService, accountCurrency: str, defaultResolution: QuantConnect.Resolution = ...) -> QuantConnect.Data.SubscriptionDataConfig:
        """
        Ensures that we have a data feed to convert this currency into the base currency.
        This will add a SubscriptionDataConfig and create a Security at the lowest resolution if one is not found.
        
        :param securities: The security manager
        :param subscriptions: The subscription manager used for searching and adding subscriptions
        :param marketMap: The market map that decides which market the new security should be in
        :param changes: Will be used to consume SecurityChanges.AddedSecurities
        :param securityService: Will be used to create required new Security
        :param accountCurrency: The account currency
        :param defaultResolution: The default resolution to use for the internal subscriptions
        :returns: Returns the added SubscriptionDataConfig, otherwise null.
        """
        ...

    @typing.overload
    def ToString(self) -> str:
        """
        Returns a string that represents the current Cash.
        
        :returns: A string that represents the current Cash.
        """
        ...

    @typing.overload
    def ToString(self, accountCurrency: str) -> str:
        """
        Returns a string that represents the current Cash.
        
        :returns: A string that represents the current Cash.
        """
        ...


class SymbolProperties(System.Object):
    """Represents common properties for a specific security, uniquely identified by market, symbol and security type"""

    @property
    def Description(self) -> str:
        """The description of the security"""
        ...

    @property
    def QuoteCurrency(self) -> str:
        """The quote currency of the security"""
        ...

    @property
    def ContractMultiplier(self) -> float:
        """The contract multiplier for the security"""
        ...

    @ContractMultiplier.setter
    def ContractMultiplier(self, value: float):
        """The contract multiplier for the security"""
        ...

    @property
    def MinimumPriceVariation(self) -> float:
        """The minimum price variation (tick size) for the security"""
        ...

    @property
    def LotSize(self) -> float:
        """The lot size (lot size of the order) for the security"""
        ...

    @property
    def MarketTicker(self) -> str:
        """The market ticker"""
        ...

    def __init__(self, description: str, quoteCurrency: str, contractMultiplier: float, minimumPriceVariation: float, lotSize: float, marketTicker: str) -> None:
        """Creates an instance of the SymbolProperties class"""
        ...

    @staticmethod
    def GetDefault(quoteCurrency: str) -> QuantConnect.Securities.SymbolProperties:
        """
        Gets a default instance of the SymbolProperties class for the specified
        
        :param quoteCurrency: The quote currency of the symbol
        :returns: A default instance of theSymbolProperties class.
        """
        ...


class SecurityCache(System.Object):
    """Base class caching caching spot for security data and any other temporary properties."""

    @property
    def Price(self) -> float:
        """Gets the most recent price submitted to this cache"""
        ...

    @Price.setter
    def Price(self, value: float):
        """Gets the most recent price submitted to this cache"""
        ...

    @property
    def Open(self) -> float:
        """Gets the most recent open submitted to this cache"""
        ...

    @Open.setter
    def Open(self, value: float):
        """Gets the most recent open submitted to this cache"""
        ...

    @property
    def High(self) -> float:
        """Gets the most recent high submitted to this cache"""
        ...

    @High.setter
    def High(self, value: float):
        """Gets the most recent high submitted to this cache"""
        ...

    @property
    def Low(self) -> float:
        """Gets the most recent low submitted to this cache"""
        ...

    @Low.setter
    def Low(self, value: float):
        """Gets the most recent low submitted to this cache"""
        ...

    @property
    def Close(self) -> float:
        """Gets the most recent close submitted to this cache"""
        ...

    @Close.setter
    def Close(self, value: float):
        """Gets the most recent close submitted to this cache"""
        ...

    @property
    def BidPrice(self) -> float:
        """Gets the most recent bid submitted to this cache"""
        ...

    @BidPrice.setter
    def BidPrice(self, value: float):
        """Gets the most recent bid submitted to this cache"""
        ...

    @property
    def AskPrice(self) -> float:
        """Gets the most recent ask submitted to this cache"""
        ...

    @AskPrice.setter
    def AskPrice(self, value: float):
        """Gets the most recent ask submitted to this cache"""
        ...

    @property
    def BidSize(self) -> float:
        """Gets the most recent bid size submitted to this cache"""
        ...

    @BidSize.setter
    def BidSize(self, value: float):
        """Gets the most recent bid size submitted to this cache"""
        ...

    @property
    def AskSize(self) -> float:
        """Gets the most recent ask size submitted to this cache"""
        ...

    @AskSize.setter
    def AskSize(self, value: float):
        """Gets the most recent ask size submitted to this cache"""
        ...

    @property
    def Volume(self) -> float:
        """Gets the most recent volume submitted to this cache"""
        ...

    @Volume.setter
    def Volume(self, value: float):
        """Gets the most recent volume submitted to this cache"""
        ...

    @property
    def OpenInterest(self) -> int:
        """Gets the most recent open interest submitted to this cache"""
        ...

    @OpenInterest.setter
    def OpenInterest(self, value: int):
        """Gets the most recent open interest submitted to this cache"""
        ...

    def AddDataList(self, data: System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData], dataType: typing.Type, containsFillForwardData: typing.Optional[bool] = None) -> None:
        """Add a list of market data points to the local security cache for the current market price."""
        ...

    def AddData(self, data: QuantConnect.Data.BaseData) -> None:
        """
        Add a new market data point to the local security cache for the current market price.
        Rules:
            Don't cache fill forward data.
            Always return the last observation.
            If two consecutive data has the same time stamp and one is Quotebars and the other Tradebar, prioritize the Quotebar.
        """
        ...

    def StoreData(self, data: System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData], dataType: typing.Type) -> None:
        """
        Stores the specified data list in the cache WITHOUT updating any of the cache properties, such as Price
        
        :param data: The collection of data to store in this cache
        :param dataType: The data type
        """
        ...

    @typing.overload
    def GetData(self) -> QuantConnect.Data.BaseData:
        """
        Get last data packet received for this security
        
        :returns: BaseData type of the security.
        """
        ...

    @typing.overload
    def GetData(self) -> QuantConnect_Securities_SecurityCache_GetData_T:
        """
        Get last data packet received for this security of the specified type
        
        :returns: The last data packet, null if none received of type.
        """
        ...

    def GetAll(self) -> System.Collections.Generic.IEnumerable[QuantConnect_Securities_SecurityCache_GetAll_T]:
        """
        Gets all data points of the specified type from the most recent time step
        that produced data for that type
        """
        ...

    def Reset(self) -> None:
        """Reset cache storage and free memory"""
        ...

    def HasData(self, type: typing.Type) -> bool:
        """Gets whether or not this dynamic data instance has data stored for the specified type"""
        ...

    def TryGetValue(self, type: typing.Type, data: System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData]) -> bool:
        """Gets whether or not this dynamic data instance has data stored for the specified type"""
        ...

    @staticmethod
    def ShareTypeCacheInstance(sourceToShare: QuantConnect.Securities.SecurityCache, targetToModify: QuantConnect.Securities.SecurityCache) -> None:
        """
        Helper method that modifies the target security cache instance to use the
        type cache of the source
        
        :param sourceToShare: The source cache to use
        :param targetToModify: The target security cache that will be modified
        """
        ...


class CashAmount:
    """Represents a cash amount which can be converted to account currency using a currency converter"""

    @property
    def Amount(self) -> float:
        """The amount of cash"""
        ...

    @property
    def Currency(self) -> str:
        """The currency in which the cash amount is denominated"""
        ...

    def __init__(self, amount: float, currency: str) -> None:
        """
        Initializes a new instance of the CashAmount class
        
        :param amount: The amount
        :param currency: The currency
        """
        ...

    def Equals(self, obj: typing.Any) -> bool:
        """
        Used to compare two CashAmount instances.
        Useful to compare against the default instance
        
        :param obj: The other object to compare with
        :returns: True if Currency and Amount are equal.
        """
        ...


class ICurrencyConverter(metaclass=abc.ABCMeta):
    """Provides the ability to convert cash amounts to the account currency"""

    @property
    @abc.abstractmethod
    def AccountCurrency(self) -> str:
        """Gets account currency"""
        ...

    def ConvertToAccountCurrency(self, cashAmount: QuantConnect.Securities.CashAmount) -> QuantConnect.Securities.CashAmount:
        """
        Converts a cash amount to the account currency
        
        :param cashAmount: The CashAmount instance to convert
        :returns: A new CashAmount instance denominated in the account currency.
        """
        ...


class SecurityHolding(System.Object):
    """SecurityHolding is a base class for purchasing and holding a market item which manages the asset portfolio"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """
        The security being held
        
        This property is protected.
        """
        ...

    @property
    def Target(self) -> QuantConnect.Algorithm.Framework.Portfolio.IPortfolioTarget:
        """Gets the current target holdings for this security"""
        ...

    @Target.setter
    def Target(self, value: QuantConnect.Algorithm.Framework.Portfolio.IPortfolioTarget):
        """Gets the current target holdings for this security"""
        ...

    @property
    def AveragePrice(self) -> float:
        """Average price of the security holdings."""
        ...

    @AveragePrice.setter
    def AveragePrice(self, value: float):
        """Average price of the security holdings."""
        ...

    @property
    def Quantity(self) -> float:
        """Quantity of the security held."""
        ...

    @Quantity.setter
    def Quantity(self, value: float):
        """Quantity of the security held."""
        ...

    @property
    def Symbol(self) -> QuantConnect.Symbol:
        """Symbol identifier of the underlying security."""
        ...

    @property
    def Type(self) -> int:
        """
        The security type of the symbol
        
        This property contains the int value of a member of the QuantConnect.SecurityType enum.
        """
        ...

    @property
    def Leverage(self) -> float:
        """Leverage of the underlying security."""
        ...

    @property
    def HoldingsCost(self) -> float:
        """Acquisition cost of the security total holdings in units of the account's currency."""
        ...

    @property
    def UnleveredHoldingsCost(self) -> float:
        """Unlevered Acquisition cost of the security total holdings in units of the account's currency."""
        ...

    @property
    def Price(self) -> float:
        """Current market price of the security."""
        ...

    @Price.setter
    def Price(self, value: float):
        """Current market price of the security."""
        ...

    @property
    def AbsoluteHoldingsCost(self) -> float:
        """Absolute holdings cost for current holdings in units of the account's currency."""
        ...

    @property
    def UnleveredAbsoluteHoldingsCost(self) -> float:
        """Unlevered absolute acquisition cost of the security total holdings in units of the account's currency."""
        ...

    @property
    def HoldingsValue(self) -> float:
        """Market value of our holdings in units of the account's currency."""
        ...

    @property
    def AbsoluteHoldingsValue(self) -> float:
        """Absolute of the market value of our holdings in units of the account's currency."""
        ...

    @property
    def HoldStock(self) -> bool:
        """Boolean flat indicating if we hold any of the security"""
        ...

    @property
    def Invested(self) -> bool:
        """Boolean flat indicating if we hold any of the security"""
        ...

    @property
    def TotalSaleVolume(self) -> float:
        """The total transaction volume for this security since the algorithm started in units of the account's currency."""
        ...

    @property
    def TotalFees(self) -> float:
        """Total fees for this company since the algorithm started in units of the account's currency."""
        ...

    @property
    def IsLong(self) -> bool:
        """Boolean flag indicating we have a net positive holding of the security."""
        ...

    @property
    def IsShort(self) -> bool:
        """BBoolean flag indicating we have a net negative holding of the security."""
        ...

    @property
    def AbsoluteQuantity(self) -> float:
        """Absolute quantity of holdings of this security"""
        ...

    @property
    def LastTradeProfit(self) -> float:
        """Record of the closing profit from the last trade conducted in units of the account's currency."""
        ...

    @property
    def Profit(self) -> float:
        """Calculate the total profit for this security in units of the account's currency."""
        ...

    @property
    def NetProfit(self) -> float:
        """Return the net for this company measured by the profit less fees in units of the account's currency."""
        ...

    @property
    def UnrealizedProfitPercent(self) -> float:
        """Gets the unrealized profit as a percenage of holdings cost"""
        ...

    @property
    def UnrealizedProfit(self) -> float:
        """Unrealized profit of this security when absolute quantity held is more than zero in units of the account's currency."""
        ...

    @typing.overload
    def __init__(self, security: QuantConnect.Securities.Security, currencyConverter: QuantConnect.Securities.ICurrencyConverter) -> None:
        """
        Create a new holding class instance setting the initial properties to $0.
        
        :param security: The security being held
        :param currencyConverter: A currency converter instance
        """
        ...

    @typing.overload
    def __init__(self, holding: QuantConnect.Securities.SecurityHolding) -> None:
        """
        Create a new holding class instance copying the initial properties
        
        This method is protected.
        
        :param holding: The security being held
        """
        ...

    def AddNewFee(self, newFee: float) -> None:
        """Adds a fee to the running total of total fees in units of the account's currency."""
        ...

    def AddNewProfit(self, profitLoss: float) -> None:
        """
        Adds a profit record to the running total of profit in units of the account's currency.
        
        :param profitLoss: The cash change in portfolio from closing a position
        """
        ...

    def AddNewSale(self, saleValue: float) -> None:
        """Adds a new sale value to the running total trading volume in units of the account's currency."""
        ...

    def SetLastTradeProfit(self, lastTradeProfit: float) -> None:
        """
        Set the last trade profit for this security from a Portfolio.ProcessFill call in units of the account's currency.
        
        :param lastTradeProfit: Value of the last trade profit
        """
        ...

    @typing.overload
    def SetHoldings(self, averagePrice: float, quantity: int) -> None:
        """Set the quantity of holdings and their average price after processing a portfolio fill."""
        ...

    @typing.overload
    def SetHoldings(self, averagePrice: float, quantity: float) -> None:
        """Set the quantity of holdings and their average price after processing a portfolio fill."""
        ...

    def UpdateMarketPrice(self, closingPrice: float) -> None:
        """
        Update local copy of closing price value.
        
        :param closingPrice: Price of the underlying asset to be used for calculating market price / portfolio value
        """
        ...

    def TotalCloseProfit(self) -> float:
        """Profit if we closed the holdings right now including the approximate fees in units of the account's currency."""
        ...


class MarketHoursState(System.Enum):
    """Specifies the open/close state for a MarketHoursSegment"""

    Closed = 0
    """The market is not open"""

    PreMarket = 1
    """The market is open, but before normal trading hours"""

    Market = 2
    """The market is open and within normal trading hours"""

    PostMarket = 3
    """The market is open, but after normal trading hours"""


class MarketHoursSegment(System.Object):
    """Represents the state of an exchange during a specified time range"""

    @property
    def Start(self) -> datetime.timedelta:
        """Gets the start time for this segment"""
        ...

    @Start.setter
    def Start(self, value: datetime.timedelta):
        """Gets the start time for this segment"""
        ...

    @property
    def End(self) -> datetime.timedelta:
        """Gets the end time for this segment"""
        ...

    @End.setter
    def End(self, value: datetime.timedelta):
        """Gets the end time for this segment"""
        ...

    @property
    def State(self) -> int:
        """
        Gets the market hours state for this segment
        
        This property contains the int value of a member of the QuantConnect.Securities.MarketHoursState enum.
        """
        ...

    @State.setter
    def State(self, value: int):
        """
        Gets the market hours state for this segment
        
        This property contains the int value of a member of the QuantConnect.Securities.MarketHoursState enum.
        """
        ...

    def __init__(self, state: QuantConnect.Securities.MarketHoursState, start: datetime.timedelta, end: datetime.timedelta) -> None:
        """
        Initializes a new instance of the MarketHoursSegment class
        
        :param state: The state of the market during the specified times
        :param start: The start time of the segment
        :param end: The end time of the segment
        """
        ...

    @staticmethod
    def OpenAllDay() -> QuantConnect.Securities.MarketHoursSegment:
        """Gets a new market hours segment representing being open all day"""
        ...

    @staticmethod
    def ClosedAllDay() -> QuantConnect.Securities.MarketHoursSegment:
        """Gets a new market hours segment representing being open all day"""
        ...

    @staticmethod
    def GetMarketHoursSegments(extendedMarketOpen: datetime.timedelta, marketOpen: datetime.timedelta, marketClose: datetime.timedelta, extendedMarketClose: datetime.timedelta) -> typing.List[QuantConnect.Securities.MarketHoursSegment]:
        """
        Creates the market hours segments for the specified market open/close times
        
        :param extendedMarketOpen: The extended market open time. If no pre market, set to market open
        :param marketOpen: The regular market open time
        :param marketClose: The regular market close time
        :param extendedMarketClose: The extended market close time. If no post market, set to market close
        :returns: An array of MarketHoursSegment representing the specified market open/close times.
        """
        ...

    def Contains(self, time: datetime.timedelta) -> bool:
        """
        Determines whether or not the specified time is contained within this segment
        
        :param time: The time to check
        :returns: True if this segment contains the specified time, false otherwise.
        """
        ...

    def Overlaps(self, start: datetime.timedelta, end: datetime.timedelta) -> bool:
        """
        Determines whether or not the specified time range overlaps with this segment
        
        :param start: The start of the range
        :param end: The end of the range
        :returns: True if the specified range overlaps this time segment, false otherwise.
        """
        ...

    def ToString(self) -> str:
        """
        Returns a string that represents the current object.
        
        :returns: A string that represents the current object.
        """
        ...


class LocalMarketHours(System.Object):
    """Represents the market hours under normal conditions for an exchange and a specific day of the week in terms of local time"""

    @property
    def IsClosedAllDay(self) -> bool:
        """Gets whether or not this exchange is closed all day"""
        ...

    @property
    def IsOpenAllDay(self) -> bool:
        """Gets whether or not this exchange is closed all day"""
        ...

    @property
    def DayOfWeek(self) -> int:
        """
        Gets the day of week these hours apply to
        
        This property contains the int value of a member of the System.DayOfWeek enum.
        """
        ...

    @property
    def MarketDuration(self) -> datetime.timedelta:
        """
        Gets the tradable time during the market day.
        For a normal US equity trading day this is 6.5 hours.
        This does NOT account for extended market hours and only
        considers MarketHoursState.Market
        """
        ...

    @property
    def Segments(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Securities.MarketHoursSegment]:
        """Gets the individual market hours segments that define the hours of operation for this day"""
        ...

    @typing.overload
    def __init__(self, day: System.DayOfWeek, *segments: QuantConnect.Securities.MarketHoursSegment) -> None:
        """
        Initializes a new instance of the LocalMarketHours class
        
        :param day: The day of the week these hours are applicable
        :param segments: The open/close segments defining the market hours for one day
        """
        ...

    @typing.overload
    def __init__(self, day: System.DayOfWeek, segments: System.Collections.Generic.IEnumerable[QuantConnect.Securities.MarketHoursSegment]) -> None:
        """
        Initializes a new instance of the LocalMarketHours class
        
        :param day: The day of the week these hours are applicable
        :param segments: The open/close segments defining the market hours for one day
        """
        ...

    @typing.overload
    def __init__(self, day: System.DayOfWeek, extendedMarketOpen: datetime.timedelta, marketOpen: datetime.timedelta, marketClose: datetime.timedelta, extendedMarketClose: datetime.timedelta) -> None:
        """
        Initializes a new instance of the LocalMarketHours class from the specified open/close times
        
        :param day: The day of week these hours apply to
        :param extendedMarketOpen: The extended market open time
        :param marketOpen: The regular market open time, must be greater than or equal to the extended market open time
        :param marketClose: The regular market close time, must be greater than the regular market open time
        :param extendedMarketClose: The extended market close time, must be greater than or equal to the regular market close time
        """
        ...

    @typing.overload
    def __init__(self, day: System.DayOfWeek, marketOpen: datetime.timedelta, marketClose: datetime.timedelta) -> None:
        """
        Initializes a new instance of the LocalMarketHours class from the specified open/close times
        using the market open as the extended market open and the market close as the extended market close, effectively
        removing any 'extended' session from these exchange hours
        
        :param day: The day of week these hours apply to
        :param marketOpen: The regular market open time
        :param marketClose: The regular market close time, must be greater than the regular market open time
        """
        ...

    def GetMarketOpen(self, time: datetime.timedelta, extendedMarket: bool) -> typing.Optional[datetime.timedelta]:
        """
        Gets the market opening time of day
        
        :param time: The reference time, the open returned will be the first open after the specified time if there are multiple market open segments
        :param extendedMarket: True to include extended market hours, false for regular market hours
        :returns: The market's opening time of day.
        """
        ...

    def GetMarketClose(self, time: datetime.timedelta, extendedMarket: bool) -> typing.Optional[datetime.timedelta]:
        """
        Gets the market closing time of day
        
        :param time: The reference time, the close returned will be the first close after the specified time if there are multiple market open segments
        :param extendedMarket: True to include extended market hours, false for regular market hours
        :returns: The market's closing time of day.
        """
        ...

    @typing.overload
    def IsOpen(self, time: datetime.timedelta, extendedMarket: bool) -> bool:
        """
        Determines if the exchange is open at the specified time
        
        :param time: The time of day to check
        :param extendedMarket: True to check exended market hours, false to check regular market hours
        :returns: True if the exchange is considered open, false otherwise.
        """
        ...

    @typing.overload
    def IsOpen(self, start: datetime.timedelta, end: datetime.timedelta, extendedMarket: bool) -> bool:
        """
        Determines if the exchange is open during the specified interval
        
        :param start: The start time of the interval
        :param end: The end time of the interval
        :param extendedMarket: True to check exended market hours, false to check regular market hours
        :returns: True if the exchange is considered open, false otherwise.
        """
        ...

    @staticmethod
    def ClosedAllDay(dayOfWeek: System.DayOfWeek) -> QuantConnect.Securities.LocalMarketHours:
        """
        Gets a LocalMarketHours instance that is always closed
        
        :param dayOfWeek: The day of week
        :returns: A LocalMarketHours instance that is always closed.
        """
        ...

    @staticmethod
    def OpenAllDay(dayOfWeek: System.DayOfWeek) -> QuantConnect.Securities.LocalMarketHours:
        """
        Gets a LocalMarketHours instance that is always open
        
        :param dayOfWeek: The day of week
        :returns: A LocalMarketHours instance that is always open.
        """
        ...

    def ToString(self) -> str:
        """
        Returns a string that represents the current object.
        
        :returns: A string that represents the current object.
        """
        ...


class SecurityExchangeHours(System.Object):
    """
    Represents the schedule of a security exchange. This includes daily regular and extended market hours
    as well as holidays, early closes and late opens.
    """

    @property
    def TimeZone(self) -> typing.Any:
        """Gets the time zone this exchange resides in"""
        ...

    @property
    def Holidays(self) -> System.Collections.Generic.HashSet[datetime.datetime]:
        """Gets the holidays for the exchange"""
        ...

    @property
    def MarketHours(self) -> System.Collections.Generic.IReadOnlyDictionary[System.DayOfWeek, QuantConnect.Securities.LocalMarketHours]:
        """Gets the market hours for this exchange"""
        ...

    @property
    def EarlyCloses(self) -> System.Collections.Generic.IReadOnlyDictionary[datetime.datetime, datetime.timedelta]:
        """Gets the early closes for this exchange"""
        ...

    @property
    def LateOpens(self) -> System.Collections.Generic.IReadOnlyDictionary[datetime.datetime, datetime.timedelta]:
        """Gets the late opens for this exchange"""
        ...

    @property
    def RegularMarketDuration(self) -> datetime.timedelta:
        """
        Gets the most common tradable time during the market week.
        For a normal US equity trading day this is 6.5 hours.
        This does NOT account for extended market hours and only
        considers MarketHoursState.Market
        """
        ...

    @staticmethod
    def AlwaysOpen(timeZone: typing.Any) -> QuantConnect.Securities.SecurityExchangeHours:
        """Gets a SecurityExchangeHours instance that is always open"""
        ...

    def __init__(self, timeZone: typing.Any, holidayDates: System.Collections.Generic.IEnumerable[datetime.datetime], marketHoursForEachDayOfWeek: System.Collections.Generic.IReadOnlyDictionary[System.DayOfWeek, QuantConnect.Securities.LocalMarketHours], earlyCloses: System.Collections.Generic.IReadOnlyDictionary[datetime.datetime, datetime.timedelta], lateOpens: System.Collections.Generic.IReadOnlyDictionary[datetime.datetime, datetime.timedelta]) -> None:
        """
        Initializes a new instance of the SecurityExchangeHours class
        
        :param timeZone: The time zone the dates and hours are represented in
        :param holidayDates: The dates this exchange is closed for holiday
        :param marketHoursForEachDayOfWeek: The exchange's schedule for each day of the week
        :param earlyCloses: The dates this exchange has an early close
        :param lateOpens: The dates this exchange has a late open
        """
        ...

    @typing.overload
    def IsOpen(self, localDateTime: datetime.datetime, extendedMarket: bool) -> bool:
        """
        Determines if the exchange is open at the specified local date time.
        
        :param localDateTime: The time to check represented as a local time
        :param extendedMarket: True to use the extended market hours, false for just regular market hours
        :returns: True if the exchange is considered open at the specified time, false otherwise.
        """
        ...

    @typing.overload
    def IsOpen(self, startLocalDateTime: datetime.datetime, endLocalDateTime: datetime.datetime, extendedMarket: bool) -> bool:
        """
        Determines if the exchange is open at any point in time over the specified interval.
        
        :param startLocalDateTime: The start of the interval in local time
        :param endLocalDateTime: The end of the interval in local time
        :param extendedMarket: True to use the extended market hours, false for just regular market hours
        :returns: True if the exchange is considered open at the specified time, false otherwise.
        """
        ...

    def IsDateOpen(self, localDateTime: datetime.datetime) -> bool:
        """
        Determines if the exchange will be open on the date specified by the local date time
        
        :param localDateTime: The date time to check if the day is open
        :returns: True if the exchange will be open on the specified date, false otherwise.
        """
        ...

    def GetMarketHours(self, localDateTime: datetime.datetime) -> QuantConnect.Securities.LocalMarketHours:
        """
        Helper to access the market hours field based on the day of week
        
        :param localDateTime: The local date time to retrieve market hours for
        """
        ...

    def GetNextMarketOpen(self, localDateTime: datetime.datetime, extendedMarket: bool) -> datetime.datetime:
        """
        Gets the local date time corresponding to the next market open following the specified time
        
        :param localDateTime: The time to begin searching for market open (non-inclusive)
        :param extendedMarket: True to include extended market hours in the search
        :returns: The next market opening date time following the specified local date time.
        """
        ...

    def GetNextMarketClose(self, localDateTime: datetime.datetime, extendedMarket: bool) -> datetime.datetime:
        """
        Gets the local date time corresponding to the next market close following the specified time
        
        :param localDateTime: The time to begin searching for market close (non-inclusive)
        :param extendedMarket: True to include extended market hours in the search
        :returns: The next market closing date time following the specified local date time.
        """
        ...

    def GetPreviousTradingDay(self, localDate: datetime.datetime) -> datetime.datetime:
        """
        Gets the previous trading day
        
        :param localDate: The date to start searching at in this exchange's time zones
        :returns: The previous trading day.
        """
        ...

    def GetNextTradingDay(self, date: datetime.datetime) -> datetime.datetime:
        """
        Gets the next trading day
        
        :param date: The date to start searching at
        :returns: The next trading day.
        """
        ...


class SecurityExchange(System.Object):
    """Base exchange class providing information and helper tools for reading the current exchange situation"""

    @property
    def Hours(self) -> QuantConnect.Securities.SecurityExchangeHours:
        """Gets the SecurityExchangeHours for this exchange"""
        ...

    @Hours.setter
    def Hours(self, value: QuantConnect.Securities.SecurityExchangeHours):
        """Gets the SecurityExchangeHours for this exchange"""
        ...

    @property
    def TimeZone(self) -> typing.Any:
        """Gets the time zone for this exchange"""
        ...

    @property
    def TradingDaysPerYear(self) -> int:
        """Number of trading days per year for this security. By default the market is open 365 days per year."""
        ...

    @property
    def LocalTime(self) -> datetime.datetime:
        """Time from the most recent data"""
        ...

    @LocalTime.setter
    def LocalTime(self, value: datetime.datetime):
        """Time from the most recent data"""
        ...

    @property
    def ExchangeOpen(self) -> bool:
        """Boolean property for quickly testing if the exchange is open."""
        ...

    @property
    def ClosingSoon(self) -> bool:
        """Boolean property for quickly testing if the exchange is 10 minutes away from closing."""
        ...

    def __init__(self, exchangeHours: QuantConnect.Securities.SecurityExchangeHours) -> None:
        """
        Initializes a new instance of the SecurityExchange class using the specified
        exchange hours to determine open/close times
        
        :param exchangeHours: Contains the weekly exchange schedule plus holidays
        """
        ...

    def SetLocalDateTimeFrontier(self, newLocalTime: datetime.datetime) -> None:
        """
        Set the current datetime in terms of the exchange's local time zone
        
        :param newLocalTime: Most recent data tick
        """
        ...

    def DateIsOpen(self, dateToCheck: datetime.datetime) -> bool:
        """
        Check if the *date* is open.
        
        :param dateToCheck: Date to check
        :returns: Return true if the exchange is open for this date.
        """
        ...

    def DateTimeIsOpen(self, dateTime: datetime.datetime) -> bool:
        """
        Check if this DateTime is open.
        
        :param dateTime: DateTime to check
        :returns: Boolean true if the market is open.
        """
        ...

    def IsOpenDuringBar(self, barStartTime: datetime.datetime, barEndTime: datetime.datetime, isExtendedMarketHours: bool) -> bool:
        """Determines if the exchange was open at any time between start and stop"""
        ...

    def IsClosingSoon(self, minutesToClose: int) -> bool:
        """
        Determines if the exchange is going to close in the next provided minutes
        
        :param minutesToClose: Minutes to close to check
        :returns: Returns true if the exchange is going to close in the next provided minutes.
        """
        ...

    def SetMarketHours(self, marketHoursSegments: System.Collections.Generic.IEnumerable[QuantConnect.Securities.MarketHoursSegment], *days: System.DayOfWeek) -> None:
        """
        Sets the regular market hours for the specified days If no days are specified then
        all days will be updated.
        
        :param marketHoursSegments: Specifies each segment of the market hours, such as premarket/market/postmark
        :param days: The days of the week to set these times for
        """
        ...


class HasSufficientBuyingPowerForOrderResult(System.Object):
    """Contains the information returned by IBuyingPowerModel.HasSufficientBuyingPowerForOrder"""

    @property
    def IsSufficient(self) -> bool:
        """Returns true if there is sufficient buying power to execute an order"""
        ...

    @property
    def Reason(self) -> str:
        """Returns the reason for insufficient buying power to execute an order"""
        ...

    def __init__(self, isSufficient: bool, reason: str = None) -> None:
        """
        Initializes a new instance of the HasSufficientBuyingPowerForOrderResult class
        
        :param isSufficient: True if the order can be executed
        :param reason: The reason for insufficient buying power
        """
        ...


class GetMaximumOrderQuantityResult(System.Object):
    """
    Contains the information returned by IBuyingPowerModel.GetMaximumOrderQuantityForTargetBuyingPower
    and  IBuyingPowerModel.GetMaximumOrderQuantityForDeltaBuyingPower
    """

    @property
    def Quantity(self) -> float:
        """Returns the maximum quantity for the order"""
        ...

    @property
    def Reason(self) -> str:
        """Returns the reason for which the maximum order quantity is zero"""
        ...

    @property
    def IsError(self) -> bool:
        """Returns true if the zero order quantity is an error condition and will be shown to the user."""
        ...

    @typing.overload
    def __init__(self, quantity: float, reason: str = None) -> None:
        """
        Initializes a new instance of the GetMaximumOrderQuantityResult class
        
        :param quantity: Returns the maximum quantity for the order
        :param reason: The reason for which the maximum order quantity is zero
        """
        ...

    @typing.overload
    def __init__(self, quantity: float, reason: str, isError: bool = True) -> None:
        """
        Initializes a new instance of the GetMaximumOrderQuantityResult class
        
        :param quantity: Returns the maximum quantity for the order
        :param reason: The reason for which the maximum order quantity is zero
        :param isError: True if the zero order quantity is an error condition
        """
        ...


class ReservedBuyingPowerForPosition(System.Object):
    """Defines the result for IBuyingPowerModel.GetReservedBuyingPowerForPosition"""

    @property
    def AbsoluteUsedBuyingPower(self) -> float:
        """Gets the reserved buying power"""
        ...

    def __init__(self, reservedBuyingPowerForPosition: float) -> None:
        """
        Initializes a new instance of the ReservedBuyingPowerForPosition class
        
        :param reservedBuyingPowerForPosition: The reserved buying power for the security's holdings
        """
        ...


class BuyingPower(System.Object):
    """Defines the result for IBuyingPowerModel.GetBuyingPower"""

    @property
    def Value(self) -> float:
        """Gets the buying power"""
        ...

    def __init__(self, buyingPower: float) -> None:
        """
        Initializes a new instance of the BuyingPower class
        
        :param buyingPower: The buying power
        """
        ...


class IBuyingPowerModel(metaclass=abc.ABCMeta):
    """Represents a security's model of buying power"""

    def GetLeverage(self, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the current leverage of the security
        
        :param security: The security to get leverage for
        :returns: The current leverage in the security.
        """
        ...

    def SetLeverage(self, security: QuantConnect.Securities.Security, leverage: float) -> None:
        """
        Sets the leverage for the applicable securities, i.e, equities
        
        :param security: The security to set leverage for
        :param leverage: The new leverage
        """
        ...

    def HasSufficientBuyingPowerForOrder(self, parameters: QuantConnect.Securities.HasSufficientBuyingPowerForOrderParameters) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """
        Check if there is sufficient buying power to execute this order.
        
        :param parameters: An object containing the portfolio, the security and the order
        :returns: Returns buying power information for an order.
        """
        ...

    def GetMaximumOrderQuantityForTargetBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForTargetBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a position with a given buying power percentage.
        Will not take into account free buying power.
        
        :param parameters: An object containing the portfolio, the security and the target signed buying power percentage
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetMaximumOrderQuantityForDeltaBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForDeltaBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a delta in the buying power used by a security.
        The deltas sign defines the position side to apply it to, positive long, negative short.
        
        :param parameters: An object containing the portfolio, the security and the delta buying power
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetReservedBuyingPowerForPosition(self, parameters: QuantConnect.Securities.ReservedBuyingPowerForPositionParameters) -> QuantConnect.Securities.ReservedBuyingPowerForPosition:
        """
        Gets the amount of buying power reserved to maintain the specified position
        
        :param parameters: A parameters object containing the security
        :returns: The reserved buying power in account currency.
        """
        ...

    def GetBuyingPower(self, parameters: QuantConnect.Securities.BuyingPowerParameters) -> QuantConnect.Securities.BuyingPower:
        """
        Gets the buying power available for a trade
        
        :param parameters: A parameters object containing the algorithm's potrfolio, security, and order direction
        :returns: The buying power available for the trade.
        """
        ...


class ISettlementModel(metaclass=abc.ABCMeta):
    """Represents the model responsible for applying cash settlement rules"""

    def ApplyFunds(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, applicationTimeUtc: datetime.datetime, currency: str, amount: float) -> None:
        """
        Applies cash settlement rules
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param applicationTimeUtc: The fill time (in UTC)
        :param currency: The currency symbol
        :param amount: The amount of cash to apply
        """
        ...


class IVolatilityModel(metaclass=abc.ABCMeta):
    """Represents a model that computes the volatility of a security"""

    @property
    @abc.abstractmethod
    def Volatility(self) -> float:
        """Gets the volatility of the security as a percentage"""
        ...

    def Update(self, security: QuantConnect.Securities.Security, data: QuantConnect.Data.BaseData) -> None:
        """
        Updates this model using the new price information in
        the specified security instance
        
        :param security: The security to calculate volatility for
        :param data: The new data used to update the model
        """
        ...

    def GetHistoryRequirements(self, security: QuantConnect.Securities.Security, utcTime: datetime.datetime) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.HistoryRequest]:
        """
        Returns history requirements for the volatility model expressed in the form of history request
        
        :param security: The security of the request
        :param utcTime: The date/time of the request
        :returns: History request object list, or empty if no requirements.
        """
        ...


class IPriceVariationModel(metaclass=abc.ABCMeta):
    """Gets the minimum price variation of a given security"""

    def GetMinimumPriceVariation(self, parameters: QuantConnect.Securities.GetMinimumPriceVariationParameters) -> float:
        """
        Get the minimum price variation from a security
        
        :param parameters: An object containing the method parameters
        :returns: Decimal minimum price variation of a given security.
        """
        ...


class IRegisteredSecurityDataTypesProvider(metaclass=abc.ABCMeta):
    """Provides the set of base data types registered in the algorithm"""

    def RegisterType(self, type: typing.Type) -> bool:
        """
        Registers the specified type w/ the provider
        
        :returns: True if the type was previously not registered.
        """
        ...

    def UnregisterType(self, type: typing.Type) -> bool:
        """
        Removes the registration for the specified type
        
        :returns: True if the type was previously registered.
        """
        ...

    def TryGetType(self, name: str, type: typing.Type) -> bool:
        """
        Determines if the specified type is registered or not and returns it
        
        :returns: True if the type was previously registered.
        """
        ...


class DynamicSecurityData(IDynamicMetaObjectProvider):
    """
    Provides access to a security's data via it's type. This implementation supports dynamic access
    by type name.
    """

    def __init__(self, registeredTypes: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider, cache: QuantConnect.Securities.SecurityCache) -> None:
        """
        Initializes a new instance of the DynamicSecurityData class
        
        :param registeredTypes: Provides all the registered data types for the algorithm
        :param cache: The security cache
        """
        ...

    def GetMetaObject(self, parameter: typing.Any) -> typing.Any:
        """
        Returns the System.Dynamic.DynamicMetaObject responsible for binding operations performed on this object.
        
        :param parameter: The expression tree representation of the runtime value.
        :returns: The System.Dynamic.DynamicMetaObject to bind this object.
        """
        ...

    def HasData(self) -> bool:
        """Gets whether or not this dynamic data instance has data stored for the specified type"""
        ...

    def HasProperty(self, name: str) -> bool:
        """
        Gets whether or not this dynamic data instance has a property with the specified name.
        This is a case-insensitive search.
        
        :param name: The property name to check for
        :returns: True if the property exists, false otherwise.
        """
        ...

    def Get(self, type: QuantConnect_Securities_DynamicSecurityData_Get_T) -> QuantConnect_Securities_DynamicSecurityData_Get_T:
        """
        Get the matching cached object in a python friendly accessor
        
        :param type: Type to search for
        :returns: Matching object.
        """
        ...

    def GetAll(self, type: QuantConnect_Securities_DynamicSecurityData_GetAll_T) -> System.Collections.Generic.IReadOnlyList[QuantConnect_Securities_DynamicSecurityData_GetAll_T]:
        """
        Get all the matching types with a python friendly overload.
        
        :param type: Search type
        :returns: List of matching objects cached.
        """
        ...

    def SetProperty(self, name: str, value: typing.Any) -> System.Object:
        """
        Sets the property with the specified name to the value. This is a case-insensitve search.
        
        :param name: The property name to set
        :param value: The new property value
        :returns: Returns the input value back to the caller.
        """
        ...

    def GetProperty(self, name: str) -> System.Object:
        """
        Gets the property's value with the specified name. This is a case-insensitve search.
        
        :param name: The property name to access
        :returns: object value of BaseData.
        """
        ...


class Security(System.Object, QuantConnect.Interfaces.ISecurityPrice):
    """A base vehicle properties class for providing a common interface to all assets in QuantConnect."""

    @property
    def SubscriptionsBag(self) -> System.Collections.Concurrent.ConcurrentBag[QuantConnect.Data.SubscriptionDataConfig]:
        """This field is protected."""
        ...

    @property
    def ShortableProvider(self) -> QuantConnect.Interfaces.IShortableProvider:
        """This property is protected."""
        ...

    @ShortableProvider.setter
    def ShortableProvider(self, value: QuantConnect.Interfaces.IShortableProvider):
        """This property is protected."""
        ...

    NullLeverage: float = 0
    """A null security leverage value"""

    @property
    def Subscriptions(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.SubscriptionDataConfig]:
        """Gets all the subscriptions for this security"""
        ...

    @property
    def Symbol(self) -> QuantConnect.Symbol:
        """Symbol for the asset."""
        ...

    @property
    def QuoteCurrency(self) -> QuantConnect.Securities.Cash:
        """Gets the Cash object used for converting the quote currency to the account currency"""
        ...

    @property
    def SymbolProperties(self) -> QuantConnect.Securities.SymbolProperties:
        """Gets the symbol properties for this security"""
        ...

    @property
    def Type(self) -> int:
        """
        Type of the security.
        
        This property contains the int value of a member of the QuantConnect.SecurityType enum.
        """
        ...

    @property
    def Resolution(self) -> int:
        """
        Resolution of data requested for this security.
        
        This property contains the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    @Resolution.setter
    def Resolution(self, value: int):
        """
        Resolution of data requested for this security.
        
        This property contains the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    @property
    def IsFillDataForward(self) -> bool:
        """Indicates the data will use previous bars when there was no trading in this time period. This was a configurable datastream setting set in initialization."""
        ...

    @IsFillDataForward.setter
    def IsFillDataForward(self, value: bool):
        """Indicates the data will use previous bars when there was no trading in this time period. This was a configurable datastream setting set in initialization."""
        ...

    @property
    def IsExtendedMarketHours(self) -> bool:
        """Indicates the security will continue feeding data after the primary market hours have closed. This was a configurable setting set in initialization."""
        ...

    @IsExtendedMarketHours.setter
    def IsExtendedMarketHours(self, value: bool):
        """Indicates the security will continue feeding data after the primary market hours have closed. This was a configurable setting set in initialization."""
        ...

    @property
    def DataNormalizationMode(self) -> int:
        """
        Gets the data normalization mode used for this security
        
        This property contains the int value of a member of the QuantConnect.DataNormalizationMode enum.
        """
        ...

    @DataNormalizationMode.setter
    def DataNormalizationMode(self, value: int):
        """
        Gets the data normalization mode used for this security
        
        This property contains the int value of a member of the QuantConnect.DataNormalizationMode enum.
        """
        ...

    @property
    def SubscriptionDataConfig(self) -> QuantConnect.Data.SubscriptionDataConfig:
        """Gets the subscription configuration for this security"""
        ...

    @property
    def HasData(self) -> bool:
        """There has been at least one datapoint since our algorithm started running for us to determine price."""
        ...

    @property
    def IsTradable(self) -> bool:
        """Gets or sets whether or not this security should be considered tradable"""
        ...

    @IsTradable.setter
    def IsTradable(self, value: bool):
        """Gets or sets whether or not this security should be considered tradable"""
        ...

    @property
    def IsDelisted(self) -> bool:
        """True if the security has been delisted from exchanges and is no longer tradable"""
        ...

    @IsDelisted.setter
    def IsDelisted(self, value: bool):
        """True if the security has been delisted from exchanges and is no longer tradable"""
        ...

    @property
    def Cache(self) -> QuantConnect.Securities.SecurityCache:
        """Data cache for the security to store previous price information."""
        ...

    @Cache.setter
    def Cache(self, value: QuantConnect.Securities.SecurityCache):
        """Data cache for the security to store previous price information."""
        ...

    @property
    def Holdings(self) -> QuantConnect.Securities.SecurityHolding:
        """Holdings class contains the portfolio, cash and processes order fills."""
        ...

    @Holdings.setter
    def Holdings(self, value: QuantConnect.Securities.SecurityHolding):
        """Holdings class contains the portfolio, cash and processes order fills."""
        ...

    @property
    def Exchange(self) -> QuantConnect.Securities.SecurityExchange:
        """Exchange class contains the market opening hours, along with pre-post market hours."""
        ...

    @Exchange.setter
    def Exchange(self, value: QuantConnect.Securities.SecurityExchange):
        """Exchange class contains the market opening hours, along with pre-post market hours."""
        ...

    @property
    def FeeModel(self) -> QuantConnect.Orders.Fees.IFeeModel:
        """Fee model used to compute order fees for this security"""
        ...

    @FeeModel.setter
    def FeeModel(self, value: QuantConnect.Orders.Fees.IFeeModel):
        """Fee model used to compute order fees for this security"""
        ...

    @property
    def FillModel(self) -> QuantConnect.Orders.Fills.IFillModel:
        """Fill model used to produce fill events for this security"""
        ...

    @FillModel.setter
    def FillModel(self, value: QuantConnect.Orders.Fills.IFillModel):
        """Fill model used to produce fill events for this security"""
        ...

    @property
    def SlippageModel(self) -> QuantConnect.Orders.Slippage.ISlippageModel:
        """Slippage model use to compute slippage of market orders"""
        ...

    @SlippageModel.setter
    def SlippageModel(self, value: QuantConnect.Orders.Slippage.ISlippageModel):
        """Slippage model use to compute slippage of market orders"""
        ...

    @property
    def PortfolioModel(self) -> QuantConnect.Securities.ISecurityPortfolioModel:
        """Gets the portfolio model used by this security"""
        ...

    @PortfolioModel.setter
    def PortfolioModel(self, value: QuantConnect.Securities.ISecurityPortfolioModel):
        """Gets the portfolio model used by this security"""
        ...

    @property
    def BuyingPowerModel(self) -> QuantConnect.Securities.IBuyingPowerModel:
        """Gets the buying power model used for this security"""
        ...

    @BuyingPowerModel.setter
    def BuyingPowerModel(self, value: QuantConnect.Securities.IBuyingPowerModel):
        """Gets the buying power model used for this security"""
        ...

    @property
    def MarginModel(self) -> QuantConnect.Securities.IBuyingPowerModel:
        """Gets the buying power model used for this security, an alias for BuyingPowerModel"""
        ...

    @MarginModel.setter
    def MarginModel(self, value: QuantConnect.Securities.IBuyingPowerModel):
        """Gets the buying power model used for this security, an alias for BuyingPowerModel"""
        ...

    @property
    def SettlementModel(self) -> QuantConnect.Securities.ISettlementModel:
        """Gets the settlement model used for this security"""
        ...

    @SettlementModel.setter
    def SettlementModel(self, value: QuantConnect.Securities.ISettlementModel):
        """Gets the settlement model used for this security"""
        ...

    @property
    def VolatilityModel(self) -> QuantConnect.Securities.IVolatilityModel:
        """Gets the volatility model used for this security"""
        ...

    @VolatilityModel.setter
    def VolatilityModel(self, value: QuantConnect.Securities.IVolatilityModel):
        """Gets the volatility model used for this security"""
        ...

    @property
    def DataFilter(self) -> QuantConnect.Securities.Interfaces.ISecurityDataFilter:
        """
        Customizable data filter to filter outlier ticks before they are passed into user event handlers.
        By default all ticks are passed into the user algorithms.
        """
        ...

    @DataFilter.setter
    def DataFilter(self, value: QuantConnect.Securities.Interfaces.ISecurityDataFilter):
        """
        Customizable data filter to filter outlier ticks before they are passed into user event handlers.
        By default all ticks are passed into the user algorithms.
        """
        ...

    @property
    def PriceVariationModel(self) -> QuantConnect.Securities.IPriceVariationModel:
        """
        Customizable price variation model used to define the minimum price variation of this security.
        By default minimum price variation is a constant find in the symbol-properties-database.
        """
        ...

    @PriceVariationModel.setter
    def PriceVariationModel(self, value: QuantConnect.Securities.IPriceVariationModel):
        """
        Customizable price variation model used to define the minimum price variation of this security.
        By default minimum price variation is a constant find in the symbol-properties-database.
        """
        ...

    @property
    def Data(self) -> QuantConnect.Securities.DynamicSecurityData:
        """Provides dynamic access to data in the cache"""
        ...

    @property
    def HoldStock(self) -> bool:
        """Read only property that checks if we currently own stock in the company."""
        ...

    @property
    def Invested(self) -> bool:
        """Alias for HoldStock - Do we have any of this security"""
        ...

    @property
    def LocalTime(self) -> datetime.datetime:
        """Local time for this market"""
        ...

    @property
    def Price(self) -> float:
        """Get the current value of the security."""
        ...

    @property
    def Leverage(self) -> float:
        """Leverage for this Security."""
        ...

    @property
    def High(self) -> float:
        """If this uses tradebar data, return the most recent high."""
        ...

    @property
    def Low(self) -> float:
        """If this uses tradebar data, return the most recent low."""
        ...

    @property
    def Close(self) -> float:
        """If this uses tradebar data, return the most recent close."""
        ...

    @property
    def Open(self) -> float:
        """If this uses tradebar data, return the most recent open."""
        ...

    @property
    def Volume(self) -> float:
        """Access to the volume of the equity today"""
        ...

    @property
    def BidPrice(self) -> float:
        """Gets the most recent bid price if available"""
        ...

    @property
    def BidSize(self) -> float:
        """Gets the most recent bid size if available"""
        ...

    @property
    def AskPrice(self) -> float:
        """Gets the most recent ask price if available"""
        ...

    @property
    def AskSize(self) -> float:
        """Gets the most recent ask size if available"""
        ...

    @property
    def OpenInterest(self) -> int:
        """Access to the open interest of the security today"""
        ...

    @property
    def Fundamentals(self) -> QuantConnect.Data.Fundamental.Fundamentals:
        """Gets the fundamental data associated with the security if there is any, otherwise null."""
        ...

    @typing.overload
    def __init__(self, exchangeHours: QuantConnect.Securities.SecurityExchangeHours, config: QuantConnect.Data.SubscriptionDataConfig, quoteCurrency: QuantConnect.Securities.Cash, symbolProperties: QuantConnect.Securities.SymbolProperties, currencyConverter: QuantConnect.Securities.ICurrencyConverter, registeredTypesProvider: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider, cache: QuantConnect.Securities.SecurityCache) -> None:
        """Construct a new security vehicle based on the user options."""
        ...

    @typing.overload
    def __init__(self, symbol: typing.Union[QuantConnect.Symbol, str], exchangeHours: QuantConnect.Securities.SecurityExchangeHours, quoteCurrency: QuantConnect.Securities.Cash, symbolProperties: QuantConnect.Securities.SymbolProperties, currencyConverter: QuantConnect.Securities.ICurrencyConverter, registeredTypesProvider: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider, cache: QuantConnect.Securities.SecurityCache) -> None:
        """Construct a new security vehicle based on the user options."""
        ...

    @typing.overload
    def __init__(self, symbol: typing.Union[QuantConnect.Symbol, str], quoteCurrency: QuantConnect.Securities.Cash, symbolProperties: QuantConnect.Securities.SymbolProperties, exchange: QuantConnect.Securities.SecurityExchange, cache: QuantConnect.Securities.SecurityCache, portfolioModel: QuantConnect.Securities.ISecurityPortfolioModel, fillModel: QuantConnect.Orders.Fills.IFillModel, feeModel: QuantConnect.Orders.Fees.IFeeModel, slippageModel: QuantConnect.Orders.Slippage.ISlippageModel, settlementModel: QuantConnect.Securities.ISettlementModel, volatilityModel: QuantConnect.Securities.IVolatilityModel, buyingPowerModel: QuantConnect.Securities.IBuyingPowerModel, dataFilter: QuantConnect.Securities.Interfaces.ISecurityDataFilter, priceVariationModel: QuantConnect.Securities.IPriceVariationModel, currencyConverter: QuantConnect.Securities.ICurrencyConverter, registeredTypesProvider: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider) -> None:
        """
        Construct a new security vehicle based on the user options.
        
        This method is protected.
        """
        ...

    @typing.overload
    def __init__(self, config: QuantConnect.Data.SubscriptionDataConfig, quoteCurrency: QuantConnect.Securities.Cash, symbolProperties: QuantConnect.Securities.SymbolProperties, exchange: QuantConnect.Securities.SecurityExchange, cache: QuantConnect.Securities.SecurityCache, portfolioModel: QuantConnect.Securities.ISecurityPortfolioModel, fillModel: QuantConnect.Orders.Fills.IFillModel, feeModel: QuantConnect.Orders.Fees.IFeeModel, slippageModel: QuantConnect.Orders.Slippage.ISlippageModel, settlementModel: QuantConnect.Securities.ISettlementModel, volatilityModel: QuantConnect.Securities.IVolatilityModel, buyingPowerModel: QuantConnect.Securities.IBuyingPowerModel, dataFilter: QuantConnect.Securities.Interfaces.ISecurityDataFilter, priceVariationModel: QuantConnect.Securities.IPriceVariationModel, currencyConverter: QuantConnect.Securities.ICurrencyConverter, registeredTypesProvider: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider) -> None:
        """
        Temporary convenience constructor
        
        This method is protected.
        """
        ...

    def GetLastData(self) -> QuantConnect.Data.BaseData:
        """
        Get the last price update set to the security.
        
        :returns: BaseData object for this security.
        """
        ...

    def SetLocalTimeKeeper(self, localTimeKeeper: QuantConnect.LocalTimeKeeper) -> None:
        """
        Sets the LocalTimeKeeper to be used for this Security.
        This is the source of this instance's time.
        
        :param localTimeKeeper: The source of this Security's time.
        """
        ...

    def SetMarketPrice(self, data: QuantConnect.Data.BaseData) -> None:
        """
        Update any security properties based on the latest market data and time
        
        :param data: New data packet from LEAN
        """
        ...

    def Update(self, data: System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData], dataType: typing.Type, containsFillForwardData: typing.Optional[bool] = None) -> None:
        """
        Updates all of the security properties, such as price/OHLCV/bid/ask based
        on the data provided. Data is also stored into the security's data cache
        
        :param data: The security update data
        :param dataType: The data type
        :param containsFillForwardData: Flag indicating whether  contains any fill forward bar or not
        """
        ...

    def IsCustomData(self) -> bool:
        """Returns true if the security contains at least one subscription that represents custom data"""
        ...

    def SetLeverage(self, leverage: float) -> None:
        """
        Set the leverage parameter for this security
        
        :param leverage: Leverage for this asset
        """
        ...

    def SetDataNormalizationMode(self, mode: QuantConnect.DataNormalizationMode) -> None:
        """Sets the data normalization mode to be used by this security"""
        ...

    def RefreshDataNormalizationModeProperty(self) -> None:
        """
        This method will refresh the value of the DataNormalizationMode property.
        This is required for backward-compatibility.
        TODO: to be deleted with the DataNormalizationMode property
        """
        ...

    @typing.overload
    def SetFeeModel(self, feelModel: QuantConnect.Orders.Fees.IFeeModel) -> None:
        """
        Sets the fee model
        
        :param feelModel: Model that represents a fee model
        """
        ...

    @typing.overload
    def SetFeeModel(self, feelModel: typing.Any) -> None:
        """
        Sets the fee model
        
        :param feelModel: Model that represents a fee model
        """
        ...

    @typing.overload
    def SetFillModel(self, fillModel: QuantConnect.Orders.Fills.IFillModel) -> None:
        """
        Sets the fill model
        
        :param fillModel: Model that represents a fill model
        """
        ...

    @typing.overload
    def SetFillModel(self, fillModel: typing.Any) -> None:
        """
        Sets the fill model
        
        :param fillModel: Model that represents a fill model
        """
        ...

    @typing.overload
    def SetSlippageModel(self, slippageModel: QuantConnect.Orders.Slippage.ISlippageModel) -> None:
        """
        Sets the slippage model
        
        :param slippageModel: Model that represents a slippage model
        """
        ...

    @typing.overload
    def SetSlippageModel(self, slippageModel: typing.Any) -> None:
        """
        Sets the slippage model
        
        :param slippageModel: Model that represents a slippage model
        """
        ...

    @typing.overload
    def SetVolatilityModel(self, volatilityModel: QuantConnect.Securities.IVolatilityModel) -> None:
        """
        Sets the volatility model
        
        :param volatilityModel: Model that represents a volatility model
        """
        ...

    @typing.overload
    def SetVolatilityModel(self, volatilityModel: typing.Any) -> None:
        """
        Sets the volatility model
        
        :param volatilityModel: Model that represents a volatility model
        """
        ...

    @typing.overload
    def SetBuyingPowerModel(self, buyingPowerModel: QuantConnect.Securities.IBuyingPowerModel) -> None:
        """
        Sets the buying power model
        
        :param buyingPowerModel: Model that represents a security's model of buying power
        """
        ...

    @typing.overload
    def SetBuyingPowerModel(self, pyObject: typing.Any) -> None:
        """
        Sets the buying power model
        
        :param pyObject: Model that represents a security's model of buying power
        """
        ...

    @typing.overload
    def SetMarginModel(self, marginModel: QuantConnect.Securities.IBuyingPowerModel) -> None:
        """
        Sets the margin model
        
        :param marginModel: Model that represents a security's model of buying power
        """
        ...

    @typing.overload
    def SetMarginModel(self, pyObject: typing.Any) -> None:
        """
        Sets the margin model
        
        :param pyObject: Model that represents a security's model of buying power
        """
        ...

    def SetShortableProvider(self, shortableProvider: QuantConnect.Interfaces.IShortableProvider) -> None:
        ...

    def ToString(self) -> str:
        """
        Returns a string that represents the current object.
        
        :returns: A string that represents the current object.
        """
        ...


class ISecurityInitializer(metaclass=abc.ABCMeta):
    """Represents a type capable of initializing a new security"""

    def Initialize(self, security: QuantConnect.Securities.Security) -> None:
        """
        Initializes the specified security
        
        :param security: The security to be initialized
        """
        ...


class SecurityInitializer(System.Object):
    """Provides static access to the Null security initializer"""

    Null: QuantConnect.Securities.ISecurityInitializer = ...
    """Gets an implementation of ISecurityInitializer that is a no-op"""


class InitialMarginRequiredForOrderParameters(System.Object):
    """Defines the parameters for BuyingPowerModel.GetInitialMarginRequiredForOrder"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def Order(self) -> QuantConnect.Orders.Order:
        """Gets the order"""
        ...

    @property
    def CurrencyConverter(self) -> QuantConnect.Securities.ICurrencyConverter:
        """Gets the currency converter"""
        ...

    def __init__(self, currencyConverter: QuantConnect.Securities.ICurrencyConverter, security: QuantConnect.Securities.Security, order: QuantConnect.Orders.Order) -> None:
        """
        Initializes a new instance of the InitialMarginRequiredForOrderParameters class
        
        :param currencyConverter: The currency converter
        :param security: The security
        :param order: The order
        """
        ...


class ISecurityProvider(metaclass=abc.ABCMeta):
    """Represents a type capable of fetching the holdings for the specified symbol"""

    def GetSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Securities.Security:
        """
        Retrieves a summary of the holdings for the specified symbol
        
        :param symbol: The symbol to get holdings for
        :returns: The holdings for the symbol or null if the symbol is invalid and/or not in the portfolio.
        """
        ...


class CashBook(System.Object, System.Collections.Generic.IDictionary[str, QuantConnect.Securities.Cash], QuantConnect.Securities.ICurrencyConverter, typing.Iterable[System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]]):
    """Provides a means of keeping track of the different cash holdings of an algorithm"""

    class UpdateType(System.Enum):
        """The different types of Updated events"""

        Added = 0
        """A new Cash.Symbol was added"""

        Removed = 1
        """One or more Cash instances were removed"""

        Updated = 2
        """An existing Cash.Symbol was updated"""

    @property
    def Updated(self) -> typing.List[System_EventHandler]:
        """
        Event fired when a Cash instance is added or removed, and when
        the Cash.Updated is triggered for the currently hold instances
        """
        ...

    @Updated.setter
    def Updated(self, value: typing.List[System_EventHandler]):
        """
        Event fired when a Cash instance is added or removed, and when
        the Cash.Updated is triggered for the currently hold instances
        """
        ...

    @property
    def AccountCurrency(self) -> str:
        """Gets the base currency used"""
        ...

    @AccountCurrency.setter
    def AccountCurrency(self, value: str):
        """Gets the base currency used"""
        ...

    @property
    def TotalValueInAccountCurrency(self) -> float:
        """Gets the total value of the cash book in units of the base currency"""
        ...

    @property
    def Count(self) -> int:
        ...

    @property
    def IsReadOnly(self) -> bool:
        """Gets a value indicating whether this instance is read only."""
        ...

    @property
    def Keys(self) -> System.Collections.Generic.ICollection[str]:
        """Gets the keys."""
        ...

    @property
    def Values(self) -> System.Collections.Generic.ICollection[QuantConnect.Securities.Cash]:
        """Gets the values."""
        ...

    def __init__(self) -> None:
        """Initializes a new instance of the CashBook class."""
        ...

    @typing.overload
    def Add(self, symbol: str, quantity: float, conversionRate: float) -> None:
        """
        Adds a new cash of the specified symbol and quantity
        
        :param symbol: The symbol used to reference the new cash
        :param quantity: The amount of new cash to start
        :param conversionRate: The conversion rate used to determine the initial portfolio value/starting capital impact caused by this currency position.
        """
        ...

    def EnsureCurrencyDataFeeds(self, securities: QuantConnect.Securities.SecurityManager, subscriptions: QuantConnect.Data.SubscriptionManager, marketMap: System.Collections.Generic.IReadOnlyDictionary[QuantConnect.SecurityType, str], changes: QuantConnect.Data.UniverseSelection.SecurityChanges, securityService: QuantConnect.Interfaces.ISecurityService, defaultResolution: QuantConnect.Resolution = ...) -> System.Collections.Generic.List[QuantConnect.Data.SubscriptionDataConfig]:
        """
        Checks the current subscriptions and adds necessary currency pair feeds to provide real time conversion data
        
        :param securities: The SecurityManager for the algorithm
        :param subscriptions: The SubscriptionManager for the algorithm
        :param marketMap: The market map that decides which market the new security should be in
        :param changes: Will be used to consume SecurityChanges.AddedSecurities
        :param securityService: Will be used to create required new Security
        :param defaultResolution: The default resolution to use for the internal subscriptions
        :returns: Returns a list of added currency SubscriptionDataConfig.
        """
        ...

    def Convert(self, sourceQuantity: float, sourceCurrency: str, destinationCurrency: str) -> float:
        """
        Converts a quantity of source currency units into the specified destination currency
        
        :param sourceQuantity: The quantity of source currency to be converted
        :param sourceCurrency: The source currency symbol
        :param destinationCurrency: The destination currency symbol
        :returns: The converted value.
        """
        ...

    @typing.overload
    def ConvertToAccountCurrency(self, sourceQuantity: float, sourceCurrency: str) -> float:
        """
        Converts a quantity of source currency units into the account currency
        
        :param sourceQuantity: The quantity of source currency to be converted
        :param sourceCurrency: The source currency symbol
        :returns: The converted value.
        """
        ...

    def ToString(self) -> str:
        """
        Returns a string that represents the current object.
        
        :returns: A string that represents the current object.
        """
        ...

    @typing.overload
    def Add(self, item: System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]) -> None:
        """
        Add the specified item to this CashBook.
        
        :param item: KeyValuePair of symbol -> Cash item
        """
        ...

    @typing.overload
    def Add(self, symbol: str, value: QuantConnect.Securities.Cash) -> None:
        """
        Add the specified key and value.
        
        :param symbol: The symbol of the Cash value.
        :param value: Value.
        """
        ...

    def Clear(self) -> None:
        """Clear this instance of all Cash entries."""
        ...

    @typing.overload
    def Remove(self, symbol: str) -> bool:
        """
        Remove the Cash item corresponding to the specified symbol
        
        :param symbol: The symbolto be removed
        """
        ...

    @typing.overload
    def Remove(self, item: System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]) -> bool:
        """
        Remove the specified item.
        
        :param item: Item.
        """
        ...

    def ContainsKey(self, symbol: str) -> bool:
        """
        Determines whether the current instance contains an entry with the specified symbol.
        
        :param symbol: Key.
        :returns: true, if key was contained, false otherwise.
        """
        ...

    def TryGetValue(self, symbol: str, value: QuantConnect.Securities.Cash) -> bool:
        """
        Try to get the value.
        
        :param symbol: The symbol.
        :param value: Value.
        :returns: true, if get value was tryed, false otherwise.
        """
        ...

    def Contains(self, item: System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]) -> bool:
        """
        Determines whether the current collection contains the specified value.
        
        :param item: Item.
        """
        ...

    def CopyTo(self, array: typing.List[System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]], arrayIndex: int) -> None:
        """
        Copies to the specified array.
        
        :param array: Array.
        :param arrayIndex: Array index.
        """
        ...

    def __getitem__(self, symbol: str) -> QuantConnect.Securities.Cash:
        """
        Gets or sets the QuantConnect.Securities.Cash with the specified symbol.
        
        :param symbol: Symbol.
        """
        ...

    def __setitem__(self, symbol: str, value: QuantConnect.Securities.Cash) -> None:
        """
        Gets or sets the QuantConnect.Securities.Cash with the specified symbol.
        
        :param symbol: Symbol.
        """
        ...

    @typing.overload
    def GetEnumerator(self) -> System.Collections.Generic.IEnumerator[System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]]:
        """
        Gets the enumerator.
        
        :returns: The enumerator.
        """
        ...

    @typing.overload
    def GetEnumerator(self) -> System.Collections.IEnumerator:
        ...

    @typing.overload
    def ConvertToAccountCurrency(self, cashAmount: QuantConnect.Securities.CashAmount) -> QuantConnect.Securities.CashAmount:
        ...


class SecurityDatabaseKey(System.Object, System.IEquatable[QuantConnect_Securities_SecurityDatabaseKey]):
    """Represents the key to a single entry in the MarketHoursDatabase or the SymbolPropertiesDatabase"""

    Wildcard: str = "[*]"
    """Represents that the specified symbol or market field will match all"""

    @property
    def Market(self) -> str:
        """The market. If null, ignore market filtering"""
        ...

    @property
    def Symbol(self) -> str:
        """The symbol. If null, ignore symbol filtering"""
        ...

    @property
    def SecurityType(self) -> QuantConnect.SecurityType:
        """The security type"""
        ...

    def __init__(self, market: str, symbol: str, securityType: QuantConnect.SecurityType) -> None:
        """
        Initializes a new instance of the SecurityDatabaseKey class
        
        :param market: The market
        :param symbol: The symbol. specify null to apply to all symbols in market/security type
        :param securityType: The security type
        """
        ...

    @staticmethod
    def Parse(key: str) -> QuantConnect.Securities.SecurityDatabaseKey:
        """
        Parses the specified string as a SecurityDatabaseKey
        
        :param key: The string representation of the key
        :returns: A new SecurityDatabaseKey instance.
        """
        ...

    @typing.overload
    def Equals(self, other: QuantConnect.Securities.SecurityDatabaseKey) -> bool:
        ...

    @typing.overload
    def Equals(self, obj: typing.Any) -> bool:
        """
        Determines whether the specified object is equal to the current object.
        
        :param obj: The object to compare with the current object.
        :returns: true if the specified object  is equal to the current object; otherwise, false.
        """
        ...

    def GetHashCode(self) -> int:
        """
        Serves as the default hash function.
        
        :returns: A hash code for the current object.
        """
        ...

    def ToString(self) -> str:
        ...


class MarketHoursDatabase(System.Object):
    """Provides access to exchange hours and raw data times zones in various markets"""

    class Entry(System.Object):
        """Represents a single entry in the MarketHoursDatabase"""

        @property
        def DataTimeZone(self) -> typing.Any:
            """Gets the raw data time zone for this entry"""
            ...

        @property
        def ExchangeHours(self) -> QuantConnect.Securities.SecurityExchangeHours:
            """Gets the exchange hours for this entry"""
            ...

        def __init__(self, dataTimeZone: typing.Any, exchangeHours: QuantConnect.Securities.SecurityExchangeHours) -> None:
            """
            Initializes a new instance of the Entry class
            
            :param dataTimeZone: The raw data time zone
            :param exchangeHours: The security exchange hours for this entry
            """
            ...

    class AlwaysOpenMarketHoursDatabaseImpl(QuantConnect_Securities_MarketHoursDatabase):
        """This class has no documentation."""

        def GetEntry(self, market: str, symbol: str, securityType: QuantConnect.SecurityType) -> QuantConnect.Securities.MarketHoursDatabase.Entry:
            ...

        def __init__(self) -> None:
            ...

    @property
    def ExchangeHoursListing(self) -> System.Collections.Generic.List[System.Collections.Generic.KeyValuePair[QuantConnect.Securities.SecurityDatabaseKey, QuantConnect.Securities.MarketHoursDatabase.Entry]]:
        """Gets all the exchange hours held by this provider"""
        ...

    AlwaysOpen: QuantConnect.Securities.MarketHoursDatabase
    """Gets a MarketHoursDatabase that always returns SecurityExchangeHours.AlwaysOpen"""

    def __init__(self, exchangeHours: System.Collections.Generic.IReadOnlyDictionary[QuantConnect.Securities.SecurityDatabaseKey, QuantConnect.Securities.MarketHoursDatabase.Entry]) -> None:
        """
        Initializes a new instance of the MarketHoursDatabase class
        
        :param exchangeHours: The full listing of exchange hours by key
        """
        ...

    @typing.overload
    def GetExchangeHours(self, configuration: QuantConnect.Data.SubscriptionDataConfig) -> QuantConnect.Securities.SecurityExchangeHours:
        """
        Convenience method for retrieving exchange hours from market hours database using a subscription config
        
        :param configuration: The subscription data config to get exchange hours for
        :returns: The configure exchange hours for the specified configuration.
        """
        ...

    @typing.overload
    def GetExchangeHours(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType) -> QuantConnect.Securities.SecurityExchangeHours:
        """
        Convenience method for retrieving exchange hours from market hours database using a subscription config
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :returns: The exchange hours for the specified security.
        """
        ...

    def GetDataTimeZone(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType) -> typing.Any:
        """
        Performs a lookup using the specified information and returns the data's time zone if found,
        if an entry is not found, an exception is thrown
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :returns: The raw data time zone for the specified security.
        """
        ...

    @staticmethod
    def Reset() -> None:
        """
        Resets the market hours database, forcing a reload when reused.
        Called in tests where multiple algorithms are run sequentially,
        and we need to guarantee that every test starts with the same environment.
        """
        ...

    @staticmethod
    @typing.overload
    def FromDataFolder() -> QuantConnect.Securities.MarketHoursDatabase:
        """
        Gets the instance of the MarketHoursDatabase class produced by reading in the market hours
        data found in /Data/market-hours/
        
        :returns: A MarketHoursDatabase class that represents the data in the market-hours folder.
        """
        ...

    @staticmethod
    @typing.overload
    def FromDataFolder(dataFolder: str) -> QuantConnect.Securities.MarketHoursDatabase:
        """
        Gets the instance of the MarketHoursDatabase class produced by reading in the market hours
        data found in /Data/market-hours/
        
        :param dataFolder: Path to the data folder
        :returns: A MarketHoursDatabase class that represents the data in the market-hours folder.
        """
        ...

    @staticmethod
    def FromFile(path: str) -> QuantConnect.Securities.MarketHoursDatabase:
        """
        Reads the specified file as a market hours database instance
        
        :param path: The market hours database file path
        :returns: A new instance of the MarketHoursDatabase class.
        """
        ...

    def SetEntry(self, market: str, symbol: str, securityType: QuantConnect.SecurityType, exchangeHours: QuantConnect.Securities.SecurityExchangeHours, dataTimeZone: typing.Any = None) -> QuantConnect.Securities.MarketHoursDatabase.Entry:
        """
        Sets the entry for the specified market/symbol/security-type.
        This is intended to be used by custom data and other data sources that don't have explicit
        entries in market-hours-database.csv. At run time, the algorithm can update the market hours
        database via calls to AddData.
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :param exchangeHours: The exchange hours for the specified symbol
        :param dataTimeZone: The time zone of the symbol's raw data. Optional, defaults to the exchange time zone
        :returns: The entry matching the specified market/symbol/security-type.
        """
        ...

    def SetEntryAlwaysOpen(self, market: str, symbol: str, securityType: QuantConnect.SecurityType, timeZone: typing.Any) -> QuantConnect.Securities.MarketHoursDatabase.Entry:
        """
        Convenience method for the common custom data case.
        Sets the entry for the specified symbol using SecurityExchangeHours.AlwaysOpen(timeZone)
        This sets the data time zone equal to the exchange time zone as well.
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :param timeZone: The time zone of the symbol's exchange and raw data
        :returns: The entry matching the specified market/symbol/security-type.
        """
        ...

    @typing.overload
    def GetEntry(self, market: str, symbol: str, securityType: QuantConnect.SecurityType) -> QuantConnect.Securities.MarketHoursDatabase.Entry:
        """
        Gets the entry for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :returns: The entry matching the specified market/symbol/security-type.
        """
        ...

    @typing.overload
    def TryGetEntry(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType, entry: QuantConnect.Securities.MarketHoursDatabase.Entry) -> bool:
        """
        Tries to get the entry for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :param entry: The entry found if any
        :returns: True if the entry was present, else false.
        """
        ...

    @typing.overload
    def TryGetEntry(self, market: str, symbol: str, securityType: QuantConnect.SecurityType, entry: QuantConnect.Securities.MarketHoursDatabase.Entry) -> bool:
        """
        Tries to get the entry for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :param entry: The entry found if any
        :returns: True if the entry was present, else false.
        """
        ...

    @typing.overload
    def GetEntry(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType) -> QuantConnect.Securities.MarketHoursDatabase.Entry:
        """
        Gets the entry for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded (Symbol class)
        :param securityType: The security type of the symbol
        :returns: The entry matching the specified market/symbol/security-type.
        """
        ...

    @staticmethod
    def GetDatabaseSymbolKey(symbol: typing.Union[QuantConnect.Symbol, str]) -> str:
        """
        Gets the correct string symbol to use as a database key
        
        :param symbol: The symbol
        :returns: The symbol string used in the database ke.
        """
        ...

    def ContainsKey(self, key: QuantConnect.Securities.SecurityDatabaseKey) -> bool:
        """
        Determines if the database contains the specified key
        
        This method is protected.
        
        :param key: The key to search for
        :returns: True if an entry is found, otherwise false.
        """
        ...


class SymbolPropertiesDatabase(System.Object):
    """Provides access to specific properties for various symbols"""

    @typing.overload
    def ContainsKey(self, market: str, symbol: str, securityType: QuantConnect.SecurityType) -> bool:
        """
        Check whether symbol properties exists for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        """
        ...

    @typing.overload
    def ContainsKey(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType) -> bool:
        """
        Check whether symbol properties exists for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded (Symbol class)
        :param securityType: The security type of the symbol
        """
        ...

    def TryGetMarket(self, symbol: str, securityType: QuantConnect.SecurityType, market: str) -> bool:
        """
        Tries to get the market for the provided symbol/security type
        
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :param market: The market the exchange resides in Market
        :returns: True if market was retrieved, false otherwise.
        """
        ...

    def GetSymbolProperties(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType, defaultQuoteCurrency: str) -> QuantConnect.Securities.SymbolProperties:
        """
        Gets the symbol properties for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded (Symbol class)
        :param securityType: The security type of the symbol
        :param defaultQuoteCurrency: Specifies the quote currency to be used when returning a default instance of an entry is not found in the database
        :returns: The symbol properties matching the specified market/symbol/security-type or null if not found.
        """
        ...

    @typing.overload
    def GetSymbolPropertiesList(self, market: str, securityType: QuantConnect.SecurityType) -> System.Collections.Generic.IEnumerable[System.Collections.Generic.KeyValuePair[QuantConnect.Securities.SecurityDatabaseKey, QuantConnect.Securities.SymbolProperties]]:
        """
        Gets a list of symbol properties for the specified market/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param securityType: The security type of the symbol
        :returns: An IEnumerable of symbol properties matching the specified market/security-type.
        """
        ...

    @typing.overload
    def GetSymbolPropertiesList(self, market: str) -> System.Collections.Generic.IEnumerable[System.Collections.Generic.KeyValuePair[QuantConnect.Securities.SecurityDatabaseKey, QuantConnect.Securities.SymbolProperties]]:
        """
        Gets a list of symbol properties for the specified market
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :returns: An IEnumerable of symbol properties matching the specified market.
        """
        ...

    def SetEntry(self, market: str, symbol: str, securityType: QuantConnect.SecurityType, properties: QuantConnect.Securities.SymbolProperties) -> bool:
        """
        Set SymbolProperties entry for a particular market, symbol and security type.
        
        :param market: Market of the entry
        :param symbol: Symbol of the entry
        :param securityType: Type of security for the entry
        :param properties: The new symbol properties to store
        :returns: True if successful.
        """
        ...

    @staticmethod
    def FromDataFolder() -> QuantConnect.Securities.SymbolPropertiesDatabase:
        """
        Gets the instance of the SymbolPropertiesDatabase class produced by reading in the symbol properties
        data found in /Data/symbol-properties/
        
        :returns: A SymbolPropertiesDatabase class that represents the data in the symbol-properties folder.
        """
        ...


class SecurityCacheProvider(System.Object):
    """A helper class that will provide SecurityCache instances"""

    def __init__(self, securityProvider: QuantConnect.Securities.ISecurityProvider) -> None:
        """
        Creates a new instance
        
        :param securityProvider: The security provider to use
        """
        ...

    def GetSecurityCache(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Securities.SecurityCache:
        """
        Will return the SecurityCache instance to use for a give Symbol.
        If the provided Symbol is a custom type which has an underlying we will try to use the
        underlying SecurityCache type cache, if the underlying is not present we will keep track
        of the custom Symbol in case it is added later.
        
        :returns: The cache instance to use.
        """
        ...


class SecurityService(System.Object, QuantConnect.Interfaces.ISecurityService):
    """This class implements interface ISecurityService providing methods for creating new Security"""

    def __init__(self, cashBook: QuantConnect.Securities.CashBook, marketHoursDatabase: QuantConnect.Securities.MarketHoursDatabase, symbolPropertiesDatabase: QuantConnect.Securities.SymbolPropertiesDatabase, securityInitializerProvider: QuantConnect.Interfaces.ISecurityInitializerProvider, registeredTypes: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider, cacheProvider: QuantConnect.Securities.SecurityCacheProvider, primaryExchangeProvider: QuantConnect.Interfaces.IPrimaryExchangeProvider = None) -> None:
        """Creates a new instance of the SecurityService class"""
        ...

    @typing.overload
    def CreateSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str], subscriptionDataConfigList: System.Collections.Generic.List[QuantConnect.Data.SubscriptionDataConfig], leverage: float = 0, addToSymbolCache: bool = True) -> QuantConnect.Securities.Security:
        """Creates a new security"""
        ...

    @typing.overload
    def CreateSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str], subscriptionDataConfig: QuantConnect.Data.SubscriptionDataConfig, leverage: float = 0, addToSymbolCache: bool = True) -> QuantConnect.Securities.Security:
        """Creates a new security"""
        ...

    def SetLiveMode(self, isLiveMode: bool) -> None:
        """
        Set live mode state of the algorithm
        
        :param isLiveMode: True, live mode is enabled
        """
        ...


class SecurityManager(QuantConnect.ExtendedDictionary[QuantConnect.Securities.Security], System.Collections.Generic.IDictionary[QuantConnect.Symbol, QuantConnect.Securities.Security], INotifyCollectionChanged, typing.Iterable[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]]):
    """Enumerable security management class for grouping security objects into an array and providing any common properties."""

    @property
    def CollectionChanged(self) -> typing.List[typing.Any]:
        """Event fired when a security is added or removed from this collection"""
        ...

    @CollectionChanged.setter
    def CollectionChanged(self, value: typing.List[typing.Any]):
        """Event fired when a security is added or removed from this collection"""
        ...

    @property
    def UtcTime(self) -> datetime.datetime:
        """Gets the most recent time this manager was updated"""
        ...

    @property
    def Count(self) -> int:
        """Count of the number of securities in the collection."""
        ...

    @property
    def IsReadOnly(self) -> bool:
        """Flag indicating if the internal array is read only."""
        ...

    @property
    def Keys(self) -> System.Collections.Generic.ICollection[QuantConnect.Symbol]:
        """List of the symbol-keys in the collection of securities."""
        ...

    @property
    def GetKeys(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Symbol]:
        """
        Gets an System.Collections.Generic.ICollection`1 containing the Symbol objects of the System.Collections.Generic.IDictionary`2.
        
        This property is protected.
        """
        ...

    @property
    def GetValues(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Securities.Security]:
        """
        Gets an System.Collections.Generic.ICollection`1 containing the values in the System.Collections.Generic.IDictionary`2.
        
        This property is protected.
        """
        ...

    @property
    def Values(self) -> System.Collections.Generic.ICollection[QuantConnect.Securities.Security]:
        """Get a list of the security objects for this collection."""
        ...

    def __init__(self, timeKeeper: QuantConnect.Interfaces.ITimeKeeper) -> None:
        """Initialise the algorithm security manager with two empty dictionaries"""
        ...

    @typing.overload
    def Add(self, symbol: typing.Union[QuantConnect.Symbol, str], security: QuantConnect.Securities.Security) -> None:
        """
        Add a new security with this symbol to the collection.
        
        :param symbol: symbol for security we're trading
        :param security: security object
        """
        ...

    @typing.overload
    def Add(self, security: QuantConnect.Securities.Security) -> None:
        """
        Add a new security with this symbol to the collection.
        
        :param security: security object
        """
        ...

    @typing.overload
    def Add(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]) -> None:
        """Add a symbol-security by its key value pair."""
        ...

    def Clear(self) -> None:
        """Clear the securities array to delete all the portfolio and asset information."""
        ...

    def Contains(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]) -> bool:
        """
        Check if this collection contains this key value pair.
        
        :param pair: Search key-value pair
        :returns: Bool true if contains this key-value pair.
        """
        ...

    def ContainsKey(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Check if this collection contains this symbol.
        
        :param symbol: Symbol we're checking for.
        :returns: Bool true if contains this symbol pair.
        """
        ...

    def CopyTo(self, array: typing.List[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]], number: int) -> None:
        """
        Copy from the internal array to an external array.
        
        :param array: Array we're outputting to
        :param number: Starting index of array
        """
        ...

    @typing.overload
    def Remove(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]) -> bool:
        """
        Remove a key value of of symbol-securities from the collections.
        
        :param pair: Key Value pair of symbol-security to remove
        :returns: Boolean true on success.
        """
        ...

    @typing.overload
    def Remove(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Remove this symbol security: Dictionary interface implementation.
        
        :param symbol: Symbol we're searching for
        :returns: true success.
        """
        ...

    def TryGetValue(self, symbol: typing.Union[QuantConnect.Symbol, str], security: QuantConnect.Securities.Security) -> bool:
        """
        Try and get this security object with matching symbol and return true on success.
        
        :param symbol: String search symbol
        :param security: Output Security object
        :returns: True on successfully locating the security object.
        """
        ...

    @typing.overload
    def GetEnumerator(self) -> System.Collections.Generic.IEnumerator[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]]:
        """
        Get the enumerator for this security collection.
        
        :returns: Enumerable key value pair.
        """
        ...

    @typing.overload
    def GetEnumerator(self) -> System.Collections.IEnumerator:
        """
        Get the enumerator for this securities collection.
        
        :returns: Enumerator.
        """
        ...

    def __getitem__(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Securities.Security:
        """
        Indexer method for the security manager to access the securities objects by their symbol.
        
        :param symbol: Symbol object indexer
        :returns: Security.
        """
        ...

    def __setitem__(self, symbol: typing.Union[QuantConnect.Symbol, str], value: QuantConnect.Securities.Security) -> None:
        """
        Indexer method for the security manager to access the securities objects by their symbol.
        
        :param symbol: Symbol object indexer
        :returns: Security.
        """
        ...

    def OnCollectionChanged(self, changedEventArgs: typing.Any) -> None:
        """
        Event invocator for the CollectionChanged event
        
        This method is protected.
        
        :param changedEventArgs: Event arguments for the CollectionChanged event
        """
        ...

    def SetSecurityService(self, securityService: QuantConnect.Securities.SecurityService) -> None:
        """Sets the Security Service to be used"""
        ...

    @typing.overload
    def CreateSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str], subscriptionDataConfigList: System.Collections.Generic.List[QuantConnect.Data.SubscriptionDataConfig], leverage: float = 0, addToSymbolCache: bool = True) -> QuantConnect.Securities.Security:
        """Creates a new security"""
        ...

    @typing.overload
    def CreateSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str], subscriptionDataConfig: QuantConnect.Data.SubscriptionDataConfig, leverage: float = 0, addToSymbolCache: bool = True) -> QuantConnect.Securities.Security:
        """Creates a new security"""
        ...

    def SetLiveMode(self, isLiveMode: bool) -> None:
        """
        Set live mode state of the algorithm
        
        :param isLiveMode: True, live mode is enabled
        """
        ...


class IOrderProvider(metaclass=abc.ABCMeta):
    """Represents a type capable of fetching Order instances by its QC order id or by a brokerage id"""

    @property
    @abc.abstractmethod
    def OrdersCount(self) -> int:
        """Gets the current number of orders that have been processed"""
        ...

    def GetOrderById(self, orderId: int) -> QuantConnect.Orders.Order:
        """
        Get the order by its id
        
        :param orderId: Order id to fetch
        :returns: A clone of the order with the specified id, or null if no match is found.
        """
        ...

    def GetOrderByBrokerageId(self, brokerageId: str) -> QuantConnect.Orders.Order:
        """
        Gets the order by its brokerage id
        
        :param brokerageId: The brokerage id to fetch
        :returns: The first order matching the brokerage id, or null if no match is found.
        """
        ...

    def GetOrderTickets(self, filter: typing.Callable[[QuantConnect.Orders.OrderTicket], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Gets and enumerable of OrderTicket matching the specified
        
        :param filter: The filter predicate used to find the required order tickets. If null is specified then all tickets are returned
        :returns: An enumerable of OrderTicket matching the specified.
        """
        ...

    def GetOpenOrderTickets(self, filter: typing.Callable[[QuantConnect.Orders.OrderTicket], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Gets and enumerable of opened OrderTicket matching the specified
        
        :param filter: The filter predicate used to find the required order tickets. If null is specified then all tickets are returned
        :returns: An enumerable of opened OrderTicket matching the specified.
        """
        ...

    def GetOrderTicket(self, orderId: int) -> QuantConnect.Orders.OrderTicket:
        """
        Gets the order ticket for the specified order id. Returns null if not found
        
        :param orderId: The order's id
        :returns: The order ticket with the specified id, or null if not found.
        """
        ...

    def GetOrders(self, filter: typing.Callable[[QuantConnect.Orders.Order], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.Order]:
        """
        Gets all orders matching the specified filter. Specifying null will return an enumerable
        of all orders.
        
        :param filter: Delegate used to filter the orders
        :returns: All orders this order provider currently holds by the specified filter.
        """
        ...

    def GetOpenOrders(self, filter: typing.Callable[[QuantConnect.Orders.Order], bool] = None) -> System.Collections.Generic.List[QuantConnect.Orders.Order]:
        """
        Gets open orders matching the specified filter. Specifying null will return an enumerable
        of all open orders.
        
        :param filter: Delegate used to filter the orders
        :returns: All filtered open orders this order provider currently holds.
        """
        ...


class IOrderProcessor(QuantConnect.Securities.IOrderProvider, metaclass=abc.ABCMeta):
    """Represents a type capable of processing orders"""

    def Process(self, request: QuantConnect.Orders.OrderRequest) -> QuantConnect.Orders.OrderTicket:
        """
        Adds the specified order to be processed
        
        :param request: The OrderRequest to be processed
        :returns: The OrderTicket for the corresponding OrderRequest.OrderId.
        """
        ...


class SecurityTransactionManager(System.Object, QuantConnect.Securities.IOrderProvider):
    """Algorithm Transactions Manager - Recording Transactions"""

    @property
    def UtcTime(self) -> datetime.datetime:
        """Gets the time the security information was last updated"""
        ...

    @property
    def TransactionRecord(self) -> System.Collections.Generic.Dictionary[datetime.datetime, float]:
        """Trade record of profits and losses for each trade statistics calculations"""
        ...

    @property
    def MinimumOrderSize(self) -> float:
        """Configurable minimum order value to ignore bad orders, or orders with unrealistic sizes"""
        ...

    @property
    def MinimumOrderQuantity(self) -> int:
        """Configurable minimum order size to ignore bad orders, or orders with unrealistic sizes"""
        ...

    @property
    def LastOrderId(self) -> int:
        """Get the last order id."""
        ...

    @property
    def MarketOrderFillTimeout(self) -> datetime.timedelta:
        """Configurable timeout for market order fills"""
        ...

    @MarketOrderFillTimeout.setter
    def MarketOrderFillTimeout(self, value: datetime.timedelta):
        """Configurable timeout for market order fills"""
        ...

    @property
    def OrdersCount(self) -> int:
        """Gets the current number of orders that have been processed"""
        ...

    def __init__(self, algorithm: QuantConnect.Interfaces.IAlgorithm, security: QuantConnect.Securities.SecurityManager) -> None:
        """Initialise the transaction manager for holding and processing orders."""
        ...

    def ProcessRequest(self, request: QuantConnect.Orders.OrderRequest) -> QuantConnect.Orders.OrderTicket:
        """
        Processes the order request
        
        :param request: The request to be processed
        :returns: The order ticket for the request.
        """
        ...

    def AddOrder(self, request: QuantConnect.Orders.SubmitOrderRequest) -> QuantConnect.Orders.OrderTicket:
        """
        Add an order to collection and return the unique order id or negative if an error.
        
        :param request: A request detailing the order to be submitted
        :returns: New unique, increasing orderid.
        """
        ...

    def UpdateOrder(self, request: QuantConnect.Orders.UpdateOrderRequest) -> QuantConnect.Orders.OrderTicket:
        """
        Update an order yet to be filled such as stop or limit orders.
        
        :param request: Request detailing how the order should be updated
        """
        ...

    def CancelOrder(self, orderId: int, orderTag: str = None) -> QuantConnect.Orders.OrderTicket:
        """
        Added alias for RemoveOrder -
        
        :param orderId: Order id we wish to cancel
        :param orderTag: Tag to indicate from where this method was called
        """
        ...

    @typing.overload
    def CancelOpenOrders(self) -> System.Collections.Generic.List[QuantConnect.Orders.OrderTicket]:
        """
        Cancels all open orders for all symbols
        
        :returns: List containing the cancelled order tickets.
        """
        ...

    @typing.overload
    def CancelOpenOrders(self, symbol: typing.Union[QuantConnect.Symbol, str], tag: str = None) -> System.Collections.Generic.List[QuantConnect.Orders.OrderTicket]:
        """
        Cancels all open orders for the specified symbol
        
        :param symbol: The symbol whose orders are to be cancelled
        :param tag: Custom order tag
        :returns: List containing the cancelled order tickets.
        """
        ...

    def RemoveOrder(self, orderId: int, tag: str = None) -> QuantConnect.Orders.OrderTicket:
        """
        Remove this order from outstanding queue: user is requesting a cancel.
        
        :param orderId: Specific order id to remove
        :param tag: Tag request
        """
        ...

    def GetOrderTickets(self, filter: typing.Callable[[QuantConnect.Orders.OrderTicket], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Gets an enumerable of OrderTicket matching the specified
        
        :param filter: The filter predicate used to find the required order tickets
        :returns: An enumerable of OrderTicket matching the specified.
        """
        ...

    @typing.overload
    def GetOpenOrderTickets(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Get an enumerable of open OrderTicket for the specified symbol
        
        :param symbol: The symbol for which to return the order tickets
        :returns: An enumerable of open OrderTicket.
        """
        ...

    @typing.overload
    def GetOpenOrderTickets(self, filter: typing.Callable[[QuantConnect.Orders.OrderTicket], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Gets an enumerable of opened OrderTicket matching the specified
        
        :param filter: The filter predicate used to find the required order tickets
        :returns: An enumerable of opened OrderTicket matching the specified.
        """
        ...

    @typing.overload
    def GetOpenOrdersRemainingQuantity(self, filter: typing.Callable[[QuantConnect.Orders.OrderTicket], bool] = None) -> float:
        """
        Gets the remaining quantity to be filled from open orders, i.e. order size minus quantity filled
        
        :param filter: Filters the order tickets to be included in the aggregate quantity remaining to be filled
        :returns: Total quantity that hasn't been filled yet for all orders that were not filtered.
        """
        ...

    @typing.overload
    def GetOpenOrdersRemainingQuantity(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> float:
        """
        Gets the remaining quantity to be filled from open orders for a Symbol, i.e. order size minus quantity filled
        
        :param symbol: Symbol to get the remaining quantity of currently open orders
        :returns: Total quantity that hasn't been filled yet for orders matching the Symbol.
        """
        ...

    def GetOrderTicket(self, orderId: int) -> QuantConnect.Orders.OrderTicket:
        """
        Gets the order ticket for the specified order id. Returns null if not found
        
        :param orderId: The order's id
        :returns: The order ticket with the specified id, or null if not found.
        """
        ...

    def WaitForOrder(self, orderId: int) -> bool:
        """
        Wait for a specific order to be either Filled, Invalid or Canceled
        
        :param orderId: The id of the order to wait for
        :returns: True if we successfully wait for the fill, false if we were unable to wait. This may be because it is not a market order or because the timeout was reached.
        """
        ...

    @typing.overload
    def GetOpenOrders(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> System.Collections.Generic.List[QuantConnect.Orders.Order]:
        """
        Get a list of all open orders for a symbol.
        
        :param symbol: The symbol for which to return the orders
        :returns: List of open orders.
        """
        ...

    @typing.overload
    def GetOpenOrders(self, filter: typing.Callable[[QuantConnect.Orders.Order], bool] = None) -> System.Collections.Generic.List[QuantConnect.Orders.Order]:
        """
        Gets open orders matching the specified filter. Specifying null will return an enumerable
        of all open orders.
        
        :param filter: Delegate used to filter the orders
        :returns: All filtered open orders this order provider currently holds.
        """
        ...

    def GetOrderById(self, orderId: int) -> QuantConnect.Orders.Order:
        """
        Get the order by its id
        
        :param orderId: Order id to fetch
        :returns: A clone of the order with the specified id, or null if no match is found.
        """
        ...

    def GetOrderByBrokerageId(self, brokerageId: str) -> QuantConnect.Orders.Order:
        """
        Gets the order by its brokerage id
        
        :param brokerageId: The brokerage id to fetch
        :returns: The first order matching the brokerage id, or null if no match is found.
        """
        ...

    def GetOrders(self, filter: typing.Callable[[QuantConnect.Orders.Order], bool]) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.Order]:
        """
        Gets all orders matching the specified filter. Specifying null will return an enumerable
        of all orders.
        
        :param filter: Delegate used to filter the orders
        :returns: All orders this order provider currently holds by the specified filter.
        """
        ...

    def GetIncrementOrderId(self) -> int:
        """
        Get a new order id, and increment the internal counter.
        
        :returns: New unique int order id.
        """
        ...

    def SetOrderProcessor(self, orderProvider: QuantConnect.Securities.IOrderProcessor) -> None:
        """
        Sets the IOrderProvider used for fetching orders for the algorithm
        
        :param orderProvider: The IOrderProvider to be used to manage fetching orders
        """
        ...

    def AddTransactionRecord(self, time: datetime.datetime, transactionProfitLoss: float) -> None:
        """
        Record the transaction value and time in a list to later be processed for statistics creation.
        
        :param time: Time of order processed
        :param transactionProfitLoss: Profit Loss.
        """
        ...


class IMarginCallModel(metaclass=abc.ABCMeta):
    """Represents the model responsible for picking which orders should be executed during a margin call"""

    def GetMarginCallOrders(self, issueMarginCallWarning: bool) -> System.Collections.Generic.List[QuantConnect.Orders.SubmitOrderRequest]:
        """
        Scan the portfolio and the updated data for a potential margin call situation which may get the holdings below zero!
        If there is a margin call, liquidate the portfolio immediately before the portfolio gets sub zero.
        
        :param issueMarginCallWarning: Set to true if a warning should be issued to the algorithm
        :returns: True for a margin call on the holdings.
        """
        ...

    def ExecuteMarginCall(self, generatedMarginCallOrders: System.Collections.Generic.IEnumerable[QuantConnect.Orders.SubmitOrderRequest]) -> System.Collections.Generic.List[QuantConnect.Orders.OrderTicket]:
        """
        Executes synchronous orders to bring the account within margin requirements.
        
        :param generatedMarginCallOrders: These are the margin call orders that were generated by individual security margin models.
        :returns: The list of orders that were actually executed.
        """
        ...


class UnsettledCashAmount(System.Object):
    """Represents a pending cash amount waiting for settlement time"""

    @property
    def SettlementTimeUtc(self) -> datetime.datetime:
        """The settlement time (in UTC)"""
        ...

    @SettlementTimeUtc.setter
    def SettlementTimeUtc(self, value: datetime.datetime):
        """The settlement time (in UTC)"""
        ...

    @property
    def Currency(self) -> str:
        """The currency symbol"""
        ...

    @Currency.setter
    def Currency(self, value: str):
        """The currency symbol"""
        ...

    @property
    def Amount(self) -> float:
        """The amount of cash"""
        ...

    @Amount.setter
    def Amount(self, value: float):
        """The amount of cash"""
        ...

    def __init__(self, settlementTimeUtc: datetime.datetime, currency: str, amount: float) -> None:
        """Creates a new instance of the UnsettledCashAmount class"""
        ...


class SecurityPortfolioManager(QuantConnect.ExtendedDictionary[QuantConnect.Securities.SecurityHolding], System.Collections.Generic.IDictionary[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding], QuantConnect.Securities.ISecurityProvider, typing.Iterable[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]]):
    """
    Portfolio manager class groups popular properties and makes them accessible through one interface.
    It also provide indexing by the vehicle symbol to get the Security.Holding objects.
    """

    @property
    def Securities(self) -> QuantConnect.Securities.SecurityManager:
        """Local access to the securities collection for the portfolio summation."""
        ...

    @Securities.setter
    def Securities(self, value: QuantConnect.Securities.SecurityManager):
        """Local access to the securities collection for the portfolio summation."""
        ...

    @property
    def Transactions(self) -> QuantConnect.Securities.SecurityTransactionManager:
        """Local access to the transactions collection for the portfolio summation and updates."""
        ...

    @Transactions.setter
    def Transactions(self, value: QuantConnect.Securities.SecurityTransactionManager):
        """Local access to the transactions collection for the portfolio summation and updates."""
        ...

    @property
    def CashBook(self) -> QuantConnect.Securities.CashBook:
        """Gets the cash book that keeps track of all currency holdings (only settled cash)"""
        ...

    @property
    def UnsettledCashBook(self) -> QuantConnect.Securities.CashBook:
        """Gets the cash book that keeps track of all currency holdings (only unsettled cash)"""
        ...

    @property
    def Count(self) -> int:
        """Count the securities objects in the portfolio."""
        ...

    @property
    def IsReadOnly(self) -> bool:
        """Check if the underlying securities array is read only."""
        ...

    @property
    def GetKeys(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Symbol]:
        """
        Gets an System.Collections.Generic.ICollection`1 containing the Symbol objects of the System.Collections.Generic.IDictionary`2.
        
        This property is protected.
        """
        ...

    @property
    def GetValues(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Securities.SecurityHolding]:
        """
        Gets an System.Collections.Generic.ICollection`1 containing the values in the System.Collections.Generic.IDictionary`2.
        
        This property is protected.
        """
        ...

    @property
    def Keys(self) -> System.Collections.Generic.ICollection[QuantConnect.Symbol]:
        """Symbol keys collection of the underlying assets in the portfolio."""
        ...

    @property
    def Values(self) -> System.Collections.Generic.ICollection[QuantConnect.Securities.SecurityHolding]:
        """Collection of securities objects in the portfolio."""
        ...

    @property
    def Cash(self) -> float:
        ...

    @property
    def UnsettledCash(self) -> float:
        """Sum of all currencies in account in US dollars (only unsettled cash)"""
        ...

    @property
    def TotalUnleveredAbsoluteHoldingsCost(self) -> float:
        """Absolute value of cash discounted from our total cash by the holdings we own."""
        ...

    @property
    def TotalAbsoluteHoldingsCost(self) -> float:
        """
        Gets the total absolute holdings cost of the portfolio. This sums up the individual
        absolute cost of each holding
        """
        ...

    @property
    def TotalHoldingsValue(self) -> float:
        """Absolute sum the individual items in portfolio."""
        ...

    @property
    def HoldStock(self) -> bool:
        """Boolean flag indicating we have any holdings in the portfolio."""
        ...

    @property
    def Invested(self) -> bool:
        """Alias for HoldStock. Check if we have and holdings."""
        ...

    @property
    def TotalUnrealisedProfit(self) -> float:
        """Get the total unrealised profit in our portfolio from the individual security unrealized profits."""
        ...

    @property
    def TotalUnrealizedProfit(self) -> float:
        """Get the total unrealised profit in our portfolio from the individual security unrealized profits."""
        ...

    @property
    def TotalPortfolioValue(self) -> float:
        """Total portfolio value if we sold all holdings at current market rates."""
        ...

    @property
    def TotalFees(self) -> float:
        """Total fees paid during the algorithm operation across all securities in portfolio."""
        ...

    @property
    def TotalProfit(self) -> float:
        """Sum of all gross profit across all securities in portfolio."""
        ...

    @property
    def TotalSaleVolume(self) -> float:
        """Total sale volume since the start of algorithm operations."""
        ...

    @property
    def TotalMarginUsed(self) -> float:
        """Gets the total margin used across all securities in the account's currency"""
        ...

    @property
    def MarginRemaining(self) -> float:
        """Gets the remaining margin on the account in the account's currency"""
        ...

    @property
    def MarginCallModel(self) -> QuantConnect.Securities.IMarginCallModel:
        """
        Gets or sets the MarginCallModel for the portfolio. This
        is used to executed margin call orders.
        """
        ...

    @MarginCallModel.setter
    def MarginCallModel(self, value: QuantConnect.Securities.IMarginCallModel):
        """
        Gets or sets the MarginCallModel for the portfolio. This
        is used to executed margin call orders.
        """
        ...

    def __init__(self, securityManager: QuantConnect.Securities.SecurityManager, transactions: QuantConnect.Securities.SecurityTransactionManager, defaultOrderProperties: QuantConnect.Interfaces.IOrderProperties = None) -> None:
        """Initialise security portfolio manager."""
        ...

    @typing.overload
    def Add(self, symbol: typing.Union[QuantConnect.Symbol, str], holding: QuantConnect.Securities.SecurityHolding) -> None:
        ...

    @typing.overload
    def Add(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]) -> None:
        """
        Add a new securities key value pair to the portfolio.
        
        :param pair: Key value pair of dictionary
        """
        ...

    def Clear(self) -> None:
        """Clear the portfolio of securities objects."""
        ...

    @typing.overload
    def Remove(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]) -> bool:
        """
        Remove this keyvalue pair from the portfolio.
        
        :param pair: Key value pair of dictionary
        """
        ...

    @typing.overload
    def Remove(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Remove this symbol from the portfolio.
        
        :param symbol: Symbol of dictionary
        """
        ...

    def ContainsKey(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Check if the portfolio contains this symbol string.
        
        :param symbol: String search symbol for the security
        :returns: Boolean true if portfolio contains this symbol.
        """
        ...

    def Contains(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]) -> bool:
        """
        Check if the key-value pair is in the portfolio.
        
        :param pair: Pair we're searching for
        :returns: True if we have this object.
        """
        ...

    def CopyTo(self, array: typing.List[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]], index: int) -> None:
        """
        Copy contents of the portfolio collection to a new destination.
        
        :param array: Destination array
        :param index: Position in array to start copying
        """
        ...

    def TryGetValue(self, symbol: typing.Union[QuantConnect.Symbol, str], holding: QuantConnect.Securities.SecurityHolding) -> bool:
        """
        Attempt to get the value of the securities holding class if this symbol exists.
        
        :param symbol: String search symbol
        :param holding: Holdings object of this security
        :returns: Boolean true if successful locating and setting the holdings object.
        """
        ...

    @typing.overload
    def GetEnumerator(self) -> System.Collections.Generic.IEnumerator[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]]:
        """
        Get the enumerator for the underlying securities collection.
        
        :returns: Enumerable key value pair.
        """
        ...

    @typing.overload
    def GetEnumerator(self) -> System.Collections.IEnumerator:
        """
        Get the enumerator for the underlying securities collection.
        
        :returns: Enumerator.
        """
        ...

    def InvalidateTotalPortfolioValue(self) -> None:
        """
        Will flag the current TotalPortfolioValue as invalid
        so it is recalculated when gotten
        """
        ...

    @typing.overload
    def GetMarginRemaining(self, totalPortfolioValue: float) -> float:
        """
        Gets the remaining margin on the account in the account's currency
        for the given total portfolio value
        
        :param totalPortfolioValue: The total portfolio value TotalPortfolioValue
        """
        ...

    def __getitem__(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Securities.SecurityHolding:
        """
        Indexer for the PortfolioManager class to access the underlying security holdings objects.
        
        :param symbol: Symbol object indexer
        :returns: SecurityHolding class from the algorithm securities.
        """
        ...

    def __setitem__(self, symbol: typing.Union[QuantConnect.Symbol, str], value: QuantConnect.Securities.SecurityHolding) -> None:
        """
        Indexer for the PortfolioManager class to access the underlying security holdings objects.
        
        :param symbol: Symbol object indexer
        :returns: SecurityHolding class from the algorithm securities.
        """
        ...

    def SetAccountCurrency(self, accountCurrency: str) -> None:
        """
        Sets the account currency cash symbol this algorithm is to manage.
        
        :param accountCurrency: The account currency cash symbol to set
        """
        ...

    @typing.overload
    def SetCash(self, cash: float) -> None:
        """
        Set the account currency cash this algorithm is to manage.
        
        :param cash: Decimal cash value of portfolio
        """
        ...

    @typing.overload
    def SetCash(self, symbol: str, cash: float, conversionRate: float) -> None:
        """
        Set the cash for the specified symbol
        
        :param symbol: The cash symbol to set
        :param cash: Decimal cash value of portfolio
        :param conversionRate: The current conversion rate for the
        """
        ...

    @typing.overload
    def GetMarginRemaining(self, symbol: typing.Union[QuantConnect.Symbol, str], direction: QuantConnect.Orders.OrderDirection = ...) -> float:
        """
        Gets the margin available for trading a specific symbol in a specific direction.
        
        :param symbol: The symbol to compute margin remaining for
        :param direction: The order/trading direction
        :returns: The maximum order size that is currently executable in the specified direction.
        """
        ...

    def GetBuyingPower(self, symbol: typing.Union[QuantConnect.Symbol, str], direction: QuantConnect.Orders.OrderDirection = ...) -> float:
        """
        Gets the margin available for trading a specific symbol in a specific direction.
        Alias for GetMarginRemaining
        
        :param symbol: The symbol to compute margin remaining for
        :param direction: The order/trading direction
        :returns: The maximum order size that is currently executable in the specified direction.
        """
        ...

    def ProcessFill(self, fill: QuantConnect.Orders.OrderEvent) -> None:
        """Calculate the new average price after processing a partial/complete order fill event."""
        ...

    def ApplyDividend(self, dividend: QuantConnect.Data.Market.Dividend, liveMode: bool, mode: QuantConnect.DataNormalizationMode) -> None:
        """
        Applies a dividend to the portfolio
        
        :param dividend: The dividend to be applied
        :param liveMode: True if live mode, false for backtest
        :param mode: The DataNormalizationMode for this security
        """
        ...

    def ApplySplit(self, split: QuantConnect.Data.Market.Split, liveMode: bool, mode: QuantConnect.DataNormalizationMode) -> None:
        """
        Applies a split to the portfolio
        
        :param split: The split to be applied
        :param liveMode: True if live mode, false for backtest
        :param mode: The DataNormalizationMode for this security
        """
        ...

    def AddTransactionRecord(self, time: datetime.datetime, transactionProfitLoss: float) -> None:
        """
        Record the transaction value and time in a list to later be processed for statistics creation.
        
        :param time: Time of order processed
        :param transactionProfitLoss: Profit Loss.
        """
        ...

    def GetSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Securities.Security:
        """
        Retrieves a summary of the holdings for the specified symbol
        
        :param symbol: The symbol to get holdings for
        :returns: The holdings for the symbol or null if the symbol is invalid and/or not in the portfolio.
        """
        ...

    def AddUnsettledCashAmount(self, item: QuantConnect.Securities.UnsettledCashAmount) -> None:
        """
        Adds an item to the list of unsettled cash amounts
        
        :param item: The item to add
        """
        ...

    def ScanForCashSettlement(self, timeUtc: datetime.datetime) -> None:
        """Scan the portfolio to check if unsettled funds should be settled"""
        ...

    def LogMarginInformation(self, orderRequest: QuantConnect.Orders.OrderRequest = None) -> None:
        """Logs margin information for debugging"""
        ...

    @typing.overload
    def SetMarginCallModel(self, marginCallModel: QuantConnect.Securities.IMarginCallModel) -> None:
        """
        Sets the margin call model
        
        :param marginCallModel: Model that represents a portfolio's model to executed margin call orders.
        """
        ...

    @typing.overload
    def SetMarginCallModel(self, pyObject: typing.Any) -> None:
        """
        Sets the margin call model
        
        :param pyObject: Model that represents a portfolio's model to executed margin call orders.
        """
        ...


class HasSufficientBuyingPowerForOrderParameters(System.Object):
    """Defines the parameters for IBuyingPowerModel.HasSufficientBuyingPowerForOrder"""

    @property
    def Portfolio(self) -> QuantConnect.Securities.SecurityPortfolioManager:
        """Gets the algorithm's portfolio"""
        ...

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def Order(self) -> QuantConnect.Orders.Order:
        """Gets the order"""
        ...

    def __init__(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, order: QuantConnect.Orders.Order) -> None:
        """
        Initializes a new instance of the HasSufficientBuyingPowerForOrderParameters class
        
        :param portfolio: The algorithm's portfolio
        :param security: The security
        :param order: The order
        """
        ...


class GetMaximumOrderQuantityForDeltaBuyingPowerParameters(System.Object):
    """Defines the parameters for IBuyingPowerModel.GetMaximumOrderQuantityForDeltaBuyingPower"""

    @property
    def Portfolio(self) -> QuantConnect.Securities.SecurityPortfolioManager:
        """Gets the algorithm's portfolio"""
        ...

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def DeltaBuyingPower(self) -> float:
        """The delta buying power."""
        ...

    @property
    def SilenceNonErrorReasons(self) -> bool:
        """
        True enables the IBuyingPowerModel to skip setting GetMaximumOrderQuantityResult.Reason
        for non error situations, for performance
        """
        ...

    def __init__(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, deltaBuyingPower: float, silenceNonErrorReasons: bool = False) -> None:
        """
        Initializes a new instance of the GetMaximumOrderQuantityForDeltaBuyingPowerParameters class
        
        :param portfolio: The algorithm's portfolio
        :param security: The security
        :param deltaBuyingPower: The delta buying power to apply. Sign defines the position side to apply the delta
        :param silenceNonErrorReasons: True will not return GetMaximumOrderQuantityResult.Reason set for non error situation, this is for performance
        """
        ...


class GetMaximumOrderQuantityForTargetBuyingPowerParameters(System.Object):
    """Defines the parameters for IBuyingPowerModel.GetMaximumOrderQuantityForTargetBuyingPower"""

    @property
    def Portfolio(self) -> QuantConnect.Securities.SecurityPortfolioManager:
        """Gets the algorithm's portfolio"""
        ...

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def TargetBuyingPower(self) -> float:
        """Gets the target signed percentage buying power"""
        ...

    @property
    def SilenceNonErrorReasons(self) -> bool:
        """
        True enables the IBuyingPowerModel to skip setting GetMaximumOrderQuantityResult.Reason
        for non error situations, for performance
        """
        ...

    def __init__(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, targetBuyingPower: float, silenceNonErrorReasons: bool = False) -> None:
        """
        Initializes a new instance of the GetMaximumOrderQuantityForTargetBuyingPowerParameters class
        
        :param portfolio: The algorithm's portfolio
        :param security: The security
        :param targetBuyingPower: The target percentage buying power
        :param silenceNonErrorReasons: True will not return GetMaximumOrderQuantityResult.Reason set for non error situation, this is for performance
        """
        ...


class ReservedBuyingPowerForPositionParameters(System.Object):
    """Defines the parameters for IBuyingPowerModel.GetReservedBuyingPowerForPosition"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    def __init__(self, security: QuantConnect.Securities.Security) -> None:
        """
        Initializes a new instance of the ReservedBuyingPowerForPositionParameters class
        
        :param security: The security
        """
        ...

    def ResultInAccountCurrency(self, reservedBuyingPower: float) -> QuantConnect.Securities.ReservedBuyingPowerForPosition:
        """
        Creates the result using the specified reserved buying power in units of the account currency
        
        :param reservedBuyingPower: The reserved buying power in units of the account currency
        :returns: The reserved buying power.
        """
        ...


class BuyingPowerParameters(System.Object):
    """Defines the parameters for IBuyingPowerModel.GetBuyingPower"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def Portfolio(self) -> QuantConnect.Securities.SecurityPortfolioManager:
        """Gets the algorithm's portfolio"""
        ...

    @property
    def Direction(self) -> int:
        """
        Gets the direction in which buying power is to be computed
        
        This property contains the int value of a member of the QuantConnect.Orders.OrderDirection enum.
        """
        ...

    def __init__(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, direction: QuantConnect.Orders.OrderDirection) -> None:
        """
        Initializes a new instance of the BuyingPowerParameters class
        
        :param portfolio: The algorithm's portfolio
        :param security: The security
        :param direction: The direction to compute buying power in
        """
        ...

    def Result(self, buyingPower: float, currency: str) -> QuantConnect.Securities.BuyingPower:
        """
        Creates the result using the specified buying power
        
        :param buyingPower: The buying power
        :param currency: The units the buying power is denominated in
        :returns: The buying power.
        """
        ...

    def ResultInAccountCurrency(self, buyingPower: float) -> QuantConnect.Securities.BuyingPower:
        """
        Creates the result using the specified buying power in units of the account currency
        
        :param buyingPower: The buying power
        :returns: The buying power.
        """
        ...


class BuyingPowerModel(System.Object, QuantConnect.Securities.IBuyingPowerModel):
    """Provides a base class for all buying power models"""

    @property
    def RequiredFreeBuyingPowerPercent(self) -> float:
        """
        The percentage used to determine the required unused buying power for the account.
        
        This field is protected.
        """
        ...

    @RequiredFreeBuyingPowerPercent.setter
    def RequiredFreeBuyingPowerPercent(self, value: float):
        """
        The percentage used to determine the required unused buying power for the account.
        
        This field is protected.
        """
        ...

    @typing.overload
    def __init__(self) -> None:
        """Initializes a new instance of the BuyingPowerModel with no leverage (1x)"""
        ...

    @typing.overload
    def __init__(self, initialMarginRequirement: float, maintenanceMarginRequirement: float, requiredFreeBuyingPowerPercent: float) -> None:
        """
        Initializes a new instance of the BuyingPowerModel
        
        :param initialMarginRequirement: The percentage of an order's absolute cost that must be held in free cash in order to place the order
        :param maintenanceMarginRequirement: The percentage of the holding's absolute cost that must be held in free cash in order to avoid a margin call
        :param requiredFreeBuyingPowerPercent: The percentage used to determine the required unused buying power for the account.
        """
        ...

    @typing.overload
    def __init__(self, leverage: float, requiredFreeBuyingPowerPercent: float = 0) -> None:
        """
        Initializes a new instance of the BuyingPowerModel
        
        :param leverage: The leverage
        :param requiredFreeBuyingPowerPercent: The percentage used to determine the required unused buying power for the account.
        """
        ...

    def GetLeverage(self, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the current leverage of the security
        
        :param security: The security to get leverage for
        :returns: The current leverage in the security.
        """
        ...

    def SetLeverage(self, security: QuantConnect.Securities.Security, leverage: float) -> None:
        """
        Sets the leverage for the applicable securities, i.e, equities
        
        :param leverage: The new leverage
        """
        ...

    def GetInitialMarginRequiredForOrder(self, parameters: QuantConnect.Securities.InitialMarginRequiredForOrderParameters) -> float:
        """
        Gets the total margin required to execute the specified order in units of the account currency including fees
        
        This method is protected.
        
        :param parameters: An object containing the portfolio, the security and the order
        :returns: The total margin in terms of the currency quoted in the order.
        """
        ...

    def GetMaintenanceMargin(self, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the margin currently allocated to the specified holding
        
        This method is protected.
        
        :param security: The security to compute maintenance margin for
        :returns: The maintenance margin required for the.
        """
        ...

    def GetMarginRemaining(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, direction: QuantConnect.Orders.OrderDirection) -> float:
        """
        Gets the margin cash available for a trade
        
        This method is protected.
        
        :param portfolio: The algorithm's portfolio
        :param security: The security to be traded
        :param direction: The direction of the trade
        :returns: The margin available for the trade.
        """
        ...

    def GetInitialMarginRequirement(self, security: QuantConnect.Securities.Security, quantity: float) -> float:
        """
        The margin that must be held in order to increase the position by the provided quantity
        
        This method is protected.
        """
        ...

    def HasSufficientBuyingPowerForOrder(self, parameters: QuantConnect.Securities.HasSufficientBuyingPowerForOrderParameters) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """
        Check if there is sufficient buying power to execute this order.
        
        :param parameters: An object containing the portfolio, the security and the order
        :returns: Returns buying power information for an order.
        """
        ...

    def GetMaximumOrderQuantityForDeltaBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForDeltaBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a delta in the buying power used by a security.
        The deltas sign defines the position side to apply it to, positive long, negative short.
        
        :param parameters: An object containing the portfolio, the security and the delta buying power
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetMaximumOrderQuantityForTargetBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForTargetBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a position with a given buying power percentage.
        Will not take into account free buying power.
        
        :param parameters: An object containing the portfolio, the security and the target signed buying power percentage
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetReservedBuyingPowerForPosition(self, parameters: QuantConnect.Securities.ReservedBuyingPowerForPositionParameters) -> QuantConnect.Securities.ReservedBuyingPowerForPosition:
        """
        Gets the amount of buying power reserved to maintain the specified position
        
        :param parameters: A parameters object containing the security
        :returns: The reserved buying power in account currency.
        """
        ...

    def GetBuyingPower(self, parameters: QuantConnect.Securities.BuyingPowerParameters) -> QuantConnect.Securities.BuyingPower:
        """
        Gets the buying power available for a trade
        
        :param parameters: A parameters object containing the algorithm's portfolio, security, and order direction
        :returns: The buying power available for the trade.
        """
        ...


class AccountEvent(System.Object):
    """Messaging class signifying a change in a user's account"""

    @property
    def CashBalance(self) -> float:
        """Gets the total cash balance of the account in units of CurrencySymbol"""
        ...

    @CashBalance.setter
    def CashBalance(self, value: float):
        """Gets the total cash balance of the account in units of CurrencySymbol"""
        ...

    @property
    def CurrencySymbol(self) -> str:
        """Gets the currency symbol"""
        ...

    @CurrencySymbol.setter
    def CurrencySymbol(self, value: str):
        """Gets the currency symbol"""
        ...

    def __init__(self, currencySymbol: str, cashBalance: float) -> None:
        """
        Creates an AccountEvent
        
        :param currencySymbol: The currency's symbol
        :param cashBalance: The total cash balance of the account
        """
        ...

    def ToString(self) -> str:
        """
        Returns a string that represents the current object.
        
        :returns: A string that represents the current object.
        """
        ...


class AccountCurrencyImmediateSettlementModel(System.Object, QuantConnect.Securities.ISettlementModel):
    """Represents the model responsible for applying cash settlement rules"""

    def ApplyFunds(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, applicationTimeUtc: datetime.datetime, currency: str, amount: float) -> None:
        """
        Applies cash settlement rules
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param applicationTimeUtc: The fill time (in UTC)
        :param currency: The currency symbol
        :param amount: The amount of cash to apply
        """
        ...


class MarginCallModel(System.Object):
    """Provides access to a null implementation for IMarginCallModel"""

    Null: QuantConnect.Securities.IMarginCallModel = ...
    """
    Gets an instance of IMarginCallModel that will always
    return an empty list of executed orders.
    """


class DelayedSettlementModel(System.Object, QuantConnect.Securities.ISettlementModel):
    """Represents the model responsible for applying cash settlement rules"""

    def __init__(self, numberOfDays: int, timeOfDay: datetime.timedelta) -> None:
        """
        Creates an instance of the DelayedSettlementModel class
        
        :param numberOfDays: The number of days required for settlement
        :param timeOfDay: The time of day used for settlement
        """
        ...

    def ApplyFunds(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, applicationTimeUtc: datetime.datetime, currency: str, amount: float) -> None:
        """
        Applies cash settlement rules
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param applicationTimeUtc: The fill time (in UTC)
        :param currency: The currency symbol
        :param amount: The amount of cash to apply
        """
        ...


class BuyingPowerModelExtensions(System.Object):
    """Provides extension methods as backwards compatibility shims"""

    @staticmethod
    def GetReservedBuyingPowerForPosition(model: QuantConnect.Securities.IBuyingPowerModel, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the amount of buying power reserved to maintain the specified position
        
        :param model: The IBuyingPowerModel
        :param security: The security
        :returns: The reserved buying power in account currency.
        """
        ...

    @staticmethod
    def HasSufficientBuyingPowerForOrder(model: QuantConnect.Securities.IBuyingPowerModel, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, order: QuantConnect.Orders.Order) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """
        Check if there is sufficient buying power to execute this order.
        
        :param model: The IBuyingPowerModel
        :param portfolio: The algorithm's portfolio
        :param security: The security to be traded
        :param order: The order
        :returns: Returns buying power information for an order.
        """
        ...

    @staticmethod
    def GetMaximumOrderQuantityForTargetBuyingPower(model: QuantConnect.Securities.IBuyingPowerModel, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, target: float) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a position with a given value in account currency
        
        :param model: The IBuyingPowerModel
        :param portfolio: The algorithm's portfolio
        :param security: The security to be traded
        :param target: The target percent holdings
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    @staticmethod
    def GetBuyingPower(model: QuantConnect.Securities.IBuyingPowerModel, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, direction: QuantConnect.Orders.OrderDirection) -> float:
        """
        Gets the buying power available for a trade
        
        :param model: The IBuyingPowerModel
        :param portfolio: The algorithm's portfolio
        :param security: The security to be traded
        :param direction: The direction of the trade
        :returns: The buying power available for the trade.
        """
        ...


class ISecuritySeeder(metaclass=abc.ABCMeta):
    """Used to seed the security with the correct price"""

    def SeedSecurity(self, security: QuantConnect.Securities.Security) -> bool:
        """
        Seed the security
        
        :param security: Security being seeded
        :returns: true if the security was seeded, false otherwise.
        """
        ...


class FuncSecuritySeeder(System.Object, QuantConnect.Securities.ISecuritySeeder):
    """Seed a security price from a history function"""

    def __init__(self, seedFunction: typing.Callable[[QuantConnect.Securities.Security], QuantConnect.Data.BaseData]) -> None:
        """Constructor that takes as a parameter the security used to seed the price"""
        ...

    def SeedSecurity(self, security: QuantConnect.Securities.Security) -> bool:
        """
        Seed the security
        
        :param security: Security being seeded
        :returns: true if the security was seeded, false otherwise.
        """
        ...


class ISecurityPortfolioModel(metaclass=abc.ABCMeta):
    """Performs order fill application to portfolio"""

    def ProcessFill(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, fill: QuantConnect.Orders.OrderEvent) -> None:
        """
        Performs application of an OrderEvent to the portfolio
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param fill: The order event fill object to be applied
        """
        ...


class IDerivativeSecurity(metaclass=abc.ABCMeta):
    """Defines a security as a derivative of another security"""

    @property
    @abc.abstractmethod
    def Underlying(self) -> QuantConnect.Securities.Security:
        """Gets or sets the underlying security for the derivative"""
        ...

    @Underlying.setter
    @abc.abstractmethod
    def Underlying(self, value: QuantConnect.Securities.Security):
        """Gets or sets the underlying security for the derivative"""
        ...


class IOrderEventProvider(metaclass=abc.ABCMeta):
    """Represents a type with a new OrderEvent event EventHandler."""

    @property
    @abc.abstractmethod
    def NewOrderEvent(self) -> typing.List[System_EventHandler]:
        """Event fired when there is a new OrderEvent"""
        ...

    @NewOrderEvent.setter
    @abc.abstractmethod
    def NewOrderEvent(self, value: typing.List[System_EventHandler]):
        """Event fired when there is a new OrderEvent"""
        ...


class SecurityMarginModel(QuantConnect.Securities.BuyingPowerModel):
    """Represents a simple, constant margin model by specifying the percentages of required margin."""

    @typing.overload
    def __init__(self) -> None:
        """Initializes a new instance of the SecurityMarginModel with no leverage (1x)"""
        ...

    @typing.overload
    def __init__(self, initialMarginRequirement: float, maintenanceMarginRequirement: float, requiredFreeBuyingPowerPercent: float) -> None:
        """
        Initializes a new instance of the SecurityMarginModel
        
        :param initialMarginRequirement: The percentage of an order's absolute cost that must be held in free cash in order to place the order
        :param maintenanceMarginRequirement: The percentage of the holding's absolute cost that must be held in free cash in order to avoid a margin call
        :param requiredFreeBuyingPowerPercent: The percentage used to determine the required unused buying power for the account.
        """
        ...

    @typing.overload
    def __init__(self, leverage: float, requiredFreeBuyingPowerPercent: float = 0) -> None:
        """
        Initializes a new instance of the SecurityMarginModel
        
        :param leverage: The leverage
        :param requiredFreeBuyingPowerPercent: The percentage used to determine the required unused buying power for the account.
        """
        ...


class DefaultMarginCallModel(System.Object, QuantConnect.Securities.IMarginCallModel):
    """Represents the model responsible for picking which orders should be executed during a margin call"""

    @property
    def Portfolio(self) -> QuantConnect.Securities.SecurityPortfolioManager:
        """
        Gets the portfolio that margin calls will be transacted against
        
        This property is protected.
        """
        ...

    @property
    def DefaultOrderProperties(self) -> QuantConnect.Interfaces.IOrderProperties:
        """
        Gets the default order properties to be used in margin call orders
        
        This property is protected.
        """
        ...

    def __init__(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, defaultOrderProperties: QuantConnect.Interfaces.IOrderProperties) -> None:
        """
        Initializes a new instance of the DefaultMarginCallModel class
        
        :param portfolio: The portfolio object to receive margin calls
        :param defaultOrderProperties: The default order properties to be used in margin call orders
        """
        ...

    def GetMarginCallOrders(self, issueMarginCallWarning: bool) -> System.Collections.Generic.List[QuantConnect.Orders.SubmitOrderRequest]:
        """
        Scan the portfolio and the updated data for a potential margin call situation which may get the holdings below zero!
        If there is a margin call, liquidate the portfolio immediately before the portfolio gets sub zero.
        
        :param issueMarginCallWarning: Set to true if a warning should be issued to the algorithm
        :returns: True for a margin call on the holdings.
        """
        ...

    def GenerateMarginCallOrder(self, security: QuantConnect.Securities.Security, totalPortfolioValue: float, totalUsedMargin: float) -> QuantConnect.Orders.SubmitOrderRequest:
        """
        Generates a new order for the specified security taking into account the total margin
        used by the account. Returns null when no margin call is to be issued.
        
        This method is protected.
        
        :param security: The security to generate a margin call order for
        :param totalPortfolioValue: The net liquidation value for the entire account
        :param totalUsedMargin: The total margin used by the account in units of base currency
        :returns: An order object representing a liquidation order to be executed to bring the account within margin requirements.
        """
        ...

    def ExecuteMarginCall(self, generatedMarginCallOrders: System.Collections.Generic.IEnumerable[QuantConnect.Orders.SubmitOrderRequest]) -> System.Collections.Generic.List[QuantConnect.Orders.OrderTicket]:
        """
        Executes synchronous orders to bring the account within margin requirements.
        
        :param generatedMarginCallOrders: These are the margin call orders that were generated by individual security margin models.
        :returns: The list of orders that were actually executed.
        """
        ...


class ErrorCurrencyConverter(System.Object, QuantConnect.Securities.ICurrencyConverter):
    """
    Provides an implementation of ICurrencyConverter for use in
    tests that don't depend on this behavior.
    """

    @property
    def AccountCurrency(self) -> str:
        """Gets account currency"""
        ...

    Instance: QuantConnect.Securities.ICurrencyConverter = ...
    """
    Provides access to the single instance of ErrorCurrencyConverter.
    This is done this way to ensure usage is explicit.
    """

    def ConvertToAccountCurrency(self, cashAmount: QuantConnect.Securities.CashAmount) -> QuantConnect.Securities.CashAmount:
        """
        Converts a cash amount to the account currency
        
        :param cashAmount: The CashAmount instance to convert
        :returns: A new CashAmount instance denominated in the account currency.
        """
        ...


class SecurityProviderExtensions(System.Object):
    """Provides extension methods for the ISecurityProvider interface."""

    @staticmethod
    def GetHoldingsQuantity(provider: QuantConnect.Securities.ISecurityProvider, symbol: typing.Union[QuantConnect.Symbol, str]) -> float:
        """
        Extension method to return the quantity of holdings, if no holdings are present, then zero is returned.
        
        :param provider: The ISecurityProvider
        :param symbol: The symbol we want holdings quantity for
        :returns: The quantity of holdings for the specified symbol.
        """
        ...


class SecurityDataFilter(System.Object, QuantConnect.Securities.Interfaces.ISecurityDataFilter):
    """Base class implementation for packet by packet data filtering mechanism to dynamically detect bad ticks."""

    def __init__(self) -> None:
        """Initialize data filter class"""
        ...

    def Filter(self, vehicle: QuantConnect.Securities.Security, data: QuantConnect.Data.BaseData) -> bool:
        """
        Filter the data packet passing through this method by returning true to accept, or false to fail/reject the data point.
        
        :param vehicle: Security vehicle for filter
        :param data: BasData data object we're filtering
        """
        ...


class BrokerageModelSecurityInitializer(System.Object, QuantConnect.Securities.ISecurityInitializer):
    """
    Provides an implementation of ISecurityInitializer that initializes a security
    by settings the Security.FillModel, Security.FeeModel,
    Security.SlippageModel, and the Security.SettlementModel properties
    """

    @typing.overload
    def __init__(self) -> None:
        """
        Initializes a new instance of the BrokerageModelSecurityInitializer class
        for the specified algorithm
        """
        ...

    @typing.overload
    def __init__(self, brokerageModel: QuantConnect.Brokerages.IBrokerageModel, securitySeeder: QuantConnect.Securities.ISecuritySeeder) -> None:
        """
        Initializes a new instance of the BrokerageModelSecurityInitializer class
        for the specified algorithm
        
        :param brokerageModel: The brokerage model used to initialize the security models
        :param securitySeeder: An ISecuritySeeder used to seed the initial price of the security
        """
        ...

    def Initialize(self, security: QuantConnect.Securities.Security) -> None:
        """
        Initializes the specified security by setting up the models
        
        :param security: The security to be initialized
        """
        ...


class GetMinimumPriceVariationParameters(System.Object):
    """Defines the parameters for IPriceVariationModel.GetMinimumPriceVariation"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def ReferencePrice(self) -> float:
        """Gets the reference price to be used for the calculation"""
        ...

    def __init__(self, security: QuantConnect.Securities.Security, referencePrice: float) -> None:
        """
        Initializes a new instance of the GetMinimumPriceVariationParameters class
        
        :param security: The security
        :param referencePrice: The reference price to be used for the calculation
        """
        ...


class AdjustedPriceVariationModel(System.Object, QuantConnect.Securities.IPriceVariationModel):
    """
    Provides an implementation of IPriceVariationModel
    for use when data is DataNormalizationMode.Adjusted.
    """

    def GetMinimumPriceVariation(self, parameters: QuantConnect.Securities.GetMinimumPriceVariationParameters) -> float:
        """
        Get the minimum price variation from a security
        
        :param parameters: An object containing the method parameters
        :returns: Zero.
        """
        ...


class IDerivativeSecurityFilterUniverse(System.Collections.Generic.IEnumerable[QuantConnect.Symbol], metaclass=abc.ABCMeta):
    """Represents derivative symbols universe used in filtering."""

    @property
    @abc.abstractmethod
    def Underlying(self) -> QuantConnect.Data.BaseData:
        """The underlying price data"""
        ...

    @property
    @abc.abstractmethod
    def IsDynamic(self) -> bool:
        """True if the universe is dynamic and filter needs to be reapplied during trading day"""
        ...


class IDerivativeSecurityFilter(metaclass=abc.ABCMeta):
    """Filters a set of derivative symbols using the underlying price data."""

    def Filter(self, universe: QuantConnect.Securities.IDerivativeSecurityFilterUniverse) -> QuantConnect.Securities.IDerivativeSecurityFilterUniverse:
        """
        Filters the input set of symbols represented by the universe
        
        :param universe: derivative symbols universe used in filtering
        :returns: The filtered set of symbols.
        """
        ...


class FuncSecurityDerivativeFilter(System.Object, QuantConnect.Securities.IDerivativeSecurityFilter):
    """Provides a functional implementation of IDerivativeSecurityFilter"""

    def __init__(self, filter: typing.Callable[[QuantConnect.Securities.IDerivativeSecurityFilterUniverse], QuantConnect.Securities.IDerivativeSecurityFilterUniverse]) -> None:
        """
        Initializes a new instance of the FuncSecurityDerivativeFilter class
        
        :param filter: The functional implementation of the Filter method
        """
        ...

    def Filter(self, universe: QuantConnect.Securities.IDerivativeSecurityFilterUniverse) -> QuantConnect.Securities.IDerivativeSecurityFilterUniverse:
        """
        Filters the input set of symbols represented by the universe
        
        :param universe: Derivative symbols universe used in filtering
        :returns: The filtered set of symbols.
        """
        ...


class ImmediateSettlementModel(System.Object, QuantConnect.Securities.ISettlementModel):
    """Represents the model responsible for applying cash settlement rules"""

    def ApplyFunds(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, applicationTimeUtc: datetime.datetime, currency: str, amount: float) -> None:
        """
        Applies cash settlement rules
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param applicationTimeUtc: The fill time (in UTC)
        :param currency: The currency symbol
        :param amount: The amount of cash to apply
        """
        ...


class RegisteredSecurityDataTypesProvider(System.Object, QuantConnect.Securities.IRegisteredSecurityDataTypesProvider):
    """
    Provides an implementation of IRegisteredSecurityDataTypesProvider that permits the
    consumer to modify the expected types
    """

    Null: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider = ...
    """Provides a reference to an instance of IRegisteredSecurityDataTypesProvider that contains no registered types"""

    def RegisterType(self, type: typing.Type) -> bool:
        """
        Registers the specified type w/ the provider
        
        :returns: True if the type was previously not registered.
        """
        ...

    def UnregisterType(self, type: typing.Type) -> bool:
        """
        Removes the registration for the specified type
        
        :returns: True if the type was previously registered.
        """
        ...

    def TryGetType(self, name: str, type: typing.Type) -> bool:
        """Gets an enumerable of data types expected to be contained in a DynamicSecurityData instance"""
        ...


class PatternDayTradingMarginModel(QuantConnect.Securities.SecurityMarginModel):
    """
    Represents a simple margining model where margin/leverage depends on market state (open or close).
    During regular market hours, leverage is 4x, otherwise 2x
    """

    @typing.overload
    def __init__(self) -> None:
        """Initializes a new instance of the PatternDayTradingMarginModel"""
        ...

    @typing.overload
    def __init__(self, closedMarketLeverage: float, openMarketLeverage: float) -> None:
        """
        Initializes a new instance of the PatternDayTradingMarginModel
        
        :param closedMarketLeverage: Leverage used outside regular market hours
        :param openMarketLeverage: Leverage used during regular market hours
        """
        ...

    def SetLeverage(self, security: QuantConnect.Securities.Security, leverage: float) -> None:
        """
        Sets the leverage for the applicable securities, i.e, equities
        
        :param security: The security to set leverage to
        :param leverage: The new leverage
        """
        ...

    def GetLeverage(self, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the current leverage of the security
        
        :param security: The security to get leverage for
        :returns: The current leverage in the security.
        """
        ...

    def GetInitialMarginRequirement(self, security: QuantConnect.Securities.Security, quantity: float) -> float:
        """
        The percentage of an order's absolute cost that must be held in free cash in order to place the order
        
        This method is protected.
        """
        ...

    def GetMaintenanceMargin(self, security: QuantConnect.Securities.Security) -> float:
        """
        The percentage of the holding's absolute cost that must be held in free cash in order to avoid a margin call
        
        This method is protected.
        """
        ...


class ContractSecurityFilterUniverse(typing.Generic[QuantConnect_Securities_ContractSecurityFilterUniverse_T], System.Object, QuantConnect.Securities.IDerivativeSecurityFilterUniverse, typing.Iterable[QuantConnect.Symbol], metaclass=abc.ABCMeta):
    """
    Base class for contract symbols filtering universes.
    Used by OptionFilterUniverse and FutureFilterUniverse
    """

    class ContractExpirationType(System.Enum):
        """
        Defines listed contract types with Flags attribute
        
        This class is protected.
        """

        Standard = 1
        """Standard contracts"""

        Weekly = 2
        """Non standard weekly contracts"""

    @property
    def Type(self) -> QuantConnect.Securities.ContractSecurityFilterUniverse.ContractExpirationType:
        """
        Expiration Types allowed through the filter
        Standards only by default
        
        This field is protected.
        """
        ...

    @Type.setter
    def Type(self, value: QuantConnect.Securities.ContractSecurityFilterUniverse.ContractExpirationType):
        """
        Expiration Types allowed through the filter
        Standards only by default
        
        This field is protected.
        """
        ...

    @property
    def UnderlyingInternal(self) -> QuantConnect.Data.BaseData:
        """
        The underlying price data
        
        This field is protected.
        """
        ...

    @UnderlyingInternal.setter
    def UnderlyingInternal(self, value: QuantConnect.Data.BaseData):
        """
        The underlying price data
        
        This field is protected.
        """
        ...

    @property
    def AllSymbols(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Symbol]:
        """
        All Symbols in this filter
        Marked internal for use by extensions
        """
        ...

    @AllSymbols.setter
    def AllSymbols(self, value: System.Collections.Generic.IEnumerable[QuantConnect.Symbol]):
        """
        All Symbols in this filter
        Marked internal for use by extensions
        """
        ...

    @property
    def IsDynamicInternal(self) -> bool:
        """
        Mark this filter dynamic for regular reapplying
        Marked internal for use by extensions
        """
        ...

    @IsDynamicInternal.setter
    def IsDynamicInternal(self, value: bool):
        """
        Mark this filter dynamic for regular reapplying
        Marked internal for use by extensions
        """
        ...

    @property
    def IsDynamic(self) -> bool:
        """True if the universe is dynamic and filter needs to be reapplied"""
        ...

    @property
    def Underlying(self) -> QuantConnect.Data.BaseData:
        """The underlying price data"""
        ...

    @typing.overload
    def __init__(self) -> None:
        """
        Constructs ContractSecurityFilterUniverse
        
        This method is protected.
        """
        ...

    @typing.overload
    def __init__(self, allSymbols: System.Collections.Generic.IEnumerable[QuantConnect.Symbol], underlying: QuantConnect.Data.BaseData) -> None:
        """
        Constructs ContractSecurityFilterUniverse
        
        This method is protected.
        """
        ...

    def IsStandard(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Function to determine if the given symbol is a standard contract
        
        This method is protected.
        
        :returns: True if standard type.
        """
        ...

    def Refresh(self, allSymbols: System.Collections.Generic.IEnumerable[QuantConnect.Symbol], underlying: QuantConnect.Data.BaseData) -> None:
        """
        Refreshes this filter universe
        
        :param allSymbols: All the contract symbols for the Universe
        :param underlying: The current underlying last data point
        """
        ...

    def StandardsOnly(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Sets universe of standard contracts (if any) as selection
        Contracts by default are standards; only needed to switch back if changed
        
        :returns: Universe with filter applied.
        """
        ...

    def IncludeWeeklys(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Includes universe of non-standard weeklys contracts (if any) into selection
        
        :returns: Universe with filter applied.
        """
        ...

    def WeeklysOnly(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Sets universe of weeklys contracts (if any) as selection
        
        :returns: Universe with filter applied.
        """
        ...

    def FrontMonth(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Returns front month contract
        
        :returns: Universe with filter applied.
        """
        ...

    def BackMonths(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Returns a list of back month contracts
        
        :returns: Universe with filter applied.
        """
        ...

    def BackMonth(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Returns first of back month contracts
        
        :returns: Universe with filter applied.
        """
        ...

    @typing.overload
    def Expiration(self, minExpiry: datetime.timedelta, maxExpiry: datetime.timedelta) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Applies filter selecting options contracts based on a range of expiration dates relative to the current day
        
        :param minExpiry: The minimum time until expiry to include, for example, TimeSpan.FromDays(10) would exclude contracts expiring in more than 10 days
        :param maxExpiry: The maximum time until expiry to include, for example, TimeSpan.FromDays(10) would exclude contracts expiring in less than 10 days
        :returns: Universe with filter applied.
        """
        ...

    @typing.overload
    def Expiration(self, minExpiryDays: int, maxExpiryDays: int) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Applies filter selecting contracts based on a range of expiration dates relative to the current day
        
        :param minExpiryDays: The minimum time, expressed in days, until expiry to include, for example, 10 would exclude contracts expiring in more than 10 days
        :param maxExpiryDays: The maximum time, expressed in days, until expiry to include, for example, 10 would exclude contracts expiring in less than 10 days
        :returns: Universe with filter applied.
        """
        ...

    @typing.overload
    def Contracts(self, contracts: typing.Any) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Explicitly sets the selected contract symbols for this universe.
        This overrides and and all other methods of selecting symbols assuming it is called last.
        
        :param contracts: The option contract symbol objects to select
        :returns: Universe with filter applied.
        """
        ...

    @typing.overload
    def Contracts(self, contracts: System.Collections.Generic.IEnumerable[QuantConnect.Symbol]) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Explicitly sets the selected contract symbols for this universe.
        This overrides and and all other methods of selecting symbols assuming it is called last.
        
        :param contracts: The option contract symbol objects to select
        :returns: Universe with filter applied.
        """
        ...

    @typing.overload
    def Contracts(self, contractSelector: typing.Callable[[System.Collections.Generic.IEnumerable[QuantConnect.Symbol]], System.Collections.Generic.IEnumerable[QuantConnect.Symbol]]) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Sets a function used to filter the set of available contract filters. The input to the 'contractSelector'
        function will be the already filtered list if any other filters have already been applied.
        
        :param contractSelector: The option contract symbol objects to select
        :returns: Universe with filter applied.
        """
        ...

    def OnlyApplyFilterAtMarketOpen(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Instructs the engine to only filter contracts on the first time step of each market day.
        
        :returns: Universe with filter applied.
        """
        ...

    @typing.overload
    def GetEnumerator(self) -> System.Collections.Generic.IEnumerator[QuantConnect.Symbol]:
        """
        IEnumerable interface method implementation
        
        :returns: IEnumerator of Symbols in Universe.
        """
        ...

    @typing.overload
    def GetEnumerator(self) -> System.Collections.IEnumerator:
        """IEnumerable interface method implementation"""
        ...


class CashBuyingPowerModel(QuantConnect.Securities.BuyingPowerModel):
    """Represents a buying power model for cash accounts"""

    def GetLeverage(self, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the current leverage of the security
        
        :param security: The security to get leverage for
        :returns: The current leverage in the security.
        """
        ...

    def SetLeverage(self, security: QuantConnect.Securities.Security, leverage: float) -> None:
        """
        Sets the leverage for the applicable securities, i.e, equities
        
        :param security: The security to set leverage for
        :param leverage: The new leverage
        """
        ...

    def HasSufficientBuyingPowerForOrder(self, parameters: QuantConnect.Securities.HasSufficientBuyingPowerForOrderParameters) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """
        Check if there is sufficient buying power to execute this order.
        
        :param parameters: An object containing the portfolio, the security and the order
        :returns: Returns buying power information for an order.
        """
        ...

    def GetMaximumOrderQuantityForDeltaBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForDeltaBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a delta in the buying power used by a security.
        The deltas sign defines the position side to apply it to, positive long, negative short.
        
        :param parameters: An object containing the portfolio, the security and the delta buying power
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetMaximumOrderQuantityForTargetBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForTargetBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a position with a given buying power percentage.
        Will not take into account free buying power.
        
        :param parameters: An object containing the portfolio, the security and the target signed buying power percentage
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetReservedBuyingPowerForPosition(self, parameters: QuantConnect.Securities.ReservedBuyingPowerForPositionParameters) -> QuantConnect.Securities.ReservedBuyingPowerForPosition:
        """
        Gets the amount of buying power reserved to maintain the specified position
        
        :param parameters: A parameters object containing the security
        :returns: The reserved buying power in account currency.
        """
        ...

    def GetBuyingPower(self, parameters: QuantConnect.Securities.BuyingPowerParameters) -> QuantConnect.Securities.BuyingPower:
        """
        Gets the buying power available for a trade
        
        :param parameters: A parameters object containing the algorithm's potrfolio, security, and order direction
        :returns: The buying power available for the trade.
        """
        ...


class SecurityPriceVariationModel(System.Object, QuantConnect.Securities.IPriceVariationModel):
    """
    Provides default implementation of IPriceVariationModel
    for use in defining the minimum price variation.
    """

    def GetMinimumPriceVariation(self, parameters: QuantConnect.Securities.GetMinimumPriceVariationParameters) -> float:
        """
        Get the minimum price variation from a security
        
        :param parameters: An object containing the method parameters
        :returns: Decimal minimum price variation of a given security.
        """
        ...


class IBaseCurrencySymbol(metaclass=abc.ABCMeta):
    """This class has no documentation."""

    @property
    @abc.abstractmethod
    def BaseCurrencySymbol(self) -> str:
        """Gets the currency acquired by going long this currency pair"""
        ...


class SecuritySeeder(System.Object):
    """Provides access to a null implementation for ISecuritySeeder"""

    Null: QuantConnect.Securities.ISecuritySeeder = ...
    """Gets an instance of ISecuritySeeder that is a no-op"""


class UniverseManager(System.Object, System.Collections.Generic.IDictionary[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe], INotifyCollectionChanged, typing.Iterable[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]]):
    """Manages the algorithm's collection of universes"""

    @property
    def CollectionChanged(self) -> typing.List[typing.Any]:
        """Event fired when a universe is added or removed"""
        ...

    @CollectionChanged.setter
    def CollectionChanged(self, value: typing.List[typing.Any]):
        """Event fired when a universe is added or removed"""
        ...

    @property
    def ActiveSecurities(self) -> System.Collections.Generic.IReadOnlyDictionary[QuantConnect.Symbol, QuantConnect.Securities.Security]:
        """
        Read-only dictionary containing all active securities. An active security is
        a security that is currently selected by the universe or has holdings or open orders.
        """
        ...

    @property
    def Count(self) -> int:
        """Gets the number of elements contained in the System.Collections.Generic.ICollection`1."""
        ...

    @property
    def IsReadOnly(self) -> bool:
        """Gets a value indicating whether the System.Collections.Generic.ICollection`1 is read-only."""
        ...

    @property
    def Keys(self) -> System.Collections.Generic.ICollection[QuantConnect.Symbol]:
        """Gets an System.Collections.Generic.ICollection`1 containing the keys of the System.Collections.Generic.IDictionary`2."""
        ...

    @property
    def Values(self) -> System.Collections.Generic.ICollection[QuantConnect.Data.UniverseSelection.Universe]:
        """Gets an System.Collections.Generic.ICollection`1 containing the values in the System.Collections.Generic.IDictionary`2."""
        ...

    def __init__(self) -> None:
        """Initializes a new instance of the UniverseManager class"""
        ...

    @typing.overload
    def GetEnumerator(self) -> System.Collections.Generic.IEnumerator[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]]:
        ...

    @typing.overload
    def GetEnumerator(self) -> System.Collections.IEnumerator:
        """
        Returns an enumerator that iterates through a collection.
        
        :returns: An System.Collections.IEnumerator object that can be used to iterate through the collection.
        """
        ...

    @typing.overload
    def Add(self, item: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]) -> None:
        """
        Adds an item to the System.Collections.Generic.ICollection`1.
        
        :param item: The object to add to the System.Collections.Generic.ICollection`1.
        """
        ...

    def Clear(self) -> None:
        """Removes all items from the System.Collections.Generic.ICollection`1."""
        ...

    def Contains(self, item: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]) -> bool:
        """
        Determines whether the System.Collections.Generic.ICollection`1 contains a specific value.
        
        :param item: The object to locate in the System.Collections.Generic.ICollection`1.
        :returns: true if  is found in the System.Collections.Generic.ICollection`1; otherwise, false.
        """
        ...

    def CopyTo(self, array: typing.List[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]], arrayIndex: int) -> None:
        """
        Copies the elements of the System.Collections.Generic.ICollection`1 to an System.Array, starting at a particular System.Array index.
        
        :param array: The one-dimensional System.Array that is the destination of the elements copied from System.Collections.Generic.ICollection`1. The System.Array must have zero-based indexing.
        :param arrayIndex: The zero-based index in  at which copying begins.
        """
        ...

    @typing.overload
    def Remove(self, item: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]) -> bool:
        """
        Removes the first occurrence of a specific object from the System.Collections.Generic.ICollection`1.
        
        :param item: The object to remove from the System.Collections.Generic.ICollection`1.
        :returns: true if  was successfully removed from the System.Collections.Generic.ICollection`1; otherwise, false. This method also returns false if  is not found in the original System.Collections.Generic.ICollection`1.
        """
        ...

    def ContainsKey(self, key: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Determines whether the System.Collections.Generic.IDictionary`2 contains an element with the specified key.
        
        :param key: The key to locate in the System.Collections.Generic.IDictionary`2.
        :returns: true if the System.Collections.Generic.IDictionary`2 contains an element with the key; otherwise, false.
        """
        ...

    @typing.overload
    def Add(self, key: typing.Union[QuantConnect.Symbol, str], universe: QuantConnect.Data.UniverseSelection.Universe) -> None:
        """
        Adds an element with the provided key and value to the System.Collections.Generic.IDictionary`2.
        
        :param key: The object to use as the key of the element to add.
        :param universe: The object to use as the value of the element to add.
        """
        ...

    @typing.overload
    def Remove(self, key: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Removes the element with the specified key from the System.Collections.Generic.IDictionary`2.
        
        :param key: The key of the element to remove.
        :returns: true if the element is successfully removed; otherwise, false.  This method also returns false if  was not found in the original System.Collections.Generic.IDictionary`2.
        """
        ...

    def TryGetValue(self, key: typing.Union[QuantConnect.Symbol, str], value: QuantConnect.Data.UniverseSelection.Universe) -> bool:
        """
        Gets the value associated with the specified key.
        
        :param key: The key whose value to get.
        :param value: When this method returns, the value associated with the specified key, if the key is found; otherwise, the default value for the type of the  parameter. This parameter is passed uninitialized.
        :returns: true if the object that implements System.Collections.Generic.IDictionary`2 contains an element with the specified key; otherwise, false.
        """
        ...

    def __getitem__(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Data.UniverseSelection.Universe:
        """
        Gets or sets the element with the specified key.
        
        :param symbol: The key of the element to get or set.
        :returns: The element with the specified key.
        """
        ...

    def __setitem__(self, symbol: typing.Union[QuantConnect.Symbol, str], value: QuantConnect.Data.UniverseSelection.Universe) -> None:
        """
        Gets or sets the element with the specified key.
        
        :param symbol: The key of the element to get or set.
        :returns: The element with the specified key.
        """
        ...

    def OnCollectionChanged(self, e: typing.Any) -> None:
        """This method is protected."""
        ...


class CompositeSecurityInitializer(System.Object, QuantConnect.Securities.ISecurityInitializer):
    """
    Provides an implementation of ISecurityInitializer that executes
    each initializer in order
    """

    def __init__(self, *initializers: QuantConnect.Securities.ISecurityInitializer) -> None:
        """
        Initializes a new instance of the CompositeSecurityInitializer class
        
        :param initializers: The initializers to execute in order
        """
        ...

    def Initialize(self, security: QuantConnect.Securities.Security) -> None:
        """
        Execute each of the internally held initializers in sequence
        
        :param security: The security to be initialized
        """
        ...


class FuncSecurityInitializer(System.Object, QuantConnect.Securities.ISecurityInitializer):
    """Provides a functional implementation of ISecurityInitializer"""

    def __init__(self, initializer: typing.Callable[[QuantConnect.Securities.Security], None]) -> None:
        """
        Initializes a new instance of the FuncSecurityInitializer class
        
        :param initializer: The functional implementation of ISecurityInitializer.Initialize
        """
        ...

    def Initialize(self, security: QuantConnect.Securities.Security) -> None:
        """
        Initializes the specified security
        
        :param security: The security to be initialized
        """
        ...


class EquityPriceVariationModel(QuantConnect.Securities.SecurityPriceVariationModel):
    """
    Provides an implementation of IPriceVariationModel
    for use in defining the minimum price variation for a given equity
    under Regulation NMS – Rule 612 (a.k.a – the “sub-penny rule”)
    """

    def GetMinimumPriceVariation(self, parameters: QuantConnect.Securities.GetMinimumPriceVariationParameters) -> float:
        """
        Get the minimum price variation from a security
        
        :param parameters: An object containing the method parameters
        :returns: Decimal minimum price variation of a given security.
        """
        ...


class OrderProviderExtensions(System.Object):
    """Provides extension methods for the IOrderProvider interface"""

    @staticmethod
    @typing.overload
    def GetOrderByBrokerageId(orderProvider: QuantConnect.Securities.IOrderProvider, brokerageId: int) -> QuantConnect.Orders.Order:
        """
        Gets the order by its brokerage id
        
        :param orderProvider: The order provider to search
        :param brokerageId: The brokerage id to fetch
        :returns: The first order matching the brokerage id, or null if no match is found.
        """
        ...

    @staticmethod
    @typing.overload
    def GetOrderByBrokerageId(orderProvider: QuantConnect.Securities.IOrderProvider, brokerageId: int) -> QuantConnect.Orders.Order:
        """
        Gets the order by its brokerage id
        
        :param orderProvider: The order provider to search
        :param brokerageId: The brokerage id to fetch
        :returns: The first order matching the brokerage id, or null if no match is found.
        """
        ...


class IdentityCurrencyConverter(System.Object, QuantConnect.Securities.ICurrencyConverter):
    """
    Provides an implementation of ICurrencyConverter that does NOT perform conversions.
    This implementation will throw if the specified cashAmount is not in units of account currency.
    """

    @property
    def AccountCurrency(self) -> str:
        """Gets account currency"""
        ...

    def __init__(self, accountCurrency: str) -> None:
        """
        Initializes a new instance of the ICurrencyConverter class
        
        :param accountCurrency: The algorithm's account currency
        """
        ...

    def ConvertToAccountCurrency(self, cashAmount: QuantConnect.Securities.CashAmount) -> QuantConnect.Securities.CashAmount:
        """
        Converts a cash amount to the account currency.
        This implementation can only handle cash amounts in units of the account currency.
        
        :param cashAmount: The CashAmount instance to convert
        :returns: A new CashAmount instance denominated in the account currency.
        """
        ...


class SecurityPortfolioModel(System.Object, QuantConnect.Securities.ISecurityPortfolioModel):
    """
    Provides a default implementation of ISecurityPortfolioModel that simply
    applies the fills to the algorithm's portfolio. This implementation is intended to
    handle all security types.
    """

    def ProcessFill(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, fill: QuantConnect.Orders.OrderEvent) -> None:
        """
        Performs application of an OrderEvent to the portfolio
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param fill: The order event fill object to be applied
        """
        ...


class SecurityCacheDataStoredEventArgs(System.EventArgs):
    """Event args for SecurityCache.DataStored event"""

    @property
    def DataType(self) -> typing.Type:
        """The type of data that was stored, such as TradeBar"""
        ...

    @property
    def Data(self) -> System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData]:
        """The list of data points stored"""
        ...

    def __init__(self, dataType: typing.Type, data: System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData]) -> None:
        """
        Initializes a new instance of the SecurityCacheDataStoredEventArgs class
        
        :param dataType: The type of data
        :param data: The list of data points
        """
        ...


class VolatilityModel(System.Object):
    """Provides access to a null implementation for IVolatilityModel"""

    Null: QuantConnect.Securities.IVolatilityModel = ...
    """
    Gets an instance of IVolatilityModel that will always
    return 0 for its volatility and does nothing during Update.
    """


class StandardDeviationOfReturnsVolatilityModel(QuantConnect.Securities.Volatility.BaseVolatilityModel):
    """
    Provides an implementation of IVolatilityModel that computes the
    annualized sample standard deviation of daily returns as the volatility of the security
    """

    @property
    def Volatility(self) -> float:
        """Gets the volatility of the security as a percentage"""
        ...

    def __init__(self, periods: int) -> None:
        """
        Initializes a new instance of the StandardDeviationOfReturnsVolatilityModel class
        
        :param periods: The number of periods (days) to wait until updating the value
        """
        ...

    def Update(self, security: QuantConnect.Securities.Security, data: QuantConnect.Data.BaseData) -> None:
        """
        Updates this model using the new price information in
        the specified security instance
        
        :param security: The security to calculate volatility for
        """
        ...

    def GetHistoryRequirements(self, security: QuantConnect.Securities.Security, utcTime: datetime.datetime) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.HistoryRequest]:
        """
        Returns history requirements for the volatility model expressed in the form of history request
        
        :param security: The security of the request
        :param utcTime: The date of the request
        :returns: History request object list, or empty if no requirements.
        """
        ...


class RelativeStandardDeviationVolatilityModel(QuantConnect.Securities.Volatility.BaseVolatilityModel):
    """
    Provides an implementation of IVolatilityModel that computes the
    relative standard deviation as the volatility of the security
    """

    @property
    def Volatility(self) -> float:
        """Gets the volatility of the security as a percentage"""
        ...

    def __init__(self, periodSpan: datetime.timedelta, periods: int) -> None:
        """
        Initializes a new instance of the RelativeStandardDeviationVolatilityModel class
        
        :param periodSpan: The time span representing one 'period' length
        :param periods: The nuber of 'period' lengths to wait until updating the value
        """
        ...

    def Update(self, security: QuantConnect.Securities.Security, data: QuantConnect.Data.BaseData) -> None:
        """
        Updates this model using the new price information in
        the specified security instance
        
        :param security: The security to calculate volatility for
        """
        ...

    def GetHistoryRequirements(self, security: QuantConnect.Securities.Security, utcTime: datetime.datetime) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.HistoryRequest]:
        """
        Returns history requirements for the volatility model expressed in the form of history request
        
        :param security: The security of the request
        :param utcTime: The date/time of the request
        :returns: History request object list, or empty if no requirements.
        """
        ...


class IndicatorVolatilityModel(typing.Generic[QuantConnect_Securities_IndicatorVolatilityModel_T], System.Object, QuantConnect.Securities.IVolatilityModel):
    """
    Provides an implementation of IVolatilityModel that uses an indicator
    to compute its value
    """

    @property
    def Volatility(self) -> float:
        """Gets the volatility of the security as a percentage"""
        ...

    @typing.overload
    def __init__(self, indicator: QuantConnect.Indicators.IIndicator[QuantConnect_Securities_IndicatorVolatilityModel_T]) -> None:
        """
        Initializes a new instance of the IVolatilityModel using
        the specified . The 
        is assumed to but updated externally from this model, such as being registered
        into the consolidator system.
        
        :param indicator: The auto-updating indicator
        """
        ...

    @typing.overload
    def __init__(self, indicator: QuantConnect.Indicators.IIndicator[QuantConnect_Securities_IndicatorVolatilityModel_T], indicatorUpdate: typing.Callable[[QuantConnect.Securities.Security, QuantConnect.Data.BaseData, QuantConnect.Indicators.IIndicator[QuantConnect_Securities_IndicatorVolatilityModel_T]], None]) -> None:
        """
        Initializes a new instance of the IVolatilityModel using
        the specified . The 
        is assumed to but updated externally from this model, such as being registered
        into the consolidator system.
        
        :param indicator: The auto-updating indicator
        :param indicatorUpdate: Function delegate used to update the indicator on each call to Update
        """
        ...

    def Update(self, security: QuantConnect.Securities.Security, data: QuantConnect.Data.BaseData) -> None:
        """
        Updates this model using the new price information in
        the specified security instance
        
        :param security: The security to calculate volatility for
        :param data: The new piece of data for the security
        """
        ...

    def GetHistoryRequirements(self, security: QuantConnect.Securities.Security, utcTime: datetime.datetime) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.HistoryRequest]:
        """
        Returns history requirements for the volatility model expressed in the form of history request
        
        :param security: The security of the request
        :param utcTime: The date/time of the request
        :returns: History request object list, or empty if no requirements.
        """
        ...


class OptionFilterUniverse(QuantConnect.Securities.ContractSecurityFilterUniverse[QuantConnect_Securities_OptionFilterUniverse]):
    """Represents options symbols universe used in filtering."""

    @typing.overload
    def __init__(self) -> None:
        """Constructs OptionFilterUniverse"""
        ...

    @typing.overload
    def __init__(self, allSymbols: System.Collections.Generic.IEnumerable[QuantConnect.Symbol], underlying: QuantConnect.Data.BaseData) -> None:
        """Constructs OptionFilterUniverse"""
        ...

    def Refresh(self, allSymbols: System.Collections.Generic.IEnumerable[QuantConnect.Symbol], underlying: QuantConnect.Data.BaseData, exchangeDateChange: bool = True) -> None:
        """
        Refreshes this option filter universe and allows specifying if the exchange date changed from last call
        
        :param allSymbols: All the options contract symbols
        :param underlying: The current underlying last data point
        :param exchangeDateChange: True if the exchange data has chanced since the last call or construction
        """
        ...

    def IsStandard(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Determine if the given Option contract symbol is standard
        
        This method is protected.
        
        :returns: True if standard.
        """
        ...

    def Strikes(self, minStrike: int, maxStrike: int) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Applies filter selecting options contracts based on a range of strikes in relative terms
        
        :param minStrike: The minimum strike relative to the underlying price, for example, -1 would filter out contracts further than 1 strike below market price
        :param maxStrike: The maximum strike relative to the underlying price, for example, +1 would filter out contracts further than 1 strike above market price
        :returns: Universe with filter applied.
        """
        ...

    def CallsOnly(self) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Sets universe of call options (if any) as a selection
        
        :returns: Universe with filter applied.
        """
        ...

    def PutsOnly(self) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Sets universe of put options (if any) as a selection
        
        :returns: Universe with filter applied.
        """
        ...


class OptionFilterUniverseEx(System.Object):
    """Extensions for Linq support"""

    @staticmethod
    def Where(universe: QuantConnect.Securities.OptionFilterUniverse, predicate: typing.Callable[[QuantConnect.Symbol], bool]) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Filters universe
        
        :param universe: Universe to apply the filter too
        :param predicate: Bool function to determine which Symbol are filtered
        :returns: Universe with filter applied.
        """
        ...

    @staticmethod
    def Select(universe: QuantConnect.Securities.OptionFilterUniverse, mapFunc: typing.Callable[[QuantConnect.Symbol], QuantConnect.Symbol]) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Maps universe
        
        :param universe: Universe to apply the filter too
        :param mapFunc: Symbol function to determine which Symbols are filtered
        :returns: Universe with filter applied.
        """
        ...

    @staticmethod
    def SelectMany(universe: QuantConnect.Securities.OptionFilterUniverse, mapFunc: typing.Callable[[QuantConnect.Symbol], System.Collections.Generic.IEnumerable[QuantConnect.Symbol]]) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Binds universe
        
        :param universe: Universe to apply the filter too
        :param mapFunc: Symbol function to determine which Symbols are filtered
        :returns: Universe with filter applied.
        """
        ...

    @staticmethod
    def WhereContains(universe: QuantConnect.Securities.OptionFilterUniverse, filterList: System.Collections.Generic.List[QuantConnect.Symbol]) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Updates universe to only contain the symbols in the list
        
        :param universe: Universe to apply the filter too
        :param filterList: List of Symbols to keep in the Universe
        :returns: Universe with filter applied.
        """
        ...


class FutureExpirationCycles(System.Object):
    """Static class contains definitions of popular futures expiration cycles"""

    January: typing.List[int] = ...
    """January Cycle: Expirations in January, April, July, October (the first month of each quarter)"""

    February: typing.List[int] = ...
    """February Cycle: Expirations in February, May, August, November (second month)"""

    March: typing.List[int] = ...
    """March Cycle: Expirations in March, June, September, December (third month)"""

    AllYear: typing.List[int] = ...
    """All Year Cycle: Expirations in every month of the year"""

    HMUZ: typing.List[int] = ...
    """HMUZ Cycle"""

    HKNUZ: typing.List[int] = ...
    """HKNUZ Cycle"""

    HKNUVZ: typing.List[int] = ...
    """HKNUVZ Cycle"""

    FHKNQUVZ: typing.List[int] = ...
    """FHKNQUVZ Cycle"""

    FHKNQUX: typing.List[int] = ...
    """FHKNQUX Cycle"""

    FGHJKMNQUVXZ: typing.List[int] = ...
    """FGHJKMNQUVXZ Cycle"""


class Futures(System.Object):
    """Futures static class contains shortcut definitions of major futures contracts available for trading"""

    class Grains(System.Object):
        """Grains and Oilseeds group"""

        BlackSeaCornFinanciallySettledPlatts: str = "BCF"
        """Black Sea Corn Financially Settled (Platts) Futures"""

        BlackSeaWheatFinanciallySettledPlatts: str = "BWF"
        """Black Sea Wheat Financially Settled (Platts) Futures"""

        SRWWheat: str = "ZW"
        """Chicago SRW Wheat Futures"""

        Wheat: str = ...
        """Default wheat contract is SRWWheat"""

        HRWWheat: str = "KE"
        """KC HRW Wheat Futures"""

        Corn: str = "ZC"
        """Corn Futures"""

        Soybeans: str = "ZS"
        """Soybeans Futures"""

        SoybeanMeal: str = "ZM"
        """Soybean Meal Futures"""

        SoybeanOil: str = "ZL"
        """Soybean Oil Futures"""

        Oats: str = "ZO"
        """Oats Futures"""

    class Currencies(System.Object):
        """Currencies group"""

        USD: str = "DX"
        """U.S. Dollar Index Futures"""

        GBP: str = "6B"
        """British Pound Futures"""

        CAD: str = "6C"
        """Canadian Dollar Futures"""

        JPY: str = "6J"
        """Japanese Yen Futures"""

        CHF: str = "6S"
        """Swiss Franc Futures"""

        EUR: str = "6E"
        """Euro FX Futures"""

        AUD: str = "6A"
        """Australian Dollar Futures"""

        NZD: str = "6N"
        """New Zealand Dollar Futures"""

        RUB: str = "6R"
        """Russian Ruble Futures"""

        BRL: str = "6L"
        """Brazillian Real Futures"""

        MXN: str = "6M"
        """Mexican Peso Futures"""

        ZAR: str = "6Z"
        """South African Rand Futures"""

        AUDCAD: str = "ACD"
        """Australian Dollar/Canadian Dollar Futures"""

        AUDJPY: str = "AJY"
        """Australian Dollar/Japanese Yen Futures"""

        AUDNZD: str = "ANE"
        """Australian Dollar/New Zealand Dollar Futures"""

        BTC: str = "BTC"
        """Bitcoin Futures"""

        CADJPY: str = "CJY"
        """Canadian Dollar/Japanese Yen Futures"""

        StandardSizeUSDOffshoreRMBCNH: str = "CNH"
        """Standard-Size USD/Offshore RMB (CNH) Futures"""

        EuroFXEmini: str = "E7"
        """E-mini Euro FX Futures"""

        EURAUD: str = "EAD"
        """Euro/Australian Dollar Futures"""

        EURCAD: str = "ECD"
        """Euro/Canadian Dollar Futures"""

        EURSEK: str = "ESK"
        """Euro/Swedish Krona Futures"""

        JapaneseYenEmini: str = "J7"
        """E-mini Japanese Yen Futures"""

    class Energies(System.Object):
        """Energies group"""

        PropaneNonLDHMontBelvieu: str = "1S"
        """Propane Non LDH Mont Belvieu (OPIS) BALMO Futures"""

        ArgusPropaneFarEastIndexBALMO: str = "22"
        """Argus Propane Far East Index BALMO Futures"""

        MiniEuropeanThreePointPercentFiveFuelOilBargesPlatts: str = "A0D"
        """Mini European 3.5% Fuel Oil Barges FOB Rdam (Platts) Futures"""

        MiniSingaporeFuelOil180CstPlatts: str = "A0F"
        """Mini Singapore Fuel Oil 180 cst (Platts) Futures"""

        GulfCoastULSDPlattsUpDownBALMO: str = "A1L"
        """Gulf Coast ULSD (Platts) Up-Down BALMO Futures"""

        GulfCoastJetPlattsUpDownBALMO: str = "A1M"
        """Gulf Coast Jet (Platts) Up-Down BALMO Futures"""

        PropaneNonLDHMontBelvieuOPIS: str = "A1R"
        """Propane Non-LDH Mont Belvieu (OPIS) Futures"""

        EuropeanPropaneCIFARAArgusBALMO: str = "A32"
        """European Propane CIF ARA (Argus) BALMO Futures"""

        PremiumUnleadedGasoline10ppmFOBMEDPlatts: str = "A3G"
        """Premium Unleaded Gasoline 10 ppm FOB MED (Platts) Futures"""

        ArgusPropaneFarEastIndex: str = "A7E"
        """Argus Propane Far East Index Futures"""

        GasolineEurobobOxyNWEBargesArgusCrackSpreadBALMO: str = "A7I"
        """Gasoline Euro-bob Oxy NWE Barges (Argus) Crack Spread BALMO Futures"""

        MontBelvieuNaturalGasolineOPIS: str = "A7Q"
        """Mont Belvieu Natural Gasoline (OPIS) Futures"""

        MontBelvieuNormalButaneOPISBALMO: str = "A8J"
        """Mont Belvieu Normal Butane (OPIS) BALMO Futures"""

        ConwayPropaneOPIS: str = "A8K"
        """Conway Propane (OPIS) Futures"""

        MontBelvieuLDHPropaneOPISBALMO: str = "A8O"
        """Mont Belvieu LDH Propane (OPIS) BALMO Futures"""

        ArgusPropaneFarEastIndexVsEuropeanPropaneCIFARAArgus: str = "A91"
        """Argus Propane Far East Index vs. European Propane CIF ARA (Argus) Futures"""

        ArgusPropaneSaudiAramco: str = "A9N"
        """Argus Propane (Saudi Aramco) Futures"""

        GroupThreeULSDPlattsVsNYHarborULSD: str = "AA6"
        """Group Three ULSD (Platts) vs. NY Harbor ULSD Futures"""

        GroupThreeSuboctaneGasolinePlattsVsRBOB: str = "AA8"
        """Group Three Sub-octane Gasoliine (Platts) vs. RBOB Futures"""

        SingaporeFuelOil180cstPlattsBALMO: str = "ABS"
        """Singapore Fuel Oil 180 cst (Platts) BALMO Futures"""

        SingaporeFuelOil380cstPlattsBALMO: str = "ABT"
        """Singapore Fuel Oil 380 cst (Platts) BALMO Futures"""

        MontBelvieuEthaneOPIS: str = "AC0"
        """Mont Belvieu Ethane (OPIS) Futures"""

        MontBelvieuNormalButaneOPIS: str = "AD0"
        """Mont Belvieu Normal Butane (OPIS) Futures"""

        BrentCrudeOilVsDubaiCrudeOilPlatts: str = "ADB"
        """Brent Crude Oil vs. Dubai Crude Oil (Platts) Futures"""

        ArgusLLSvsWTIArgusTradeMonth: str = "AE5"
        """Argus LLS vs. WTI (Argus) Trade Month Futures"""

        SingaporeGasoilPlattsVsLowSulphurGasoilFutures: str = "AGA"
        """Singapore Gasoil (Platts) vs. Low Sulphur Gasoil Futures"""

        LosAngelesCARBOBGasolineOPISvsRBOBGasoline: str = "AJL"
        """Los Angeles CARBOB Gasoline (OPIS) vs. RBOB Gasoline Futures"""

        LosAngelesJetOPISvsNYHarborULSD: str = "AJS"
        """Los Angeles Jet (OPIS) vs. NY Harbor ULSD Futures"""

        LosAngelesCARBDieselOPISvsNYHarborULSD: str = "AKL"
        """Los Angeles CARB Diesel (OPIS) vs. NY Harbor ULSD Futures"""

        EuropeanNaphthaPlattsBALMO: str = "AKZ"
        """European Naphtha (Platts) BALMO Futures"""

        EuropeanPropaneCIFARAArgus: str = "APS"
        """European Propane CIF ARA (Argus) Futures"""

        MontBelvieuNaturalGasolineOPISBALMO: str = "AR0"
        """Mont Belvieu Natural Gasoline (OPIS) BALMO Futures"""

        RBOBGasolineCrackSpread: str = "ARE"
        """RBOB Gasoline Crack Spread Futures"""

        GulfCoastHSFOPlattsBALMO: str = "AVZ"
        """Gulf Coast HSFO (Platts) BALMO Futures"""

        MarsArgusVsWTITradeMonth: str = "AYV"
        """Mars (Argus) vs. WTI Trade Month Futures"""

        MarsArgusVsWTIFinancial: str = "AYX"
        """Mars (Argus) vs. WTI Financial Futures"""

        EthanolT2FOBRdamIncludingDutyPlatts: str = "AZ1"
        """Ethanol T2 FOB Rdam Including Duty (Platts) Futures"""

        MontBelvieuLDHPropaneOPIS: str = "B0"
        """Mont Belvieu LDH Propane (OPIS) Futures"""

        GasolineEurobobOxyNWEBargesArgus: str = "B7H"
        """Gasoline Euro-bob Oxy NWE Barges (Argus) Futures"""

        WTIBrentFinancial: str = "BK"
        """WTI-Brent Financial Futures"""

        ThreePointFivePercentFuelOilBargesFOBRdamPlattsCrackSpread1000mt: str = "BOO"
        """3.5% Fuel Oil Barges FOB Rdam (Platts) Crack Spread (1000mt) Futures"""

        GasolineEurobobOxyNWEBargesArgusBALMO: str = "BR7"
        """Gasoline Euro-bob Oxy NWE Barges (Argus) BALMO Futures"""

        BrentLastDayFinancial: str = "BZ"
        """Brent Last Day Financial Futures"""

        CrudeOilWTI: str = "CL"
        """Crude Oil WTI Futures"""

        GulfCoastCBOBGasolineA2PlattsVsRBOBGasoline: str = "CRB"
        """Gulf Coast CBOB Gasoline A2 (Platts) vs. RBOB Gasoline Futures"""

        ClearbrookBakkenSweetCrudeOilMonthlyIndexNetEnergy: str = "CSW"
        """Clearbrook Bakken Sweet Crude Oil Monthly Index (Net Energy) Futures"""

        WTIFinancial: str = "CSX"
        """WTI Financial Futures"""

        ChicagoEthanolPlatts: str = "CU"
        """Chicago Ethaanol (Platts) Futures"""

        SingaporeMogas92UnleadedPlattsBrentCrackSpread: str = "D1N"
        """Singapore Mogas 92 Unleaded (Platts) Brent Crack Spread Futures"""

        DubaiCrudeOilPlattsFinancial: str = "DCB"
        """Dubai Crude Oil (Platts) Financial Futures"""

        JapanCnFNaphthaPlattsBALMO: str = "E6"
        """Japan C&F Naphtha (Platts) BALMO Futures"""

        Ethanol: str = "EH"
        """Ethanol Futures"""

        EuropeanNaphthaPlattsCrackSpread: str = "EN"
        """European Naphtha (Platts) Crack Spread Futures"""

        EuropeanPropaneCIFARAArgusVsNaphthaCargoesCIFNWEPlatts: str = "EPN"
        """European Propane CIF ARA (Argus) vs. Naphtha Cargoes CIF NWE (Platts) Futures"""

        SingaporeFuelOil380cstPlattsVsEuropeanThreePointFivePercentFuelOilBargesFOBRdamPlatts: str = "EVC"
        """Singapore Fuel Oil 380 cst (Platts) vs. European 3.5% Fuel Oil Barges FOB Rdam (Platts) Futures"""

        EastWestGasolineSpreadPlattsArgus: str = "EWG"
        """East-West Gasoline Spread (Platts-Argus) Futures"""

        EastWestNaphthaJapanCFvsCargoesCIFNWESpreadPlatts: str = "EWN"
        """East-West Naphtha: Japan C&F vs. Cargoes CIF NWE Spread (Platts) Futures"""

        RBOBGasolineVsEurobobOxyNWEBargesArgusThreeHundredFiftyThousandGallons: str = "EXR"
        """RBOB Gasoline vs. Euro-bob Oxy NWE Barges (Argus) (350,000 gallons) Futures"""

        ThreePointFivePercentFuelOilBargesFOBRdamPlattsCrackSpread: str = "FO"
        """3.5% Fuel Oil Barges FOB Rdam (Platts) Crack Spread Futures"""

        FreightRouteTC14Baltic: str = "FRC"
        """Freight Route TC14 (Baltic) Futures"""

        OnePercentFuelOilCargoesFOBNWEPlattsVsThreePointFivePercentFuelOilBargesFOBRdamPlatts: str = "FSS"
        """1% Fuel Oil Cargoes FOB NWE (Platts) vs. 3.5% Fuel Oil Barges FOB Rdam (Platts) Futures"""

        GulfCoastHSFOPlattsVsEuropeanThreePointFivePercentFuelOilBargesFOBRdamPlatts: str = "GCU"
        """Gulf Coast HSFO (Platts) vs. European 3.5% Fuel Oil Barges FOB Rdam (Platts) Futures"""

        WTIHoustonCrudeOil: str = "HCL"
        """WTI Houston Crude Oil Futures"""

        NaturalGasHenryHubLastDayFinancial: str = "HH"
        """Natural Gas (Henry Hub) Last-day Financial Futures"""

        HeatingOil: str = "HO"
        """Heating Oil Futures"""

        NaturalGasHenryHubPenultimateFinancial: str = "HP"
        """Natural Gas (Henry Hub) Penultimate Financial Futures"""

        WTIHoustonArgusVsWTITradeMonth: str = "HTT"
        """WTI Houston (Argus) vs. WTI Trade Month Futures"""

        Gasoline: str = "RB"
        """Gasoline RBOB Futures"""

        NaturalGas: str = "NG"
        """Natural Gas Futures"""

        BrentCrude: str = "B"
        """Brent Crude Futures"""

        LowSulfurGasoil: str = "G"
        """Low Sulfur Gasoil"""

    class Financials(System.Object):
        """Financials group"""

        Y30TreasuryBond: str = "ZB"
        """30Y U.S. Treasury Bond Futures"""

        Y10TreasuryNote: str = "ZN"
        """10Y U.S. Treasury Note Futures"""

        Y5TreasuryNote: str = "ZF"
        """5Y U.S. Treasury Note Futures"""

        Y2TreasuryNote: str = "ZT"
        """2Y U.S. Treasury Note Futures"""

        EuroDollar: str = "GE"
        """EuroDollar Futures"""

        FiveYearUSDMACSwap: str = "F1U"
        """5-Year USD MAC Swap Futures"""

        UltraUSTreasuryBond: str = "UB"
        """Ultra U.S. Treasury Bond Futures"""

        UltraTenYearUSTreasuryNote: str = "TN"
        """Ultra 10-Year U.S. Treasury Note Futures"""

    class Indices(System.Object):
        """Indices group"""

        SP500EMini: str = "ES"
        """E-mini S&P 500 Futures"""

        NASDAQ100EMini: str = "NQ"
        """E-mini NASDAQ 100 Futures"""

        Dow30EMini: str = "YM"
        """E-mini Dow Indu 30 Futures"""

        VIX: str = "VX"
        """CBOE Volatility Index Futures"""

        Russell2000EMini: str = "RTY"
        """E-mini Russell 2000 Futures"""

        Nikkei225Dollar: str = "NKD"
        """Nikkei-225 Dollar Futures"""

        BloombergCommodityIndex: str = "AW"
        """Bloomberg Commodity Index Futures"""

        NASDAQ100BiotechnologyEMini: str = "BIO"
        """E-mini Nasdaq-100 Biotechnology Index Futures"""

        FTSEEmergingEmini: str = "EI"
        """E-mini FTSE Emerging Index Futures"""

        SP400MidCapEmini: str = "EMD"
        """E-mini S&P MidCap 400 Futures"""

        SPGSCICommodity: str = "GD"
        """S&P-GSCI Commodity Index Futures"""

        USDDenominatedIbovespa: str = "IBV"
        """USD-Denominated Ibovespa Index Futures"""

        MSCITaiwanIndex: str = "TW"
        """USD-Denominated MSCI Taiwan Index Futures"""

        Nikkei225Yen: str = "NK"
        """Nikkei-225 Yen denominated Index Futures"""

        Nifty50: str = "IN"
        """Nifty 50  Index Futures"""

        HangSeng: str = "HSI"
        """Hang Seng Index"""

    class Forestry(System.Object):
        """Forestry group"""

        RandomLengthLumber: str = "LBS"
        """Random Length Lumber Futures"""

    class Meats(System.Object):
        """Meats group"""

        LiveCattle: str = "LE"
        """Live Cattle Futures"""

        FeederCattle: str = "GF"
        """Feeder Cattle Futures"""

        LeanHogs: str = "HE"
        """Lean Hogs Futures"""

    class Metals(System.Object):
        """Metals group"""

        Gold: str = "GC"
        """Gold Futures"""

        Silver: str = "SI"
        """Silver Futures"""

        Platinum: str = "PL"
        """Platinum Futures"""

        Palladium: str = "PA"
        """Palladium Futures"""

        AluminumMWUSTransactionPremiumPlatts25MT: str = "AUP"
        """Aluminum MW U.S. Transaction Premium Platts (25MT) Futures"""

        AluminiumEuropeanPremiumDutyPaidMetalBulletin: str = "EDP"
        """Aluminium European Premium Duty-Paid (Metal Bulletin) Futures"""

        Copper: str = "HG"
        """Copper Futures"""

        USMidwestDomesticHotRolledCoilSteelCRUIndex: str = "HRC"
        """U.S. Midwest Domestic Hot-Rolled Coil Steel (CRU) Index Futures"""

    class Softs(System.Object):
        """Softs group"""

        Cotton2: str = "CT"
        """Cotton #2 Futures"""

        OrangeJuice: str = "OJ"
        """Orange Juice Futures"""

        Coffee: str = "KC"
        """Coffee C Arabica Futures"""

        Sugar11: str = "SB"
        """Sugar #11 Futures ICE"""

        Sugar11CME: str = "YO"
        """Sugar #11 Futures CME"""

        Cocoa: str = "CC"
        """Cocoa Futures"""

    class Dairy(System.Object):
        """Dairy group"""

        CashSettledButter: str = "CB"
        """Cash-settled Butter Futures"""

        CashSettledCheese: str = "CSC"
        """Cash-settled Cheese Futures"""

        ClassIIIMilk: str = "DC"
        """Class III Milk Futures"""

        DryWhey: str = "DY"
        """Dry Whey Futures"""

        ClassIVMilk: str = "GDK"
        """Class IV Milk Futures"""

        NonfatDryMilk: str = "GNF"
        """Non-fat Dry Milk Futures"""


class FutureFilterUniverse(QuantConnect.Securities.ContractSecurityFilterUniverse[QuantConnect_Securities_FutureFilterUniverse]):
    """Represents futures symbols universe used in filtering."""

    def __init__(self, allSymbols: System.Collections.Generic.IEnumerable[QuantConnect.Symbol], underlying: QuantConnect.Data.BaseData) -> None:
        """Constructs FutureFilterUniverse"""
        ...

    def IsStandard(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Determine if the given Future contract symbol is standard
        
        This method is protected.
        
        :returns: True if contract is standard.
        """
        ...

    def ExpirationCycle(self, months: typing.List[int]) -> QuantConnect.Securities.FutureFilterUniverse:
        """
        Applies filter selecting futures contracts based on expiration cycles. See FutureExpirationCycles for details
        
        :param months: Months to select contracts from
        :returns: Universe with filter applied.
        """
        ...


class FutureFilterUniverseEx(System.Object):
    """Extensions for Linq support"""

    @staticmethod
    def Where(universe: QuantConnect.Securities.FutureFilterUniverse, predicate: typing.Callable[[QuantConnect.Symbol], bool]) -> QuantConnect.Securities.FutureFilterUniverse:
        """
        Filters universe
        
        :param universe: Universe to apply the filter too
        :param predicate: Bool function to determine which Symbol are filtered
        :returns: FutureFilterUniverse with filter applied.
        """
        ...

    @staticmethod
    def Select(universe: QuantConnect.Securities.FutureFilterUniverse, mapFunc: typing.Callable[[QuantConnect.Symbol], QuantConnect.Symbol]) -> QuantConnect.Securities.FutureFilterUniverse:
        """
        Maps universe
        
        :param universe: Universe to apply the filter too
        :param mapFunc: Symbol function to determine which Symbols are filtered
        :returns: FutureFilterUniverse with filter applied.
        """
        ...

    @staticmethod
    def SelectMany(universe: QuantConnect.Securities.FutureFilterUniverse, mapFunc: typing.Callable[[QuantConnect.Symbol], System.Collections.Generic.IEnumerable[QuantConnect.Symbol]]) -> QuantConnect.Securities.FutureFilterUniverse:
        """
        Binds universe
        
        :param universe: Universe to apply the filter too
        :param mapFunc: Symbols function to determine which Symbols are filtered
        :returns: FutureFilterUniverse with filter applied.
        """
        ...


