import pytest

import tests.resources.ethercat
from ingenialink.dictionary import Interface, SubnodeType
from ingenialink.ethercat.dictionary import EthercatDictionaryV2, EthercatDictionaryV3

SINGLE_AXIS_BASE_SUBNODES = {0: SubnodeType.COMMUNICATION, 1: SubnodeType.MOTION}


@pytest.mark.no_connection
def test_read_dictionary():
    dictionary_path = tests.resources.ethercat.TEST_DICT_ETHERCAT
    expected_device_attr = {
        "path": dictionary_path,
        "version": "2",
        "firmware_version": "2.0.1",
        "product_code": 57745409,
        "part_number": "CAP-NET-E",
        "revision_number": 196635,
        "interface": Interface.ECAT,
        "subnodes": SINGLE_AXIS_BASE_SUBNODES,
        "is_safe": False,
    }

    ethercat_dict = EthercatDictionaryV2(dictionary_path)

    for attr, value in expected_device_attr.items():
        assert getattr(ethercat_dict, attr) == value


@pytest.mark.no_connection
def test_read_dictionary_file_not_found():
    dictionary_path = "false.xdf"

    with pytest.raises(FileNotFoundError):
        EthercatDictionaryV2(dictionary_path)


@pytest.mark.no_connection
def test_read_dictionary_registers():
    dictionary_path = tests.resources.ethercat.TEST_DICT_ETHERCAT
    expected_regs_per_subnode = {
        0: [
            "DRV_DIAG_ERROR_LAST_COM",
            "DIST_CFG_REG0_MAP",
            "COMMS_ETH_IP",
            "COMMS_ETH_NET_MASK",
            "DRV_BOOT_COCO_VERSION",
            "MON_CFG_EOC_TYPE",
            "ETG_COMMS_RPDO_ASSIGN_TOTAL",
            "ETG_COMMS_RPDO_ASSIGN_1",
            "ETG_COMMS_RPDO_ASSIGN_2",
            "ETG_COMMS_RPDO_ASSIGN_3",
            "ETG_COMMS_RPDO_MAP1_TOTAL",
            *[f"ETG_COMMS_RPDO_MAP1_{i_element}" for i_element in range(1, 16)],
            "ETG_COMMS_RPDO_MAP2_TOTAL",
            *[f"ETG_COMMS_RPDO_MAP2_{i_element}" for i_element in range(1, 16)],
            "ETG_COMMS_RPDO_MAP3_TOTAL",
            *[f"ETG_COMMS_RPDO_MAP3_{i_element}" for i_element in range(1, 16)],
            "ETG_COMMS_TPDO_ASSIGN_TOTAL",
            "ETG_COMMS_TPDO_ASSIGN_1",
            "ETG_COMMS_TPDO_ASSIGN_2",
            "ETG_COMMS_TPDO_ASSIGN_3",
            "ETG_COMMS_TPDO_MAP1_TOTAL",
            *[f"ETG_COMMS_TPDO_MAP1_{i_element}" for i_element in range(1, 16)],
            "ETG_COMMS_TPDO_MAP2_TOTAL",
            *[f"ETG_COMMS_TPDO_MAP2_{i_element}" for i_element in range(1, 16)],
            "ETG_COMMS_TPDO_MAP3_TOTAL",
            *[f"ETG_COMMS_TPDO_MAP3_{i_element}" for i_element in range(1, 16)],
        ],
        1: [
            "CL_POS_SET_POINT_VALUE",
            "CL_POS_FBK_VALUE",
            "CL_VEL_SET_POINT_VALUE",
            "CL_VEL_FBK_VALUE",
            "COMMU_ANGLE_SENSOR",
        ],
    }

    ethercat_dict = EthercatDictionaryV2(dictionary_path)

    for subnode in expected_regs_per_subnode:
        assert expected_regs_per_subnode[subnode] == list(ethercat_dict.registers(subnode))


@pytest.mark.no_connection
def test_pdo_maps_equivalence():
    """This tests checks that registers generated by hardcoded
    EthercatDictionaryV2 are equivalent to a xdf v3 reference file."""
    dictionary_path = tests.resources.ethercat.TEST_DICT_ETHERCAT
    ethercat_dict_v2 = EthercatDictionaryV2(dictionary_path)
    ethercat_dict_v3 = EthercatDictionaryV3(tests.resources.DEN_NET_E_2_8_0_xdf_v3, Interface.ECAT)

    for register_v3 in ethercat_dict_v3.all_registers():
        reg_id = register_v3.identifier
        if reg_id.startswith(("ETG_COMMS_TPDO", "ETG_COMMS_RPDO")):
            register_v2 = ethercat_dict_v2.get_register(reg_id)
            assert register_v3.idx == register_v2.idx
            assert register_v3.subidx == register_v2.subidx
            assert register_v3.is_node_id_dependent == register_v2.is_node_id_dependent
            assert register_v3.dtype == register_v2.dtype
            assert register_v3.access == register_v2.access
            assert register_v3.identifier == register_v2.identifier
            assert register_v3.units == register_v2.units
            assert register_v3.pdo_access == register_v2.pdo_access
            assert register_v3.subnode == register_v2.subnode
            assert register_v3.storage == register_v2.storage
            assert register_v3.labels == register_v2.labels
            assert register_v3.cat_id == register_v2.cat_id
            assert register_v3.address_type == register_v2.address_type
            assert register_v3.monitoring == register_v2.monitoring

    item_uids = {obj.uid for obj in ethercat_dict_v2.all_objs()}
    assert {
        "ETG_COMMS_RPDO_MAP3",
        "ETG_COMMS_TPDO_MAP3",
        "ETG_COMMS_TPDO_MAP1",
        "ETG_COMMS_RPDO_ASSIGN",
        "ETG_COMMS_RPDO_MAP2",
        "ETG_COMMS_TPDO_ASSIGN",
        "ETG_COMMS_TPDO_MAP2",
        "ETG_COMMS_RPDO_MAP1",
    } == item_uids

    for item_v3 in ethercat_dict_v3.all_objs():
        if item_v3.uid.startswith(("ETG_COMMS_TPDO", "ETG_COMMS_RPDO")):
            item_v2 = ethercat_dict_v2.get_object(item_v3.uid)
            assert item_v3.uid == item_v2.uid
            assert item_v3.idx == item_v2.idx
            assert item_v3.object_type == item_v2.object_type
            assert len(item_v3.registers) == len(item_v2.registers)


@pytest.mark.no_connection
def test_read_dictionary_registers_multiaxis():
    expected_num_registers_per_subnode = {0: 106, 1: 2, 2: 2}
    dictionary_path = tests.resources.ethercat.TEST_DICT_ETHERCAT_AXIS

    ethercat_dict = EthercatDictionaryV2(dictionary_path)
    assert ethercat_dict.subnodes == {
        0: SubnodeType.COMMUNICATION,
        1: SubnodeType.MOTION,
        2: SubnodeType.MOTION,
    }

    for subnode in expected_num_registers_per_subnode:
        num_registers = len(ethercat_dict.registers(subnode))
        assert num_registers == expected_num_registers_per_subnode[subnode]


@pytest.mark.no_connection
def test_read_dictionary_categories():
    expected_categories = [
        "IDENTIFICATION",
        "COMMUTATION",
        "COMMUNICATIONS",
        "REPORTING",
        "MONITORING",
        "TARGET",
    ]
    dictionary_path = tests.resources.ethercat.TEST_DICT_ETHERCAT

    ethercat_dict = EthercatDictionaryV2(dictionary_path)

    assert ethercat_dict.categories.category_ids == expected_categories


@pytest.mark.no_connection
def test_read_dictionary_errors():
    expected_errors = [
        0x00003280,
        0x00007380,
        0x00007385,
        0x06010000,
    ]
    dictionary_path = tests.resources.ethercat.TEST_DICT_ETHERCAT

    ethercat_dict = EthercatDictionaryV2(dictionary_path)

    assert list(ethercat_dict.errors) == expected_errors


@pytest.mark.no_connection
def test_read_xdf_register():
    dictionary_path = tests.resources.ethercat.TEST_DICT_ETHERCAT
    idx = 0x580F
    subidx = 0x00
    reg_id = "DRV_DIAG_ERROR_LAST_COM"
    subnode = 0

    ethercat_dict = EthercatDictionaryV2(dictionary_path)

    assert ethercat_dict.registers(subnode)[reg_id].idx == idx
    assert ethercat_dict.registers(subnode)[reg_id].subidx == subidx


@pytest.mark.no_connection
@pytest.mark.parametrize(
    "register_uid, subnode, idx",
    [("DIST_CFG_REG0_MAP", 0, 0x5890), ("DRV_OP_CMD", 1, 0x2014), ("DRV_STATE_CONTROL", 2, 0x2810)],
)
def test_mcb_to_can_mapping(register_uid, subnode, idx):
    dictionary_path = tests.resources.ethercat.TEST_DICT_ETHERCAT_AXIS

    ethercat_dict = EthercatDictionaryV2(dictionary_path)

    ethercat_register = ethercat_dict.registers(subnode)[register_uid]
    assert ethercat_register.idx == idx


@pytest.mark.no_connection
def test_object_not_exist():
    dictionary_path = tests.resources.ethercat.TEST_DICT_ETHERCAT
    ethercat_dict = EthercatDictionaryV2(dictionary_path)
    with pytest.raises(KeyError):
        ethercat_dict.get_object("NOT_EXISTING_UID", 0)


@pytest.mark.no_connection
def test_safety_pdo_not_implemented():
    dictionary_path = tests.resources.ethercat.TEST_DICT_ETHERCAT
    ethercat_dict = EthercatDictionaryV2(dictionary_path)
    with pytest.raises(NotImplementedError):
        ethercat_dict.get_safety_rpdo("NOT_EXISTING_UID")
    with pytest.raises(NotImplementedError):
        ethercat_dict.get_safety_tpdo("NOT_EXISTING_UID")
