from typing import Literal

import msgspec
from msgspec import Struct


class TagsSearchFilters(Struct, omit_defaults=True):
    guild_id: int
    # filters
    name: str | None = None  # raw name
    fuzzy: bool = False  # fuzzy (pg_trgm)
    include_aliases: bool = True
    only_aliases: bool = False
    owner_id: int | None = None
    # special modes
    random: bool = False  # random tag
    by_id: int | None = None  # lookup by tag_id
    # output controls
    include_content: bool = False  # return content
    include_rank: bool = False
    # sorting & paging
    sort_by: Literal["name", "uses", "created_at"] = "name"
    sort_dir: Literal["asc", "desc"] = "asc"
    limit: int = 20
    offset: int = 0


class TagRowDTO(Struct, omit_defaults=True):
    id: int
    guild_id: int
    name: str
    owner_id: int
    is_alias: bool = False
    canonical_name: str | None = None
    uses: int | None = None
    content: str | None = None
    rank: int | None = None


class TagsSearchResponse(Struct):
    items: list[TagRowDTO]
    total: int | None = None  # optionally returned for paging UIs
    suggestions: list[str] | None = None  # for fuzzy fallbacks


class OpBase(Struct, tag_field="op"):
    """Discriminated union base; JSON will include an 'op' field with the tag."""

    @property
    def op(self) -> str:
        """Return the operation type."""
        ti = msgspec.inspect.type_info(type(self))
        return "" if ti.tag is None else ti.tag  # pyright: ignore[reportAttributeAccessIssue]


class OpCreate(OpBase, tag="create"):
    guild_id: int
    name: str
    content: str
    owner_id: int


class OpAlias(OpBase, tag="alias"):
    guild_id: int
    new_name: str
    old_name: str
    owner_id: int


class OpEdit(OpBase, tag="edit"):
    guild_id: int
    name: str
    new_content: str
    owner_id: int


class OpRemove(OpBase, tag="remove"):
    guild_id: int
    name: str
    requester_id: int


class OpRemoveById(OpBase, tag="remove_by_id"):
    guild_id: int
    tag_id: int
    requester_id: int


class OpClaim(OpBase, tag="claim"):
    guild_id: int
    name: str
    requester_id: int


class OpTransfer(OpBase, tag="transfer"):
    guild_id: int
    name: str
    new_owner_id: int
    requester_id: int


class OpPurge(OpBase, tag="purge"):
    guild_id: int
    owner_id: int
    requester_id: int


class OpIncrementUsage(OpBase, tag="increment_usage"):
    guild_id: int
    name: str


TagOp = OpCreate | OpAlias | OpEdit | OpRemove | OpRemoveById | OpClaim | OpTransfer | OpPurge | OpIncrementUsage


class TagsMutateRequest(Struct):
    ops: list[TagOp]


class TagsMutateResult(Struct, omit_defaults=True):
    ok: bool
    message: str | None = None
    affected: int | None = None
    tag_id: int | None = None


class TagsMutateResponse(Struct):
    results: list[TagsMutateResult]


class TagsAutocompleteRequest(Struct):
    guild_id: int
    q: str
    mode: Literal["aliased", "non_aliased", "owned_aliased", "owned_non_aliased"] = "aliased"
    owner_id: int | None = None
    limit: int = 12


class TagsAutocompleteResponse(Struct):
    items: list[str]
