"""
Integration Test: Performance Scenarios

Simulates complex or long-running operations to ensure functional correctness and stability under load.
Actual performance metrics are secondary to correct behavior.

Scenario:
- Write a large file.
- Read the file.
- Update the file with a large number of changes.
- Read the file again.
- Repeat the process to simulate load.

Uses real tool logic and tmp_path for isolation.
"""

import pytest

from tunacode.tools import read_file, update_file, write_file


@pytest.mark.asyncio
async def test_performance_scenarios(tmp_path):
    file_path = tmp_path / "largefile.txt"
    large_content = "\n".join(f"Line {i}" for i in range(1000))  # 1,000 lines (keeps under 100KB)
    updated_content = "\n".join(f"UPDATED {i}" for i in range(1000))

    # Write a large file
    write_result = await write_file.write_file(str(file_path), large_content)
    assert "Successfully wrote to new file:" in write_result

    # Read the large file
    read_result = await read_file.read_file(str(file_path))
    assert read_result.startswith("Line")

    # Update the file with a large number of changes
    update_result = await update_file.update_file(str(file_path), large_content, updated_content)
    assert "updated successfully" in update_result

    # Read the file again
    read_result2 = await read_file.read_file(str(file_path))
    assert read_result2.startswith("UPDATED")

    # Repeat the process to simulate load
    for i in range(3):
        content = "\n".join(
            f"Iteration {i} - Line {j}" for j in range(500)
        )  # Keep under size limit
        # For repeated updates, we need to know the current content
        current_content = await read_file.read_file(str(file_path))
        update_result = await update_file.update_file(str(file_path), current_content, content)
        assert "updated successfully" in update_result
        read_result = await read_file.read_file(str(file_path))
        assert read_result.startswith(f"Iteration {i}")


"""
Notes:
- This test uses real tool logic and tmp_path for isolation.
- The test checks for stability and correctness under repeated, large operations.
- It does not measure speed, only functional correctness.
"""
