# YNAB CLI - Copilot Instructions

## Project Overview

This is a command-line interface for interacting with YNAB (You Need A Budget) written in Python. The project follows clean architecture principles with ports and adapters pattern.

## Architecture

The project uses a layered architecture:

- `src/ynab_cli/domain/`: Core business logic, models, and ports (interfaces)
- `src/ynab_cli/adapters/`: External adapters (YNAB API, Rich UI, Textual TUI)
- `src/ynab_cli/host/`: Application entry points and framework integration (Click CLI, Textual app)

### Key Patterns

1. **Ports and Adapters**: Domain defines protocols/interfaces, adapters implement them
2. **Dependency Injection**: Dependencies are injected through constructors
3. **Async/Await**: Most operations are async for better I/O handling
4. **Type Safety**: Heavy use of type hints and protocols

## Coding Standards

### Python Version & Dependencies
- Target Python 3.11+
- Use `uv` for dependency management
- All dependencies defined in `pyproject.toml`

### Code Style
- Follow PEP 8
- Use type hints for all function parameters and return values
- Prefer `typing_extensions` imports for newer typing features
- Use `attrs` for data classes with `@define` decorator
- Use `@override` decorator from `typing_extensions` when overriding methods in protocols or base classes

### Async Patterns
- Most I/O operations should be async
- Use `anyio` for async utilities when needed
- Progress tracking should be async-compatible

### Error Handling
- Use proper exception types
- Handle API errors gracefully
- Provide meaningful error messages to users

## Testing Guidelines

### Test Structure
- Unit tests in `tests/unit/` mirroring `src/` structure
- Use pytest as the testing framework
- All test functions should have proper type annotations
- Test classes should be named `Test<ClassName>`
- Test methods should be descriptive: `test_<method_name>_<scenario>`

### Mocking
- Use `unittest.mock` for mocking external dependencies
- Mock at the adapter level, not domain level
- Use `MagicMock` for complex objects, `Mock` for simple ones

### Test Patterns
```python
from typing import Any
from unittest.mock import MagicMock, Mock
import pytest

class TestMyClass:
    def test_method_name_success_scenario(self) -> None:
        """Test description of what is being tested."""
        # Arrange
        mock_dependency = Mock()
        instance = MyClass(mock_dependency)
        
        # Act
        result = instance.method_name()
        
        # Assert
        assert result == expected_value
        mock_dependency.some_method.assert_called_once()
```

## Domain Models

### Data Classes
- Use `@attrs.define` for data classes
- Implement `to_dict()` and `from_dict()` for serialization when needed
- Use proper type hints including `Self` for class methods

```python
from typing import Any, Self
from attrs import define

@define
class MyModel:
    field: str
    optional_field: int | None = None
    
    def to_dict(self) -> dict[str, Any]:
        return {"field": self.field, "optional_field": self.optional_field}
    
    @classmethod
    def from_dict(cls, data: dict[str, Any]) -> Self:
        return cls(field=data["field"], optional_field=data.get("optional_field"))
```

### Protocols
- Define interfaces as protocols in `domain/ports/`
- Use `typing.Protocol` for structural typing
- Mark implementation methods with `@override`

## UI Frameworks

### Rich (Progress & Prompts)
- Used for CLI progress bars and user prompts
- Implement `Progress` and `IO` protocols
- Handle console stop/start for prompts

### Textual (TUI)
- Used for interactive terminal UI
- Follow Textual app patterns
- Use proper CSS for styling
- Handle async operations properly

### Click (CLI Framework)
- Use Click for command-line interface
- Use `click-default-group` for default commands
- Implement proper help text and options
- Use context objects for passing settings

## API Integration

### YNAB API
- All API models should be in `adapters/ynab/models/`
- Use proper error handling for HTTP requests
- Implement rate limiting if needed
- Use `httpx` for HTTP client

## File Organization

### Import Order
1. Standard library imports
2. Third-party imports
3. Local imports (domain, then adapters, then host)

### Module Structure
- Keep modules focused and cohesive
- Use `__init__.py` for public APIs
- Private modules should have underscore prefix

## Common Patterns

### Progress Tracking
```python
async def long_running_operation(io: IO) -> None:
    await io.progress.update(total=100)
    for i in range(100):
        # Do work
        await io.progress.update(advance=1)
```

### Settings & Configuration
- Use environment variables with `python-dotenv`
- Settings objects should be immutable
- Pass settings through dependency injection

### Error Messages
- Provide actionable error messages
- Include context about what was being attempted
- Use appropriate log levels

## Development Workflow

1. Create feature branches from main
2. Write tests first (TDD when possible)
3. Implement functionality
4. Ensure all tests pass
5. Run linting and formatting (`make lint`)
6. Create pull request with descriptive title and description

## Performance Considerations

- Use async I/O for network operations
- Implement progress tracking for long operations
- Consider memory usage for large datasets
- Use appropriate data structures for lookups

## Security

- Never log sensitive data (API keys, personal info)
- Use environment variables for secrets
- Validate user inputs
- Handle API rate limits gracefully
