# AIShield - Code Obfuscation Against AI Detection

A language-agnostic code obfuscator designed to protect proprietary code from AI-based analysis and reverse engineering. AIShield preserves framework identifiers while intelligently obfuscating custom code, maintaining functionality while disrupting AI pattern recognition.

[![License](https://img.shields.io/badge/License-BSD%203--Clause-blue.svg)](LICENSE)
[![Python 3.8+](https://img.shields.io/badge/python-3.8+-blue.svg)](https://www.python.org/downloads/)
[![GitHub](https://img.shields.io/badge/GitHub-AIShield-blue?logo=github)](https://github.com/webinadvance/aishield)

## Features

- **Smart Obfuscation:** Preserves framework APIs (React, AspNetCore, etc.) while obfuscating custom logic
- **Multi-Language:** C#, JavaScript/TypeScript with extensible plugin architecture
- **AI-Resistant:** Disrupts pattern matching and intent inference by AI models
- **Deterministic:** Consistent hashing ensures reproducible results across builds
- **Framework-Aware:** Preserves common patterns (Async, Controller, Service, Handler, etc.)
- **Reversible:** Generates mapping files for unminification if needed
- **Developer-Friendly:** Auto-language detection and clipboard integration

## Installation

### Prerequisites
- Python 3.8+
- pip

### Quick Setup

```bash
pip install -r requirements.txt
```

Or install directly from distribution:
```bash
pip install code-defender
```

## Quick Start

### Basic Usage

```bash
code-defender              # Obfuscate code (interactive)
code-defender -u           # Unminify code (interactive)
code-defender --help       # Show help message
code-defender --version    # Show version
```

### Workflow

1. Copy code to clipboard
2. Run `code-defender` (or `code-defender -u` for unminifying)
3. Select language (auto-detected) or press Enter to skip
4. Result copied to clipboard
5. Mapping saved to `obfuscation_mapping_<language>.json`

### Supported Languages

- **C# / .NET**
- **JavaScript / TypeScript**

## How It Works

### Obfuscation Strategy

AIShield uses **intelligent preservation** to balance security and maintainability:

| Element | Handling | Reason |
|---------|----------|--------|
| Framework APIs | ✅ Preserved | Needed for compilation/runtime |
| Custom identifiers | ❌ Obfuscated | Protects business logic |
| Method suffixes | ✅ Preserved | Date, Service, Handler, Controller, etc. |
| Imports/Namespaces | ✅ Preserved | Required for code functionality |
| String literals | ❌ Encrypted | Hides hardcoded values |

### Example Impact

- `UserController` → `E1234Controller` (class name hidden, pattern preserved)
- `calculateTotalAmount()` → `C5892E1234()` (logic hidden)
- `HttpClient.Get()` → `HttpClient.Get()` (framework unchanged)
- Custom namespaces → Obfuscated
- Framework namespaces → Preserved

## Configuration

### Global Preservation

Edit `preserve_custom.json` to add project-specific terms:

```json
{
  "preserve": [
    "MyCompany",
    "ProjectX",
    "PartnerAPI"
  ]
}
```

### Language-Specific Lists

Framework words are configured per language:
- `languages/csharp/preserve_language.json` - C# framework terms
- `languages/javascript/preserve_language.json` - JS/TypeScript framework terms

**Don't edit these unless adding new framework words.**

### Custom Suffixes

Modify `get_preserve_suffixes()` in language adapters to preserve domain-specific patterns:

```python
def get_preserve_suffixes(self):
    return ['Async', 'Controller', 'Service', 'Repository', 'Handler']
```

## Advanced Usage

### Reverse Mapping

Obfuscation mappings are saved in `obfuscation_mapping_<language>.json`:

```json
{
  "created_at": "2025-10-30T12:00:00",
  "language": "csharp",
  "reverse_map": {
    "C1234": "MyCompany",
    "E5678": "UserService"
  }
}
```

Use this to track obfuscated identifiers or unminify code if needed.

### Add New Language Support

Create a language plugin:

1. **Create directory:**
   ```bash
   mkdir languages/python
   ```

2. **Create `adapter.py`** (inherit from `LanguageAdapter`):
   - Implement parser setup
   - Define identifier patterns
   - Add detection patterns

3. **Create `preserve_language.json`** with framework terms

4. **Install tree-sitter parser:**
   ```bash
   pip install tree-sitter-python
   ```

See existing adapters (`languages/csharp/adapter.py`, `languages/javascript/adapter.py`) for reference.

## Use Cases

- **AI Protection:** Defend proprietary algorithms from AI code analysis and regeneration
- **IP Protection:** Obfuscate business logic while maintaining code readability for developers
- **Supply Chain Security:** Secure code from AI training dataset absorption
- **Demo Code:** Share samples without exposing business logic
- **Code Size Reduction:** Minify while maintaining AI resistance

## Limitations

- **Not Encryption:** Obfuscation is reversible; determined reverse engineering can recover logic
- **Parser Dependent:** Relies on tree-sitter; edge cases may not parse correctly
- **String Literals:** Obfuscates all strings; may break hardcoded dependencies
- **Dynamic Code:** Cannot handle runtime code generation or reflection-based logic

## 🚀 Coming Soon

### Claude Code IDE Integration with Custom Proxy

**Seamless in-editor obfuscation directly from Claude Code environment**

A native Claude Code extension enabling real-time code obfuscation through a custom HTTP proxy layer:

- **Proxy Architecture:** Custom reverse proxy middleware intercepts HTTP requests between Claude Code LSP client and obfuscation engine
- **Streaming Support:** Chunked transfer encoding for large file obfuscation with progress callbacks
- **AST Streaming:** Incremental Abstract Syntax Tree parsing for near-instantaneous feedback on identifier changes
- **LSP Integration:** Language Server Protocol hooks for syntax highlighting, diagnostics, and code lens features
- **Multi-Adapter Pipeline:** Automatic language detection via file extension + content sniffing with fallback detection
- **Session-Aware Mapping:** Persistent obfuscation mappings per workspace session with automatic cleanup
- **Real-time Diffing:** Visual diff display showing original vs. obfuscated code side-by-side with line-by-line mapping
- **Configuration Schema:** JSON Schema validation for `aishield.json` workspace configuration
- **Telemetry-Free:** No external analytics; all processing stays local with optional audit logging

**Expected Features:**
- ✨ One-click obfuscation from context menu
- 📊 Obfuscation metrics dashboard
- 🔄 Batch processing for entire projects
- 🔐 Encrypted mapping storage with passphrase protection
- 🧪 Test suite generation for obfuscated code validation

---

## Architecture

```
aishield/
├── base.py                    # Base language adapter interface
├── minify.py                  # Obfuscation engine
├── unminify.py                # Reverse obfuscation utility
├── cli.py                     # Command-line interface
├── preserve_custom.json       # Global preservation list
├── languages/
│   ├── csharp/
│   │   ├── adapter.py
│   │   └── preserve_language.json
│   └── javascript/
│       ├── adapter.py
│       └── preserve_language.json
```

## Contributing

See [CONTRIBUTING.md](CONTRIBUTING.md) for contribution guidelines.

**Areas for contribution:**
- Additional language support (Python, Go, Java, Rust, etc.)
- Framework word list expansion
- CLI enhancements
- Documentation improvements

## License

BSD 3-Clause License - see [LICENSE](LICENSE)

## Support

- **Issues:** [GitHub Issues](https://github.com/webinadvance/aishield/issues)
- **Discussions:** [GitHub Discussions](https://github.com/webinadvance/aishield/discussions)
- **GitHub:** [webinadvance/aishield](https://github.com/webinadvance/aishield)
- **PyPI:** [code-defender](https://pypi.org/project/code-defender/)

---

**⚠️ Note:** AIShield is a defensive obfuscation tool. Use responsibly and in compliance with applicable laws and regulations.

---

💜 **Made with love and ❤️ in collaboration with Claude** 🤖
*Powered by intelligent code analysis and obfuscation techniques*
