# Contributing to Comp-LEO SDK

Thank you for your interest in contributing! This guide will help you get started.

## Development Setup

### Prerequisites

- Python 3.10+
- Git
- pip

### Setup

```bash
# Clone repository
git clone https://github.com/compiledger/comp-leo-sdk
cd comp-leo-sdk

# Create virtual environment
python -m venv venv
source venv/bin/activate  # On Windows: venv\Scripts\activate

# Install in development mode
make install-dev
# or
pip install -e ".[dev,api]"
```

### Run Tests

```bash
# Run all tests
make test

# Run specific test file
pytest tests/test_parser.py -v

# Run with coverage
pytest --cov=comp_leo --cov-report=html
open htmlcov/index.html
```

## Code Style

We use:
- **Black** for code formatting
- **Ruff** for linting
- **MyPy** for type checking

```bash
# Format code
make format

# Run linters
make lint

# All checks
make pre-commit
```

## Adding New Rules

### 1. Define Rule in Policy Pack

Edit `comp_leo/policies/aleo-baseline.json`:

```json
{
  "id": "my-new-check",
  "name": "My New Security Check",
  "description": "Detailed description of what this checks",
  "severity": "high",
  "violation_type": "security",
  "controls": [
    {
      "framework": "NIST-800-53",
      "control_id": "XX-1",
      "control_name": "Control Name",
      "description": "What this maps to"
    }
  ],
  "enabled": true
}
```

### 2. Implement Check Function

In `comp_leo/analyzer/checker.py`:

```python
def _check_my_new_check(self, program: LeoProgram, rule: Dict) -> List[Violation]:
    """
    Check for [specific issue].
    
    This rule detects [what it detects] and is important because [why].
    
    Example violation:
        transition unsafe() {
            // Bad pattern here
        }
    
    Args:
        program: Parsed Leo program
        rule: Rule definition from policy pack
    
    Returns:
        List of violations found
    """
    violations = []
    
    for transition in program.transitions:
        # Your detection logic here
        if self._detects_issue(transition):
            violations.append(self._create_violation(
                rule=rule,
                program=program,
                line=transition.line_start,
                message=f"Specific issue in '{transition.name}'",
                code_snippet=self._get_code_snippet(program, transition.line_start, 3),
                remediation=Remediation(
                    description="How to fix this",
                    code_example="// Fixed code example",
                    automated=False,  # Set to True if auto-fixable
                    confidence=0.0    # 0.0-1.0 for auto-fix confidence
                )
            ))
    
    return violations
```

### 3. Add Tests

Create test in `tests/test_checker.py`:

```python
def test_my_new_check():
    """Test detection of my new security issue."""
    source = """
program test.aleo {
    transition vulnerable() {
        // Code that should trigger the check
    }
}
"""
    
    checker = ComplianceChecker(policy_pack="aleo-baseline")
    program = checker.parser.parse_source(source, "test.leo")
    violations = checker._run_checks(program)
    
    # Should find the violation
    my_violations = [v for v in violations if v.rule_id == "my-new-check"]
    assert len(my_violations) > 0
    assert my_violations[0].severity == Severity.HIGH
```

### 4. Document

Add to `QUICKSTART.md` or `README.md` if it's a significant check.

## Adding New Policy Packs

### 1. Create Policy Pack JSON

Create `comp_leo/policies/your-framework.json`:

```json
{
  "id": "your-framework",
  "name": "Your Framework Name",
  "description": "What this framework covers",
  "version": "1.0.0",
  "frameworks": ["Framework1", "Framework2"],
  "enabled_rules": [
    "rule-1",
    "rule-2"
  ],
  "fail_threshold": 80,
  "rules": [...]
}
```

### 2. Map Controls

For each rule, map to framework controls:

```json
{
  "id": "example-rule",
  "controls": [
    {
      "framework": "Your-Framework",
      "control_id": "YF-1.2.3",
      "control_name": "Control Name",
      "description": "What this control requires"
    }
  ]
}
```

### 3. Test

```python
def test_your_framework_policy():
    checker = ComplianceChecker(policy_pack="your-framework")
    result = checker.check_file("test.leo")
    assert result.policy_pack == "your-framework"
```

## Parser Improvements

To support new Leo syntax:

### 1. Add Pattern

In `comp_leo/analyzer/parser.py`:

```python
def _extract_new_construct(self, source: str, lines: List[str]) -> List[NewConstruct]:
    """Extract new Leo construct."""
    constructs = []
    pattern = r'new_keyword\s+(\w+)\s*\{'
    
    for match in re.finditer(pattern, source):
        # Extract details
        constructs.append(NewConstruct(...))
    
    return constructs
```

### 2. Update LeoProgram

```python
@dataclass
class LeoProgram:
    # ... existing fields ...
    new_constructs: List[NewConstruct] = field(default_factory=list)
```

### 3. Test

```python
def test_parse_new_construct():
    source = """
program test.aleo {
    new_keyword example {
        // ...
    }
}
"""
    
    parser = LeoParser()
    program = parser.parse_source(source)
    
    assert len(program.new_constructs) == 1
    assert program.new_constructs[0].name == "example"
```

## Pull Request Process

### 1. Create Branch

```bash
git checkout -b feature/my-new-feature
# or
git checkout -b fix/issue-123
```

### 2. Make Changes

- Write code
- Add tests (maintain >80% coverage)
- Update documentation
- Run `make pre-commit`

### 3. Commit

```bash
git add .
git commit -m "feat: add new security check for X"

# Follow conventional commits:
# feat: new feature
# fix: bug fix
# docs: documentation
# test: tests
# refactor: code refactoring
# chore: maintenance
```

### 4. Push & Create PR

```bash
git push origin feature/my-new-feature
```

Then create PR on GitHub with:
- Clear description of changes
- Link to related issues
- Test results
- Screenshots (if UI changes)

### 5. Code Review

- Address reviewer feedback
- Keep commits clean and atomic
- Ensure CI passes

## Release Process

(For maintainers)

### 1. Update Version

In `comp_leo/__init__.py`:
```python
__version__ = "0.2.0"
```

In `pyproject.toml`:
```toml
version = "0.2.0"
```

### 2. Update Changelog

Add to `CHANGELOG.md`:
```markdown
## [0.2.0] - 2025-XX-XX

### Added
- New feature X
- New policy pack Y

### Fixed
- Bug Z

### Changed
- Improved performance of parser
```

### 3. Build & Test

```bash
make clean
make build
make test
```

### 4. Publish

```bash
make publish
```

### 5. Tag Release

```bash
git tag v0.2.0
git push origin v0.2.0
```

## Documentation

### Code Documentation

- Add docstrings to all public functions/classes
- Include examples in docstrings
- Document parameters and return values

```python
def check_file(self, file_path: str, threshold: int = 75) -> CheckResult:
    """
    Run compliance checks on a Leo file.
    
    Args:
        file_path: Path to .leo file
        threshold: Minimum passing score (0-100)
    
    Returns:
        CheckResult with violations and score
    
    Example:
        >>> checker = ComplianceChecker()
        >>> result = checker.check_file("main.leo")
        >>> print(result.score)
        85
    """
```

### User Documentation

- Update README.md for major features
- Add examples to `examples/`
- Update QUICKSTART.md for common use cases

## Community

- **Discord:** https://discord.gg/compiledger
- **Issues:** https://github.com/compiledger/comp-leo-sdk/issues
- **Discussions:** https://github.com/compiledger/comp-leo-sdk/discussions

## Code of Conduct

- Be respectful and inclusive
- Provide constructive feedback
- Help newcomers
- Focus on what's best for the community

## License

By contributing, you agree that your contributions will be licensed under the Apache 2.0 License.

---

**Questions?** Open a discussion or join our Discord!
