# Comp-LEO SDK Quick Start Guide

Get up and running with compliance checking for Leo smart contracts in 5 minutes.

## Installation

```bash
# Install via pip
pip install comp-leo

# Or install from source
git clone https://github.com/compiledger/comp-leo-sdk
cd comp-leo-sdk
pip install -e .
```

## Basic Usage

### 1. Check a Single Leo File

```bash
comp-leo check programs/my_contract/src/main.leo
```

**Output:**
```
Comp-LEO SDK v0.1.0
Policy: aleo-baseline | Threshold: 75

⚠️  Compliance Check: 2 issues found

HIGH: Missing Input Validation [SI-10, NIST 800-53]
  → programs/my_contract/src/main.leo:15
  💡 Add assertion: assert(amount > 0u64);

MEDIUM: Insufficient logging [AU-2, NIST 800-53]
  → programs/my_contract/src/main.leo:28
  💡 Log transaction hash before state mutation

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
✅ 8 checks passed  ⚠️ 2 warnings  ❌ 0 critical
Score: 82/100 (Threshold: 75)
```

### 2. Check Entire Directory

```bash
comp-leo check programs/ --policy nist-800-53 --threshold 80
```

### 3. Generate Audit Report

```bash
# JSON format
comp-leo report programs/ -o audit-report.json

# HTML format for stakeholders
comp-leo report programs/ --format html -o report.html

# Markdown for documentation
comp-leo report programs/ --format markdown -o COMPLIANCE.md
```

## Python API

### Basic Check

```python
from comp_leo import ComplianceChecker

# Initialize checker
checker = ComplianceChecker(policy_pack="aleo-baseline")

# Run check
result = checker.check_file("programs/my_contract/src/main.leo")

# Access results
print(f"Score: {result.score}/100")
print(f"Passed: {result.passed}")
print(f"Violations: {len(result.violations)}")

for violation in result.violations:
    print(f"  {violation.severity}: {violation.message}")
    print(f"  Line {violation.line_number}: {violation.file_path}")
```

### Check Directory

```python
from comp_leo import ComplianceChecker

checker = ComplianceChecker(policy_pack="pci-dss")
result = checker.check_directory("programs/", threshold=85)

# Export to JSON
import json
with open("report.json", "w") as f:
    json.dump(result.model_dump(mode='json'), f, indent=2, default=str)
```

### Parse Leo Code

```python
from comp_leo import LeoParser

parser = LeoParser()
program = parser.parse_file("main.leo")

print(f"Program: {program.name}")
print(f"Transitions: {[t.name for t in program.transitions]}")
print(f"Mappings: {[m.name for m in program.mappings]}")

# Analyze specific function
for transition in program.transitions:
    if transition.modifies_state:
        print(f"⚠️  {transition.name} modifies state")
```

## CI/CD Integration

### GitHub Actions

Create `.github/workflows/compliance.yml`:

```yaml
name: Compliance Check

on: [pull_request, push]

jobs:
  compliance:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4
      
      - name: Set up Python
        uses: actions/setup-python@v4
        with:
          python-version: '3.10'
      
      - name: Install Comp-LEO
        run: pip install comp-leo
      
      - name: Run compliance check
        run: |
          comp-leo check programs/ \
            --policy nist-800-53 \
            --threshold 75 \
            --fail-on-critical \
            --output report.json
      
      - name: Upload report
        uses: actions/upload-artifact@v3
        with:
          name: compliance-report
          path: report.json
```

### Pre-Commit Hook

```bash
# Copy pre-commit hook
cp examples/pre-commit-hook.sh .git/hooks/pre-commit
chmod +x .git/hooks/pre-commit

# Now every commit will run compliance checks
git commit -m "feat: add transfer function"
```

## API Service

### Start Local API

```bash
# Install with API dependencies
pip install comp-leo[api]

# Start server
python -m comp_leo.api.main

# Access docs at http://localhost:8000/docs
```

### Make API Request

```python
import requests

response = requests.post(
    "http://localhost:8000/v1/check",
    headers={"X-API-Key": "your_key"},
    json={
        "source": """
            program example.aleo {
                transition hello() {
                    return;
                }
            }
        """,
        "policy_pack": "aleo-baseline",
        "threshold": 75
    }
)

result = response.json()
print(f"Score: {result['result']['score']}")
```

## Policy Packs

List available policies:

```bash
comp-leo list-policies
```

**Available Packs:**

| Pack | Description | Use Case |
|------|-------------|----------|
| `aleo-baseline` | Core Leo security best practices | Default for all projects |
| `nist-800-53` | Federal security controls | Government/enterprise |
| `iso-27001` | Information security management | International compliance |
| `pci-dss` | Payment card security | Financial applications |
| `gdpr` | Data protection & privacy | EU data handling |

## Common Patterns

### Fix Critical Issues First

```bash
comp-leo check programs/ --fail-on-critical
```

### Gradual Adoption

```bash
# Start with low threshold
comp-leo check programs/ --threshold 50

# Gradually increase as you fix issues
comp-leo check programs/ --threshold 65
comp-leo check programs/ --threshold 75
```

### Focus on Specific Rules

Create custom policy pack:

```json
{
  "id": "payment-security",
  "name": "Payment Security",
  "enabled_rules": [
    "input-validation-missing",
    "state-mutation-unprotected",
    "integer-overflow-risk"
  ],
  "fail_threshold": 90
}
```

## Troubleshooting

### "No .leo files found"

Make sure you're pointing to the correct directory:
```bash
comp-leo check programs/my_contract/src/
```

### "Policy pack not found"

List available policies:
```bash
comp-leo list-policies
```

### Parse Errors

The SDK supports Leo 1.x syntax. If you encounter parse errors with newer Leo versions, please open an issue.

## Next Steps

- Read the [full documentation](https://docs.compiledger.com)
- Explore [example projects](./examples/)
- Join our [Discord community](https://discord.gg/compiledger)
- Get an [API key](https://compiledger.com/api-keys) for production use

## Need Help?

- 📧 Email: support@compiledger.com
- 💬 Discord: https://discord.gg/compiledger
- 🐛 Issues: https://github.com/compiledger/comp-leo-sdk/issues
