# 🔒 Comp-LEO SDK

[![PyPI version](https://badge.fury.io/py/comp-leo.svg)](https://pypi.org/project/comp-leo/)
[![License](https://img.shields.io/badge/License-Apache_2.0-blue.svg)](https://opensource.org/licenses/Apache-2.0)
[![Python 3.10+](https://img.shields.io/badge/python-3.10+-blue.svg)](https://www.python.org/downloads/)

**Compliance & Security for Leo Smart Contracts | 100% Local | Zero Network Calls**

Comp-LEO brings shift-left compliance to the Aleo ecosystem. Check your Leo smart contracts for security vulnerabilities and compliance issues in seconds, not months. Find issues during development, not after deployment.

---

## ✨ Features

🔍 **Static Analysis** - Parse Leo code with regex-based AST extraction  
🛡️ **10+ Security Rules** - Access control, input validation, overflow risks  
📊 **Smart Scoring** - Severity-weighted compliance scores (0-100)  
🎨 **Beautiful CLI** - Interactive menu with auto-scan and selection  
📈 **Multiple Formats** - Export reports as JSON, HTML, or Markdown  
🤖 **CI/CD Ready** - GitHub Actions, GitLab CI, pre-commit hooks  
🔒 **100% Private** - Code never leaves your machine  
⚡ **Blazing Fast** - <100ms per file, 25x faster than AI tools  
🆓 **Free & Open Source** - Apache 2.0 license

---

## 🚀 Quick Start

### Installation

```bash
pip install comp-leo

# With interactive menu mode
pip install comp-leo[interactive]

# With file watching
pip install comp-leo[watch]

# Full install
pip install comp-leo[all]
```

### Usage

#### 🎨 Interactive Menu Mode (Recommended)

The interactive menu mode provides a beautiful, user-friendly interface with auto-scanning:

```bash
comp-leo --interactive
```

**Features:**
- 🔍 Auto-scans for `.leo` files in current directory, parent, and `programs/` folders
- 📋 Shows up to 5 files directly in the main menu for quick access
- ⌨️ Navigate with arrow keys, select with Enter
- 🔄 Rescan on demand to find new files
- 📊 View detailed results and statistics
- 💾 Export reports in multiple formats

**Example Session:**

```
  ██████╗ ██████╗ ███╗   ███╗██████╗       ██╗     ███████╗ ██████╗ 
 ██╔════╝██╔═══██╗████╗ ████║██╔══██╗      ██║     ██╔════╝██╔═══██╗
 ██║     ██║   ██║██╔████╔██║██████╔╝█████╗██║     █████╗  ██║   ██║
 ██║     ██║   ██║██║╚██╔╝██║██╔═══╝ ╚════╝██║     ██╔══╝  ██║   ██║
 ╚██████╗╚██████╔╝██║ ╚═╝ ██║██║           ███████╗███████╗╚██████╔╝
  ╚═════╝ ╚═════╝ ╚═╝     ╚═╝╚═╝           ╚══════╝╚══════╝ ╚═════╝ 

Compliance & Security for Leo Smart Contracts
v0.1.1 | Zero-Knowledge Compliance | 100% Local

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
📋 Interactive Menu Mode
Use arrow keys to navigate, Enter to select

🔍 Scanning for Leo files...
✓ Found 8 Leo file(s)

› What would you like to do?
  ─── Quick Check ───
▸ ✓ programs/sbom_registry/src/main.leo
  ✓ programs/compliance_oracle/src/main.leo
  ✓ programs/token/src/main.leo
  ... and 5 more files
  ─── More Options ───
  🔍 Browse & Check File
  📁 Check Directory
  🔄 Rescan for Leo Files
  📋 List Available Policies
  🔧 Change Policy Pack
  ❓ Help
  ❌ Exit
```

**Menu Navigation:**
- Use **↑/↓** arrow keys to move
- Press **Enter** to select
- Press **Ctrl+C** to cancel (returns to menu)
- Select "❌ Exit" to quit

**Quick Check:**
Select any file from the "Quick Check" section to instantly analyze it. Results show violations, severity, and compliance score.

**After Running a Check:**
The menu dynamically updates to show additional options:
```
  📊 View Last Results      # See all violations with details
  📈 Show Statistics        # View detailed metrics
  💾 Export Report          # Save as JSON/HTML/Markdown
```

**Browse Mode:**
When you select "🔍 Browse & Check File":
1. Enter a path (defaults to current directory)
2. If it's a directory, see all `.leo` files
3. Select a file to check
4. See up to 30 files, with option to show all

#### 💻 Command Line Mode

For scripting and CI/CD, use direct commands:

```bash
# Check a single file
comp-leo check programs/my_contract/src/main.leo

# Check entire directory
comp-leo check programs/

# Check with custom threshold
comp-leo check programs/ --threshold 90

# Fail on any high severity issues
comp-leo check programs/ --fail-on-high

# Generate HTML report
comp-leo report programs/ --format html -o report.html

# Generate Markdown report
comp-leo report programs/ --format markdown -o COMPLIANCE.md

# List available policies
comp-leo list-policies

# Generate CI/CD configs
comp-leo init-ci

# Watch mode (auto-check on file changes)
comp-leo watch programs/
```

## 📋 Commands

| Command | Description |
|---------|-------------|
| `comp-leo` | Show banner and help |
| `comp-leo --interactive` | Launch interactive menu mode |
| `comp-leo check <path>` | Check Leo file or directory |
| `comp-leo report <path>` | Generate compliance report |
| `comp-leo list-policies` | List available policy packs |
| `comp-leo init-ci` | Generate CI/CD configurations |
| `comp-leo watch <path>` | Watch files for changes (requires `[watch]`) |
| `comp-leo --version` | Show version |
| `comp-leo --help` | Show full help |

## 🎯 Use Cases

### Pre-Commit Hook
```bash
# .git/hooks/pre-commit
#!/bin/bash
comp-leo check programs/ --threshold 75 --fail-on-critical || exit 1
```

### GitHub Actions
```yaml
# .github/workflows/compliance.yml
name: Compliance Check
on: [pull_request]

jobs:
  compliance:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4
      - uses: actions/setup-python@v4
        with:
          python-version: '3.10'
      - run: pip install comp-leo
      - run: comp-leo check programs/ --fail-on-critical
```

### Python API
```python
from comp_leo.analyzer.checker import ComplianceChecker

checker = ComplianceChecker(policy_pack="aleo-baseline")
result = checker.check_file("programs/my_contract/src/main.leo")

print(f"Score: {result.score}/100")
print(f"Violations: {len(result.violations)}")

for v in result.violations:
    print(f"{v.severity}: {v.message} at line {v.line_number}")
```

## 📚 Policy Packs

| Pack | Status | Controls | Focus Area |
|------|--------|----------|-----------|
| **aleo-baseline** | ✅ Available | 10+ | Leo security best practices |
| **nist-800-53** | 🚧 v0.2.0 | 1,200+ | Federal security baseline |
| **iso-27001** | 🚧 v0.2.0 | 114 | Information security |
| **pci-dss** | 🚧 v0.3.0 | 300+ | Payment card security |
| **gdpr** | 🚧 v0.3.0 | 99 | Data protection & privacy |

### Current Rules (aleo-baseline)

- ✅ Missing access control checks
- ✅ Unvalidated inputs in transitions
- ✅ Unprotected state mutations
- ✅ Integer overflow risks
- ✅ Missing event logging
- ✅ Hardcoded credentials
- ✅ Weak randomness
- ✅ Reentrancy patterns
- ✅ Gas optimization issues
- ✅ Documentation gaps

## Architecture

```
┌─────────────────────────────────────────────────────────────┐
│                      Comp-LEO SDK                           │
├─────────────────────────────────────────────────────────────┤
│  CLI Tool              API Service           CI Integration  │
│  comp-leo check        /v1/check             GitHub Actions  │
│  comp-leo fix          /v1/report            GitLab CI       │
│  comp-leo report       Authentication         PR Comments    │
├─────────────────────────────────────────────────────────────┤
│               Static Analysis Engine                         │
│  Leo Parser → AST → Pattern Matcher → Scorer                │
├─────────────────────────────────────────────────────────────┤
│               Policy Engine                                  │
│  Rules | Severity | Evidence | Control Mapping              │
├─────────────────────────────────────────────────────────────┤
│               Remediation Engine (Future)                    │
│  Fix Generator → AI Agent → PR Creator                       │
└─────────────────────────────────────────────────────────────┘
```

## Example Output

```
⚠️  Compliance Check: 3 issues found

HIGH: Missing input validation [AC-3.1, NIST 800-53]
  → programs/payment/src/main.leo:45
  💡 Add assertion: assert(amount > 0u64);

MEDIUM: Insufficient logging [AU-2, NIST 800-53]
  → programs/payment/src/main.leo:78
  💡 Log transaction hash before state mutation

LOW: Public field exposure [privacy-001]
  → programs/payment/src/main.leo:12
  💡 Consider using private modifier for sensitive data

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
✅ 47 checks passed  ⚠️ 3 warnings  ❌ 0 critical
Score: 85/100 (Threshold: 75)
```

## 🆕 What's New in v0.1.1

✨ **Interactive Menu Mode** - Beautiful TUI with auto-scanning  
📁 **Smart File Discovery** - Auto-finds Leo files in current/parent directories  
🎯 **Quick Check** - One-click checking from scanned files  
🔄 **Rescan on Demand** - Refresh file list without restarting  
📊 **Dynamic Menus** - Context-aware options based on state  
🎨 **Enhanced CLI** - Improved error messages and help  
📦 **Optional Dependencies** - Install only what you need (`[interactive]`, `[watch]`, `[all]`)  

## 🔒 Why 100% Local?

Your code never leaves your machine. No AI APIs. No network calls. True privacy for ZK blockchain development.

- **No Data Leakage** - Code stays on your machine
- **Works Offline** - Zero network dependency
- **Deterministic** - Same code = same results always
- **Fast** - <100ms vs 2-5s with cloud AI
- **Free Forever** - No per-check costs
- **Auditable** - Open source, verify everything

See [WHY_LOCAL.md](WHY_LOCAL.md) for full philosophy.

## Pricing

| Tier | Checks/Month | Price | Features |
|------|-------------|-------|----------|
| **Freemium** | 100 | Free | Core policies, CLI access |
| **Pro** | 1,000 | $99/mo | All policies, API access, CI integration |
| **Enterprise** | Unlimited | $999/mo | Custom rules, SLA, white-label |

## Project Structure

```
comp-leo-sdk/
├─ cli/                    # Command-line tool
├─ api/                    # FastAPI service
├─ analyzer/              # Static analysis engine
│  ├─ parser.py           # Leo AST parser
│  ├─ checker.py          # Pattern matcher
│  └─ scorer.py           # Severity & scoring
├─ policies/              # Compliance rule definitions
│  ├─ nist_800_53.json
│  ├─ iso_27001.json
│  ├─ pci_dss.json
│  └─ aleo_baseline.json
├─ integrations/          # CI/CD plugins
│  ├─ github/
│  └─ gitlab/
└─ tests/                 # Test suite
```

## Development Roadmap

### Phase 1: Foundation (Weeks 1-4)
- [x] Leo parser & AST builder
- [x] Core static analysis patterns
- [x] NIST 800-53 baseline (80% of ISO overlap)
- [x] CLI tool with local checks
- [ ] Unit test suite (>80% coverage)

### Phase 2: API & Monetization (Weeks 5-8)
- [ ] FastAPI service with authentication
- [ ] Rate limiting & usage tracking
- [ ] API key management portal
- [ ] Stripe integration for paid tiers

### Phase 3: CI/CD & Ecosystem (Weeks 9-12)
- [ ] GitHub Actions integration
- [ ] PR comment bot
- [ ] Policy pack expansion (PCI, GDPR)
- [ ] VS Code extension

### Phase 4: AI Auto-Fix (Weeks 13-16)
- [ ] Fix suggestion engine
- [ ] LLM integration (GPT-4/Claude)
- [ ] Automated PR generation
- [ ] Confidence scoring for fixes

## Contributing

See [CONTRIBUTING.md](CONTRIBUTING.md) for development setup and guidelines.

## License

Apache 2.0 for core SDK (open-source)
Proprietary for API service & enterprise features

---

**Built for the Aleo ecosystem** | [Website](https://compiledger.com) |
