# %%
import os
import tempfile
import pytest
from pandas.testing import assert_frame_equal

import brodata


def test_get_bronhouders():
    brodata.bro.get_bronhouders()


def test_get_brondocumenten_per_bronhouder():
    brodata.bro.get_brondocumenten_per_bronhouder()


def test_get_gmw_of_bronhouder():
    brodata.gmw.get_bro_ids_of_bronhouder(30277172)


def test_get_gmw_characteristics():
    extent = [117700, 118700, 439400, 440400]
    brodata.gmw.get_characteristics(extent=extent)


def test_gmw_get_gld_data_in_extent():
    extent = [118200, 118400, 439700, 440000]
    tempdir = tempfile.gettempdir()
    fname_zip = os.path.join(tempdir, "test_gmw_get_gld_data_in_extent.zip")
    gdf1 = brodata.gmw.get_data_in_extent(
        extent=extent,
        combine=True,
        as_csv=False,
        to_zip=fname_zip,
        redownload=True,
    )
    gdf2 = brodata.gmw.get_data_in_extent(
        extent=extent,
        combine=True,
        as_csv=False,
        to_zip=fname_zip,
        redownload=False,
    )
    gdf3 = brodata.gmw.get_data_in_extent(fname_zip, combine=True)

    assert_frame_equal(gdf1, gdf2)
    assert_frame_equal(gdf1, gdf3)


def test_gmw_get_gld_data_in_extent_as_csv():
    extent = [118200, 118400, 439700, 440000]
    brodata.gmw.get_data_in_extent(extent=extent, combine=True, as_csv=True)


def test_gmw_get_gar_data_in_extent():
    extent = [115000, 120000, 438000, 441000]
    brodata.gmw.get_data_in_extent(extent=extent, kind="gar", combine=True)


# def test_gmw_get_frd_data_in_extent():
#    extent = [115000, 120000, 438000, 441000]
#    gdf, frd = brodata.gmw.get_data_in_extent(extent=extent, kind="frd")


def tets_get_well_code():
    brodata.gmw.get_well_code("GMW000000049567")


def test_get_gmw():
    brodata.gmw.GroundwaterMonitoringWell.from_bro_id("GMW000000049567")


def test_groundwater_monitoring_well():
    fname = os.path.join("tests", "data", "GMW000000036287.xml")
    brodata.gmw.GroundwaterMonitoringWell(fname)


def test_groundwater_level_dossier():
    fname = os.path.join("tests", "data", "GLD000000012893.xml")
    brodata.gld.GroundwaterLevelDossier(fname)


def test_gld_observations_summary():
    brodata.gld.get_observations_summary("GLD000000012893")


def test_gld_get_objects_as_csv():
    brodata.gld.get_objects_as_csv("GLD000000012893")


def test_gld_get_series_as_csv():
    brodata.gld.get_series_as_csv("GLD000000012893")


def test_geotechnical_borehole_research():
    fname = os.path.join("tests", "data", "BHR000000353924.xml")
    bhrgt = brodata.bhr.GeotechnicalBoreholeResearch(fname)
    # also test the plot
    brodata.plot.bro_lithology(bhrgt.descriptiveBoreholeLog[0]["layer"])
    # and test the plot generated by BRO
    brodata.bhr.bhrgt_graph(fname)


def test_pedological_borehole_research():
    fname = os.path.join("tests", "data", "BHR000000175723.xml")
    brodata.bhr.PedologicalBoreholeResearch(fname)


def test_gar_report_from_xml_file_multiple_analysis_processes():
    fname = os.path.join("tests", "data", "GAR000000019636.xml")
    brodata.gar.GroundwaterAnalysisReport(fname)


def test_gar_report_from_xml_file_single_analysis_process():
    fname = os.path.join("tests", "data", "GAR000000042563.xml")
    brodata.gar.GroundwaterAnalysisReport(fname)


def test_gar_report_from_csv_file():
    fname = os.path.join("tests", "data", "GAR000000042563.csv")
    brodata.gar.GroundwaterAnalysisReport(fname)


def test_gar_report():
    brodata.gar.GroundwaterAnalysisReport.from_bro_id("GAR000000019636")


def test_gar_get_parameter_list():
    brodata.gar.get_parameter_list()


def test_soil_face_research():
    fname = os.path.join("tests", "data", "SFR000000000243.xml")
    brodata.sfr.SoilFaceResearch(fname)


def test_groundwater_monitoring_network():
    fname = os.path.join("tests", "data", "GMN000000000163.xml")
    brodata.gmn.GroundwaterMonitoringNetwork(fname)


def test_get_cpt_characteristics():
    extent = [117700, 118700, 439400, 440400]
    brodata.cpt.get_characteristics(extent=extent)


def test_get_cpt_graph_types():
    brodata.cpt.get_graph_types()


def test_get_cpt():
    fname = os.path.join("tests", "data", "CPT000000005925.xml")
    cpt = brodata.cpt.ConePenetrationTest(fname)
    # also test the plot
    brodata.plot.cone_penetration_test(cpt)
    # and test the plot generated by BRO
    brodata.cpt.graph(fname)


def test_get_cpt_test_with_dissipation_test():
    fname = os.path.join("tests", "data", "CPT000000115243.xml")
    brodata.cpt.ConePenetrationTest(fname)


def test_groundwater_utilisation_facility():
    fname = os.path.join("tests", "data", "GUF000000016723.xml")
    brodata.guf.GroundwaterUtilisationFacility(fname)


def test_gm_gar():
    extent = [112_000, 119_000, 430_000, 445_000]
    tempdir = tempfile.gettempdir()
    fname_zip = os.path.join(tempdir, "test_gm_gar.zip")
    gdf1 = brodata.gm.get_data_in_extent(
        extent, kind="gar", tmin="2000", tmax="2010", to_zip=fname_zip
    )
    gdf2 = brodata.gm.get_data_in_extent(
        extent, kind="gar", tmin="2008", tmax="2010", to_zip=fname_zip
    )
    assert len(gdf2) < len(gdf1)


def test_gm_gar2():
    extent = [112_000, 119_000, 430_000, 445_000]
    tubes = brodata.gm.gmw_monitoringtube_items(extent)
    gars_gdf = brodata.gm.gar_items(extent)
    # add information from tubes to gar-data
    gars_gdf = gars_gdf.join(
        tubes.set_index("gm_gmw_monitoringtube_pk").drop(columns="geometry"),
        on="gm_gmw_monitoringtube_fk",
    )
    assert "screen_top_position" in gars_gdf.columns


def test_gm_gld():
    extent = [112_000, 119_000, 430_000, 445_000]
    tempdir = tempfile.gettempdir()
    fname_zip = os.path.join(tempdir, "test_gm_gld.zip")
    brodata.gm.get_data_in_extent(
        extent, kind="gld", tmin="2000", tmax="2010", to_zip=fname_zip
    )


def test_get_guf_data_in_extent():
    extent = [117700, 118700, 439400, 440400]
    brodata.guf.get_characteristics(extent=extent)

    tempdir = tempfile.gettempdir()
    fname_zip = os.path.join(tempdir, "test_get_guf_data_in_extent.zip")
    gdf1 = brodata.guf.get_data_in_extent(extent=extent, to_zip=fname_zip)

    extent = [118300, 118700, 439400, 440400]
    gdf2 = brodata.guf.get_data_in_extent(extent=extent, to_zip=fname_zip)
    assert len(gdf2) < len(gdf1)


def test_get_bhr_in_extent():
    extent = [119000, 120000, 440500, 441000]
    gdf = brodata.bhr.get_data_in_extent(extent=extent)
    line = [(extent[0], extent[2]), (extent[1], extent[3])]
    colors = brodata.plot.lithology_colors.copy()
    colors["kleiigeHumus"] = colors["klei"]
    brodata.plot.lithology_along_line(gdf, line, "bro", colors=colors)


def test_get_kvk_df():
    brodata.bro.get_kvk_df()


def test_unknwon_gmw_raises_value_error():
    # make sure a not existant bro-id of a gmw returns a ValueError
    with pytest.raises(ValueError):
        brodata.gmw.GroundwaterMonitoringWell.from_bro_id("GMW000000000000")
