import ast
import os
import re
from collections import OrderedDict
from typing import Any, Dict, List, Literal, Mapping, Optional, Set, Tuple, Union

import msgspec
from msgspec import Meta, Struct, defstruct
from typing_extensions import Annotated

from msgflux.dotdict import dotdict
from msgflux.logger import logger
from msgflux.utils.common import type_mapping
from msgflux.utils.convert import convert_none_to_string, convert_string_to_none


class StructFactory:
    """Rebuild msgspec.Struct from a JSON-schema or str signature."""

    def __init__(self):
        self.reconstructed_classes = OrderedDict()

    @classmethod
    def from_schema(cls, schema: Dict[str, Any]) -> Struct:
        """Rebuild all Struct classes defined in the schema.

        Args:
            schema: The JSON schema generated by msgspec

        Returns:
            Dict mapping class names to the reconstructed classes
        """
        self = cls()  # Temp instance
        if "$defs" not in schema:
            raise ValueError("Schema must contain definitions in `$defs`")

        definitions = schema["$defs"]

        # First, we identify the dependency order
        dependency_order = self._get_dependency_order(definitions)

        # We rebuild the classes in the correct order
        for class_name in dependency_order:
            class_def = definitions[class_name]
            self._reconstruct_class(class_name, class_def, definitions)

        main_struct = self.reconstructed_classes.popitem(last=True)[1]
        return main_struct

    def _get_dependency_order(self, definitions: Dict[str, Any]) -> List[str]:
        """Determines the construction order of classes based on dependencies."""
        dependencies = {}

        # Map dependencies for each class
        for class_name, class_def in definitions.items():
            deps = set()
            self._find_dependencies(class_def, deps, definitions.keys())
            dependencies[class_name] = deps

        # Sort topologically
        ordered = []
        remaining = set(definitions.keys())

        while remaining:
            # Find classes without unresolved dependencies
            ready = [
                name for name in remaining if dependencies[name].issubset(set(ordered))
            ]

            if not ready:  # Circular dependency - uses alphabetical order as fallback
                ready = [min(remaining)]

            for name in ready:
                ordered.append(name)
                remaining.remove(name)

        return ordered

    def _find_dependencies(
        self, definition: Dict[str, Any], deps: Set[str], available_classes: Set[str]
    ):
        """Recursively finds all dependencies of a definition."""
        if isinstance(definition, dict):
            # Check direct deps
            if "$ref" in definition:
                ref_name = definition["$ref"].split("/")[-1]
                if ref_name in available_classes:
                    deps.add(ref_name)

            # Check in properties
            if "properties" in definition:
                for prop_def in definition["properties"].values():
                    self._find_dependencies(prop_def, deps, available_classes)

            # Checks on anyOf (for optional fields)
            if "anyOf" in definition:
                for item in definition["anyOf"]:
                    self._find_dependencies(item, deps, available_classes)

            # Checks on items (for arrays)
            if "items" in definition:
                self._find_dependencies(definition["items"], deps, available_classes)

    def _reconstruct_class(
        self,
        class_name: str,
        class_def: Dict[str, Any],
        all_definitions: Dict[str, Any],
    ):
        """Rebuild single class."""
        if class_name in self.reconstructed_classes:
            return self.reconstructed_classes[class_name]

        properties = class_def.get("properties", {})
        required_fields = set(class_def.get("required", []))

        fields = []

        for field_name, field_def in properties.items():
            field_type = self._resolve_field_type(field_def, all_definitions)

            if field_name in required_fields:  # Checks if the field is required
                fields.append(
                    (field_name, field_type)
                )  # Required field - type can be annotated
            else:
                fields.append(
                    (field_name, field_type, None)
                )  # Optional field - can be annotated

        reconstructed_class = defstruct(
            class_name,
            fields,
            kw_only=True,  # default
            module=__name__,
        )

        self.reconstructed_classes[class_name] = reconstructed_class
        return reconstructed_class

    def _resolve_field_type(
        self, field_def: Dict[str, Any], all_definitions: Dict[str, Any]
    ):
        """Resolves the type of a field based on the schema definition."""
        if "anyOf" in field_def:  # If it has anyOf, treat it as Union (optional field)
            types = []
            description = None

            for option in field_def["anyOf"]:
                if option.get("type") == "null":
                    continue  # Skip null, it will be treated as Optional

                if "description" in option:  # Capture description if exists
                    description = option["description"]

                option_type = self._resolve_single_type(option, all_definitions)
                if option_type and option_type is not type(None):
                    types.append(option_type)

            if len(types) == 1:
                base_type = types[0]
                if description:
                    annotated_type = Annotated[base_type, Meta(description=description)]
                    return Optional[annotated_type]
                else:
                    return Optional[base_type]
            elif len(types) > 1:
                if description:
                    union_type = Union[tuple(types)]
                    annotated_type = Annotated[
                        union_type, Meta(description=description)
                    ]
                    return Optional[annotated_type]
                else:
                    return Optional[Union[tuple(types)]]
            else:
                return Optional[str]  # Fallback

        return self._resolve_single_type(field_def, all_definitions)

    def _resolve_single_type(
        self, type_def: Dict[str, Any], all_definitions: Dict[str, Any]
    ):
        """Resolves a unique type based on the definition."""
        if "$ref" in type_def:  # Ref to other struct
            ref_name = type_def["$ref"].split("/")[-1]
            if ref_name in self.reconstructed_classes:
                return self.reconstructed_classes[ref_name]
            else:
                # Class not yet constructed, returns by name
                # This should be resolved by dependency order
                return ref_name

        if "enum" in type_def:  # Enum/Literal
            enum_values = type_def["enum"]
            description = type_def.get("description")

            # If all values ​​are of the same type, use Literal
            if len(enum_values) > 0:
                if len(enum_values) == 1:
                    literal_type = Literal[enum_values[0]]
                else:
                    literal_type = Literal[tuple(enum_values)]

                if description:  # If has description, use with Annotated
                    return Annotated[literal_type, Meta(description=description)]
                else:
                    return literal_type

        if "type" in type_def:  # Built-in type
            type_name = type_def["type"]

            if type_name == "array":
                if "items" in type_def:  # Array with specific items
                    item_type = self._resolve_single_type(
                        type_def["items"], all_definitions
                    )
                    return List[item_type]
                else:
                    return List[Any]

            return type_mapping.get(type_name, str)

        return str  # Fallback

    @classmethod
    def from_signature(
        cls, signature: str, struct_name: Optional[str] = "DynamicStruct"
    ) -> Struct:
        annotations = cls._parse_annotations(signature)
        struct_fields = []

        for name, type_str in annotations:
            try:
                parsed_type = cls._parse_type_string(type_str)
                struct_fields.append((name, parsed_type))
            except ValueError as e:
                raise ValueError(
                    f"Error parsing field `{name}` (type='{type_str}')"
                ) from e
            except Exception as e:
                raise RuntimeError(f"Unexpected error parsing field `{name}`") from e

        if not struct_fields and signature.strip():
            raise ValueError("No valid fields parsed from the signature.")

        try:
            DynamicStruct = msgspec.defstruct(struct_name, struct_fields) # noqa: N806
        except Exception as e:
            raise RuntimeError(f"Error creating struct `{struct_name}`") from e

        return DynamicStruct

    @classmethod
    def _parse_literal_args(cls, args_str: str) -> Tuple:
        try:
            parsed_node = ast.parse(f"[{args_str}]", mode="eval")
            values = [ast.literal_eval(node) for node in parsed_node.body.elts]
            return tuple(values)
        except (SyntaxError, ValueError, TypeError) as e:
            logger.error(str(e))
            raise ValueError(f"Invalid literal arguments: `{args_str}`") from e

    @classmethod
    def _split_args(cls, args_str: str) -> list[str]:
        args = []
        level = 0
        current_arg_start = 0
        in_quotes = None

        if not args_str.strip():
            return []

        for i, char in enumerate(args_str):
            if char in ("[", "{", "(") and not in_quotes:
                level += 1
            elif char in ("]", "}", ")") and not in_quotes:
                level -= 1
            elif char in ("'", '"'):
                if in_quotes == char:
                    in_quotes = None
                elif in_quotes is None:
                    in_quotes = char
            elif char == "," and level == 0 and not in_quotes:
                args.append(args_str[current_arg_start:i].strip())
                current_arg_start = i + 1

        args.append(args_str[current_arg_start:].strip())
        return [arg for arg in args if arg]

    @classmethod
    def _parse_type_string(cls, type_str: str) -> type: # noqa: C901
        type_str = type_str.strip()
        if not type_str:
            raise ValueError("The type string cannot be empty.")

        type_str_lower = type_str.lower()
        simple_types_lower = {"str", "int", "float", "bool", "any", "none"}
        if type_str_lower in simple_types_lower and type_str_lower in type_mapping:
            return type_mapping[type_str_lower]

        match = re.match(r"^\s*(\w+)\s*\[(.*)\]\s*$", type_str, re.DOTALL)
        if match:
            base_type_name, args_str = match.groups()
            base_type_name_lower = base_type_name.lower()

            if base_type_name_lower not in type_mapping:
                raise ValueError(
                    f"Base type not supported: `{base_type_name}` in `{type_str}`"
                )

            base_type = type_mapping[base_type_name_lower]

            if base_type is Literal:
                parsed_args = cls._parse_literal_args(args_str)
                if not parsed_args:
                    raise ValueError("Literal[...] cannot be empty")
                return Literal[parsed_args]

            elif base_type is Optional:
                arg_strs_list = cls._split_args(args_str)
                if len(arg_strs_list) != 1:
                    raise ValueError("Optional[...] requires exactly 1 argument")
                inner_type = cls._parse_type_string(arg_strs_list[0])
                return Union[inner_type, type(None)]

            elif base_type in (List, Dict, Union, Tuple):
                arg_strs_list = cls._split_args(args_str)

                if base_type is Tuple:
                    if not arg_strs_list:
                        return Tuple[()]
                    if len(arg_strs_list) == 2 and arg_strs_list[1] == "...":
                        item_type = cls._parse_type_string(arg_strs_list[0])
                        return Tuple[item_type, ...]

                parsed_args = tuple(
                    cls._parse_type_string(arg) for arg in arg_strs_list
                )

                if base_type is Dict and len(parsed_args) != 2:
                    raise ValueError("Dict requires exactly 2 arguments")
                if base_type is List and len(parsed_args) != 1:
                    raise ValueError("List requires exactly 1 argument")
                if base_type is Union:
                    if len(parsed_args) == 0:
                        raise ValueError("Union[...] cannot be empty")
                    if len(parsed_args) == 1:
                        return parsed_args[0]

                return base_type[parsed_args]

        if type_str_lower in type_mapping:
            return type_mapping[type_str_lower]

        raise ValueError(f"Unsupported or malformed type string: `{type_str}`")

    @classmethod
    def _parse_annotations(cls, signature: str) -> List[Tuple[str, str]]: # noqa: C901
        fields = []
        current_pos = 0
        level = 0
        in_quotes = None
        current_field_start = 0
        signature = signature.strip()

        if not signature:
            return []

        while current_pos < len(signature):
            char = signature[current_pos]
            if char in ("[", "{", "(") and not in_quotes:
                level += 1
            elif char in ("]", "}", ")") and not in_quotes:
                level -= 1
                if level < 0:
                    raise ValueError(
                        f"Unbalanced nesting near `{signature[current_pos:]}`"
                    )
            elif char in ("'", '"'):
                if in_quotes == char:
                    in_quotes = None
                elif in_quotes is None:
                    in_quotes = char

            if char == "," and level == 0 and not in_quotes:
                field_str = signature[current_field_start:current_pos].strip()
                if field_str:
                    fields.append(field_str)
                current_field_start = current_pos + 1
            current_pos += 1

        if level != 0:
            raise ValueError("Unbalanced brackets/parentheses in signature.")
        if in_quotes:
            raise ValueError("Unclosed quotation marks in signature.")

        last_field_str = signature[current_field_start:].strip()
        if last_field_str:
            fields.append(last_field_str)

        result = []
        for field_str in fields:
            parts = field_str.split(":", 1)
            if len(parts) == 2:
                key = parts[0].strip()
                value_type = parts[1].strip()
                if not key:
                    raise ValueError(f"Field name cannot be empty in `{field_str}`")
                if not value_type:
                    raise ValueError(f"Type cannot be empty after ':' in `{field_str}`")
            else:
                key = field_str.strip()
                value_type = "str"
                if not key:
                    raise ValueError(f"Field name cannot be empty in `{field_str}`")
            result.append((key, value_type))

        return result


def export_to_toml(obj: object, filepath: Union[str, os.PathLike]):
    with open(filepath, "wb") as f:
        safe_obj = convert_none_to_string(obj)
        f.write(msgspec.toml.encode(safe_obj))


def export_to_json(
    obj: object, filepath: Union[str, os.PathLike], indent: Optional[int] = 4
):
    with open(filepath, "wb") as f:
        obj_b = msgspec.json.encode(obj)
        formatted_obj_b = msgspec.json.format(obj_b, indent=indent)
        f.write(formatted_obj_b)


def save(obj: object, f: Union[str, os.PathLike]):
    """Save a Python object to a file in either TOML or JSON format.

    Args:
        data:
            Saved object.
        filepath:
            A string or os.PathLike object containing a file name.

    Raises:
        ValueError:
            If the file format is not "toml" or "json".
        FileNotFoundError:
            If the directory of the provided filepath does not exist.

    !!! example
        ``` python
        data = {"name": "Satoshi", "age": 42}
        save(data, "output.toml")
        save(data, "output.json")
        ```
    """
    directory = os.path.dirname(f)
    if directory and not os.path.exists(directory):
        raise FileNotFoundError(f"The directory `{directory}` does not exist")

    if f.endswith("toml"):
        export_to_toml(obj, f)
    elif f.endswith("json"):
        export_to_json(obj, f)
    else:
        raise ValueError(f"Unsupported format: `{f}`. Use `toml` or `json`.")


def read_json(filepath: Union[str, os.PathLike]) -> Mapping[str, Any]:
    with open(filepath, "rb") as f:
        return msgspec.json.decode(f.read())


def read_toml(filepath: Union[str, os.PathLike]) -> Mapping[str, Any]:
    with open(filepath, "rb") as f:
        obj = msgspec.toml.decode(f.read())
        safe_obj = convert_string_to_none(obj)
        return safe_obj


def load(f: Union[str, os.PathLike]) -> Any:
    """Load data from a file in either JSON or TOML format.

    Args:
        f: A string or os.PathLike object containing a file name

    Returns:
        The Python object loaded from the file

    Raises:
        FileNotFoundError: If the file does not exist
        ValueError: If the file extension is not ".json" or ".toml"

    !!! example
        ``` python
        data = load("data.toml")
        data = load("data.json")
        ```
    """
    if not os.path.exists(f):
        raise FileNotFoundError(f"The file `{f}` does not exist.")

    if f.endswith(".json"):
        return read_json(f)
    elif f.endswith(".toml"):
        return read_toml(f)
    else:
        raise ValueError(f"Unsupported file extension: `{f}`. Use `.json` or `.toml`")


def struct_to_dict(obj: object):
    """Recursively converts a msgspec.Struct object to a pure Python dictionary."""
    if isinstance(obj, msgspec.Struct):
        # Convert the struct to a dictionary and recursively process each value
        return {k: struct_to_dict(v) for k, v in msgspec.structs.asdict(obj).items()}
    elif isinstance(obj, list):
        # Convert each item in the list recursively
        return [struct_to_dict(item) for item in obj]
    elif isinstance(obj, dict):
        # If it is a dictionary, recursively convert its values
        return dotdict({k: struct_to_dict(v) for k, v in obj.items()})
    else:  # Returns the value as is for simple types
        return obj
