# Multi-Cloud Usage Guide

This guide explains how to use Riveter for multi-cloud infrastructure validation across AWS, Azure, and GCP.

## Table of Contents

- [Multi-Cloud Strategy](#multi-cloud-strategy)
- [Rule Pack Selection](#rule-pack-selection)
- [Cloud-Specific Validation](#cloud-specific-validation)
- [Multi-Cloud Security Patterns](#multi-cloud-security-patterns)
- [Kubernetes Across Clouds](#kubernetes-across-clouds)
- [CI/CD Integration](#cicd-integration)
- [Best Practices](#best-practices)

## Multi-Cloud Strategy

### Why Multi-Cloud Validation?

Multi-cloud environments require consistent security and compliance posture across different cloud providers. Riveter supports this through:

1. **Provider-Specific Rule Packs**: Tailored rules for AWS, Azure, and GCP
2. **Multi-Cloud Rule Pack**: Common security patterns across all providers
3. **Kubernetes Security**: Container security across EKS, AKS, and GKE
4. **Compliance Standards**: Framework-agnostic compliance validation

### Architecture Patterns

#### Hybrid Multi-Cloud
```
┌─────────────┐    ┌─────────────┐    ┌─────────────┐
│     AWS     │    │    Azure    │    │     GCP     │
│             │    │             │    │             │
│ ┌─────────┐ │    │ ┌─────────┐ │    │ ┌─────────┐ │
│ │   EKS   │ │    │ │   AKS   │ │    │ │   GKE   │ │
│ └─────────┘ │    │ └─────────┘ │    │ └─────────┘ │
│             │    │             │    │             │
│ ┌─────────┐ │    │ ┌─────────┐ │    │ ┌─────────┐ │
│ │   RDS   │ │    │ │SQL DB   │ │    │ │Cloud SQL│ │
│ └─────────┘ │    │ └─────────┘ │    │ └─────────┘ │
└─────────────┘    └─────────────┘    └─────────────┘
```

#### Multi-Cloud Kubernetes
```
┌─────────────────────────────────────────────────────┐
│                 Kubernetes Layer                    │
│  ┌─────────┐    ┌─────────┐    ┌─────────┐         │
│  │   EKS   │    │   AKS   │    │   GKE   │         │
│  │  (AWS)  │    │ (Azure) │    │  (GCP)  │         │
│  └─────────┘    └─────────┘    └─────────┘         │
└─────────────────────────────────────────────────────┘
```

## Rule Pack Selection

### Single Cloud Validation

#### AWS Only
```bash
# Complete AWS validation
riveter scan -p aws-security -p cis-aws -p aws-well-architected -t aws/

# AWS with compliance
riveter scan -p aws-security -p aws-hipaa -p aws-pci-dss -t aws/

# AWS with SOC 2
riveter scan -p aws-security -p cis-aws -p soc2-security -t aws/
```

#### GCP Only
```bash
# Complete GCP validation
riveter scan -p gcp-security -p cis-gcp -p gcp-well-architected -t gcp/

# GCP with SOC 2
riveter scan -p gcp-security -p cis-gcp -p soc2-security -t gcp/
```

#### Azure Only
```bash
# Complete Azure validation
riveter scan -p azure-security -p cis-azure -p azure-well-architected -t azure/

# Azure with compliance
riveter scan -p azure-security -p azure-hipaa -p soc2-security -t azure/
```

### Multi-Cloud Validation

#### All Clouds with Common Security
```bash
# Multi-cloud security patterns
riveter scan -p multi-cloud-security -t infrastructure/

# Multi-cloud with provider-specific rules
riveter scan -p multi-cloud-security -p aws-security -p gcp-security -p azure-security -t infrastructure/
```

#### Multi-Cloud with Compliance
```bash
# Healthcare compliance across clouds
riveter scan -p aws-hipaa -p azure-hipaa -p soc2-security -t healthcare-infrastructure/

# SOC 2 across all clouds
riveter scan -p soc2-security -p multi-cloud-security -t infrastructure/
```

#### Multi-Cloud with Kubernetes
```bash
# Kubernetes security across all clouds
riveter scan -p kubernetes-security -p multi-cloud-security -t k8s-infrastructure/

# Cloud-specific Kubernetes validation
riveter scan -p kubernetes-security -p aws-security -t eks-infrastructure/
riveter scan -p kubernetes-security -p azure-security -t aks-infrastructure/
riveter scan -p kubernetes-security -p gcp-security -t gke-infrastructure/
```

## Cloud-Specific Validation

### AWS Infrastructure

#### Directory Structure
```
aws/
├── compute/
│   ├── ec2.tf
│   └── autoscaling.tf
├── storage/
│   ├── s3.tf
│   └── ebs.tf
├── database/
│   └── rds.tf
├── networking/
│   ├── vpc.tf
│   └── security-groups.tf
└── iam/
    └── policies.tf
```

#### Validation Commands
```bash
# Security best practices
riveter scan -p aws-security -t aws/ --output-format table

# CIS compliance
riveter scan -p cis-aws -t aws/ --output-format json

# Well-architected validation
riveter scan -p aws-well-architected -t aws/ --output-format sarif

# Combined validation
riveter scan -p aws-security -p cis-aws -p aws-well-architected -t aws/
```

### GCP Infrastructure

#### Directory Structure
```
gcp/
├── compute/
│   ├── instances.tf
│   └── instance-groups.tf
├── storage/
│   ├── cloud-storage.tf
│   └── persistent-disks.tf
├── database/
│   └── cloud-sql.tf
├── networking/
│   ├── vpc.tf
│   └── firewall.tf
└── iam/
    └── service-accounts.tf
```

#### Validation Commands
```bash
# Security best practices
riveter scan -p gcp-security -t gcp/ --output-format table

# CIS compliance
riveter scan -p cis-gcp -t gcp/ --output-format json

# Well-architected validation
riveter scan -p gcp-well-architected -t gcp/ --output-format sarif

# Combined validation
riveter scan -p gcp-security -p cis-gcp -p gcp-well-architected -t gcp/
```

### Azure Infrastructure

#### Directory Structure
```
azure/
├── compute/
│   ├── virtual-machines.tf
│   └── scale-sets.tf
├── storage/
│   └── storage-accounts.tf
├── database/
│   └── sql-database.tf
├── networking/
│   ├── virtual-network.tf
│   └── network-security-groups.tf
└── identity/
    └── managed-identities.tf
```

#### Validation Commands
```bash
# Security best practices
riveter scan -p azure-security -t azure/ --output-format table

# CIS compliance
riveter scan -p cis-azure -t azure/ --output-format json

# Well-architected validation
riveter scan -p azure-well-architected -t azure/ --output-format sarif

# Combined validation
riveter scan -p azure-security -p cis-azure -p azure-well-architected -t azure/
```

## Multi-Cloud Security Patterns

### Common Security Requirements

The `multi-cloud-security` rule pack enforces consistent patterns:

#### Encryption Standards
```bash
# Validate encryption across all clouds
riveter scan -p multi-cloud-security -t infrastructure/ | grep -i encryption
```

**Covered Resources**:
- **AWS**: S3 buckets, RDS instances, EBS volumes
- **Azure**: Storage accounts, SQL databases, VM disks
- **GCP**: Cloud Storage buckets, Cloud SQL instances, persistent disks

#### Network Security
```bash
# Validate network security across clouds
riveter scan -p multi-cloud-security -t infrastructure/ | grep -i network
```

**Covered Patterns**:
- **AWS**: Security groups, NACLs, VPC configuration
- **Azure**: Network security groups, virtual networks
- **GCP**: Firewall rules, VPC networks

#### Identity and Access Management
```bash
# Validate IAM across clouds
riveter scan -p multi-cloud-security -t infrastructure/ | grep -i iam
```

**Covered Areas**:
- **AWS**: IAM policies, roles, MFA
- **Azure**: RBAC, managed identities, conditional access
- **GCP**: IAM policies, service accounts, Workload Identity

### Cross-Cloud Resource Mapping

| Security Pattern | AWS Resource | Azure Resource | GCP Resource |
|------------------|--------------|----------------|--------------|
| **Storage Encryption** | `aws_s3_bucket` | `azurerm_storage_account` | `google_storage_bucket` |
| **Database Encryption** | `aws_db_instance` | `azurerm_sql_database` | `google_sql_database_instance` |
| **Compute Encryption** | `aws_ebs_volume` | `azurerm_managed_disk` | `google_compute_disk` |
| **Network Firewall** | `aws_security_group` | `azurerm_network_security_group` | `google_compute_firewall` |
| **Load Balancer** | `aws_lb` | `azurerm_lb` | `google_compute_backend_service` |
| **Key Management** | `aws_kms_key` | `azurerm_key_vault_key` | `google_kms_crypto_key` |

## Kubernetes Across Clouds

### Kubernetes Security Validation

The `kubernetes-security` rule pack works across EKS, AKS, and GKE:

#### Pod Security
```bash
# Validate pod security across all K8s clusters
riveter scan -p kubernetes-security -t k8s/ | grep -i pod
```

**Covered Areas**:
- No privileged containers
- Read-only root filesystem
- Non-root users
- Security contexts
- Resource limits

#### RBAC Configuration
```bash
# Validate RBAC across clusters
riveter scan -p kubernetes-security -t k8s/ | grep -i rbac
```

**Covered Resources**:
- `kubernetes_role`
- `kubernetes_role_binding`
- `kubernetes_cluster_role`
- `kubernetes_cluster_role_binding`
- `kubernetes_service_account`

#### Network Policies
```bash
# Validate network policies
riveter scan -p kubernetes-security -t k8s/ | grep -i network
```

**Covered Patterns**:
- Default deny policies
- Ingress rules
- Egress rules
- Pod-to-pod communication

### Cloud-Specific Kubernetes Features

#### EKS (AWS)
```bash
# EKS with AWS security
riveter scan -p kubernetes-security -p aws-security -t eks/

# EKS with AWS Well-Architected
riveter scan -p kubernetes-security -p aws-well-architected -t eks/
```

#### AKS (Azure)
```bash
# AKS with Azure security
riveter scan -p kubernetes-security -p azure-security -t aks/

# AKS with Azure Well-Architected
riveter scan -p kubernetes-security -p azure-well-architected -t aks/
```

#### GKE (GCP)
```bash
# GKE with GCP security
riveter scan -p kubernetes-security -p gcp-security -t gke/

# GKE with GCP Well-Architected
riveter scan -p kubernetes-security -p gcp-well-architected -t gke/
```

## CI/CD Integration

### Multi-Cloud Pipeline Strategy

#### Parallel Validation
```yaml
# GitHub Actions example
jobs:
  validate-aws:
    # AWS validation job
  validate-azure:
    # Azure validation job
  validate-gcp:
    # GCP validation job
  validate-kubernetes:
    # Kubernetes validation job
```

#### Sequential Validation
```yaml
# GitLab CI example
stages:
  - validate-infrastructure
  - validate-compliance
  - validate-kubernetes
```

### Environment-Specific Validation

#### Development Environment
```bash
# Relaxed validation for development
riveter scan -p multi-cloud-security -t dev/ --severity warning
```

#### Staging Environment
```bash
# Standard validation for staging
riveter scan -p multi-cloud-security -p soc2-security -t staging/
```

#### Production Environment
```bash
# Strict validation for production
riveter scan -p aws-security -p gcp-security -p azure-security -p cis-aws -p cis-gcp -p cis-azure -t production/
```

## Best Practices

### Rule Pack Selection Strategy

#### 1. Start with Cloud-Specific Security
```bash
# Begin with provider-specific security rules
riveter scan -p aws-security -t aws/
riveter scan -p gcp-security -t gcp/
riveter scan -p azure-security -t azure/
```

#### 2. Add Compliance Requirements
```bash
# Add compliance based on industry requirements
riveter scan -p aws-security -p cis-aws -t aws/          # General compliance
riveter scan -p aws-security -p aws-hipaa -t aws/       # Healthcare
riveter scan -p aws-security -p aws-pci-dss -t aws/     # Payment processing
```

#### 3. Include Well-Architected Frameworks
```bash
# Add architectural best practices
riveter scan -p aws-security -p cis-aws -p aws-well-architected -t aws/
```

#### 4. Apply Multi-Cloud Patterns
```bash
# Ensure consistency across clouds
riveter scan -p multi-cloud-security -p aws-security -p gcp-security -t infrastructure/
```

### Directory Organization

#### Option 1: Cloud-Separated
```
infrastructure/
├── aws/
│   ├── production/
│   ├── staging/
│   └── development/
├── gcp/
│   ├── production/
│   ├── staging/
│   └── development/
└── azure/
    ├── production/
    ├── staging/
    └── development/
```

#### Option 2: Environment-Separated
```
infrastructure/
├── production/
│   ├── aws/
│   ├── gcp/
│   └── azure/
├── staging/
│   ├── aws/
│   ├── gcp/
│   └── azure/
└── development/
    ├── aws/
    ├── gcp/
    └── azure/
```

#### Option 3: Service-Separated
```
infrastructure/
├── compute/
│   ├── aws-ec2/
│   ├── gcp-compute/
│   └── azure-vm/
├── storage/
│   ├── aws-s3/
│   ├── gcp-storage/
│   └── azure-storage/
└── kubernetes/
    ├── eks/
    ├── gke/
    └── aks/
```

### Validation Strategies

#### Progressive Validation
1. **Level 1**: Basic security (`*-security` packs)
2. **Level 2**: Add compliance (`cis-*` packs)
3. **Level 3**: Add architecture (`*-well-architected` packs)
4. **Level 4**: Add specialized compliance (`*-hipaa`, `*-pci-dss`)

#### Risk-Based Validation
- **High Risk**: Production environments with strict compliance
- **Medium Risk**: Staging environments with standard security
- **Low Risk**: Development environments with basic security

#### Continuous Validation
```bash
# Pre-commit validation
riveter scan -p multi-cloud-security -t changed-files/

# Pull request validation
riveter scan -p aws-security -p gcp-security -p azure-security -t infrastructure/

# Deployment validation
riveter scan -p cis-aws -p cis-gcp -p cis-azure -t production/
```

### Performance Optimization

#### Targeted Scanning
```bash
# Scan specific directories
riveter scan -p aws-security -t aws/compute/
riveter scan -p gcp-security -t gcp/storage/

# Scan by resource type
riveter scan -p kubernetes-security -t k8s/deployments/
```

#### Parallel Execution
```bash
# Run multiple scans in parallel
riveter scan -p aws-security -t aws/ &
riveter scan -p gcp-security -t gcp/ &
riveter scan -p azure-security -t azure/ &
wait
```

#### Caching Strategy
- Cache rule pack loading between scans
- Use incremental scanning for large infrastructures
- Implement result caching for unchanged files

### Troubleshooting Multi-Cloud Issues

#### Common Issues

1. **Resource Type Conflicts**
   - Ensure proper directory separation
   - Use provider-specific rule packs appropriately

2. **Rule Pack Conflicts**
   - Check for duplicate rule IDs across packs
   - Use `riveter validate-rule-pack` to verify packs

3. **Performance Issues**
   - Limit scope with targeted scanning
   - Use appropriate rule pack combinations

#### Debugging Commands
```bash
# List all available rule packs
riveter list-rule-packs

# Validate specific rule pack
riveter validate-rule-pack rule_packs/multi-cloud-security.yml

# Debug rule execution
riveter scan -p aws-security -t main.tf --verbose
```

### Migration Strategies

#### From Single Cloud to Multi-Cloud
1. Start with existing cloud-specific validation
2. Add multi-cloud security pack gradually
3. Implement consistent patterns across clouds
4. Migrate to unified CI/CD pipeline

#### From Manual to Automated
1. Begin with manual rule pack execution
2. Add to development workflow
3. Integrate into CI/CD pipeline
4. Implement automated remediation

This guide provides a comprehensive approach to multi-cloud infrastructure validation with Riveter. For specific implementation details, refer to the individual rule pack documentation and the main README.
