> # 📢 Note
> 
> This is my fork of [fogleman/sdf](https://github.com/fogleman/sdf). This fork is available [on GitHub](https://github.com/nobodyinperson/sdf) (so it's clear where I forked it from) and [on GitLab](https://gitlab.com/nobodyinperson/sdf), where I'll do the deployments, automatic tests and docs at some point.
> See [here](https://gitlab.com/nobodyinperson/sdfcad/-/compare/348d1c49...main?from_project_id=45731219&straight=false) for a an effective diff since forking.   
>
> ## 📹 Video: My Talk [OpenSCAD vs PythonSDF (🇩🇪 German) at the Tübix2023 Linux Day](https://odysee.com/@nobodyinperson:6/T%C3%BCbix2023-Yann-B%C3%BCchau-OpenSCAD-vs-PythonSDF:d)
> 
> I added so many things that it doesn't really make sense to just open a PR upstream. Probably I'll just maintain my own fork. 
> 
> Documentation of the new features is sparse though...  😅
> 
> Here is a rough list:
> 
> - `sdf.ease` got a huge revamp - easings can now be manipulated and combined easily:
>     ```python
>     ease.linear.plot()             # show what it looks like
>     ease.Easing.plot(ease.linear, ease.in_cubic, ease.smoothstep) # show multiple for comparison
>     3 * ease.linear.symmetric + 1  # This makes a triangle starting at 1 and going as high as 4 in the center
>     ease.smoothstep[0.2:0.8]       # zoom into function
>     ease.linear + ease.in_sine     # add or multiply functions
>     ease.linear.append(ease.in_cubic.reverse) # stitch functions together
>     # etc, much more, try 'python -m sdf.ease' for a non-comprehensive overview
>     ```
> - New objects:
>     - There's now `bezier()`-curves! 🥳
>         - their width can even be modulated with an easing function (https://fosstodon.org/@nobodyinperson/110367790398144919)
>     - There's now `Thread()`s 🔩
>         - just a twisted offset infinite cylinder, very nice to 3D print due to the smoothness
>         - `sphere(10)-Thread().dilate(0.4)` makes a nice thread hole with tolerance
>         - `Screw()` with head for convenience
>     - `pieslice()`: a vertically infinite pie slice, useful to cut out parts
> - New operations:
>     - `union()`, `intersection()` and `difference()` now support both `fillet=` and `chamfer=`
>     - `mirror()` an object at an arbitrary point into an any direction
>     - `stretch()` an object from here to there
>     - `rotate_stretch()` an object to do a kind of partial rotary sweep
>     - `shear()` an object from between two points along a direction
>     - `modulate_between()`: modify an object's thickness between two points with an easing function
>     - `twist_between()`: twist an object between two points with a variable rotation angle specified by an easing function
>     - `chamfercut()`: cut a chamfer along a plane (a wrapper around `modulate_between()`)
>     - `shell()` can now also do inner and outer shell, not just around boundary
>   Shape analysis (not very precise and reliable yet):
>     - `.bounds` et al.: finding boundaries/closest surface points/intersections via optimization of the SDF
>     - `volume()`: brute-force-approximate the volume Monte-Carlo-style
> - Many usability fixes here and there
>     - `save()` is now properly interruptible, no more zombie worker threads
>     - `save()` now optionally shows the mesh with `pyvista` (useful in Notebooks)
>     - docstrings are shown properly in interactive Python shells
>     - `k()` now creates a copy and doesn't modify the object itself
> 
> 
> Some things on my TODO list:
> 
> - [ ]  deleting   this point from the README (impossible on mobile apparently)
> - [ ] add tests!
> - [ ] auto-generated sphinx documentation
>     - [ ] auto-generated examples with code besides renders
> - [ ] importing SVGs 🤔
> - [ ] exporting SVGs (with marching squares / contourplots for 2D SDFs)
> - [ ] importing STLs 🤔
> 

[![coverage report](https://gitlab.com/nobodyinperson/sdfCAD/badges/main/coverage.svg)](https://nobodyinperson.gitlab.io/sdfcad/)

[![PyPI version](https://badge.fury.io/py/sdfcad.svg)](https://badge.fury.io/py/sdfcad)

[![Downloads](https://static.pepy.tech/badge/sdfcad)](https://pepy.tech/project/sdfcad)

# sdfCAD

Generate 3D meshes based on SDFs (signed distance functions) with a dirt simple Python API.

Special thanks to Michael Fogleman for initializing this codebase in [GitHub:fogleman/sdf](https://github.com/fogleman/sdf).

Special thanks also to [Inigo Quilez](https://iquilezles.org/) for his excellent documentation on signed distance functions:

- [3D Signed Distance Functions](https://iquilezles.org/www/articles/distfunctions/distfunctions.htm)
- [2D Signed Distance Functions](https://iquilezles.org/www/articles/distfunctions2d/distfunctions2d.htm)

## Example

<img width=350 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/csg-canonical.png">

Here is a complete example that generates the model shown. This is the
canonical [Constructive Solid Geometry](https://en.wikipedia.org/wiki/Constructive_solid_geometry)
example. Note the use of operators for union, intersection, and difference.

```python
from sdfcad import *

f = sphere(1) & box(1.5)

c = cylinder(0.5)
f -= c.orient(X) | c.orient(Y) | c.orient(Z)

f.save('out.stl')
```

Yes, that's really the entire code! You can 3D print that model or use it
in a 3D application.

## More Examples

Have a cool example? Submit a PR!

| [gearlike.py](examples/gearlike.py) | [knurling.py](examples/knurling.py) | [blobby.py](examples/blobby.py) | [weave.py](examples/weave.py) |
| --- | --- | --- | --- |
| ![gearlike](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/gearlike.png) | ![knurling](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/knurling.png) | ![blobby](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/blobby.png) | ![weave](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/weave.png) |
| ![gearlike](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/gearlike.jpg) | ![knurling](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/knurling.jpg) | ![blobby](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/blobby.jpg) | ![weave](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/weave.jpg) |

## Requirements

Note that the dependencies will be automatically installed by when following the directions below.

- Python 3
- matplotlib
- meshio
- numpy
- Pillow
- scikit-image
- scipy
- (pyvista, trame, jupyter, ...)

## 📥 Installation

Quick install with `pip`:

```bash
# optionally make a virtualenv
python -m venv sdf-venv
source sdf-venv/bin/activate
# from PyPI: together with the dependencies to run within Jupyter
pip install 'sdfcad[jupyter]'
# latest development version
pip install 'sdfcad[jupyter] @ git+https://gitlab.com/nobodyinperson/sdfCAD'
# for headless operation
pip install git+https://gitlab.com/nobodyinperson/sdfCAD
```

To hack on:

```bash
git clone https://gitlab.com/nobodyinperson/sdfCAD
cd sdfCAD
poetry install --all-extras --with=dev
poetry shell # enter a shell in the virtual environment
```

Confirm that it works:

```bash
python examples/example.py # should generate a file named out.stl
```

## File Formats

`sdf` natively writes binary STL files. For other formats, [meshio](https://github.com/nschloe/meshio)
is used (based on your output file extension). This adds support for over 20 different 3D file formats,
including OBJ, PLY, VTK, and many more.

## Viewing the Mesh

<img width=250 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/sdfCAD-octopus-in-Jupyter.png">

sdfCAD is best worked with in [Jupyter](https://jupyter.org) Lab:

```bash
# launch jupyter lab, this should open your browser with Jupyter Lab
jupyter lab
```

Alternatively, you can just `.save()` your object and open the resulting file in your Mesh viewer of choice, e.g.:

- [MeshLab](https://www.meshlab.net/)
- [OpenSCAD](https://openscad.org) (`save(openscad=True)` creates a `.scad` file you can open. OpenSCAD can also auto-reload if the file changes.)
- Michael Fogleman's [meshview](https://github.com/fogleman/meshview)

# API

In all of the below examples, `f` is any 3D SDF, such as:

```python
f = sphere()
```

## Bounds

The bounding box of the SDF is automatically estimated. Inexact SDFs such as
non-uniform scaling may cause issues with this process. In that case you can
specify the bounds to sample manually:

```python
f.save('out.stl', bounds=((-1, -1, -1), (1, 1, 1)))
```

## Resolution

The resolution of the mesh is also computed automatically. There are two ways
to specify the resolution. You can set the resolution directly with `step`:

```python
f.save('out.stl', step=0.01)
f.save('out.stl', step=(0.01, 0.02, 0.03)) # non-uniform resolution
```

Or you can specify approximately how many points to sample:

```python
f.save('out.stl', samples=2**24) # sample about 16M points
```

By default, `samples=2**18` is used.

*Tip*: Use the default resolution while developing your SDF. Then when you're done,
crank up the resolution for your final output.

## Batches

The SDF is sampled in batches. By default the batches have `32**3 = 32768`
points each. This batch size can be overridden:

```python
f.save('out.stl', batch_size=64) # instead of 32
```

The code attempts to skip any batches that are far away from the surface of
the mesh. Inexact SDFs such as non-uniform scaling may cause issues with this
process, resulting in holes in the output mesh (where batches were skipped when
they shouldn't have been). To avoid this, you can disable sparse sampling:

```python
f.save('out.stl', sparse=False) # force all batches to be completely sampled
```

## Worker Threads

The SDF is sampled in batches using worker threads. By default,
`multiprocessing.cpu_count()` worker threads are used. This can be overridden:

```python
f.save('out.stl', workers=1) # only use one worker thread
```

## Without Saving

You can of course generate a mesh without writing it to an STL file:

```python
points = f.generate() # takes the same optional arguments as `save`
print(len(points)) # print number of points (3x the number of triangles)
print(points[:3]) # print the vertices of the first triangle
```

If you want to save an STL after `generate`, just use:

```python
write_binary_stl(path, points)
```

## Visualizing the SDF

The `save()` method automatically shows the object with `pyvista` if available (e.g. when you installed with the `jupyter` extra as described above).

```python
# Turn automatic plot visualization off with this:
sphere().save(plot=False)
```

In Jupyter, you can try switching the backend between `client` (faster) or `server` (slower but probably more reliable).

```bash
import pyvista as pv
pv.set_jupyter_backend("client")
# or pv.set_jupyter_backend("server")
```

<img width=350 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/show_slice.png">

You can plot a visualization of a 2D slice of the SDF using matplotlib.
This can be useful for debugging purposes.

```python
f.show_slice(z=0)
f.show_slice(z=0, abs=True) # show abs(f)
```

You can specify a slice plane at any X, Y, or Z coordinate. You can
also specify the bounds to plot.

Note that `matplotlib` is only imported if this function is called, so it
isn't strictly required as a dependency.

<br clear="right">

## How it Works

The code simply uses the [Marching Cubes](https://en.wikipedia.org/wiki/Marching_cubes)
algorithm to generate a mesh from the [Signed Distance Function](https://en.wikipedia.org/wiki/Signed_distance_function).

This would normally be abysmally slow in Python. However, numpy is used to
evaluate the SDF on entire batches of points simultaneously. Furthermore,
multiple threads are used to process batches in parallel. The result is
surprisingly fast (for marching cubes). Meshes of adequate detail can
still be quite large in terms of number of triangles.

The core "engine" of the `sdf` library is very small and can be found in
[mesh.py](https://github.com/fogleman/sdf/blob/main/sdf/mesh.py).

In short, there is nothing algorithmically revolutionary here. The goal is
to provide a simple, fun, and easy-to-use API for generating 3D models in our
favorite language Python.

## Files

- [sdf/d2.py](https://github.com/fogleman/sdf/blob/main/sdf/d2.py): 2D signed distance functions
- [sdf/d3.py](https://github.com/fogleman/sdf/blob/main/sdf/d3.py): 3D signed distance functions
- [sdf/dn.py](https://github.com/fogleman/sdf/blob/main/sdf/dn.py): Dimension-agnostic signed distance functions
- [sdf/ease.py](https://github.com/fogleman/sdf/blob/main/sdf/ease.py): [Easing functions](https://easings.net/) that operate on numpy arrays. Some SDFs take an easing function as a parameter.
- [sdf/mesh.py](https://github.com/fogleman/sdf/blob/main/sdf/mesh.py): The core mesh-generation engine. Also includes code for estimating the bounding box of an SDF and for plotting a 2D slice of an SDF with matplotlib.
- [sdf/progress.py](https://github.com/fogleman/sdf/blob/main/sdf/progress.py): A console progress bar.
- [sdf/stl.py](https://github.com/fogleman/sdf/blob/main/sdf/stl.py): Code for writing a binary [STL file](https://en.wikipedia.org/wiki/STL_(file_format)).
- [sdf/text.py](https://github.com/fogleman/sdf/blob/main/sdf/text.py): Generate 2D SDFs for text (which can then be extruded)
- [sdf/util.py](https://github.com/fogleman/sdf/blob/main/sdf/util.py): Utility constants and functions.

## SDF Implementation

It is reasonable to write your own SDFs beyond those provided by the
built-in library. Browse the SDF implementations to understand how they are
implemented. Here are some simple examples:

```python
@sdf3
def sphere(radius=1, center=ORIGIN):
    def f(p):
        return np.linalg.norm(p - center, axis=1) - radius
    return f
```

An SDF is simply a function that takes a numpy array of points with shape `(N, 3)`
for 3D SDFs or shape `(N, 2)` for 2D SDFs and returns the signed distance for each
of those points as an array of shape `(N, 1)`. They are wrapped with the
`@sdf3` decorator (or `@sdf2` for 2D SDFs) which make boolean operators work,
add the `save` method, add the operators like `translate`, etc.

```python
@op3
def translate(other, offset):
    def f(p):
        return other(p - offset)
    return f
```

An SDF that operates on another SDF (like the above `translate`) should use
the `@op3` decorator instead. This will register the function such that SDFs
can be chained together like:

```python
f = sphere(1).translate((1, 2, 3))
```

Instead of what would otherwise be required:

```python
f = translate(sphere(1), (1, 2, 3))
```

## Remember, it's Python!

<img width=250 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/customizable_box.png">

Remember, this is Python, so it's fully programmable. You can and should split up your
model into parameterized sub-components, for example. You can use for loops and
conditionals wherever applicable. The sky is the limit!

See the [customizable box example](examples/customizable_box.py) for some starting ideas.

<br clear="right">

## Easings

Many transformations in sdfCAD take an optional **easing** parameter `e`.
This is a scalar function that takes an input between 0 and 1 and output another scalar, typically also between 0 and 1.
For convenience, there are many predefined easing functions available, the most important ones probably being `ease.linear` and `ease.smoothstep`.
Easings can be scaled, added, multiplied, chained, zoomed, etc. to customize them.
An easing function can be plotted via its `plot()` method (e.g. `ease.in_out_cubic.plot()`)
Here is an example of some operations:

```python
ease.Easing.plot(
    ease.linear,
    ease.smoothstep,
    ease.smoothstep.chain(),
    0.5 * ease.in_out_cubic.reverse,
    ease.out_elastic.symmetric,
    ease.smoothstep.symmetric,
    ease.smoothstep.between(-0.5,1),
    ease.smoothstep[0.1:0.75],
    ease.smoothstep.mirror(),
    -3*ease.linear + ease.smoothstep,
)
```

![easings](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/easings.png)

# Function Reference

## 3D Primitives

### sphere

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/sphere.png">

`sphere(radius=1, center=ORIGIN)`

```python
f = sphere() # unit sphere
f = sphere(2) # specify radius
f = sphere(1, (1, 2, 3)) # translated sphere
```

### box

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/box2.png">

`box(size=1, center=ORIGIN, a=None, b=None)`

```python
f = box(1) # all side lengths = 1
f = box((1, 2, 3)) # different side lengths
f = box(a=(-1, -1, -1), b=(3, 4, 5)) # specified by bounds
```

### rounded_box

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/rounded_box.png">

`rounded_box(size, radius)`

```python
f = rounded_box((1, 2, 3), 0.25)
```

### wireframe_box
<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/wireframe_box.png">

`wireframe_box(size, thickness)`

```python
f = wireframe_box((1, 2, 3), 0.05)
```

### torus
<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/torus.png">

`torus(r1, r2)`

```python
f = torus(1, 0.25)
```

### capsule
<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/capsule.png">

`capsule(a, b, radius)`

```python
f = capsule(-Z, Z, 0.5)
```

### capped_cylinder
<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/capped_cylinder.png">

`capped_cylinder(a, b, radius)`

```python
f = capped_cylinder(-Z, Z, 0.5)
```

### rounded_cylinder
<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/rounded_cylinder.png">

`rounded_cylinder(ra, rb, h)`

```python
f = rounded_cylinder(0.5, 0.1, 2)
```

### capped_cone

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/capped_cone.png">

`capped_cone(a, b, ra, rb)`

```python
f = capped_cone(-Z, Z, 1, 0.5)
```

### rounded_cone

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/rounded_cone.png">

`rounded_cone(r1, r2, h)`

```python
f = rounded_cone(0.75, 0.25, 2)
```

### ellipsoid

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/ellipsoid.png">

`ellipsoid(size)`

```python
f = ellipsoid((1, 2, 3))
```

### pyramid

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/pyramid.png">

`pyramid(h)`

```python
f = pyramid(1)
```

## Platonic Solids

### tetrahedron

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/tetrahedron.png">

`tetrahedron(r)`

```python
f = tetrahedron(1)
```

### octahedron

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/octahedron.png">

`octahedron(r)`

```python
f = octahedron(1)
```

### dodecahedron

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/dodecahedron.png">

`dodecahedron(r)`

```python
f = dodecahedron(1)
```

### icosahedron

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/icosahedron.png">

`icosahedron(r)`

```python
f = icosahedron(1)
```

## Infinite 3D Primitives

The following SDFs extend to infinity in some or all axes.
They can only effectively be used in combination with other shapes, as shown in the examples below.

### plane

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/plane.png">

`plane(normal=UP, point=ORIGIN)`

`plane` is an infinite plane, with one side being positive (outside) and one side being negative (inside).

```python
f = sphere() & plane()
```

### slab

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/slab.png">

```python
slab(x0=None, y0=None, z0=None, x1=None, y1=None, z1=None, r=None)
slab(dx=None, dy=None, dz=None) # symmetric version
```

`slab` is useful for cutting a shape on one or more axis-aligned planes.

```python
f = sphere() & slab(dz=1, x0=0)
```

### cylinder

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/cylinder.png">

`cylinder(radius)`

`cylinder` is an infinite cylinder along the Z axis.

```python
f = sphere() - cylinder(0.5)
```

## Text

Yes, even text is supported!

![Text](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/text-large.png)

`text(font_name, text, width=None, height=None, pixels=PIXELS, points=512)`

```python
FONT = 'Arial'
TEXT = 'Hello, world!'

w, h = measure_text(FONT, TEXT)

f = rounded_box((w + 1, h + 1, 0.2), 0.1)
f -= text(FONT, TEXT).extrude(1)
```

Note: [PIL.ImageFont](https://pillow.readthedocs.io/en/stable/reference/ImageFont.html),
which is used to load fonts, does not search for the font by name on all operating systems.
For example, on Ubuntu the full path to the font has to be provided.
(e.g. `/usr/share/fonts/truetype/freefont/FreeMono.ttf`)

## Images

Image masks can be extruded and incorporated into your 3D model.

![Image Mask](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/butterfly.png)

`image(path_or_array, width=None, height=None, pixels=PIXELS)`

```python
IMAGE = 'examples/butterfly.png'

w, h = measure_image(IMAGE)

f = rounded_box((w * 1.1, h * 1.1, 0.1), 0.05)
f |= image(IMAGE).extrude(1) & slab(z0=0, z1=0.075)
```

## Positioning

### translate

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/translate.png">

`translate(other, offset)`

```python
f = sphere().translate((0, 0, 2))
```

### scale

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/scale.png">

`scale(other, factor)`

Note that non-uniform scaling is an inexact SDF.

```python
f = sphere().scale(2)
f = sphere().scale((1, 2, 3)) # non-uniform scaling
```

### rotate

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/rotate.png">

`rotate(other, angle, vector=Z)`

```python
f = capped_cylinder(-Z, Z, 0.5).rotate(pi / 4, X)
```

### orient

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/orient.png">

`orient(other, axis)`

`orient` rotates the shape such that whatever was pointing in the +Z direction
is now pointing in the specified direction.

```python
c = capped_cylinder(-Z, Z, 0.25)
f = c.orient(X) | c.orient(Y) | c.orient(Z)
```

## Boolean Operations

The following primitives `a` and `b` are used in all of the following
boolean operations.

```python
a = box((3, 3, 0.5))
b = sphere()
```

The named versions (`union`, `difference`, `intersection`) can all take
one or more SDFs as input. They all take an optional `radius` (`r`) and
`chamfer` (`c`) parameter to define the amount of filleting/chamfering to apply. When
using operators (`|`, `-`, `&`) the smoothing can still be applied via the
`.fillet(1)` and `.chamfer(3)` functions.

### union

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/union.png">

```python
f = a | b
f = union(a, b) # equivalent
```

<br clear="right">

### difference

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/difference.png">

```python
f = a - b
f = difference(a, b) # equivalent
```

<br clear="right">

### intersection

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/intersection.png">

```python
f = a & b
f = intersection(a, b) # equivalent
```

<br clear="right">

### smooth_union

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/smooth_union.png">

```python
f = a | b.r(0.25)
f = union(a, b, r=0.25) # equivalent
```

<br clear="right">

### smooth_difference

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/smooth_difference.png">

```python
f = a - b.r(0.25)
f = difference(a, b, r=0.25) # equivalent
```

<br clear="right">

### smooth_intersection

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/smooth_intersection.png">

```python
f = a & b.r(0.25)
f = intersection(a, b, r=0.25) # equivalent
```

<br clear="right">

## Repetition

### repeat

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/repeat.png">

`repeat(other, spacing, count=None, padding=0)`

`repeat` can repeat the underlying SDF infinitely or a finite number of times.
If finite, the number of repetitions must be odd, because the count specifies
the number of copies to make on each side of the origin. If the repeated
elements overlap or come close together, you may need to specify a `padding`
greater than zero to compute a correct SDF.

```python
f = sphere().repeat(3, (1, 1, 0))
```

### circular_array

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/circular_array.png">

`circular_array(other, count, offset)`

`circular_array` makes `count` copies of the underlying SDF, arranged in a
circle around the Z axis. `offset` specifies how far to translate the shape
in X before arraying it. The underlying SDF is only evaluated twice (instead
of `count` times), so this is more performant than instantiating `count` copies
of a shape.

```python
f = capped_cylinder(-Z, Z, 0.5).circular_array(8, 4)
```

## Miscellaneous

### blend

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/blend.png">

`blend(a, *bs, r=0.5)`

```python
f = sphere().blend(box())
```

### dilate

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/dilate.png">

`dilate(other, r)`

```python
f = example.dilate(0.1)
```

### erode

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/erode.png">

`erode(other, r)`

```python
f = example.erode(0.1)
```

### shell

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/shell.png">

`shell(other, thickness)`

```python
f = sphere().shell(0.05) & plane(-Z)
```

### elongate

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/elongate.png">

`elongate(other, size)`

```python
f = example.elongate((0.25, 0.5, 0.75))
```

### twist

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/twist.png">

`twist(other, k)`

```python
f = box().twist(pi / 2)
```

### twist_between

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/twist_between.png">

```python
slab(z0=0, dx=5, dy=20, z1=30,r=2).twist_between(
    a = 5*Z,     # start to twist here
    b = 25 * Z,  # stop twisting here
    # smoothstep funtion between 0° and 80°
    e=units("80°").to("radians").m * ease.smoothstep
    ).save()
```

### stretch

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/stretch.png">

Stretch an object from here to there.

```python
# stretch a sphere vertically to make a capsule
sphere(10).stretch(ORIGIN, 20*Z).save()
# specify symmetric=True to stretch equally in the opposite direction
```

### rotate_stretch

![rotate_stretch animation](https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/rotate-stretch-capsule.mp4)

Do a kind of rotary sweep like `stretch()`, but for rotation.

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/rotate-stretch-capsule.png">

```python
# make a quarter capsule sweep
# (”grab X, stretch it around origin to Y”)
capsule(ORIGIN,30*X,10).rotate_stretch(X,Y).save()
# the same with angle, Z is default rotation axis
capsule(ORIGIN,30*X,10).rotate_stretch(angle=units("90°")).save()
```

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/rotate-stretch-capsule-shear-thicken-fling.png">

Along the way, one can move along the rotation axis, outwards or modulate the thickness:

```python
# sweep a capsule around
capsule(30*X, 50*X, 10).rotate_stretch(
    angle=units("330°"),
    shear=40*ease.smoothstep,   # move smoothly up along rotation axis while rotating
    thicken=-6*ease.smoothstep, # gradually get thinner along the way
    fling=-20*ease.smoothstep,  # gradually move closer to rotation axis
).save(sparse=False)
```

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/rotate-stretch-transition-capsule-box.png">

Changing shape along the way is also possible:

```python
# sweep a capsule around
capsule(20 * X, 40 * X, 10).rotate_stretch(
    shear=30 * ease.smoothstep, # spiral upwards
    # transition to a slightly rotated box
    transition=slab(dx=30,dy=10,dz=10,r=1).rotate(-units("50°"),Y).translate(30*X),
    transition_ease=ease.smoothstep.chain().chain(), # transition smoothly to box
).save(sparse=False)
```

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/rotate-stretch-wavy-ring.png">

```python
# sweep a capsule around
capsule(30 * X, 50 * X, 10).rotate_stretch(
    shear=10 * (ease.smoothstep.symmetric-0.5).repeat(5), # up and down along the way
).save(sparse=False)
```

### shear

Grab a point and move it into a direction, keeping another point fix. 
This can be used to perform a shearing operation.

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/shear-smooth.png">

```python
box([20,10,50]).shear(fix=-15*Z, grab=15*Z, move=-10*X, e=ease.smoothstep).save()
```

The easing function can be used to scale the movement along the way, e.g. to make a notch.

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/shear-notch.png">

```python
box([20,10,50]).shear(-15*Z, 15*Z, -5*X, e=ease.smoothstep.symmetric).save()
```

### modulate_between

Modulate the thickness between two points.

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/modulate-between.png">

```python
# make a cylinder
(cylinder(10) & slab(dz=50)).modulate_between(
    -20 * Z, # start modulating here
    20 * Z,  # stop  modulating here
    # easing to apply.
    # symmetric smoothstep function for negative dip.
    e=-5 * ease.smoothstep.symmetric,
).save()
```

### chamfer

This is a wrapper around `stretch()` and `modulate_between()` to facilitate a chamfer cut along a plane.

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/chamfer-sphere.png">

```python
sphere(10).chamfercut(2,at=ORIGIN, direction=-Z).save()
```

### bend

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/bend.png">

`bend(other, k)`

```python
f = box().bend(1)
```

### bend_linear

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/bend_linear.png">

`bend_linear(other, p0, p1, v, e=ease.linear)`

```python
f = capsule(-Z * 2, Z * 2, 0.25).bend_linear(-Z, Z, X, ease.in_out_quad)
```

### bend_radial

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/bend_radial.png">

`bend_radial(other, r0, r1, dz, e=ease.linear)`

```python
f = box((5, 5, 0.25)).bend_radial(1, 2, -1, ease.in_out_quad)
```

### transition_linear

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/transition_linear.png">

`transition_linear(f0, f1, p0=-Z, p1=Z, e=ease.linear)`

```python
f = box().transition_linear(sphere(), e=ease.in_out_quad)
```

### transition_radial

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/transition_radial.png">

`transition_radial(f0, f1, r0=0, r1=1, e=ease.linear)`

```python
f = box().transition_radial(sphere(), e=ease.in_out_quad)
```

### wrap_around

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/wrap_around.png">

`wrap_around(other, x0, x1, r=None, e=ease.linear)`

```python
FONT = 'Arial'
TEXT = ' wrap_around ' * 3
w, h = measure_text(FONT, TEXT)
f = text(FONT, TEXT).extrude(0.1).orient(Y).wrap_around(-w / 2, w / 2)
```

## 2D to 3D Operations

### extrude

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/extrude.png">

`extrude(other, h)`

```python
f = hexagon(1).extrude(1)
```

### extrude_to

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/extrude_to.png">

`extrude_to(a, b, h, e=ease.linear)`

```python
f = rectangle(2).extrude_to(circle(1), 2, ease.in_out_quad)
```

### revolve

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/revolve.png">

`revolve(other, offset=0)`

```python
f = hexagon(1).revolve(3)
```

## 3D to 2D Operations

### slice

<img width=128 align="right" src="https://gitlab.com/nobodyinperson/sdfcad/-/raw/main/docs/images/slice.png">

`slice(other)`

```python
f = example.translate((0, 0, 0.55)).slice().extrude(0.1)
```

## 2D Primitives

### circle
### line
### rectangle
### rounded_rectangle
### equilateral_triangle
### hexagon
### rounded_x
### polygon
