"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/cdk");
const glue_generated_1 = require("./glue.generated");
/**
 * A Glue database.
 */
class Database extends cdk.Construct {
    /**
     * Creates a Database construct that represents an external database.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param props A `DatabaseImportProps` object. Can be obtained from a call to `database.export()` or manually created.
     */
    static import(scope, id, props) {
        return new ImportedDatabase(scope, id, props);
    }
    constructor(scope, id, props) {
        super(scope, id);
        if (props.locationUri) {
            this.locationUri = props.locationUri;
        }
        else {
            const bucket = new s3.Bucket(this, 'Bucket');
            this.locationUri = `s3://${bucket.bucketName}/${props.databaseName}`;
        }
        this.catalogId = this.node.stack.accountId;
        const resource = new glue_generated_1.CfnDatabase(this, 'Resource', {
            catalogId: this.catalogId,
            databaseInput: {
                name: props.databaseName,
                locationUri: this.locationUri
            }
        });
        // see https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html#data-catalog-resource-arns
        this.databaseName = resource.databaseName;
        this.databaseArn = this.node.stack.formatArn({
            service: 'glue',
            resource: 'database',
            resourceName: this.databaseName
        });
        // catalogId is implicitly the accountId, which is why we don't pass the catalogId here
        this.catalogArn = this.node.stack.formatArn({
            service: 'glue',
            resource: 'catalog'
        });
    }
    /**
     * Exports this database from the stack.
     */
    export() {
        return {
            catalogArn: new cdk.CfnOutput(this, 'CatalogArn', { value: this.catalogArn }).makeImportValue().toString(),
            catalogId: new cdk.CfnOutput(this, 'CatalogId', { value: this.catalogId }).makeImportValue().toString(),
            databaseArn: new cdk.CfnOutput(this, 'DatabaseArn', { value: this.databaseArn }).makeImportValue().toString(),
            databaseName: new cdk.CfnOutput(this, 'DatabaseName', { value: this.databaseName }).makeImportValue().toString(),
            locationUri: new cdk.CfnOutput(this, 'LocationURI', { value: this.locationUri }).makeImportValue().toString()
        };
    }
}
exports.Database = Database;
class ImportedDatabase extends cdk.Construct {
    constructor(parent, name, props) {
        super(parent, name);
        this.props = props;
        this.catalogArn = props.catalogArn;
        this.catalogId = props.catalogId;
        this.databaseArn = props.databaseArn;
        this.databaseName = props.databaseName;
        this.locationUri = props.locationUri;
    }
    export() {
        return this.props;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGF0YWJhc2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJkYXRhYmFzZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHNDQUF1QztBQUN2QyxvQ0FBcUM7QUFDckMscURBQStDO0FBcUQvQzs7R0FFRztBQUNILE1BQWEsUUFBUyxTQUFRLEdBQUcsQ0FBQyxTQUFTO0lBQ3pDOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBMEI7UUFDL0UsT0FBTyxJQUFJLGdCQUFnQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQTJCRCxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQW9CO1FBQ2hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQ3JCLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztTQUN0QzthQUFNO1lBQ0wsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztZQUM3QyxJQUFJLENBQUMsV0FBVyxHQUFHLFFBQVEsTUFBTSxDQUFDLFVBQVUsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFLENBQUM7U0FDdEU7UUFFRCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUMzQyxNQUFNLFFBQVEsR0FBRyxJQUFJLDRCQUFXLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNqRCxTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVM7WUFDekIsYUFBYSxFQUFFO2dCQUNiLElBQUksRUFBRSxLQUFLLENBQUMsWUFBWTtnQkFDeEIsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXO2FBQzlCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsK0dBQStHO1FBQy9HLElBQUksQ0FBQyxZQUFZLEdBQUcsUUFBUSxDQUFDLFlBQVksQ0FBQztRQUMxQyxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUMzQyxPQUFPLEVBQUUsTUFBTTtZQUNmLFFBQVEsRUFBRSxVQUFVO1lBQ3BCLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtTQUNoQyxDQUFDLENBQUM7UUFDSCx1RkFBdUY7UUFDdkYsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7WUFDMUMsT0FBTyxFQUFFLE1BQU07WUFDZixRQUFRLEVBQUUsU0FBUztTQUNwQixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNO1FBQ1gsT0FBTztZQUNMLFVBQVUsRUFBRSxJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxRQUFRLEVBQUU7WUFDMUcsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsV0FBVyxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDLFFBQVEsRUFBRTtZQUN2RyxXQUFXLEVBQUUsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsZUFBZSxFQUFFLENBQUMsUUFBUSxFQUFFO1lBQzdHLFlBQVksRUFBRSxJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxRQUFRLEVBQUU7WUFDaEgsV0FBVyxFQUFFLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDLFFBQVEsRUFBRTtTQUM5RyxDQUFDO0lBQ0osQ0FBQztDQUNGO0FBbEZELDRCQWtGQztBQUVELE1BQU0sZ0JBQWlCLFNBQVEsR0FBRyxDQUFDLFNBQVM7SUFPMUMsWUFBWSxNQUFxQixFQUFFLElBQVksRUFBbUIsS0FBMEI7UUFDMUYsS0FBSyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQztRQUQ0QyxVQUFLLEdBQUwsS0FBSyxDQUFxQjtRQUUxRixJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7UUFDbkMsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyxJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFDdkMsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO0lBQ3ZDLENBQUM7SUFFTSxNQUFNO1FBQ1gsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDO0lBQ3BCLENBQUM7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBzMyA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1zMycpO1xuaW1wb3J0IGNkayA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2NkaycpO1xuaW1wb3J0IHsgQ2ZuRGF0YWJhc2UgfSBmcm9tICcuL2dsdWUuZ2VuZXJhdGVkJztcblxuZXhwb3J0IGludGVyZmFjZSBJRGF0YWJhc2UgZXh0ZW5kcyBjZGsuSUNvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoZSBjYXRhbG9nLlxuICAgKi9cbiAgcmVhZG9ubHkgY2F0YWxvZ0Fybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY2F0YWxvZyBpZCBvZiB0aGUgZGF0YWJhc2UgKHVzdWFsbHksIHRoZSBBV1MgYWNjb3VudCBpZClcbiAgICovXG4gIHJlYWRvbmx5IGNhdGFsb2dJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoZSBkYXRhYmFzZS5cbiAgICovXG4gIHJlYWRvbmx5IGRhdGFiYXNlQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBkYXRhYmFzZS5cbiAgICovXG4gIHJlYWRvbmx5IGRhdGFiYXNlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbG9jYXRpb24gb2YgdGhlIGRhdGFiYXNlIChmb3IgZXhhbXBsZSwgYW4gSERGUyBwYXRoKS5cbiAgICovXG4gIHJlYWRvbmx5IGxvY2F0aW9uVXJpOiBzdHJpbmc7XG5cbiAgZXhwb3J0KCk6IERhdGFiYXNlSW1wb3J0UHJvcHM7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VJbXBvcnRQcm9wcyB7XG4gIHJlYWRvbmx5IGNhdGFsb2dBcm46IHN0cmluZztcbiAgcmVhZG9ubHkgY2F0YWxvZ0lkOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGRhdGFiYXNlQXJuOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGRhdGFiYXNlTmFtZTogc3RyaW5nO1xuICByZWFkb25seSBsb2NhdGlvblVyaTogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIERhdGFiYXNlUHJvcHMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGRhdGFiYXNlLlxuICAgKi9cbiAgcmVhZG9ubHkgZGF0YWJhc2VOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBsb2NhdGlvbiBvZiB0aGUgZGF0YWJhc2UgKGZvciBleGFtcGxlLCBhbiBIREZTIHBhdGgpLlxuICAgKlxuICAgKiBAZGVmYXVsdCBhIGJ1Y2tldCBpcyBjcmVhdGVkIGFuZCB0aGUgZGF0YWJhc2UgaXMgc3RvcmVkIHVuZGVyIHMzOi8vPGJ1Y2tldC1uYW1lPi88ZGF0YWJhc2UtbmFtZT5cbiAgICovXG4gIHJlYWRvbmx5IGxvY2F0aW9uVXJpPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEEgR2x1ZSBkYXRhYmFzZS5cbiAqL1xuZXhwb3J0IGNsYXNzIERhdGFiYXNlIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgRGF0YWJhc2UgY29uc3RydWN0IHRoYXQgcmVwcmVzZW50cyBhbiBleHRlcm5hbCBkYXRhYmFzZS5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIFRoZSBzY29wZSBjcmVhdGluZyBjb25zdHJ1Y3QgKHVzdWFsbHkgYHRoaXNgKS5cbiAgICogQHBhcmFtIGlkIFRoZSBjb25zdHJ1Y3QncyBpZC5cbiAgICogQHBhcmFtIHByb3BzIEEgYERhdGFiYXNlSW1wb3J0UHJvcHNgIG9iamVjdC4gQ2FuIGJlIG9idGFpbmVkIGZyb20gYSBjYWxsIHRvIGBkYXRhYmFzZS5leHBvcnQoKWAgb3IgbWFudWFsbHkgY3JlYXRlZC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgaW1wb3J0KHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGF0YWJhc2VJbXBvcnRQcm9wcyk6IElEYXRhYmFzZSB7XG4gICAgcmV0dXJuIG5ldyBJbXBvcnRlZERhdGFiYXNlKHNjb3BlLCBpZCwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFSTiBvZiB0aGUgR2x1ZSBjYXRhbG9nIGluIHdoaWNoIHRoaXMgZGF0YWJhc2UgaXMgc3RvcmVkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNhdGFsb2dBcm46IHN0cmluZztcblxuICAvKipcbiAgICogSUQgb2YgdGhlIEdsdWUgY2F0YWxvZyBpbiB3aGljaCB0aGlzIGRhdGFiYXNlIGlzIHN0b3JlZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjYXRhbG9nSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogQVJOIG9mIHRoaXMgZGF0YWJhc2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGF0YWJhc2VBcm46IHN0cmluZztcblxuICAvKipcbiAgICogTmFtZSBvZiB0aGlzIGRhdGFiYXNlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRhdGFiYXNlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBMb2NhdGlvbiBVUkkgb2YgdGhpcyBkYXRhYmFzZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBsb2NhdGlvblVyaTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGF0YWJhc2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBpZiAocHJvcHMubG9jYXRpb25VcmkpIHtcbiAgICAgIHRoaXMubG9jYXRpb25VcmkgPSBwcm9wcy5sb2NhdGlvblVyaTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLCAnQnVja2V0Jyk7XG4gICAgICB0aGlzLmxvY2F0aW9uVXJpID0gYHMzOi8vJHtidWNrZXQuYnVja2V0TmFtZX0vJHtwcm9wcy5kYXRhYmFzZU5hbWV9YDtcbiAgICB9XG5cbiAgICB0aGlzLmNhdGFsb2dJZCA9IHRoaXMubm9kZS5zdGFjay5hY2NvdW50SWQ7XG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuRGF0YWJhc2UodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgY2F0YWxvZ0lkOiB0aGlzLmNhdGFsb2dJZCxcbiAgICAgIGRhdGFiYXNlSW5wdXQ6IHtcbiAgICAgICAgbmFtZTogcHJvcHMuZGF0YWJhc2VOYW1lLFxuICAgICAgICBsb2NhdGlvblVyaTogdGhpcy5sb2NhdGlvblVyaVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9nbHVlL2xhdGVzdC9kZy9nbHVlLXNwZWNpZnlpbmctcmVzb3VyY2UtYXJucy5odG1sI2RhdGEtY2F0YWxvZy1yZXNvdXJjZS1hcm5zXG4gICAgdGhpcy5kYXRhYmFzZU5hbWUgPSByZXNvdXJjZS5kYXRhYmFzZU5hbWU7XG4gICAgdGhpcy5kYXRhYmFzZUFybiA9IHRoaXMubm9kZS5zdGFjay5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ2dsdWUnLFxuICAgICAgcmVzb3VyY2U6ICdkYXRhYmFzZScsXG4gICAgICByZXNvdXJjZU5hbWU6IHRoaXMuZGF0YWJhc2VOYW1lXG4gICAgfSk7XG4gICAgLy8gY2F0YWxvZ0lkIGlzIGltcGxpY2l0bHkgdGhlIGFjY291bnRJZCwgd2hpY2ggaXMgd2h5IHdlIGRvbid0IHBhc3MgdGhlIGNhdGFsb2dJZCBoZXJlXG4gICAgdGhpcy5jYXRhbG9nQXJuID0gdGhpcy5ub2RlLnN0YWNrLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnZ2x1ZScsXG4gICAgICByZXNvdXJjZTogJ2NhdGFsb2cnXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogRXhwb3J0cyB0aGlzIGRhdGFiYXNlIGZyb20gdGhlIHN0YWNrLlxuICAgKi9cbiAgcHVibGljIGV4cG9ydCgpOiBEYXRhYmFzZUltcG9ydFByb3BzIHtcbiAgICByZXR1cm4ge1xuICAgICAgY2F0YWxvZ0FybjogbmV3IGNkay5DZm5PdXRwdXQodGhpcywgJ0NhdGFsb2dBcm4nLCB7IHZhbHVlOiB0aGlzLmNhdGFsb2dBcm4gfSkubWFrZUltcG9ydFZhbHVlKCkudG9TdHJpbmcoKSxcbiAgICAgIGNhdGFsb2dJZDogbmV3IGNkay5DZm5PdXRwdXQodGhpcywgJ0NhdGFsb2dJZCcsIHsgdmFsdWU6IHRoaXMuY2F0YWxvZ0lkIH0pLm1ha2VJbXBvcnRWYWx1ZSgpLnRvU3RyaW5nKCksXG4gICAgICBkYXRhYmFzZUFybjogbmV3IGNkay5DZm5PdXRwdXQodGhpcywgJ0RhdGFiYXNlQXJuJywgeyB2YWx1ZTogdGhpcy5kYXRhYmFzZUFybiB9KS5tYWtlSW1wb3J0VmFsdWUoKS50b1N0cmluZygpLFxuICAgICAgZGF0YWJhc2VOYW1lOiBuZXcgY2RrLkNmbk91dHB1dCh0aGlzLCAnRGF0YWJhc2VOYW1lJywgeyB2YWx1ZTogdGhpcy5kYXRhYmFzZU5hbWUgfSkubWFrZUltcG9ydFZhbHVlKCkudG9TdHJpbmcoKSxcbiAgICAgIGxvY2F0aW9uVXJpOiBuZXcgY2RrLkNmbk91dHB1dCh0aGlzLCAnTG9jYXRpb25VUkknLCB7IHZhbHVlOiB0aGlzLmxvY2F0aW9uVXJpIH0pLm1ha2VJbXBvcnRWYWx1ZSgpLnRvU3RyaW5nKClcbiAgICB9O1xuICB9XG59XG5cbmNsYXNzIEltcG9ydGVkRGF0YWJhc2UgZXh0ZW5kcyBjZGsuQ29uc3RydWN0IGltcGxlbWVudHMgSURhdGFiYXNlIHtcbiAgcHVibGljIHJlYWRvbmx5IGNhdGFsb2dBcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGNhdGFsb2dJZDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZGF0YWJhc2VBcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGRhdGFiYXNlTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgbG9jYXRpb25Vcmk6IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihwYXJlbnQ6IGNkay5Db25zdHJ1Y3QsIG5hbWU6IHN0cmluZywgcHJpdmF0ZSByZWFkb25seSBwcm9wczogRGF0YWJhc2VJbXBvcnRQcm9wcykge1xuICAgIHN1cGVyKHBhcmVudCwgbmFtZSk7XG4gICAgdGhpcy5jYXRhbG9nQXJuID0gcHJvcHMuY2F0YWxvZ0FybjtcbiAgICB0aGlzLmNhdGFsb2dJZCA9IHByb3BzLmNhdGFsb2dJZDtcbiAgICB0aGlzLmRhdGFiYXNlQXJuID0gcHJvcHMuZGF0YWJhc2VBcm47XG4gICAgdGhpcy5kYXRhYmFzZU5hbWUgPSBwcm9wcy5kYXRhYmFzZU5hbWU7XG4gICAgdGhpcy5sb2NhdGlvblVyaSA9IHByb3BzLmxvY2F0aW9uVXJpO1xuICB9XG5cbiAgcHVibGljIGV4cG9ydCgpIHtcbiAgICByZXR1cm4gdGhpcy5wcm9wcztcbiAgfVxufVxuIl19