/**
 * A column of a table.
 */
export interface Column {
    /**
     * Name of the column.
     */
    readonly name: string;
    /**
     * Type of the column.
     */
    readonly type: Type;
    /**
     * Coment describing the column.
     *
     * @default none
     */
    readonly comment?: string;
}
/**
 * Represents a type of a column in a table schema.
 */
export interface Type {
    /**
     * Indicates whether this type is a primitive data type.
     */
    readonly isPrimitive: boolean;
    /**
     * Glue InputString for this type.
     */
    readonly inputString: string;
}
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/data-types.html
 */
export declare class Schema {
    static readonly boolean: Type;
    static readonly binary: Type;
    /**
     * A 64-bit signed INTEGER in two’s complement format, with a minimum value of -2^63 and a maximum value of 2^63-1.
     */
    static readonly bigint: Type;
    static readonly double: Type;
    static readonly float: Type;
    /**
     * A 32-bit signed INTEGER in two’s complement format, with a minimum value of -2^31 and a maximum value of 2^31-1.
     */
    static readonly integer: Type;
    /**
     * A 16-bit signed INTEGER in two’s complement format, with a minimum value of -2^15 and a maximum value of 2^15-1.
     */
    static readonly smallint: Type;
    /**
     * A 8-bit signed INTEGER in two’s complement format, with a minimum value of -2^7 and a maximum value of 2^7-1
     */
    static readonly tinyint: Type;
    /**
     * Date type.
     */
    static readonly date: Type;
    /**
     * Timestamp type (date and time).
     */
    static readonly timestamp: Type;
    /**
     * Arbitrary-length string type.
     */
    static readonly string: Type;
    /**
     * Creates a decimal type.
     *
     * TODO: Bounds
     *
     * @param precision the total number of digits
     * @param scale the number of digits in fractional part, the default is 0
     */
    static decimal(precision: number, scale?: number): Type;
    /**
     * Fixed length character data, with a specified length between 1 and 255.
     *
     * @param length length between 1 and 255
     */
    static char(length: number): Type;
    /**
     * Variable length character data, with a specified length between 1 and 65535.
     *
     * @param length length between 1 and 65535.
     */
    static varchar(length: number): Type;
    /**
     * Creates an array of some other type.
     *
     * @param itemType type contained by the array.
     */
    static array(itemType: Type): Type;
    /**
     * Creates a map of some primitive key type to some value type.
     *
     * @param keyType type of key, must be a primitive.
     * @param valueType type fo the value indexed by the key.
     */
    static map(keyType: Type, valueType: Type): Type;
    /**
     * Creates a nested structure containing individually named and typed columns.
     *
     * @param columns the columns of the structure.
     */
    static struct(columns: Column[]): Type;
}
