import iam = require('@aws-cdk/aws-iam');
import kms = require('@aws-cdk/aws-kms');
import s3 = require('@aws-cdk/aws-s3');
import cdk = require('@aws-cdk/cdk');
import { DataFormat } from './data-format';
import { IDatabase } from './database';
import { Column } from './schema';
export interface ITable extends cdk.IConstruct {
    readonly tableArn: string;
    readonly tableName: string;
    export(): TableImportProps;
}
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
export declare enum TableEncryption {
    Unencrypted = "Unencrypted",
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    S3Managed = "SSE-S3",
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    Kms = "SSE-KMS",
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    KmsManaged = "SSE-KMS-MANAGED",
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    ClientSideKms = "CSE-KMS"
}
export interface TableImportProps {
    readonly tableArn: string;
    readonly tableName: string;
}
export interface TableProps {
    /**
     * Name of the table.
     */
    readonly tableName: string;
    /**
     * Description of the table.
     *
     * @default generated
     */
    readonly description?: string;
    /**
     * Database in which to store the table.
     */
    readonly database: IDatabase;
    /**
     * S3 bucket in which to store data.
     *
     * @default one is created for you
     */
    readonly bucket?: s3.IBucket;
    /**
     * S3 prefix under which table objects are stored.
     *
     * @default data/
     */
    readonly s3Prefix?: string;
    /**
     * Columns of the table.
     */
    readonly columns: Column[];
    /**
     * Partition columns of the table.
     *
     * @default table is not partitioned
     */
    readonly partitionKeys?: Column[];
    /**
     * Storage type of the table's data.
     */
    readonly dataFormat: DataFormat;
    /**
     * Indicates whether the table's data is compressed or not.
     *
     * @default false
     */
    readonly compressed?: boolean;
    /**
     * The kind of encryption to secure the data with.
     *
     * You can only provide this option if you are not explicitly passing in a bucket.
     *
     * If you choose `SSE-KMS`, you *can* provide an un-managed KMS key with `encryptionKey`.
     * If you choose `CSE-KMS`, you *must* provide an un-managed KMS key with `encryptionKey`.
     *
     * @default Unencrypted
     */
    readonly encryption?: TableEncryption;
    /**
     * External KMS key to use for bucket encryption.
     *
     * The `encryption` property must be `SSE-KMS` or `CSE-KMS`.
     *
     * @default key is managed by KMS.
     */
    readonly encryptionKey?: kms.IEncryptionKey;
    /**
     * Indicates whether the table data is stored in subdirectories.
     *
     * @default false
     */
    readonly storedAsSubDirectories?: boolean;
}
/**
 * A Glue table.
 */
export declare class Table extends cdk.Construct implements ITable {
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param props A `TableImportProps` object. Can be obtained from a call to `table.export()` or manually created.
     */
    static import(scope: cdk.Construct, id: string, props: TableImportProps): ITable;
    /**
     * Database this table belongs to.
     */
    readonly database: IDatabase;
    /**
     * The type of encryption enabled for the table.
     */
    readonly encryption: TableEncryption;
    /**
     * The KMS key used to secure the data if `encryption` is set to `CSE-KMS` or `SSE-KMS`. Otherwise, `undefined`.
     */
    readonly encryptionKey?: kms.IEncryptionKey;
    /**
     * S3 bucket in which the table's data resides.
     */
    readonly bucket: s3.IBucket;
    /**
     * S3 Key Prefix under which this table's files are stored in S3.
     */
    readonly s3Prefix: string;
    /**
     * Name of this table.
     */
    readonly tableName: string;
    /**
     * ARN of this table.
     */
    readonly tableArn: string;
    /**
     * Format of this table's data files.
     */
    readonly dataFormat: DataFormat;
    /**
     * This table's columns.
     */
    readonly columns: Column[];
    /**
     * This table's partition keys if the table is partitioned.
     */
    readonly partitionKeys?: Column[];
    constructor(scope: cdk.Construct, id: string, props: TableProps);
    export(): TableImportProps;
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param identity the principal
     */
    grantRead(identity: iam.IPrincipal): void;
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param identity the principal
     */
    grantWrite(identity: iam.IPrincipal): void;
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param identity the principal
     */
    grantReadWrite(identity: iam.IPrincipal): void;
    private grant;
}
