"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/cdk");
const glue_generated_1 = require("./glue.generated");
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["Unencrypted"] = "Unencrypted";
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    TableEncryption["S3Managed"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    TableEncryption["Kms"] = "SSE-KMS";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    TableEncryption["KmsManaged"] = "SSE-KMS-MANAGED";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    TableEncryption["ClientSideKms"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * A Glue table.
 */
class Table extends cdk.Construct {
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param props A `TableImportProps` object. Can be obtained from a call to `table.export()` or manually created.
     */
    static import(scope, id, props) {
        return new ImportedTable(scope, id, props);
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = props.s3Prefix || 'data/';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new glue_generated_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: props.tableName,
                description: props.description || `${props.tableName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    has_encrypted_data: this.encryption !== TableEncryption.Unencrypted
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: props.compressed === undefined ? false : props.compressed,
                    storedAsSubDirectories: props.storedAsSubDirectories === undefined ? false : props.storedAsSubDirectories,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className
                    },
                },
                tableType: 'EXTERNAL_TABLE'
            }
        });
        this.tableName = tableResource.tableName;
        this.tableArn = `${this.database.databaseArn}/${this.tableName}`;
    }
    export() {
        return {
            tableName: new cdk.CfnOutput(this, 'TableName', { value: this.tableName }).makeImportValue().toString(),
            tableArn: new cdk.CfnOutput(this, 'TableArn', { value: this.tableArn }).makeImportValue().toString(),
        };
    }
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param identity the principal
     */
    grantRead(identity) {
        this.grant(identity, {
            permissions: readPermissions,
            kmsActions: ['kms:Decrypt']
        });
        this.bucket.grantRead(identity, this.s3Prefix);
    }
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param identity the principal
     */
    grantWrite(identity) {
        this.grant(identity, {
            permissions: writePermissions,
            kmsActions: ['kms:Encrypt', 'kms:GenerateDataKey']
        });
        this.bucket.grantWrite(identity, this.s3Prefix);
    }
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param identity the principal
     */
    grantReadWrite(identity) {
        this.grant(identity, {
            permissions: readPermissions.concat(writePermissions),
            kmsActions: ['kms:Decrypt', 'kms:Encrypt', 'kms:GenerateDataKey']
        });
        this.bucket.grantReadWrite(identity, this.s3Prefix);
    }
    grant(identity, props) {
        identity.addToPolicy(new iam.PolicyStatement()
            .addResource(this.tableArn)
            .addActions(...props.permissions));
        if (this.encryption === TableEncryption.ClientSideKms) {
            identity.addToPolicy(new iam.PolicyStatement()
                .addResource(this.encryptionKey.keyArn)
                .addActions(...props.kmsActions));
        }
    }
}
exports.Table = Table;
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but 'p1' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3Managed]: s3.BucketEncryption.S3Managed,
    [TableEncryption.KmsManaged]: s3.BucketEncryption.KmsManaged,
    [TableEncryption.Kms]: s3.BucketEncryption.Kms,
    [TableEncryption.ClientSideKms]: s3.BucketEncryption.Unencrypted,
    [TableEncryption.Unencrypted]: s3.BucketEncryption.Unencrypted,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.Unencrypted;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.Unencrypted && encryption !== TableEncryption.ClientSideKms)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.ClientSideKms && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.EncryptionKey(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.ClientSideKms) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey
    };
}
const readPermissions = [
    'glue:BatchDeletePartition',
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersions'
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition'
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment
        };
    });
}
class ImportedTable extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.tableArn = props.tableArn;
        this.tableName = props.tableName;
    }
    export() {
        return this.props;
    }
}
//# sourceMappingURL=data:application/json;base64,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