"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/cdk");
const glue = require("../lib");
function createTable(props) {
    const stack = new cdk.Stack();
    new glue.Table(stack, 'table', Object.assign({}, props, { database: new glue.Database(stack, 'db', {
            databaseName: 'database_name'
        }), dataFormat: glue.DataFormat.Json }));
}
module.exports = {
    'unpartitioned JSON table'(test) {
        const dbStack = new cdk.Stack();
        const database = new glue.Database(dbStack, 'Database', {
            databaseName: 'database'
        });
        const tableStack = new cdk.Stack();
        const table = new glue.Table(tableStack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.string
                }],
            dataFormat: glue.DataFormat.Json,
        });
        test.equals(table.encryption, glue.TableEncryption.Unencrypted);
        assert_1.expect(tableStack).to(assert_1.haveResource('AWS::S3::Bucket', {
            Type: "AWS::S3::Bucket",
            DeletionPolicy: "Retain"
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(tableStack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                "Fn::ImportValue": "Stack:ExportsOutputRefDatabaseB269D8BB88F4B1C4"
            },
            TableInput: {
                Name: "table",
                Description: "table generated by CDK",
                Parameters: {
                    has_encrypted_data: false
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: "col",
                            Type: "string"
                        }
                    ],
                    Compressed: false,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: {
                        "Fn::Join": [
                            "",
                            [
                                "s3://",
                                {
                                    Ref: "TableBucketDA42407C"
                                },
                                "/data/"
                            ]
                        ]
                    },
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    SerdeInfo: {
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    },
                    StoredAsSubDirectories: false
                },
                TableType: "EXTERNAL_TABLE"
            }
        }));
        test.done();
    },
    'partitioned JSON table'(test) {
        const dbStack = new cdk.Stack();
        const database = new glue.Database(dbStack, 'Database', {
            databaseName: 'database'
        });
        const tableStack = new cdk.Stack();
        const table = new glue.Table(tableStack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.string
                }],
            partitionKeys: [{
                    name: 'year',
                    type: glue.Schema.smallint
                }],
            dataFormat: glue.DataFormat.Json,
        });
        test.equals(table.encryption, glue.TableEncryption.Unencrypted);
        test.equals(table.encryptionKey, undefined);
        test.equals(table.bucket.encryptionKey, undefined);
        assert_1.expect(tableStack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                "Fn::ImportValue": "Stack:ExportsOutputRefDatabaseB269D8BB88F4B1C4"
            },
            TableInput: {
                Name: "table",
                Description: "table generated by CDK",
                Parameters: {
                    has_encrypted_data: false
                },
                PartitionKeys: [
                    {
                        Name: "year",
                        Type: "smallint"
                    }
                ],
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: "col",
                            Type: "string"
                        }
                    ],
                    Compressed: false,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: {
                        "Fn::Join": [
                            "",
                            [
                                "s3://",
                                {
                                    Ref: "TableBucketDA42407C"
                                },
                                "/data/"
                            ]
                        ]
                    },
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    SerdeInfo: {
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    },
                    StoredAsSubDirectories: false
                },
                TableType: "EXTERNAL_TABLE"
            }
        }));
        test.done();
    },
    'compressed table'(test) {
        const stack = new cdk.Stack();
        const database = new glue.Database(stack, 'Database', {
            databaseName: 'database'
        });
        const table = new glue.Table(stack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.string
                }],
            compressed: true,
            dataFormat: glue.DataFormat.Json,
        });
        test.equals(table.encryptionKey, undefined);
        test.equals(table.bucket.encryptionKey, undefined);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                Ref: "DatabaseB269D8BB"
            },
            TableInput: {
                Name: "table",
                Description: "table generated by CDK",
                Parameters: {
                    has_encrypted_data: false
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: "col",
                            Type: "string"
                        }
                    ],
                    Compressed: true,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: {
                        "Fn::Join": [
                            "",
                            [
                                "s3://",
                                {
                                    Ref: "TableBucketDA42407C"
                                },
                                "/data/"
                            ]
                        ]
                    },
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    SerdeInfo: {
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    },
                    StoredAsSubDirectories: false
                },
                TableType: "EXTERNAL_TABLE"
            }
        }));
        test.done();
    },
    'encrypted table': {
        'SSE-S3'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database'
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.string
                    }],
                encryption: glue.TableEncryption.S3Managed,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.S3Managed);
            test.equals(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Name: "table",
                    Description: "table generated by CDK",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                SSEAlgorithm: "AES256"
                            }
                        }
                    ]
                }
            }));
            test.done();
        },
        'SSE-KMS (implicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database'
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.string
                    }],
                encryption: glue.TableEncryption.Kms,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.Kms);
            test.equals(table.encryptionKey, table.bucket.encryptionKey);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion"
                            ],
                            Effect: "Allow",
                            Principal: {
                                AWS: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                },
                Description: "Created by Table/Bucket"
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                KMSMasterKeyID: {
                                    "Fn::GetAtt": [
                                        "TableBucketKey3E9F984A",
                                        "Arn"
                                    ]
                                },
                                SSEAlgorithm: "aws:kms"
                            }
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Name: "table",
                    Description: "table generated by CDK",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        },
        'SSE-KMS (explicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database'
            });
            const encryptionKey = new kms.EncryptionKey(stack, 'MyKey');
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.string
                    }],
                encryption: glue.TableEncryption.Kms,
                encryptionKey,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.Kms);
            test.equals(table.encryptionKey, table.bucket.encryptionKey);
            test.notEqual(table.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion"
                            ],
                            Effect: "Allow",
                            Principal: {
                                AWS: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                KMSMasterKeyID: {
                                    "Fn::GetAtt": [
                                        "MyKey6AB29FA6",
                                        "Arn"
                                    ]
                                },
                                SSEAlgorithm: "aws:kms"
                            }
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Description: "table generated by CDK",
                    Name: "table",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        },
        'SSE-KMS_MANAGED'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database'
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.string
                    }],
                encryption: glue.TableEncryption.KmsManaged,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.KmsManaged);
            test.equals(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                SSEAlgorithm: "aws:kms"
                            }
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Name: "table",
                    Description: "table generated by CDK",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        },
        'CSE-KMS (implicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database'
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.string
                    }],
                encryption: glue.TableEncryption.ClientSideKms,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.ClientSideKms);
            test.notEqual(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion"
                            ],
                            Effect: "Allow",
                            Principal: {
                                AWS: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Description: "table generated by CDK",
                    Name: "table",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        },
        'CSE-KMS (explicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database'
            });
            const encryptionKey = new kms.EncryptionKey(stack, 'MyKey');
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.string
                    }],
                encryption: glue.TableEncryption.ClientSideKms,
                encryptionKey,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.ClientSideKms);
            test.notEqual(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion"
                            ],
                            Effect: "Allow",
                            Principal: {
                                AWS: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Description: "table generated by CDK",
                    Name: "table",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        },
        'CSE-KMS (explicitly passed bucket and key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database'
            });
            const bucket = new s3.Bucket(stack, 'Bucket');
            const encryptionKey = new kms.EncryptionKey(stack, 'MyKey');
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.string
                    }],
                bucket,
                encryption: glue.TableEncryption.ClientSideKms,
                encryptionKey,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.ClientSideKms);
            test.notEqual(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion"
                            ],
                            Effect: "Allow",
                            Principal: {
                                AWS: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Description: "table generated by CDK",
                    Name: "table",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "Bucket83908E77"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        }
    },
    'explicit s3 bucket and prefix'(test) {
        const dbStack = new cdk.Stack();
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'ExplicitBucket');
        const database = new glue.Database(dbStack, 'Database', {
            databaseName: 'database'
        });
        new glue.Table(stack, 'Table', {
            database,
            bucket,
            s3Prefix: 'prefix/',
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.string
                }],
            dataFormat: glue.DataFormat.Json,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                "Fn::ImportValue": "Stack:ExportsOutputRefDatabaseB269D8BB88F4B1C4"
            },
            TableInput: {
                Description: "table generated by CDK",
                Name: "table",
                Parameters: {
                    has_encrypted_data: false
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: "col",
                            Type: "string"
                        }
                    ],
                    Compressed: false,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: {
                        "Fn::Join": [
                            "",
                            [
                                "s3://",
                                {
                                    Ref: "ExplicitBucket0AA51A3F"
                                },
                                "/prefix/"
                            ]
                        ]
                    },
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    SerdeInfo: {
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    },
                    StoredAsSubDirectories: false
                },
                TableType: "EXTERNAL_TABLE"
            }
        }));
        test.done();
    },
    'grants': {
        'read only'(test) {
            const stack = new cdk.Stack();
            const user = new iam.User(stack, 'User');
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database'
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.string
                    }],
                compressed: true,
                dataFormat: glue.DataFormat.Json,
            });
            table.grantRead(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "glue:BatchDeletePartition",
                                "glue:BatchGetPartition",
                                "glue:GetPartition",
                                "glue:GetPartitions",
                                "glue:GetTable",
                                "glue:GetTables",
                                "glue:GetTableVersions"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":glue:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":database/",
                                        {
                                            Ref: "DatabaseB269D8BB"
                                        },
                                        "/",
                                        {
                                            Ref: "Table4C2D914F"
                                        }
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "s3:GetObject*",
                                "s3:GetBucket*",
                                "s3:List*"
                            ],
                            Effect: "Allow",
                            Resource: [
                                {
                                    "Fn::GetAtt": [
                                        "TableBucketDA42407C",
                                        "Arn"
                                    ]
                                },
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            {
                                                "Fn::GetAtt": [
                                                    "TableBucketDA42407C",
                                                    "Arn"
                                                ]
                                            },
                                            "/data/"
                                        ]
                                    ]
                                }
                            ]
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "UserDefaultPolicy1F97781E",
                Users: [
                    {
                        Ref: "User00B015A1"
                    }
                ]
            }));
            test.done();
        },
        'write only'(test) {
            const stack = new cdk.Stack();
            const user = new iam.User(stack, 'User');
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database'
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.string
                    }],
                compressed: true,
                dataFormat: glue.DataFormat.Json,
            });
            table.grantWrite(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "glue:BatchCreatePartition",
                                "glue:BatchDeletePartition",
                                "glue:CreatePartition",
                                "glue:DeletePartition",
                                "glue:UpdatePartition"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":glue:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":database/",
                                        {
                                            Ref: "DatabaseB269D8BB"
                                        },
                                        "/",
                                        {
                                            Ref: "Table4C2D914F"
                                        }
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "s3:DeleteObject*",
                                "s3:PutObject*",
                                "s3:Abort*"
                            ],
                            Effect: "Allow",
                            Resource: [
                                {
                                    "Fn::GetAtt": [
                                        "TableBucketDA42407C",
                                        "Arn"
                                    ]
                                },
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            {
                                                "Fn::GetAtt": [
                                                    "TableBucketDA42407C",
                                                    "Arn"
                                                ]
                                            },
                                            "/data/"
                                        ]
                                    ]
                                }
                            ]
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "UserDefaultPolicy1F97781E",
                Users: [
                    {
                        Ref: "User00B015A1"
                    }
                ]
            }));
            test.done();
        },
        'read and write'(test) {
            const stack = new cdk.Stack();
            const user = new iam.User(stack, 'User');
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database'
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.string
                    }],
                compressed: true,
                dataFormat: glue.DataFormat.Json,
            });
            table.grantReadWrite(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "glue:BatchDeletePartition",
                                "glue:BatchGetPartition",
                                "glue:GetPartition",
                                "glue:GetPartitions",
                                "glue:GetTable",
                                "glue:GetTables",
                                "glue:GetTableVersions",
                                "glue:BatchCreatePartition",
                                "glue:BatchDeletePartition",
                                "glue:CreatePartition",
                                "glue:DeletePartition",
                                "glue:UpdatePartition"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":glue:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":database/",
                                        {
                                            Ref: "DatabaseB269D8BB"
                                        },
                                        "/",
                                        {
                                            Ref: "Table4C2D914F"
                                        }
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "s3:GetObject*",
                                "s3:GetBucket*",
                                "s3:List*",
                                "s3:DeleteObject*",
                                "s3:PutObject*",
                                "s3:Abort*"
                            ],
                            Effect: "Allow",
                            Resource: [
                                {
                                    "Fn::GetAtt": [
                                        "TableBucketDA42407C",
                                        "Arn"
                                    ]
                                },
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            {
                                                "Fn::GetAtt": [
                                                    "TableBucketDA42407C",
                                                    "Arn"
                                                ]
                                            },
                                            "/data/"
                                        ]
                                    ]
                                }
                            ]
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "UserDefaultPolicy1F97781E",
                Users: [
                    {
                        Ref: "User00B015A1"
                    }
                ]
            }));
            test.done();
        }
    },
    'validate': {
        'at least one column'(test) {
            test.throws(() => {
                createTable({
                    columns: [],
                    tableName: 'name',
                });
            }, undefined, 'you must specify at least one column for the table');
            test.done();
        },
        'unique column names'(test) {
            test.throws(() => {
                createTable({
                    tableName: 'name',
                    columns: [{
                            name: 'col1',
                            type: glue.Schema.string
                        }, {
                            name: 'col1',
                            type: glue.Schema.string
                        }]
                });
            }, undefined, "column names and partition keys must be unique, but 'col1' is duplicated.");
            test.done();
        },
        'unique partition keys'(test) {
            test.throws(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.string
                    }],
                partitionKeys: [{
                        name: 'p1',
                        type: glue.Schema.string
                    }, {
                        name: 'p1',
                        type: glue.Schema.string
                    }]
            }), undefined, "column names and partition keys must be unique, but 'p1' is duplicated");
            test.done();
        },
        'column names and partition keys are all unique'(test) {
            test.throws(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.string
                    }],
                partitionKeys: [{
                        name: 'col1',
                        type: glue.Schema.string
                    }]
            }), "column names and partition keys must be unique, but 'col1' is duplicated");
            test.done();
        },
        'can not specify an explicit bucket and encryption'(test) {
            test.throws(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.string
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: glue.TableEncryption.Kms
            }), undefined, 'you can not specify encryption settings if you also provide a bucket');
            test.done();
        },
        'can explicitly pass bucket if Encryption undefined'(test) {
            test.doesNotThrow(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.string
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: undefined
            }));
            test.done();
        },
        'can explicitly pass bucket if Unencrypted'(test) {
            test.doesNotThrow(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.string
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: undefined
            }));
            test.done();
        },
        'can explicitly pass bucket if ClientSideKms'(test) {
            test.doesNotThrow(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.string
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: glue.TableEncryption.ClientSideKms
            }));
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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