*Design & Architecture Overview for the **toolsandogh** scientific‑data‑processing package*

*Last updated: 2025‑09‑24*

---

## Project Overview

**toolsandogh** is a Python library (≥ 3.12) that provides a lightweight, extensible toolbox for processing large‑scale scientific data sets (e.g., microscopy volumes, multi‑modal imaging, high‑throughput experiments).

The library embraces **lazy, parallel computation** via Dask, **labeled, N‑dimensional arrays** via xarray, and **type‑safe, well‑documented** code.

---

## Repository Layout

```
toolsandogh/
│
├─ DESIGN.md                 # ← THIS FILE (design & conventions)
├─ README.md                 # User‑oriented introduction, installation, examples
├─ pyproject.toml            # Build system, metadata, dependencies, ruff config
├─ .pre-commit-config.yaml   # Pre‑commit hooks (ruff, isort, pyright, etc.)
├─ .gitignore                # Files/folders that Git should ignore
├─ uv.lock                   # Deterministic lockfile generated by uv
│
├─ notebooks/                # Jupyter notebooks showing typical workflows
│   ├─ 01-quickstart.ipynb
│   └─ ...
│
└─ src/
   └─ toolsandogh/           # The actual Python package
       ├─ __init__.py
       ├─ _generate_video.py # All file names start with an underscore
       ├─ _load_video.py
       ├─ _store_video.py
       ├─ ...                # One file per logical piece of functionality
       │
       ├─ scripts/           # The scripts directory contains executable files
       │   ├─ __init__.py
       │   ├─ iscat_analysis.py
       │   └─ ...
       │
       └─ tests/               # tests for all toolsandogh functionality
           ├─ __init__.py
           └─ test_load_video.py
```

---

## Coding Conventions

### 1. Documentation – **numpydoc**

All public objects (functions, classes, methods) must be documented using the **numpydoc** style. This format is machine‑parsable (e.g., for Sphinx) and familiar to scientific Python users.

#### Comprehensive Example

```python
def median_filter(
    data: xr.DataArray,
    size: int | tuple[int, ...] = 3,
    *,
    preserve_nan: bool = True,
) -> xr.DataArray:
    """
    Apply a median filter to an ``xarray.DataArray`` using a sliding window.

    Parameters
    ----------
    data : xarray.DataArray
        The input array.  The array can be lazily backed by Dask; the
        operation will therefore be executed in parallel when the result
        is computed.

    size : int or tuple of int, optional
        The size of the filtering window along each dimension.  If a
        single integer is supplied it is used for all dimensions.  The
        default is ``3`` (a 3×3×… window).

    preserve_nan : bool, optional
        If ``True`` (default), NaN values are ignored when computing the
        median, i.e. they act as “missing data”.  If ``False`` any window
        containing a NaN will produce a NaN in the output.

    Returns
    -------
    xarray.DataArray
        A new ``DataArray`` with the same coordinates and attributes as
        ``data`` but containing the filtered values.  The returned object
        retains the original chunking layout, so the computation remains
        lazy until ``compute()`` or ``persist()`` is called.

    Notes
    -----
    * The implementation uses :func:`dask.array.map_overlap` under the
      hood, which means that the filter works on arbitrarily large
      datasets that do not fit into memory.
    * For integer ``size`` values larger than the chunk size along a
      dimension, the function will rechunk automatically.  This may
      trigger a temporary increase in memory usage.

    References
    ----------
    .. [1] J. R. Smith, *Median filtering of large image stacks*, J. Imaging,
           2023, https://doi.org/10.1234/jim.2023.001

    Examples
    --------
    >>> import xarray as xr, numpy as np
    >>> arr = xr.DataArray(np.random.rand(100, 100), dims=('y', 'x'))
    >>> filtered = median_filter(arr, size=5)
    >>> filtered.compute()  # triggers Dask execution
    <xarray.DataArray (y: 100, x: 100)>
    """
    # implementation follows …
```

*Key points*

* **Section order** – Parameters → Returns → (optional) Notes → References → Examples.
* Use **reST** syntax for cross‑references (`:func:` , `:class:`).
* Types are expressed in the signature; the docstring repeats them for clarity.
* Example code must be runnable and end with a comment showing the expected output when possible.

### 2. Linting

Code must obey the following formatting rules (which are enforced by **ruff**):

- No pycodestyle errors and warnings, i.e., full complience with PEP-8

- No pyflakes errors

- Imports must be sorted as with **isort**

### 3. Coding Conventions

The following coding conventions apply to the toolsandogh code base:

- All functions, methods, and class attributes must have full type hints.  These type hints should be as specific as possible (e.g., `Literal["r", "w"]` instead of `str`)

- Publicly visible functions should liberal in what inputs they accept, and conservative in what they emit.

- Simplicity is king.  Functionality should be provided mainly via free functions.  Each function should have a clear purpose that can be summed up in a one-line docstring.

- All parallelization should be handled via xarrays and Dask.  Kernels should be written such that they support both Numpy and Cupy.

- Data should be encapsulated as either a `xarray.DataArray` or a `xarray.Dataset`.

- Error handling should be rigorous.  Exceptions should provide a clear explanation of what went wrong and why.
